/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Bank_PLD.h"

#include "Bank_ROM.h"			// ROMBank::IsPCInRAM
#include "Byteswapping.h"		// ByteswapWords
#include "CPU_REG.h"			// Software::BusError
#include "DebugMgr.h"			// Debug::CheckStepSpy
#include "Profiling.h"			// WAITSTATES_PLD
#include "RAM_ROM.h"			// gMemoryAccess
#include "SessionFile.h"		// WritePLDRegsType, ReadPLDRegsType, etc.


typedef uae_u32 	(*ReadFunction)(uaecptr address, int size);
typedef void		(*WriteFunction)(uaecptr address, int size, uae_u32 value);

// Macro to return the Dragonball address of the specified register

#define addressof(x)	(kMemoryStart + offsetof(HwrJerryPLDType, x))


// Macros for reading/writing Dragonball registers.

#define READ_REGISTER(reg)	\
	((sizeof(gPLDRegs.reg) == 1) ? do_get_mem_byte(&gPLDRegs.reg) :	\
	 (sizeof(gPLDRegs.reg) == 2) ? do_get_mem_word(&gPLDRegs.reg) :	\
								  do_get_mem_long(&gPLDRegs.reg))

#define WRITE_REGISTER(reg, value)	\
	(sizeof(gPLDRegs.reg) == 1) ? do_put_mem_byte(&gPLDRegs.reg, value) : \
	(sizeof(gPLDRegs.reg) == 2) ? do_put_mem_word(&gPLDRegs.reg, value) : \
								 do_put_mem_long(&gPLDRegs.reg, value)

class PLDReg
{
	public:
		static void 		SetHandler			(ReadFunction, WriteFunction, uae_u32 start, int count);

		static uae_u32		UnsupportedRead 	(uaecptr address, int size);
		static void 		UnsupportedWrite	(uaecptr address, int size, uae_u32 value);

		static uae_u32		StdRead 			(uaecptr address, int size);
		static void 		StdWrite			(uaecptr address, int size, uae_u32 value);
		static void 		NullWrite			(uaecptr address, int size, uae_u32 value); // For read-only registers
};



const HwrJerryPLDType kInitialPLDRegisterValues =
{
	0x00,		// UInt8	dspOn;			// pld output turn dsp on/off
	0x00,		// UInt8	unused1;
	0x00,		// UInt8	chargeOn;		// pld output turn charger on/off
	0x00,		// UInt8	unused2;
	0x00,		// UInt8	refOn;			// pld output turn 3.0 volt reference on/off
	0x00,		// UInt8	unused3;
	0x00,		// UInt8	pipaBiasEnable;	// pld output turn PA bias on/off
	0x00,		// UInt8	unused4;
	0x00,		// UInt8	dspReset;		// pld output control dsp reset pin up/down
	0x00,		// UInt8	unused5;
	0x00,		// UInt8	ezToDsl;		// pld output comm. pin to dsp
	0x00,		// UInt8	unused6;
	0x00,		// UInt8	rs232Shdn;		// pld output control rs232 interface chip on/off
	0x00,		// UInt8	unused7;
	0x00,		// UInt8	spareOut;		// pld output not used
	0x00,		// UInt8	unused8;
	0x0000,		// UInt16	dTo3Sln;		// pld input comm. from dsp
	0x0000		// UInt16	iXtrnl2;		// pld input cradle connector pin
};


static AddressBank	gAddressBank =
{
	PLDBank::GetLong,
	PLDBank::GetWord,
	PLDBank::GetByte,
	PLDBank::SetLong,
	PLDBank::SetWord,
	PLDBank::SetByte,
	PLDBank::GetRealAddress,
	PLDBank::ValidAddress,
	NULL,
	NULL
};

static const uae_u32	kMemoryStart = hwrJerryPldBase;

static HwrJerryPLDType	gPLDRegs;

// These functions provide fetch and store access to the
// emulator's register memory.

static ReadFunction 	gReadFunctions[sizeof(HwrJerryPLDType)];
static WriteFunction	gWriteFunctions[sizeof(HwrJerryPLDType)];


#pragma mark -


/***********************************************************************
 *
 * FUNCTION:	PLDBank::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void PLDBank::Initialize (void)
{
	// Memory banks starting at kMemoryStart are managed by the functions in PLDBank.

	long	numBanks = (sizeof (SED1375RegsType) + 0x0FFFF) / 0x10000;
	Memory::InitializeBanks (gAddressBank, bankindex (kMemoryStart), numBanks);

	// Install the handlers for each register.

	PLDBank::InstallHandlers ();
}


/***********************************************************************
 *
 * FUNCTION:	PLDBank::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void PLDBank::Reset (void)
{
	gPLDRegs = kInitialPLDRegisterValues;

	Canonical (gPLDRegs);
	ByteswapWords (&gPLDRegs, sizeof(gPLDRegs));
}


/***********************************************************************
 *
 * FUNCTION:	PLDBank::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void PLDBank::Save (SessionFile& f)
{
	StWordSwapper	swapper (&gPLDRegs, sizeof(gPLDRegs));
	f.WritePLDRegsType (gPLDRegs);
}


/***********************************************************************
 *
 * FUNCTION:	PLDBank::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void PLDBank::Load (SessionFile& f)
{
	// Read in the PLD registers, and then byteswap them.

	if (f.ReadPLDRegsType (gPLDRegs))
	{
		ByteswapWords (&gPLDRegs, sizeof(gPLDRegs));
	}
	else
	{
		f.SetCanReload (false);
	}
}


/***********************************************************************
 *
 * FUNCTION:	PLDBank::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void PLDBank::Dispose (void)
{
}


// ---------------------------------------------------------------------------
//		 PLDBank::InstallHandlers
// ---------------------------------------------------------------------------

void PLDBank::InstallHandlers (void)
{
	PLDReg::SetHandler (PLDReg::UnsupportedRead, PLDReg::UnsupportedWrite,
						kMemoryStart, sizeof (HwrJerryPLDType));

	// Now add standard/specialized handers for the defined registers.

#define INSTALL_HANDLER(read, write, reg) \
	PLDReg::SetHandler (PLDReg::read, PLDReg::write, \
							addressof (reg), sizeof (gPLDRegs.reg))


	INSTALL_HANDLER (StdRead,			StdWrite,				dspOn);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused1);
	INSTALL_HANDLER (StdRead,			StdWrite,				chargeOn);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused2);
	INSTALL_HANDLER (StdRead,			StdWrite,				refOn);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused3);
	INSTALL_HANDLER (StdRead,			StdWrite,				pipaBiasEnable);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused4);
	INSTALL_HANDLER (StdRead,			StdWrite,				dspReset);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused5);
	INSTALL_HANDLER (StdRead,			StdWrite,				ezToDsl);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused6);
	INSTALL_HANDLER (StdRead,			StdWrite,				rs232Shdn);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused7);
	INSTALL_HANDLER (StdRead,			StdWrite,				spareOut);
	INSTALL_HANDLER (StdRead,			NullWrite,				unused8);
	INSTALL_HANDLER (StdRead,			StdWrite,				dTo3Sln);
	INSTALL_HANDLER (StdRead,			StdWrite,				iXtrnl2);
}


// ---------------------------------------------------------------------------
//		 PLDBank::GetLong
// ---------------------------------------------------------------------------

uae_u32 PLDBank::GetLong (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kPLDLongRead]++;
	if (iAddress & 2)
		gMemoryAccess[kPLDLongRead2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 4, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 4))
	{
		InvalidAccess (iAddress, 4, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETLONG (WAITSTATES_PLD);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	ReadFunction	fn = PLDReg::UnsupportedRead;

	if (offset < sizeof (HwrJerryPLDType))
	{
		fn = gReadFunctions[offset];
	}

	assert (fn);
	return fn (iAddress, 4);
}


// ---------------------------------------------------------------------------
//		 PLDBank::GetWord
// ---------------------------------------------------------------------------

uae_u32 PLDBank::GetWord (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kPLDWordRead]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 2, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 2))
	{
		InvalidAccess (iAddress, 2, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETWORD (WAITSTATES_PLD);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	ReadFunction	fn = PLDReg::UnsupportedRead;

	if (offset < sizeof (HwrJerryPLDType))
	{
		fn = gReadFunctions[offset];
	}

	assert (fn);
	return fn (iAddress, 2);
}


// ---------------------------------------------------------------------------
//		 PLDBank::GetByte
// ---------------------------------------------------------------------------

uae_u32 PLDBank::GetByte (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kPLDByteRead]++;
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 1, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 1))
	{
		InvalidAccess (iAddress, 1, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETBYTE (WAITSTATES_PLD);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	ReadFunction	fn = PLDReg::UnsupportedRead;

	if (offset < sizeof (HwrJerryPLDType))
	{
		fn = gReadFunctions[offset];
	}

	assert (fn);
	return fn (iAddress, 1);
}


// ---------------------------------------------------------------------------
//		 PLDBank::SetLong
// ---------------------------------------------------------------------------

void PLDBank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kPLDLongWrite]++;
	if (iAddress & 2)
		gMemoryAccess[kPLDLongWrite2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 4, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 4))
	{
		InvalidAccess (iAddress, 4, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTLONG (WAITSTATES_PLD);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = PLDReg::UnsupportedWrite;

	if (offset < sizeof (HwrJerryPLDType))
	{
		fn = gWriteFunctions[offset];
	}

	assert (fn);
	fn (iAddress, 4, iLongValue);

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 4);
}


// ---------------------------------------------------------------------------
//		 PLDBank::SetWord
// ---------------------------------------------------------------------------

void PLDBank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kPLDWordWrite]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 2, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 2))
	{
		InvalidAccess (iAddress, 2, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTWORD (WAITSTATES_PLD);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = PLDReg::UnsupportedWrite;

	if (offset < sizeof (HwrJerryPLDType))
	{
		fn = gWriteFunctions[offset];
	}

	assert (fn);
	fn (iAddress, 2, iWordValue);

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 2);
}


// ---------------------------------------------------------------------------
//		 PLDBank::SetByte
// ---------------------------------------------------------------------------

void PLDBank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kPLDByteWrite]++;
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 1, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 1))
	{
		InvalidAccess (iAddress, 1, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTBYTE (WAITSTATES_SED1375);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = PLDReg::UnsupportedWrite;

	if (offset < sizeof (HwrJerryPLDType))
	{
		fn = gWriteFunctions[offset];
	}

	assert (fn);
	fn (iAddress, 1, iByteValue);

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 1);
}


// ---------------------------------------------------------------------------
//		 PLDBank::ValidAddress
// ---------------------------------------------------------------------------

int PLDBank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	UNUSED_PARAM(iSize)

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = PLDReg::UnsupportedWrite;

	if (offset < sizeof (HwrJerryPLDType))
	{
		fn = gWriteFunctions[offset];
	}

	int result = (fn != PLDReg::UnsupportedWrite);

	assert (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 PLDBank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* PLDBank::GetRealAddress (uaecptr iAddress)
{
	uae_u8*	loc = ((uae_u8*) &gPLDRegs) + (iAddress - kMemoryStart);

	return loc;
}


// ---------------------------------------------------------------------------
//		 PLDBank::IsSerialPortOn
// ---------------------------------------------------------------------------

Bool PLDBank::IsSerialPortOn (void)
{
	Bool	isOn;

	isOn = (READ_REGISTER (iXtrnl2) & hwrJerryPld232Enable) != 0;

	return isOn;
}

// ---------------------------------------------------------------------------
//		 PLDBank::InvalidAccess
// ---------------------------------------------------------------------------

void PLDBank::InvalidAccess (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Software::BusError ();
}


// ---------------------------------------------------------------------------
//		 PLDBank::PreventedAccess
// ---------------------------------------------------------------------------

void PLDBank::PreventedAccess (uaecptr iAddress, long size, Bool forRead)
{
	Memory::PreventedAccess (iAddress, size, forRead, Errors::kRegisterAccess);
}


#pragma mark -

// ===========================================================================
//		 Register Accessors
// ===========================================================================

// ---------------------------------------------------------------------------
//		 PLDReg::SetHandler
// ---------------------------------------------------------------------------

void PLDReg::SetHandler (ReadFunction read, WriteFunction write,
							uae_u32 start, int count)
{
	int index = start - kMemoryStart;
	for (int ii = 0; ii < count; ++ii, ++index)
	{
		gReadFunctions[index] = read;
		gWriteFunctions[index] = write;
	}
}


// ---------------------------------------------------------------------------
//		 PLDReg::UnsupportedRead
// ---------------------------------------------------------------------------

uae_u32 PLDReg::UnsupportedRead (uaecptr address, int size)
{
	if (!CEnableFullAccess::AccessOK ())
	{
		PLDBank::InvalidAccess (address, size, true);
	}

	return ~0;
}


// ---------------------------------------------------------------------------
//		 PLDReg::UnsupportedWrite
// ---------------------------------------------------------------------------

void PLDReg::UnsupportedWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(value)

	if (!CEnableFullAccess::AccessOK ())
	{
		PLDBank::InvalidAccess (address, size, false);
	}
}


// ---------------------------------------------------------------------------
//		 PLDReg::StdRead
// ---------------------------------------------------------------------------

uae_u32 PLDReg::StdRead (uaecptr address, int size)
{
	if (size == 1)
		return do_get_mem_byte (PLDBank::GetRealAddress (address));

	if (size == 2)
		return do_get_mem_word (PLDBank::GetRealAddress (address));

	return do_get_mem_long (PLDBank::GetRealAddress (address));
}


// ---------------------------------------------------------------------------
//		 PLDReg::StdWrite
// ---------------------------------------------------------------------------

void PLDReg::StdWrite (uaecptr address, int size, uae_u32 value)
{
	if (size == 1)
		do_put_mem_byte (PLDBank::GetRealAddress (address), value);

	else if (size == 2)
		do_put_mem_word (PLDBank::GetRealAddress (address), value);

	else
		do_put_mem_long (PLDBank::GetRealAddress (address), value);
}


// ---------------------------------------------------------------------------
//		 PLDReg::NullWrite
// ---------------------------------------------------------------------------

void PLDReg::NullWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)
	UNUSED_PARAM(value)

	// Do nothing (for read-only registers).
}
