/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _PALMHEAP_H_
#define _PALMHEAP_H_

class SessionFile;


class PalmHeap
{
	public:
		enum EWalkerProcResult
		{
			kKeepIterating,
			kStopIterating
		};

		enum EPalmHeapResult
		{
				// The following are returned by WalkHeaps and WalkChunks.

			kWalkHeapFinished,
			kWalkHeapAborted,

			kChunkOK,

				// Handy "first error" enumerated value.

			kWalkHeapError,

				// The following are returned by ValidateFoo functions
				// as well as WalkHeap (implicitly, as it calls the
				// ValidateFoo functions).

			kCorruptedHeap_ChunkNotInHeap,
			kCorruptedHeap_ChunkTooLarge,
			kCorruptedHeap_InvalidFlags,
			kCorruptedHeap_HOffsetNotInMPT,
			kCorruptedHeap_HOffsetNotBackPointing,
			kCorruptedHeap_InvalidLockCount,

			kNULLPointerOrHandle
		};

		enum
		{
			kHeapVersionUnknown,
			kHeapVersion1,	// 1 = original
			kHeapVersion2,	// 2 = >64K
			kHeapVersion3,	// 3 = has free list
			kHeapVersion4	// 4 = free MP's on a list
		};

		struct HeapInfo
		{
			int			version;		

			uaecptr		heapHdrStart;
			uae_u32		heapHdrSize;		// Up to the master pointer table struct
			uaecptr		firstChunk;			// Just after the end of the master pointer table

			Word		flags;
			DWord		size;
			DWord		firstFree;
			UInt		chunkHdrSize;
		};

		enum
		{
			kMPTVersionUnknown,
			kMPTVersion1,	// 1 = original
			kMPTVersion2	// 2 = >64K
		};

		struct MPTInfo
		{
			int			version;		

			uaecptr		mptHdrStart;
			uae_u32		mptHdrSize;			// Up to the array of master pointers

			Word		numEntries;
			DWord		nextTblOffset;
		};

		enum
		{
			kChunkVersionUnknown,
			kChunkVersion1,	// 1 = original
			kChunkVersion2	// 2 = >64K
		};

		struct ChunkInfo
		{
			int			version;		

			uaecptr		chunkHdrStart;
			uae_u32		chunkHdrSize;

			Bool		free;				// set if free chunk
			Bool		moved;				// used by MemHeapScramble
			Bool		unused2;			// unused
			Bool		unused3;			// unused
			char		sizeAdj;			// size adjustment
			DWord		size;				// actual size of chunk

			char		lockCount;			// lock count
			char		owner;				// owner ID
			SDWord		hOffset;			// signed handle offset/2
											// used in free chunks to point to next free chunk
		};


	// Standard sub-system functions.

		static void				Initialize			(void);
		static void				Reset				(void);
		static void				Save				(SessionFile&);
		static void				Load				(SessionFile&);
		static void				Dispose				(void);


	// Client API -- getting heap info.

	// Get information on a heap given a pointer to its start.

		static void				GetHeapInfo			(uaecptr			heapPtr,
													 HeapInfo*			heapInfo);

	// Get information on a master pointer table, given its base
	// address and information on the heap it's in.

		static void				GetMPTInfo			(uaecptr			mptStart,
													 const HeapInfo&	heapInfo,
													 MPTInfo*			mptInfo);

	// Get information on a chunk, given its base address and 
	// information on the heap it's in.

		static void				GetChunkInfo		(uaecptr			hdrP,
													 const HeapInfo&	heapInfo,
													 ChunkInfo*			chunkInfo);

	// Get the starting address of the heap with the given ID,
	// or that contains the given address.

		static uaecptr			GetHeapHeader		(UInt				heapID);
		static uaecptr			FindHeapHeader		(uaecptr			addr);

	// Get the base address of a chunk given a Palm OS pointer or handle.
	// The third variation searches for the chunk containing the given address.

		static uaecptr			GetChunkHeader		(VoidPtr	p, const HeapInfo& heapInfo);
		static uaecptr			GetChunkHeader		(VoidHand	h, const HeapInfo& heapInfo);
		static uaecptr			FindChunkHeader		(uaecptr	p, const HeapInfo& heapInfo);

	// Client API -- validating heap chunks.

		static PalmHeap::EPalmHeapResult
								ValidateChunk		(const HeapInfo&	heapInfo,
													 const ChunkInfo&	chunkInfo,
													 Bool				validateHOffset);
		static PalmHeap::EPalmHeapResult
								ValidatePointer		(VoidPtr			p,
													 const HeapInfo&	heapInfo);
		static PalmHeap::EPalmHeapResult
								ValidateHandle		(VoidHand			h,
													 const HeapInfo&	heapInfo);

		static void				ValidateHeaps		(void);


	// Client API -- walking the heap.

		typedef EWalkerProcResult(*HeapWalker)		(const HeapInfo&	heapInfo,
													 void*				userData);
		typedef EWalkerProcResult(*ChunkWalker)		(const HeapInfo&	heapInfo,
													 const ChunkInfo&	chunkInfo,
													 void*				userData);

		static EPalmHeapResult	WalkHeaps			(HeapWalker			fn,
													 void*				userData);

		static EPalmHeapResult	WalkChunks			(const HeapInfo&	heapInfo,
													 ChunkWalker		fn,
													 void*				userData);


	// Internal utilities.

		static UInt				GetNumHeaps			(UInt		cardNo);
		static int				GetHeapVersion		(uaecptr	heapPtr);
		static VoidPtr			DerefHandle			(VoidHand	h);

		struct FindHeapData
		{
			uaecptr	testAddress;
			uaecptr	heapHeader;
		};

		struct FindChunkData
		{
			uaecptr	testAddress;
			uaecptr	chunkHeader;
		};

		static EWalkerProcResult	FindHeapCallback	(const HeapInfo& heapInfo,
														 void* userData);
		static EWalkerProcResult	FindChunkCallback	(const HeapInfo& heapInfo,
														 const ChunkInfo& chunkInfo,
														 void* userData);
		static EWalkerProcResult	ValidateHeapCallback(const HeapInfo& heapInfo,
														 void* userData);
		static EWalkerProcResult	ValidateChunkCallback(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);
};

#endif /* _PALMHEAP_H_ */
