/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "SLP.h"

#include "CPU_MT.h" 			// CPUStopper
#include "Crc.h"				// Crc16CalcBlock
#include "DebugMgr.h"			// Debug::HandleNewPacket
#include "Logging.h"			// LogAppendMsg
#include "EmRPC.h"				// RPC::HandleNewPacket, slkSocketRPC
#include "SocketMessaging.h"	// CSocket::Write
#include "StringData.h" 		// kExceptionNames, kPacketNames

#define PacketName(command) (kPacketNames[command])

#define PRINTF	if (!this->LogFlow ()) ; else LogAppendMsg

static void PrvPrintHeader	(const SlkPktHeaderType& header);
static void PrvPrintBody	(const SysPktBodyType& body);
static void PrvPrintFooter	(const SlkPktFooterType& footer);


/***********************************************************************
 *
 * FUNCTION:	SLP::SLP
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

SLP::SLP (void) :
	fSocket (NULL),
	fHavePacket(false),
	fSendReply (true)
{
}


/***********************************************************************
 *
 * FUNCTION:	SLP::SLP
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

SLP::SLP (CSocket* s) :
	fSocket (s),
	fHavePacket (false),
	fSendReply (true)
{
}


/***********************************************************************
 *
 * FUNCTION:	SLP::SLP
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

SLP::SLP (const SLP& other) :
	fSocket (other.fSocket),
	fHavePacket (other.fHavePacket),
	fSendReply (other.fSendReply)
{
	assert (fSocket);

	memcpy (fPacket, other.fPacket, kMaxSLPPacketSize);
}


/***********************************************************************
 *
 * FUNCTION:	SLP::~SLP
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

SLP::~SLP (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	SLP::EventCallback
 *
 * DESCRIPTION: Standard callback for handling SLP packets.  If data
 *				is received, we stop the emulator thread and call
 *				HandleDataReceived to read the dispatch the packet.
 *				Nothing is done by default on connect and disconnect
 *				events.
 *
 * PARAMETERS:	s - the socket that connected, disconnected, or received
 *					some data.
 *
 *				event - a code indicating what happened.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SLP::EventCallback (CSocket* s, int event)
{
	switch (event)
	{
		case CSocket::kConnected:
		{
			break;
		}

		case CSocket::kDataReceived:
		{
			ErrCode	result;
			do {
				SLP slp (s);
				result = slp.HandleDataReceived ();
			} while (result == errNone && s->HasUnreadData (500));
			break;
		}

		case CSocket::kDisconnected:
		{
//			if (s == fSocket)
//			{
//				fSocket = NULL;
//			}
			break;
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	SLP::HandleDataReceived
 *
 * DESCRIPTION: Called when received data is pending in the socket.
 *				Read that data, break it down into header, body, and
 *				footer sections, and call HandleNewPacket to dispatch
 *				it to the right sub-system.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	An error indicating what bad things happened.
 *
 ***********************************************************************/

ErrCode SLP::HandleDataReceived (void)
{
	ErrCode result = errNone;

//	do {
		assert (fSocket);

		SlkPktHeaderType	header;
		long				amtRead;
		ErrCode 			err = fSocket->Read (&header, sizeof (header), &amtRead);

		if (err == errNone && amtRead == sizeof (header))
		{
			Canonical (header);

			long	packetSize = sizeof (SlkPktHeaderType) +
								header.bodySize +
								sizeof (SlkPktFooterType);
			fHavePacket = true;

			this->Header () = header;

			fSocket->Read (&this->Body (), header.bodySize, NULL);
			Canonical (this->Body ());

			if (!fSocket->ShortPacketHack ())
			{
				fSocket->Read (&this->Footer (), sizeof (SlkPktFooterType), NULL);
				Canonical (this->Footer ());
			}

			result = this->HandleNewPacket ();
		}
		else
		{
//			fSocket = NULL; // Disconnected?
			if (err == errNone)
			{
				err = 1;
			}
		}
//	} while (result == errNone && fSocket && fSocket->HasUnreadData (500));

	return result;
}

				
/***********************************************************************
 *
 * FUNCTION:	SLP::HandleNewPacket
 *
 * DESCRIPTION: Dispatch the packet information to the right sub-system.
 *
 * PARAMETERS:	header - contents of header part of message.
 *
 *				body - contents of body part of message.
 *
 *				footer - contents of footer part of message.  May
 *					not be valid with all sockets.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SLP::HandleNewPacket ()
{
	ErrCode result = kError_NoError;

	// Say that we're here.  Do this after setting gLastestHeader, as the PRINTF
	// macro looks at it.

	PRINTF ("Entering SLP::HandleNewPacket.");

	// Print the logging information if requested.

	if (this->LogData ())
	{
#if !defined (__MACOS__)
		// PrvPrintBody calls GetTrapName -> FindTrapName -> GetFunctionAddress ->
		// GetSysFunctionAddress -> LowMem::GetTrapAddress, which creates
		// a CEnableFullAccess object.  In order to create one of those outside
		// the CPU thread, we must stop the CPU thread first.
		CPUStopper	stopper (kStopOnADime);
		if (stopper.Stopped ())
#endif
		{
			PrvPrintHeader (this->Header ());
			PrvPrintBody (this->Body ());
			PrvPrintFooter (this->Footer ());
		}
	}
	else if (this->LogFlow ())
	{
		if (PacketName (this->Body().command))
			LogAppendMsg (" Received %s packet.", PacketName (this->Body().command));
		else
			LogAppendMsg (" Received unknown (0x%02X) packet.", this->Body().command);
	}

	// Dispatch the packet to the right sub-system.

	switch (this->Header().dest)
	{
		case slkSocketDebugger:
		{
#if !defined (__MACOS__)
			CPUStopper	stopper (kStopOnADime);
			if (stopper.Stopped ())
#endif
			{
				result = Debug::HandleNewPacket (*this);
			}
		}
		break;

		case slkSocketConsole:
		{
#if !defined (__MACOS__)
			CPUStopper	stopper (kStopOnATrap);
			if (stopper.Stopped ())
#endif
			{
				result = Debug::HandleNewPacket (*this);
			}
		}
		break;

		case slkSocketRPC:
		{
#if !defined (__MACOS__)
			CPUStopper	stopper (kStopOnATrap);
			if (stopper.Stopped ())
#endif
			{
				result = RPC::HandleNewPacket (*this);
			}
		}
		break;

		default:
			result = slkErrWrongDestSocket;
			PRINTF ("Unknown destination: %ld.", (long) this->Header ().dest);
			break;
	}

	PRINTF ("Exiting SLP::HandleNewPacket.");

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SLP::SendPacket
 *
 * DESCRIPTION: Sends the given packet to the external debugger.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SLP::SendPacket (const void* body, long bodySize)
{
	PRINTF ("Entering SLP::SendPacket.");

	if (!fSendReply)
	{
		PRINTF ("Not sending a reply because we were asked not to.");
		PRINTF ("Exiting SLP::SendPacket.");
		return errNone;
	}

	// Fold all the parts into a buffer: header, body, footer.

	long		totalSize = sizeof (SlkPktHeaderType) + bodySize + sizeof (SlkPktFooterType);
#if 0
	// Normally, I'd do it this way, but the Mac version has this
	// problem with not being able to allocate memory while handling
	// an asynchronous packet...

	StMemory	buffer (totalSize);
	char*		bufferP = (char*) buffer.Get ();
#else
	assert (totalSize < 400);
	char			buffer[400];
	char*			bufferP = &buffer[0];
#endif

	SlkPktHeaderPtr headerP = (SlkPktHeaderPtr) &bufferP[0];
	SysPktBodyPtr	bodyP	= (SysPktBodyPtr)	&bufferP[sizeof (SlkPktHeaderType)];
	SlkPktFooterPtr footerP = (SlkPktFooterPtr) &bufferP[sizeof (SlkPktHeaderType) + bodySize];


	// Gen up a header.  If we're replying to a packet from an external
	// source, use the information from it to form a reply (socket
	// numbers and transaction ID).  Otherwise, stuff in some of our
	// own values.

	if (this->HavePacket())
	{
		headerP->signature1 = slkPktHeaderSignature1;
		headerP->signature2 = slkPktHeaderSignature2;
		headerP->dest		= this->Header().src;
		headerP->src		= this->Header().dest;
		headerP->type		= slkPktTypeSystem;
		headerP->bodySize	= bodySize;
		headerP->transId	= this->Header().transId;
		headerP->checksum	= 0;
	}
	else
	{
		headerP->signature1 = slkPktHeaderSignature1;
		headerP->signature2 = slkPktHeaderSignature2;
		headerP->dest		= slkSocketDebugger;	// !!! May want to parameterize these two.
		headerP->src		= slkSocketDebugger;	// !!! May want to parameterize these two.
		headerP->type		= slkPktTypeSystem;
		headerP->bodySize	= bodySize;
		headerP->transId	= 0;
		headerP->checksum	= 0;
	}

	// Compute and stuff the header checksum

	Canonical (*headerP);	// Canonical format
	headerP->checksum =  SLP::CalcHdrChecksum (0, (BytePtr) headerP,
					offsetof (SlkPktHeaderType, checksum));
	Canonical (headerP->checksum);	// Canonicial format
	Canonical (*headerP);	// Host format

	if (this->LogData ())
		PrvPrintHeader (*headerP);

	Canonical (*headerP);	// Canonical format


	// Copy in the packet body and byteswap it.

	memcpy (bodyP, body, bodySize);

	if (this->LogData ())
	{
		PrvPrintBody (*bodyP);
	}
	else if (this->LogFlow ())
	{
		if (PacketName (bodyP->command))
			LogAppendMsg (" Sending %s packet.", PacketName (bodyP->command));
		else
			LogAppendMsg (" Sending unknown (0x%02X) packet.", bodyP->command);
	}

	Canonical (*bodyP);	// Canonical format


	// Re-introduce byteswapping bugs if necessary for clients
	// that expect them.

	assert (fSocket);
	if (fSocket->ByteswapHack ())
	{
		if (bodyP->command == sysPktRPCRsp)
		{
			Canonical (((SysPktRPCType*) bodyP)->resultD0);
			Canonical (((SysPktRPCType*) bodyP)->resultA0);
		}
		else if (bodyP->command == sysPktReadRegsRsp)
		{
			Canonical ((SysPktReadRegsRspType&) *bodyP);
		}
	}


	// Calculate the footer checksum

	footerP->crc16 = ::Crc16CalcBlock (headerP, sizeof (SlkPktHeaderType), 0);
	footerP->crc16 = ::Crc16CalcBlock (bodyP, bodySize, footerP->crc16);
	if (this->LogData ())
		PrvPrintFooter (*footerP);
	Canonical (*footerP);


	// Send it to the external client.

	if (fSocket->ShortPacketHack ())
	{
		totalSize -= sizeof (SlkPktFooterType);
	}

	long	amtWritten;
	ErrCode result = fSocket->Write (bufferP, totalSize, &amtWritten);

	PRINTF ("Exiting SLP::SendPacket.");

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SLP::HavePacket
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

Bool SLP::HavePacket (void) const
{
//	return fPacket != NULL;
	return fHavePacket;
}


/***********************************************************************
 *
 * FUNCTION:	SLP::GetPacketSize
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

long SLP::GetPacketSize (void) const
{
	assert (this->HavePacket());

	return sizeof (SlkPktHeaderType) +
			this->Header().bodySize +
			sizeof (SlkPktFooterType);
}


/***********************************************************************
 *
 * FUNCTION:	SLP::Header
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

SlkPktHeaderType& SLP::Header (void) const
{
	assert (this->HavePacket());
	return *(SlkPktHeaderType*) fPacket;
}


/***********************************************************************
 *
 * FUNCTION:	SLP::Body
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

SysPktBodyType& SLP::Body (void) const
{
	assert (this->HavePacket());
	return *(SysPktBodyType*) (&this->Header() + 1);
}


/***********************************************************************
 *
 * FUNCTION:	SLP:: Footer
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

SlkPktFooterType& SLP:: Footer (void) const
{
	assert (this->HavePacket());
	return *(SlkPktFooterType*) (((char*) &this->Body ()) + this->Header ().bodySize);
}



/***********************************************************************
 *
 * FUNCTION:    SLP::DeferReply
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void SLP::DeferReply (Bool v)
{
	fSendReply = !v;
}


/***********************************************************************
 *
 * FUNCTION:	SLP::CalcHdrChecksum
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

SlkPktHeaderChecksum SLP::CalcHdrChecksum (SlkPktHeaderChecksum start,
										   BytePtr bufP, Long count)
{
	do {
		start += *bufP++;
	} while (--count);

	return start;
}


/***********************************************************************
 *
 * FUNCTION:	SLP::LogFlow
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

Bool SLP::LogFlow (void)
{
	Byte	dest = slkSocketDebugger;
	if (this->HavePacket())
		dest = this->Header().dest;

	switch (dest)
	{
		case slkSocketDebugger:
		case slkSocketConsole:
			return LogHLDebugger ();

		case slkSocketRPC:
			return LogRPC ();

		default:
			break;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	SLP::LogData
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

Bool SLP::LogData (void)
{
	Byte	dest = slkSocketDebugger;
	if (this->HavePacket())
		dest = this->Header().dest;

	switch (dest)
	{
		case slkSocketDebugger:
		case slkSocketConsole:
			return LogHLDebuggerData ();

		case slkSocketRPC:
			return LogRPCData ();

		default:
			break;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	PrvPrintHeader
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void PrvPrintHeader (const SlkPktHeaderType& header)
{
	LogAppendMsg (" header.signature1   = 0x%04X",	header.signature1);
	LogAppendMsg (" header.signature2   = 0x%02X",	header.signature2);
	LogAppendMsg (" header.dest         = 0x%02X",	header.dest);
	LogAppendMsg (" header.src          = 0x%02X",	header.src);
	LogAppendMsg (" header.type         = 0x%02X",	header.type);
	LogAppendMsg (" header.bodySize     = 0x%04X",	header.bodySize);
	LogAppendMsg (" header.transId      = 0x%02X",	header.transId);
	LogAppendMsg (" header.checksum     = 0x%02X",	header.checksum);
}


/***********************************************************************
 *
 * FUNCTION:	PrvPrintBody
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void PrvPrintBody (const SysPktBodyType& body)
{
	static Word gLastNumReadBytes;

	if (PacketName (body.command))
	{
		LogAppendMsg (" body.command        = 0x%02X / %s.",
			body.command,
			PacketName (body.command));
	}
	else
	{
		LogAppendMsg (" body.command        = 0x%02X / <unknown command>.",
			body.command);
	}

	switch (body.command)
	{
		case sysPktStateCmd:
			break;
		case sysPktStateRsp:
		{
			SysPktStateRspType& body2 = (SysPktStateRspType&) body;
			int what = body2.exceptionId / 4;

			LogAppendMsg (" body.resetted       = %s",		body2.resetted ? "true" : "false");
			LogAppendMsg (" body.exceptionId    = 0x%02X / %s", what, kExceptionNames [what]);

			LogAppendMsg (" body.reg.d[0]       = 0x%08X",	body2.reg.d[0]);
			LogAppendMsg (" body.reg.d[1]       = 0x%08X",	body2.reg.d[1]);
			LogAppendMsg (" body.reg.d[2]       = 0x%08X",	body2.reg.d[2]);
			LogAppendMsg (" body.reg.d[3]       = 0x%08X",	body2.reg.d[3]);
			LogAppendMsg (" body.reg.d[4]       = 0x%08X",	body2.reg.d[4]);
			LogAppendMsg (" body.reg.d[5]       = 0x%08X",	body2.reg.d[5]);
			LogAppendMsg (" body.reg.d[6]       = 0x%08X",	body2.reg.d[6]);
			LogAppendMsg (" body.reg.d[7]       = 0x%08X",	body2.reg.d[7]);

			LogAppendMsg (" body.reg.a[0]       = 0x%08X",	body2.reg.a[0]);
			LogAppendMsg (" body.reg.a[1]       = 0x%08X",	body2.reg.a[1]);
			LogAppendMsg (" body.reg.a[2]       = 0x%08X",	body2.reg.a[2]);
			LogAppendMsg (" body.reg.a[3]       = 0x%08X",	body2.reg.a[3]);
			LogAppendMsg (" body.reg.a[4]       = 0x%08X",	body2.reg.a[4]);
			LogAppendMsg (" body.reg.a[5]       = 0x%08X",	body2.reg.a[5]);
			LogAppendMsg (" body.reg.a[6]       = 0x%08X",	body2.reg.a[6]);

			LogAppendMsg (" body.reg.usp        = 0x%08X",	body2.reg.usp);
			LogAppendMsg (" body.reg.ssp        = 0x%08X",	body2.reg.ssp);
			LogAppendMsg (" body.reg.pc         = 0x%08X",	body2.reg.pc);
			LogAppendMsg (" body.reg.sr         = 0x%04X",	body2.reg.sr);

			LogAppendMsg (" body.bp[0]          = 0x%08X",	body2.bp[0]);
			LogAppendMsg (" body.bp[1]          = 0x%08X",	body2.bp[1]);
			LogAppendMsg (" body.bp[2]          = 0x%08X",	body2.bp[2]);
			LogAppendMsg (" body.bp[3]          = 0x%08X",	body2.bp[3]);
			LogAppendMsg (" body.bp[4]          = 0x%08X",	body2.bp[4]);
			LogAppendMsg (" body.bp[5]          = 0x%08X",	body2.bp[5]);

			LogAppendMsg (" body.startAddr      = 0x%08X",	body2.startAddr);
			LogAppendMsg (" body.endAddr        = 0x%08X",	body2.endAddr);
			LogAppendMsg (" body.name           = %s",		body2.name);
			LogAppendMsg (" body.trapTableRev   = %d",		body2.trapTableRev);
			break;
		}

		case sysPktReadMemCmd:
		{
			SysPktReadMemCmdType&	body2 = (SysPktReadMemCmdType&) body;
			LogAppendMsg (" body.address        = 0x%08X",	body2.address);
			LogAppendMsg (" body.numBytes       = 0x%04X",	body2.numBytes);

			gLastNumReadBytes = body2.numBytes;
			break;
		}
		case sysPktReadMemRsp:
		{
			SysPktReadMemRspType&	body2 = (SysPktReadMemRspType&) body;
			LogAppendData (&body2 + 1, gLastNumReadBytes, " body.data           = ");
			break;
		}

		case sysPktWriteMemCmd:
		{
			SysPktWriteMemCmdType&	body2 = (SysPktWriteMemCmdType&) body;
			LogAppendMsg (" body.address        = 0x%08X",	body2.address);
			LogAppendMsg (" body.numBytes       = 0x%04X",	body2.numBytes);
			LogAppendData (&body2 + 1, body2.numBytes, "    body.data           = ");
			break;
		}
		case sysPktWriteMemRsp:
			break;

		case sysPktSingleStepCmd:
			break;
//		case sysPktSingleStepRsp:
//			break;

		case sysPktGetRtnNameCmd:
			break;
		case sysPktGetRtnNameRsp:
		{
			SysPktRtnNameRspType&	body2 = (SysPktRtnNameRspType&) body;
			LogAppendMsg (" body.address        = 0x%08X",	body2.address);
			LogAppendMsg (" body.startAddr      = 0x%08X",	body2.startAddr);
			LogAppendMsg (" body.endAddr        = 0x%08X",	body2.endAddr);
			LogAppendMsg (" body.name           = %s",		body2.name);
			break;
		}

		case sysPktReadRegsCmd:
			break;
		case sysPktReadRegsRsp:
			break;

		case sysPktWriteRegsCmd:
			break;
		case sysPktWriteRegsRsp:
			break;

		case sysPktContinueCmd:
			break;
//		case sysPktContinueRsp:
//			break;

		case sysPktRPCCmd:
		case sysPktRPCRsp:
		{
			SysPktRPCType&	body2 = (SysPktRPCType&) body;
			LogAppendMsg (" body.trapWord       = 0x%04X / %s.", body2.trapWord,
																	::GetTrapName (body2.trapWord));
			if (body.command == sysPktRPCRsp)
			{
				LogAppendMsg (" body.resultD0       = 0x%08X.",  body2.resultD0);
				LogAppendMsg (" body.resultA0       = 0x%08X.",  body2.resultA0);
			}

			LogAppendMsg (" body.numParams      = 0x%04X.", body2.numParams);

			SysPktRPCParamInfo* param = body2.param;

			for (Word ii = 0; ii < body2.numParams; ++ii)
			{
				if (param->byRef)
				{
					LogAppendData (&param->data, param->size,
						"   body.param%d            = %d bytes of data.", ii, param->size);
				}
				else if (body.command == sysPktRPCCmd)
				{
					if (param->size == 1)
					{
						uae_u8	value = *(uae_u8*) param->data;
						Canonical (value);
						LogAppendMsg (" body.param%d            = 0x%02X.", ii, value);
					}
					else if (param->size == 2)
					{
						uae_u16 value = *(uae_u16*) param->data;
						Canonical (value);
						LogAppendMsg (" body.param%d            = 0x%04X.", ii, value);
					}
					else if (param->size == 4)
					{
						uae_u32 value = *(uae_u32*) param->data;
						Canonical (value);
						LogAppendMsg (" body.param%d            = 0x%08X.", ii, value);
					}
				}

				param = (SysPktRPCParamInfo*) (((char*) param) + offsetof (SysPktRPCParamInfo, data) +
												((param->size + 1) & ~1));
			}
			break;
		}

		case sysPktGetBreakpointsCmd:
			break;
		case sysPktGetBreakpointsRsp:
			break;

//		case sysPktSetBreakpointsCmd:
//			break;
		case sysPktSetBreakpointsRsp:
			break;

		case sysPktRemoteUIUpdCmd:
			break;
//		case sysPktRemoteUIUpdRsp:
//			break;

		case sysPktRemoteEvtCmd:
			break;
//		case sysPktRemoteEvtRsp:
//			break;

//		case sysPktDbgBreakToggleCmd:
//			break;
		case sysPktDbgBreakToggleRsp:
			break;

		case sysPktFlashCmd:
			break;
		case sysPktFlashRsp:
			break;

		case sysPktCommCmd:
			break;
		case sysPktCommRsp:
			break;

		case sysPktGetTrapBreaksCmd:
			break;
		case sysPktGetTrapBreaksRsp:
			break;

		case sysPktSetTrapBreaksCmd:
			break;
		case sysPktSetTrapBreaksRsp:
			break;

		case sysPktGremlinsCmd:
			break;
//		case sysPktGremlinsRsp:
//			break;

		case sysPktFindCmd:
			break;
		case sysPktFindRsp:
			break;

		case sysPktGetTrapConditionsCmd:
			break;
		case sysPktGetTrapConditionsRsp:
			break;

		case sysPktSetTrapConditionsCmd:
			break;
		case sysPktSetTrapConditionsRsp:
			break;

		case sysPktChecksumCmd:
			break;
		case sysPktChecksumRsp:
			break;

		case sysPktExecFlashCmd:
			break;
		case sysPktExecFlashRsp:
			break;

		case sysPktRemoteMsgCmd:
			break;
//		case sysPktRemoteMsgRsp:
//			break;

		default:
			break;
	}
}


/***********************************************************************
 *
 * FUNCTION:	PrvPrintFooter
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void PrvPrintFooter (const SlkPktFooterType& footer)
{
	LogAppendMsg (" footer.crc16        = 0x%02X",	  footer.crc16);
}


