/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"

#include "CPU_MT.h"				// CPU
#include "CPU_REG.h"			// Emulator::, others
#include "DebugMgr.h"			// Debug::HandleNewPacket
#include "Document.h"
#include "RAM_ROM.h"			// Memory::
#include "Hordes.h"				// Hordes::New...
#include "PreferenceMgr.h"		// Preference (kPrefKeyGremlinInfo)
#include "SocketMessaging.h"
#include "TrapPatches.h"		// Patches::fgUIInitialized

#include <limits.h>

#define down fl_down			// "down" defined both in WindowNew.h and Fl/Enumerations.h
#include <FL/Fl.H>

#include "fltk_Dialog.h"		// showNewGremlinWindow


// -----------------------------------------------------------------------------
// constructor(s) / destructor
// -----------------------------------------------------------------------------

Document::Document (const Configuration& cfg) :
	_cpu (NULL),
	_currentConfiguration (cfg)
{
	_cpu = new CPU;
	_cpu->CreateThread ();

	// Try initializing the emulator with all this information.
	ErrCode	error = errNone;

	try
	{
		Emulator::Initialize (_currentConfiguration);
	}
	catch (ErrCode e)
	{
		error = e;

		_cpu->DestroyThread ();
		delete _cpu;
		_cpu = NULL;
	}
}

Document::~Document (void)
{
	propertyChanged (documentClosing);

	if (_cpu != NULL)
	{
		Platform::GCW_Close ();

		_cpu->DestroyThread ();
		delete _cpu;

		Emulator::Dispose ();
	}
}


// -----------------------------------------------------------------------------
// public methods
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// read the document state from the given file, then get the CPU going with it.

ErrCode Document::open (FileReference& ramFileRef)
{
	ErrCode	result = errNone;

	try
	{
		Emulator::Load (ramFileRef, true, true, true);
		Hordes::PostLoad ();
	}
	catch (ErrCode e)
	{
		result = e;
	}

	if (result == errNone)
	{
		// If we loaded a session file with a Gremlin running, turn
		// it off, but let the user turn it back on.

		if (Hordes::IsOn ())
		{
			Hordes::Stop ();
			Platform::GCW_Open ();
		}

		_currentRAMRef = ramFileRef;
		_cpu->StartThread ();

		propertyChanged (documentOpening);
	}

	return result;
}

// -----------------------------------------------------------------------------
// start the emulator running

void Document::emulatorStart (void)
{
	_cpu->StartThread ();
	propertyChanged (cpuStatusChanged);
}

// -----------------------------------------------------------------------------
// stop the emulator

void Document::emulatorStop (void)
{
	Bool dummy;
	_cpu->StopThread (dummy, dummy, kStopOnADime);
	propertyChanged (cpuStatusChanged);
}

// -----------------------------------------------------------------------------
// reset the emulator

void Document::emulatorReset (void)
{
	Bool dummy;
	_cpu->StopThread (dummy, dummy, kStopOnADime);

	Emulator::Reset ();

	_cpu->StartThread ();

	propertyChanged (cpuStatusChanged);
}

// -----------------------------------------------------------------------------
// hotsync helper function

void _hotSyncUp (void* data)
{
	CPUStopper stopper (kStopOnADime);
	if (!stopper.Stopped())
		return;

	Hardware::HotSyncEvent(false);
}

// -----------------------------------------------------------------------------
// Start hotsync in the emulator

void Document::hotsync (void)
{
	CPUStopper stopper (kStopOnADime);
	if (!stopper.Stopped())
		return;

	Hardware::HotSyncEvent(true);

	// install a timer to have the hotsync button released in 1 second:
	Fl::add_timeout (1.0, _hotSyncUp, this);
}

// -----------------------------------------------------------------------------
// Pass a pen event to the emulator

void Document::pen (bool down, int x, int y)
{
	CPUStopper stopper (kStopOnADime);
	if (!stopper.Stopped())
		return;

	Hardware::PenEvent (down, x, y);
}

// -----------------------------------------------------------------------------
// Pass a button event to the emulator

void Document::button (bool down, SkinElementType button)
{
	CPUStopper stopper (kStopOnADime);
	if (!stopper.Stopped())
		return;

	Hardware::ButtonEvent (down, button);
}

// -----------------------------------------------------------------------------
// Pass a key event to the emulator

void Document::putkey (unsigned char key)
{
	CPUStopper stopper (kStopOnADime);
	if (!stopper.Stopped())
		return;

	Hardware::KeyboardEvent(key);
}

// -----------------------------------------------------------------------------
// load the given application file

void Document::loadApplication (const FileReference& appFile)
{
	FileRefList fileList;
	fileList.push_back (appFile);

	::LoadPalmFileList (fileList);
}

// -----------------------------------------------------------------------------
// export the application file

void Document::exportDatabase (const DatabaseInfo& db, const FileReference& exportFileRef)
{
	FileHandle		exportFile (exportFileRef,
								kCreateAlways | kOpenReadWrite,
								0, 0);

	::SavePalmFile (exportFile, db.cardNo, db.dbName);
}

// -----------------------------------------------------------------------------
// save the memory image to current file

void Document::saveMemory (void)
{
	saveRAMImage (_currentRAMRef, false);
	propertyChanged ();
}

// -----------------------------------------------------------------------------
// save the memory image to current file

void Document::saveMemoryAs (FileReference& dest)
{
	saveRAMImage (dest, false);
	propertyChanged ();
}

// -----------------------------------------------------------------------------
// save the memory image to the given file ref

void Document::saveScreen (FileReference& dest)
{
	//! TODO
	//_cpu->saveScreenShot( dest );
}

// -----------------------------------------------------------------------------
// add the given callback to the list of PropertyChangedHandlers

void Document::addPropertyChangedHandler (PropertyChangedHandler callback, void* param)
{
	PropertyChangedHandlerSpec	spec;
	spec.func = callback;
	spec.param = param;

	_handlers.push_back (spec);
}

// -----------------------------------------------------------------------------
// remove the given callback from the list of PropertyChangedHandlers

void Document::removePropertyChangedHandler (PropertyChangedHandler callback)
{
	list<PropertyChangedHandlerSpec>::iterator	iter;

	for (iter = _handlers.begin (); iter != _handlers.end (); ++iter)
	{
		if (iter->func == callback)
		{
			_handlers.erase (iter);
			return;
		}
	}
}

// -----------------------------------------------------------------------------
// Return true if the HW LCD is on

bool Document::LcdIsOn ()
{
	return HWRegisters::LCDIsOn ();
}

// -----------------------------------------------------------------------------
// gremlins control functions

void Document::gremlinsNew (void)
{
	if (showNewGremlinWindow ())
	{
		Preference<HordeInfo> pref (kPrefKeyHordeInfo);
		Hordes::New (*pref);
	}
}

void Document::gremlinsStep (void)
{
	Hordes::Step ();
}

void Document::gremlinsResume (void)
{
	Hordes::Resume ();
}

void Document::gremlinsStop()
{
	Hordes::Stop ();
}

// -----------------------------------------------------------------------------
// private methods
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// save the ram image to the file

void Document::saveRAMImage (const FileReference& ramFileRef, Bool forAutoSave)
{
	// See if there's a given name we should use.

	if (!ramFileRef.IsSpecified())
	{
		return;
	}

	try
	{
		Emulator::Save (ramFileRef, !forAutoSave, !forAutoSave, !forAutoSave);
	}
	catch (...)
	{
	}

	_currentRAMRef = ramFileRef;
}

// -----------------------------------------------------------------------------
// broadcast changed notices to all propertyChangedHandlers

void Document::propertyChanged (int property)
{
	list<PropertyChangedHandlerSpec>::iterator	iter;

	for (iter = _handlers.begin (); iter != _handlers.end (); ++iter)
	{
		(iter->func) (property, iter->param);
	}
}
