/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "fltk_MainWindow.h"

#include "CPU_MT.h"				// CPUStopper
#include "Hordes.h"				// Hordes::CanNew...
#include "Platform.h"			// Platform::GetString
#include "PreferenceMgr.h"		// Preference
#include "Strings.r.h"			// IDS_SAVE_RAM_PROMPT

#include "Document.h"			// docChangedCallback, documentClosing, etc.
#include "fltk_LcdWindow.h"		// Lcd_window::isOn, etc.

#include <FL/Fl.H>
#include <FL/Fl_Menu_Bar.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Box.H>
#include <FL/fl_ask.H>
#include <FL/fl_file_chooser.H>

#include "fltk_Dialog.h"


// -----------------------------------------------------------------------------
// file private constants
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// Menu Items IDs

enum
{
	kMenuFile = 0,
	kMenuNewEmulator,
	kMenuOpenMemory,
	kMenuCloseEmulator,
	kMenuSaveMemory,
	kMenuSaveMemoryAs,
	kMenuSaveScreen,
	kMenuLoadApplication,
	kMenuExportDatabase,
	kMenuHotSync,
	kMenuResetEmulator,
	kMenuDownloadROM,
	kMenuQuit,
	kMenuNull0,
	kMenuEdit,
	kMenuCut,
	kMenuCopy,
	kMenuPaste,
	kMenuClear,
	kMenuPreferences,
	kMenuLogging,
	kMenuDebug,
	kMenuSkins,
	kMenuDisplayToggle,
	kMenuNull1,
	kMenuGremlins,
	kMenuGremlinsNew,
	kMenuGremlinsStep,
	kMenuGremlinsResume,
	kMenuGremlinsStop,
	kMenuNull2,
#if HAS_PROFILING
	kMenuProfile,
	kMenuProfileInit,
	kMenuProfileStart,
	kMenuProfileStop,
	kMenuProfileDump,
	kMenuNull3,
#endif
	kMenuHelp,
	kMenuAbout
};


// -----------------------------------------------------------------------------
// Menu Items IDs

#define MENUITEM		0, 0, 0, 12, 0
#define SUBMENU			FL_SUBMENU, 0, 0, 12, 0
#define DIVIDER			FL_MENU_DIVIDER, 0, 0, 12, 0

Fl_Menu_Item menu_mbar[] =
{
	{"&File", 0,  0, 0, SUBMENU},
	{"&New", FL_ALT+'n', 0, (void*)kMenuNewEmulator, MENUITEM},
	{"&Open...", FL_ALT+'o', 0, (void*)kMenuOpenMemory, MENUITEM},
	{"&Close", FL_ALT+'w', 0, (void*)kMenuCloseEmulator, DIVIDER},
	{"&Save", FL_ALT+'s', 0, (void*)kMenuSaveMemory, MENUITEM},
	{"Save As...", 0, 0, (void*)kMenuSaveMemoryAs, MENUITEM},
	{"Save S&creen...", FL_ALT+'m', 0, (void*)kMenuSaveScreen, DIVIDER},
	{"&Install Application/Database...", 0, 0, (void*)kMenuLoadApplication, MENUITEM},
	{"Export Database...", 0, 0, (void*)kMenuExportDatabase, MENUITEM},
	{"&HotSync", FL_ALT+'h', 0, (void*)kMenuHotSync, MENUITEM},
	{"&Reset", FL_ALT+'r', 0, (void*)kMenuResetEmulator, MENUITEM},
	{"&Transfer ROM...", FL_ALT+'d', 0, (void*)kMenuDownloadROM, DIVIDER},
	{"&Quit", FL_ALT+'q', 0, (void*)kMenuQuit, MENUITEM},
	{0},
	{"&Edit", 0, 0, 0, SUBMENU},
	{"Cu&t", 0, 0, (void*)kMenuCut, MENUITEM},
	{"&Copy", 0, 0, (void*)kMenuCopy, MENUITEM},
	{"&Paste", 0, 0, (void*)kMenuPaste, MENUITEM},
	{"Clea&r", 0, 0, (void*)kMenuClear, DIVIDER},
	{"&Preferences...", FL_ALT+'/', 0, (void*)kMenuPreferences, MENUITEM},
	{"&Logging Options...", FL_ALT+'l', 0, (void*)kMenuLogging, MENUITEM},
	{"&Debug Options...", FL_ALT+'\\', 0, (void*)kMenuDebug, MENUITEM},
	{"S&kins...", FL_ALT+'k', 0, (void*)kMenuSkins, DIVIDER},
	{"Display To&ggle...", FL_ALT+'g', 0, (void*)kMenuDisplayToggle, MENUITEM},
	{0},
	{"&Gremlins", 0, 0, 0, SUBMENU},
	{"&New...", FL_ALT+'g', 0, (void*)kMenuGremlinsNew, DIVIDER},
	{"&Step", 0, 0, (void*)kMenuGremlinsStep, MENUITEM},
	{"&Resume", 0, 0, (void*)kMenuGremlinsResume, DIVIDER},
	{"Sto&p", 0, 0, (void*)kMenuGremlinsStop, MENUITEM},
	{0},
#if HAS_PROFILING
	{"&Profiling", 0, 0, 0, SUBMENU},
	{"&Init", 0, 0, (void*)kMenuProfileInit, MENUITEM},
	{"&Start", FL_ALT+'[', 0, (void*)kMenuProfileStart, MENUITEM},
	{"Sto&p", FL_ALT+']', 0, (void*)kMenuProfileStop, MENUITEM},
	{"&Dump", 0, 0, (void*)kMenuProfileDump, MENUITEM},
	{0},
#endif
	{"&Help", 0,  0, 0, SUBMENU},
	{"&About", 0, 0, (void*)kMenuAbout, MENUITEM},
	{0},
	{0}
};


// -----------------------------------------------------------------------------
// file private functions
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// menu item de/activate helper

static void setMenuItemStatus( Fl_Menu_Item& mi, bool status )
{
	if ( status )
		mi.activate();
	else
		mi.deactivate();
}


// -----------------------------------------------------------------------------
// called whenever my data model changes. Sets the various menu items to the
// corrected de/activated state based on the doc state.

void docChangedCallback( int property, void *param )
{
	MainWindow *mw = (MainWindow*) param;
	Document *doc = (mw != NULL)? mw->_doc : NULL;

	// a closing document is as good as no document...
	if ( (doc != NULL) && (property == Document::documentClosing) )
		doc = NULL;
	
	// File Menu
	setMenuItemStatus( menu_mbar[kMenuNewEmulator], true );
	setMenuItemStatus( menu_mbar[kMenuOpenMemory], true );
	setMenuItemStatus( menu_mbar[kMenuCloseEmulator], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuSaveMemory], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuSaveMemoryAs], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuSaveScreen], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuLoadApplication], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuExportDatabase], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuHotSync], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuResetEmulator], (doc != NULL) );
	setMenuItemStatus( menu_mbar[kMenuDownloadROM], true );
	setMenuItemStatus( menu_mbar[kMenuQuit], true );

	// Edit Menu
	menu_mbar[kMenuCut].deactivate();
	menu_mbar[kMenuCopy].deactivate();
	menu_mbar[kMenuPaste].deactivate();
	menu_mbar[kMenuClear].deactivate();
	setMenuItemStatus( menu_mbar[kMenuPreferences], true );
	setMenuItemStatus( menu_mbar[kMenuLogging], true );
	setMenuItemStatus( menu_mbar[kMenuDebug], true );
	setMenuItemStatus( menu_mbar[kMenuSkins], true );
	setMenuItemStatus( menu_mbar[kMenuDisplayToggle], (doc != NULL) );

	// Gremlins Menu
	setMenuItemStatus( menu_mbar[kMenuGremlinsNew], (doc != NULL && Hordes::CanNew()) );
	setMenuItemStatus( menu_mbar[kMenuGremlinsStep], (doc != NULL && Hordes::CanStep()) );
	setMenuItemStatus( menu_mbar[kMenuGremlinsResume], (doc != NULL && Hordes::CanResume()) );
	setMenuItemStatus( menu_mbar[kMenuGremlinsStop], (doc != NULL && Hordes::CanStop()) );

#if HAS_PROFILING
	// Profile Menu
	setMenuItemStatus( menu_mbar[kMenuProfileInit], (!gProfilingEnabled) );
	setMenuItemStatus( menu_mbar[kMenuProfileStart], (!gProfilingEnabled) );
	setMenuItemStatus( menu_mbar[kMenuProfileStop], (gProfilingEnabled && gProfilingOn) );
	setMenuItemStatus( menu_mbar[kMenuProfileDump], (gProfilingEnabled && gProfilingOn) );
#endif

	// Help Menu
	setMenuItemStatus( menu_mbar[kMenuAbout], true );	 
}


// -----------------------------------------------------------------------------
// callback for main application menu items

typedef void (*MenuHandler)(MainWindow*, int menuID);

static void PrvMenuNewEmulator (MainWindow* mw, int menuID)
{
	mw->doEmulatorStart();
}

static void PrvMenuOpenMemory (MainWindow* mw, int menuID)
{
	mw->doOpenMemory();
}

static void PrvMenuCloseEmulator (MainWindow* mw, int menuID)
{
	mw->doEmulatorStop();
}

static void PrvMenuSaveMemory (MainWindow* mw, int menuID)
{
}

static void PrvMenuSaveMemoryAs (MainWindow* mw, int menuID)
{
	mw->doSaveMemoryAs();
}

static void PrvMenuSaveScreen (MainWindow* mw, int menuID)
{
	mw->doSaveScreen();
}

static void PrvMenuLoadApplication (MainWindow* mw, int menuID)
{
	mw->doLoadApplication();
}

static void PrvMenuExportDatabase (MainWindow* mw, int menuID)
{
	mw->doExportDatabase();
}

static void PrvMenuHotSync (MainWindow* mw, int menuID)
{
	mw->_doc->hotsync();
}

static void PrvMenuResetEmulator (MainWindow* mw, int menuID)
{
	mw->_doc->emulatorReset();
}

static void PrvMenuDownloadROM (MainWindow* mw, int menuID)
{
}

static void PrvMenuQuit (MainWindow* mw, int menuID)
{
	mw->doQuit();
}

static void PrvMenuPreferences (MainWindow* mw, int menuID)
{
	mw->doEditPreferences();
}

static void PrvMenuLogging (MainWindow* mw, int menuID)
{
	mw->doEditLoggingPreferences();
}

static void PrvMenuDebug (MainWindow* mw, int menuID)
{
	mw->doEditDebugPreferences();
}

static void PrvMenuSkins (MainWindow* mw, int menuID)
{
	mw->doEditSkins();
}

static void PrvMenuDisplayToggle (MainWindow* mw, int menuID)
{
	mw->_lcd->setOn (!mw->_lcd->isOn ());
}

static void PrvMenuGremlinsNew (MainWindow* mw, int menuID)
{
	mw->_doc->gremlinsNew();
}

static void PrvMenuGremlinsStep (MainWindow* mw, int menuID)
{
	mw->_doc->gremlinsStep();
}

static void PrvMenuGremlinsResume (MainWindow* mw, int menuID)
{
	mw->_doc->gremlinsResume();
}

static void PrvMenuGremlinsStop (MainWindow* mw, int menuID)
{
	mw->_doc->gremlinsStop();
}

#if HAS_PROFILING
static void PrvMenuProfileInit (MainWindow* mw, int menuID)
{
	::ProfileInit (0x0C000, MAXDEPTH);
}

static void PrvMenuProfileStart (MainWindow* mw, int menuID)
{
	::ProfileStart();
}

static void PrvMenuProfileStop (MainWindow* mw, int menuID)
{
	::ProfileStop();
}

static void PrvMenuProfileDump (MainWindow* mw, int menuID)
{
	::ProfileDump (NULL);
	::ProfileCleanup ();
}
#endif

static void PrvMenuAbout (MainWindow* mw, int menuID)
{
	showAboutBox();
}

static const struct
{
	int			fCommand;
	MenuHandler	fHandler;
    int			fDummy;	// What's this one for?
	int			fStopType;
}
kCommandHandlers[] =
{
	{ kMenuNewEmulator,			PrvMenuNewEmulator,		0,		0				},
	{ kMenuOpenMemory,			PrvMenuOpenMemory,		0,		kStopOnADime   	},
	{ kMenuCloseEmulator,		PrvMenuCloseEmulator,	0,		0				},
	{ kMenuSaveMemory,			PrvMenuSaveMemory,		0,		kStopOnADime   	},
	{ kMenuSaveMemoryAs,		PrvMenuSaveMemoryAs,	0,		kStopOnADime   	},
	{ kMenuSaveScreen,			PrvMenuSaveScreen,		0,		kStopOnADime   	},
	{ kMenuLoadApplication,		PrvMenuLoadApplication,	0,		kStopOnATrap   	},
	{ kMenuExportDatabase,		PrvMenuExportDatabase,	0,		kStopOnATrap	},
	{ kMenuHotSync,				PrvMenuHotSync,			0,		kStopOnADime   	},
	{ kMenuResetEmulator,		PrvMenuResetEmulator,	0,		0				},
	{ kMenuDownloadROM,			PrvMenuDownloadROM,		0,		0				},
	{ kMenuQuit,				PrvMenuQuit,			0,		0				},
	{ kMenuCut,					NULL,					0,		0				},
	{ kMenuCopy,				NULL,					0,		0				},
	{ kMenuPaste,				NULL,					0,		0				},
	{ kMenuClear,				NULL,					0,		0				},
	{ kMenuPreferences,			PrvMenuPreferences,		0,		kStopOnATrap   	},
	{ kMenuLogging,				PrvMenuLogging,			0,		kStopOnADime   	},
	{ kMenuDebug,				PrvMenuDebug,			0,		kStopOnADime   	},
	{ kMenuSkins,				PrvMenuSkins,			0,		kStopOnADime   	},
	{ kMenuDisplayToggle,		PrvMenuDisplayToggle,	0,		0				},
	{ kMenuGremlinsNew,			PrvMenuGremlinsNew,		0,		kStopOnATrap   	},
	{ kMenuGremlinsStep,		PrvMenuGremlinsStep,	0,		kStopOnATrap   	},
	{ kMenuGremlinsResume,		PrvMenuGremlinsResume,	0,		kStopOnATrap   	},
	{ kMenuGremlinsStop,		PrvMenuGremlinsStop,	0,		kStopOnATrap   	},
#if HAS_PROFILING
	{ kMenuProfileInit,			PrvMenuProfileInit,		0,		kStopOnADime   	},
	{ kMenuProfileStart,		PrvMenuProfileStart,	0,		kStopOnADime   	},
	{ kMenuProfileStop,			PrvMenuProfileStop,		0,		kStopOnADime   	},
	{ kMenuProfileDump,			PrvMenuProfileDump,		0,		kStopOnADime   	},
#endif
	{ kMenuAbout,				PrvMenuAbout,			0,		0				}
};



void menuCallback (Fl_Widget* w, void* data)
{
	Fl_Menu_Button*		mb = (Fl_Menu_Button*) w;
	const Fl_Menu_Item*	mi = mb->mvalue ();
	int					id = mi->argument();
	MainWindow*			mw = (MainWindow*) data;

	int index = 0;
	while (kCommandHandlers[index].fCommand)
	{
		if (kCommandHandlers[index].fCommand == id)
		{
			try
			{
				if (kCommandHandlers[index].fStopType)
				{
					CPUStopper	stopper (kCommandHandlers[index].fStopType);
					if (stopper.Stopped ())
					{
						kCommandHandlers[index].fHandler (mw, id);
					}
					else
					{
						stopper.ShowError ("perform the command");
					}
				}
				else
				{
					kCommandHandlers[index].fHandler (mw, id);
				}
			}
			catch (ErrCode errCode)
			{
//				Errors::ReportIfError (kCommandHandlers[index].fOperation, errCode, 0, false);
			}

			break;
		}

		index++;
	}
}


// -----------------------------------------------------------------------------
// constructor
// -----------------------------------------------------------------------------

MainWindow::MainWindow() :
	Fl_Window( 250, 358 ), _doc( NULL )
{
	box (FL_FLAT_BOX);
	color (fl_gray_ramp (FL_NUM_GRAY - 1));

	resizable (NULL);
	
	_menuButton = new Fl_Menu_Button (0, 0, 600, 400, "POSER");
	_menuButton->type (Fl_Menu_Button::POPUP3);
	_menuButton->menu (menu_mbar);
	_menuButton->callback (menuCallback, this);

	_lcd = new LCD_window (this);

#if 0
	// if there is a skin file path stored in the preferences, load it.
	const char* spath = gHostPrefs->getSkinFile();
	if (spath && (strlen (spath) > 0))
	{
		_lcd->loadSkinFile (spath);
	}
#endif

	end();

	// call the menu setup function
	docChangedCallback (0, this);

	if (_sMainWindow == NULL)
		_sMainWindow = this;
}


// -----------------------------------------------------------------------------
// private methods
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// quit the application.

void MainWindow::doQuit()
{
	if (_doc != NULL)
		if (!doEmulatorStop())
			return;

	if (_sMainWindow == this)
		_sMainWindow = NULL;

	delete this;
}


// -----------------------------------------------------------------------------
// create a new data document & start the emulator

void MainWindow::doEmulatorStart()
{
	Preference<Configuration> pref (kPrefKeyLastConfiguration);
	Configuration cfg = *pref;

	// ask the user for doc creation parameters
	if (!showNewEmulatorWindow (cfg))
		return;

	// if there is a running document, clear it out.
	if (_doc)
		doEmulatorStop ();

	// create the new document
	_doc = new Document (cfg);
	if (_doc->initializedOk ())
	{
		_lcd->setDocument (_doc);
		_doc->addPropertyChangedHandler (docChangedCallback, (void*) this);
		_doc->emulatorStart ();
	}
	else
	{
		delete _doc;
		_doc = NULL;
		_lcd->setDocument (NULL);
	}
}


// -----------------------------------------------------------------------------
// stop the emulator and kill the existing data document. Ask the user first.
// returns false if the user cancels.

Bool MainWindow::doEmulatorStop()
{
	Preference<CloseActionType> pref (kPrefKeyCloseAction);
	CloseActionType action = *pref;

	if (action == kSaveAsk)
	{
		string saveChanges (Platform::GetString (kStr_SaveBeforeClosing));
		string untitled (Platform::GetString (kStr_Untitled));
		string app (Platform::GetString (kStr_AppName));

		ParamList paramList;
		paramList.push_back (string ("%AppName"));
		paramList.push_back (app);
		paramList.push_back (string ("%DocName"));

		if (_doc->getCurrentRAMRef().IsSpecified())
			paramList.push_back (_doc->getCurrentRAMRef().GetFileName());
		else
			paramList.push_back (untitled);

		string msg = Errors::ReplaceParameters (saveChanges, paramList);

		int result = fl_choice (msg.c_str(),
								Platform::GetString (kStr_Cancel).c_str(),
								Platform::GetString (kStr_Yes).c_str(),
								Platform::GetString (kStr_No).c_str() );

		if (result == 1) // yes
			action = kSaveAlways;
		else if (result == 2) // no
			action = kSaveNever;
		else		// result == IDCANCEL
			return false;
	}

	_doc->emulatorStop ();

	if (action == kSaveAlways)
	{
		_doc->saveMemory ();
	}

	_lcd->setDocument (NULL);
	_lcd->redraw();

	delete _doc;
	_doc = NULL;

	return true;
}


// -----------------------------------------------------------------------------
// load an application file into the current running emulator

void MainWindow::doLoadApplication()
{
	// resource-ize this stuff...
	char *fname = fl_file_chooser( "Choose an Application or Database",
								   "*.{prc,PRC,pdb,PDB,pqa,PQA}",
								   NULL );

	if ( fname != NULL )
	{
		FileReference fileRef( fname );
		_doc->loadApplication( fileRef );
	}
}


// -----------------------------------------------------------------------------
// Save a database in .pdb format.

void MainWindow::doExportDatabase()
{
	DatabaseInfo db;
	if (getDatabaseName(db))
	{
		char* fname = fl_file_chooser ("Save database as", NULL, NULL);
	
		if (fname)
		{
			FileReference f = FileReference(fname);
			_doc->exportDatabase(db, f);
		}
	}
}


// -----------------------------------------------------------------------------
// save the document with a new file name

void MainWindow::doSaveMemoryAs()
{
	char fileName[PATH_MAX] = {0};
	
	char *fname = fl_file_chooser( "Save RAM file as", "*.psf", fileName );
	
	if ( fname == NULL )
	{
		return;
	}
	
	FileReference ramFileRef(fname);

	_doc->saveMemoryAs( ramFileRef );
}


// -----------------------------------------------------------------------------
// open a pre-existing document.

void MainWindow::doOpenMemory()
{
	// get the document name from the user
	char fileName[PATH_MAX] = {0};
	char *fname = fl_file_chooser( "Open RAM file", "*.psf", fileName );
	if ( fname == NULL )
	{
		return;
	}

	// get a FileReference to the given file
	FileReference ramFileRef( fname );

	// Clear out othe old document
	if ( _doc )
	{
		this->doEmulatorStop();
	}

	// create a new document & set up the pointers
	Preference<Configuration> pref (kPrefKeyLastConfiguration);
	_doc = new Document( *pref );
	if ( _doc->initializedOk() )
	{
		_lcd->setDocument( _doc );
		_doc->addPropertyChangedHandler( docChangedCallback, (void*)this );

		// go ahead and open the document
		_doc->open( ramFileRef );
	}
	else
	{
		delete _doc;
		_doc = NULL;
	}
}


// -----------------------------------------------------------------------------
// save the screen image

void MainWindow::doSaveScreen()
{
	FileReference f;
	_doc->saveScreen (f);
}


// -----------------------------------------------------------------------------
// preference editing

void MainWindow::doEditLoggingPreferences()
{
	if (editLogging ())
	{
		// possibly take action based on new prefs...
	}
}

void MainWindow::doEditDebugPreferences()
{
	if (editDebugging ())
	{
		// possibly take action based on new prefs...
	}
}

void MainWindow::doEditSkins()
{
	if (editSkins ())
	{
		// possibly take action based on new prefs...
	}
}

void MainWindow::doEditPreferences()
{
	if (editPreferences ())
	{
		// possibly take action based on new prefs...
	}
}

MainWindow* MainWindow::_sMainWindow = NULL;


