/* GMP3 - A front end for mpg123
 * Copyright (C) 1998 Brett Kosinski
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk_imlib.h>

#include "pixmap.h"
#include "control_funcs.h"
#include "shape.h"
#include "list.h"
#include "mixerpanel.h"
#include "gmp3.h"
#include "mixer.h"
#include "rc_parser.h"
#include "opt_dialog.h"
#include "about.h"
#include "xpmbutton.h"
#include "themerc.h"

GdkWindow *root_win;

configType options;

ThemeWindow *activeWindow;
ThemeWindow bigWindow;
ThemeWindow smallWindow;

pid_t mpgPID;

int smallSize = FALSE;
int playing = FALSE;
int doneSong = TRUE;
int paused = FALSE;
int loopOne = FALSE;
int loopAll = FALSE;
int shuffle = FALSE;
long int startSecs = 0;
long int currSecs = 0;
int oldSeconds = 0;

#define TOKENCOUNT 16
tokenInfo tokens[TOKENCOUNT] = { { "buffer_size", INTEGER, NULL },
                                 { "downsample", INTEGER, NULL },
                                 { "output_type", INTEGER, NULL },
                                 { "send_to_lineout", BOOLEAN, NULL },
                                 { "managed", BOOLEAN, NULL },
                                 { "tooltips", BOOLEAN, NULL },
                                 { "autoload", STRING, NULL },
                                 { "autoparse", BOOLEAN, NULL},
                                 { "mixerapp", STRING, NULL },
                                 { "themepack", STRING, NULL },
                                 { "autoloadtags", BOOLEAN, NULL},
                                 { "mp3loaddir", STRING, NULL},
                                 { "mp3player", STRING, NULL},
                                 { "sound_device", STRING, NULL},
                                 { "http_proxy", STRING, NULL},
                                 { "http_auth_string", STRING, NULL} };

void bringToFront(GtkWidget *widget, GdkEventVisibility *event);
void createSmallWindow();
void createBigWindow();
void resizeMainWindow();
void installSigHandler();

/********** Main Guts ***********/

/***********************
 * Signal Handler Code *
 ***********************/

void childKill(int signal)
{
  int status;      
  
  if ((waitpid(-1, &status, WNOHANG | WUNTRACED) == mpgPID) &&
      (! WIFSTOPPED(status)))
    {
      mpgPID = 0;
      doneSong = TRUE;
    
    #ifdef __DEBUG__
      g_print("mpg123 exited\n");
    #endif
    }  
  
  installSigHandler();
}

void installSigHandler()
{
  signal(SIGCHLD, childKill);
}

void openDialog(GtkWidget *widget, GtkWidget *dialog)
{
  if (! GTK_WIDGET_VISIBLE(dialog))
    gtk_widget_show(dialog);
}

void closeDialog(GtkWidget *widget, GtkWidget *dialog)
{
  if (GTK_WIDGET_VISIBLE(dialog))
    gtk_widget_hide(dialog);
}

/********************************
 * RC File Loading/Saving Stuff *
 ********************************/

void adjustThemePath()
{
  char *homeDir = getenv("HOME");
  char *fullFile = (char *)malloc(1024);
  
  strcpy(fullFile, homeDir);
  strcat(fullFile, RCFILEDIR);
  strcat(fullFile, "/");
  strcat(fullFile, options.themePack);           
  
  if (access(fullFile, F_OK) < 0)
    {
      strcpy(fullFile, PIXMAPDIR);
      strcat(fullFile, "/");
      strcat(fullFile, options.themePack);
      
      options.themePath = (char *)malloc(strlen(fullFile)+1);
      strcpy(options.themePath, fullFile);
    }
  else
    {
      options.themePath = (char *)malloc(strlen(fullFile)+1);
      strcpy(options.themePath, fullFile);  
    }
    
  free(fullFile);
}

gint checkMP3Player()
{
  if (access(options.mp3player, F_OK) < 0)
    {
      fprintf(stderr, "Error, %s not found!\n", options.mp3player);
      return 0;
    }
  else
    return 1;
}

void setupRCTokens()
{
  char *homeDir = getenv("HOME");

  options.bufferSize = 1500;
  options.downSample = 0;              /* Create default settings. */
  options.output = 2;
  options.lineOut = FALSE;
  options.managed = TRUE;
  options.toolTips = TRUE;
  options.autoparse = FALSE;      
  options.autoloadTags = FALSE;
  
  options.playListName = (char *)malloc(1024);
  options.playListName[0] = '\0';  
  
  options.mixerApp = (char *)malloc(1024);
  options.mixerApp[0] = '\0';

  options.themePack = (char *)malloc(1024);  
  strcpy(options.themePack, "default");

  options.mp3loaddir = (char *)malloc(2048);
  strncpy(options.mp3loaddir, homeDir, 1023);
  strcat(options.mp3loaddir, "/");
  
  options.mp3player = (char *)malloc(1024);
  strncpy(options.mp3player, MPG123_NAME, 1023);  
  
  options.soundDevice = (char *)malloc(1024);
  strncpy(options.soundDevice, SOUND_DEV, 1023);
  
  options.httpProxy = (char *)malloc(1024);
  strncpy(options.httpProxy, HTTP_PROXY, 1023);
  
  options.httpAuthString = (char *)malloc(1024);
  strncpy(options.httpAuthString, HTTP_AUTH_STRING, 1023);
  
  tokens[0].dataBuffer = &options.bufferSize;
  tokens[1].dataBuffer = &options.downSample;
  tokens[2].dataBuffer = &options.output;     /* Set up the tokens correctly */
  tokens[3].dataBuffer = &options.lineOut;
  tokens[4].dataBuffer = &options.managed;
  tokens[5].dataBuffer = &options.toolTips;
  tokens[6].dataBuffer = options.playListName;
  tokens[7].dataBuffer = &options.autoparse;
  tokens[8].dataBuffer = options.mixerApp;
  tokens[9].dataBuffer = options.themePack;
  tokens[10].dataBuffer = &options.autoloadTags;
  tokens[11].dataBuffer = options.mp3loaddir;
  tokens[12].dataBuffer = options.mp3player;
  tokens[13].dataBuffer = options.soundDevice;
  tokens[14].dataBuffer = options.httpProxy;
  tokens[15].dataBuffer = options.httpAuthString;
}

void loadRCFile()
{                                                                 
  char *homeDir = getenv("HOME");
  char *rcFile = (char *)malloc(strlen(homeDir)+strlen(RCFILENAME)+
                                strlen(RCFILEDIR)+4); 
  strcpy(rcFile, homeDir);
  strcat(rcFile, RCFILEDIR);
  strcat(rcFile, "/");
  strcat(rcFile, RCFILENAME);

  parseFile(rcFile, tokens, TOKENCOUNT);

  if (options.playListName[0] != '\0')  /* Read playlist from autoload param */
      loadAlbumns(options.playListName);
  
  checkMP3Player();
  adjustThemePath();
  
  free(rcFile);
}  

void saveRCFile()
{
  char *homeDir = getenv("HOME");
  char *rcFile = (char *)malloc(strlen(homeDir)+strlen(RCFILENAME)+
                                strlen(RCFILEDIR)+4);
  char *rcDir = (char *)malloc(strlen(homeDir)+strlen(RCFILEDIR)+2);
  
  strcpy(rcDir, homeDir);
  strcat(rcDir, RCFILEDIR);

  strcpy(rcFile, rcDir);
  strcat(rcFile, "/");
  strcat(rcFile, RCFILENAME);
  
  if (access(rcDir, F_OK) < 0)
    mkdir(rcDir, 0755);
  
  saveFile(rcFile, tokens, TOKENCOUNT);
}

void loadCommandlineFiles(int argc, char *argv[])
{
  int i;
  
  for (i = 1; i < argc; i++)
    {
      addFileToPlayList(argv[i]);
      
      playing = TRUE;
      doneSong = TRUE;
    }
}

/*********************************
 * Callback Functions for Events *
 *********************************/

void destroyWindow(GtkWidget *widget, GtkWidget *window)
{
  gtk_widget_destroy(window);
}

void missingFileError()
{
  GtkWidget *window;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *vbox, *hbox;
  char *errorStr;
  char *filename = ((songNode *)(currentSong->data))->fileName;
  
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), "Play Error");
  gtk_window_set_wmclass(GTK_WINDOW(window), "gmp3.error_dialog", NULL);

  errorStr = (char *)malloc(strlen("Error Opening File: ")+
                            strlen(filename)+4);
  sprintf(errorStr, "Error Opening File: %s", filename);
  
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(window), hbox);
  gtk_widget_show(hbox);
  
  vbox = gtk_vbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 5);
  gtk_widget_show(vbox);

  label = gtk_label_new(errorStr);
  gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 5);
  gtk_widget_show(label);
  
  button = gtk_button_new_with_label("Close");
  gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 5);
  gtk_widget_show(button);  
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(destroyWindow), window);

  gtk_widget_show(window);
}

void printSongName()
{          
  char *currentSongName;
  char *currentFileName;
  gint i;

  if (activeWindow->song_label.eventBox == NULL)
    return;

  if ((activePlayList == NULL) || (g_list_first(activePlayList)->next == NULL))
    return;

  currentSongName = ((songNode *)(currentSong->data))->title;
  currentFileName = ((songNode *)(currentSong->data))->fileName;

  /* get rid of path in filenames [pel] */
  if (strcmp(currentSongName, currentFileName) == 0)
    {
      i = strlen(currentSongName);
  
      for (i = strlen(currentSongName); currentSongName[i] != '/'; i--);
      i++;
  
      currentSongName += i;
    }

  gtk_label_set(GTK_LABEL(activeWindow->song_label.widget->widget), currentSongName); 
}

void resetClock()
{
  if (activeWindow->time_label.eventBox == NULL)
    return;
    
  gtk_label_set(GTK_LABEL(activeWindow->time_label.widget->widget), "0:00");
}

int playLoop()
{  
  if (playing && doneSong)
    {    
      if (! nextSong(TRUE))
        {
          if (! play(NULL, &options)) missingFileError();

          playing = TRUE;
          doneSong = FALSE;
                    
          printSongName();
        }  
      else
        {
          if (shuffle)
            clearPlayedFlags();
            
          playing = FALSE;
          doneSong = TRUE;
          
          currentSong = g_list_first(activePlayList)->next;
        }  
                
      startSecs = currSecs+1;  /* Add 1 so the timer starts at 0:00 */
    }      
    
  return TRUE;    
}

int adjustTimer()
{    
  char secondStr[4];
  char timeStr[8];
  struct timeval time;
  struct timezone zone;
  int seconds;
  
  gettimeofday(&time, &zone);
  currSecs = time.tv_sec;

  if (playing && ! doneSong && ! paused)
    {           
      seconds = currSecs - startSecs;
      
      if (oldSeconds != currSecs)
        { 
          oldSeconds = currSecs;
        
          sprintf(secondStr, "%d", seconds % 60);
        
          if (strlen(secondStr) > 1)
            sprintf(timeStr, "%d:%s", seconds / 60, secondStr);
          else
            sprintf(timeStr, "%d:0%s", seconds / 60, secondStr);  
            
          if (activeWindow->time_label.eventBox != NULL)
            gtk_label_set(GTK_LABEL(activeWindow->time_label.widget->widget), timeStr);  
       }   
    }
  else if (paused)
         {
           seconds = currSecs - startSecs;  
           
           if (currSecs != oldSeconds)
             {               
               oldSeconds = currSecs;       /* This part adjusts for pausing */   
               startSecs++;                 /* time. */
             }
         }  
         
  return TRUE;         
}

void stopCallback()
{
  if (paused && activeWindow->pause_label.eventBox)
    gtk_widget_hide(activeWindow->pause_label.widget->widget);
    
  stop();  
}

void playCallback()
{
  if (playing)
    return;
      
  if (shuffle)
    {
      clearPlayedFlags();
      nextSong(TRUE);
    }

  resetClock();
  
  if (! play(NULL, &options)) missingFileError();
}

void backCallback()
{
  if (! back(NULL, &options)) missingFileError();
}

void forwardCallback()
{
  if (! forward(NULL, &options)) missingFileError();
}

void togglePause()
{    
  pauseSong();  
  
  if (activeWindow->pause_label.eventBox == NULL)
    return;

  if (paused)
    gtk_widget_show(activeWindow->pause_label.widget->widget);
  else
    gtk_widget_hide(activeWindow->pause_label.widget->widget);
}

void toggleShuffle()
{
  if (shuffle)
    {
      shuffle = FALSE;
      
      if (activeWindow->shuffle_label.eventBox != NULL)
        gtk_widget_hide(activeWindow->shuffle_label.widget->widget);
    }  
  else
    {
      shuffle = TRUE;
      
      if (activeWindow->shuffle_label.eventBox != NULL)      
        gtk_widget_show(activeWindow->shuffle_label.widget->widget);
    }  
}

void toggleLoop()
{
  if (loopAll || loopOne)
    {
      loopOne = loopAll;
      loopAll = FALSE;
      
      if (activeWindow->repeat_label.eventBox == NULL)
        return;
             
      if (loopOne)
        gtk_label_set(GTK_LABEL(activeWindow->repeat_label.widget->widget), "Repeat");
      else 
        gtk_widget_hide(activeWindow->repeat_label.widget->widget);    
    }
  else
    {
      loopAll = TRUE;
      
      if (activeWindow->repeat_label.eventBox == NULL)
        return;
        
      gtk_label_set(GTK_LABEL(activeWindow->repeat_label.widget->widget), "Repeat All");
      gtk_widget_show(activeWindow->repeat_label.widget->widget);
    }  
}

void quitCallback()
{ 
  static GtkWidget *window = NULL;
  GtkWidget *hbox, *hbox2, *vbox;
  GtkWidget *button;
  GtkWidget *label;

  if (window && GTK_WIDGET_VISIBLE(window))
    return;

  if ((playing) && (! doneSong))
    stop();

  if (albumnChanged && ! window)
    {
      window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
      gtk_window_set_title(GTK_WINDOW(window), "Alert!");
      gtk_window_set_wmclass(GTK_WINDOW(window), "gmp3.alert", NULL);
      
      gtk_signal_connect(GTK_OBJECT(window), "delete_event", 
                         GTK_SIGNAL_FUNC(gtk_widget_hide), NULL);
                         
      gtk_signal_connect(GTK_OBJECT(window), "destroy_event",
                         GTK_SIGNAL_FUNC(gtk_widget_hide), NULL);
      
      vbox = gtk_vbox_new(FALSE, 5);
      gtk_container_add(GTK_CONTAINER(window), vbox);
      gtk_widget_show(vbox);

      hbox = gtk_hbox_new(FALSE, 5);
      gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 5);
      gtk_widget_show(hbox);
      
      label = gtk_label_new("You haven't saved your albumns!");
      gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 5);
      gtk_widget_show(label);
      
      hbox2 = gtk_hbox_new(FALSE, 5);
      gtk_box_pack_start(GTK_BOX(vbox), hbox2, TRUE, TRUE, 5);
      gtk_widget_show(hbox2);
      
      hbox = gtk_hbox_new(FALSE, 0);
      gtk_box_pack_start(GTK_BOX(hbox2), hbox, TRUE, TRUE, 5);
      gtk_widget_show(hbox);
      
      button = gtk_button_new_with_label("Quit");
      gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
      gtk_widget_show(button);
      
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
      
      button = gtk_button_new_with_label("Cancel");
      gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
      gtk_widget_show(button);      
      
      gtk_signal_connect(GTK_OBJECT(button), "clicked", 
                         GTK_SIGNAL_FUNC(closeDialog), window);
      
      gtk_widget_show(window);
    }
  else if (albumnChanged)
    gtk_widget_show(window);
  else
    gtk_main_quit();
}

void volumeSliderCallback(GtkAdjustment *adjustment, gpointer *data)
{
  mixerSettings.globalVolume = adjustment->value;
  
  if (mixerExists)
    setVolume();
}

void popupMenu(GtkWidget *menu)
{
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 3, 0);
}

void toggleWindowSize()
{
  int x = activeWindow->position.x;
  int y = activeWindow->position.y;
      
  if (smallSize)
    {
       smallSize = FALSE;
       activeWindow = &bigWindow;
       gtk_widget_hide(smallWindow.main_window);
    }
  else
    {
       smallSize = TRUE;
       activeWindow = &smallWindow;       
       gtk_widget_hide(bigWindow.main_window);
    }
    
  gtk_widget_set_uposition(activeWindow->main_window, x, y);
  activeWindow->position.x = x;
  activeWindow->position.y = y;  
    
  if (activeWindow->shuffle_label.eventBox)
    {
      if (shuffle)
        gtk_widget_show(activeWindow->shuffle_label.widget->widget);
      else
        gtk_widget_hide(activeWindow->shuffle_label.widget->widget);
    }    
      
  if (activeWindow->repeat_label.eventBox)
    {     
      if (loopOne)
        {
          gtk_label_set(GTK_LABEL(activeWindow->repeat_label.widget->widget), 
                        "Repeat");
          gtk_widget_show(activeWindow->repeat_label.widget->widget);
        }
      else if (loopAll)
        {
          gtk_label_set(GTK_LABEL(activeWindow->repeat_label.widget->widget), 
                        "Repeat All");
          gtk_widget_show(activeWindow->repeat_label.widget->widget);        
        }
      else
        gtk_widget_hide(activeWindow->repeat_label.widget->widget);
    }

  if (activeWindow->song_label.eventBox != NULL)
    printSongName();
    
  gtk_widget_show(activeWindow->main_window);
}    

void widgetCommandCallback(GtkWidget *widget, gint command)
{
  switch (command)
    {
    case CMD_NO_COMMAND : { break; }
    case CMD_BACK : { backCallback(NULL, &options); break; }
    case CMD_FORWARD : { forwardCallback(NULL, &options); break; }
    case CMD_PLAY : { playCallback(NULL, &options); break; }
    case CMD_STOP : { stopCallback(NULL, &options); break; }
    case CMD_PAUSE : { togglePause(NULL, &options); break; }
    case CMD_TOGGLE_SHUFFLE : { toggleShuffle(NULL, &options); break; }
    case CMD_TOGGLE_REPEAT : { toggleLoop(NULL, &options); break; }      
    case CMD_OPEN_ALBUMN_EDITOR : { openAlbumnEditorDialog(); break; }
    case CMD_OPEN_PLAYLIST_EDITOR : { openPlaylistEditorDialog(); break; }
    case CMD_OPEN_OPTIONS : { createOptionsDialog(); break; }
    case CMD_OPEN_ABOUT_BOX : { createAboutBox(); break; }
    case CMD_OPEN_MIXER : { openMixer(); break; }      
    case CMD_TOGGLE_SIZE : { toggleWindowSize(); break; }
    case CMD_QUIT : { quitCallback(); break; }
    default :
      {
        g_print("Warning, invalid signal %i.\n", command);
        break;
      }        
    }
}

gint widgetLClickCommandCallback(GtkWidget *widget, GdkEventButton *event,
                                 gint command)
{
  if (event->button != 1)
    return FALSE;

  switch (command)
    {
    case CMD_POPUP_MENU : 
      {
        gtk_menu_popup(GTK_MENU(activeWindow->main_menu),
                       NULL, NULL, NULL, NULL,
                       3, event->time);
        return TRUE;
      }  
    case CMD_DRAG_CAPTURE : shape_pressed(widget, event, activeWindow);
    case CMD_NO_COMMAND : return FALSE;
    default : { widgetCommandCallback(widget, command); return TRUE; }
    }
}                                 

gint widgetMClickCommandCallback(GtkWidget *widget, GdkEventButton *event,
                                 gint command)
{
  if (event->button != 2)
    return FALSE;

  switch (command)
    {
    case CMD_POPUP_MENU : 
      {
        gtk_menu_popup(GTK_MENU(activeWindow->main_menu),
                       NULL, NULL, NULL, NULL,                             
                       3, event->time);
        break;
      }  
    case CMD_DRAG_CAPTURE : shape_pressed(widget, event, activeWindow);
    case CMD_NO_COMMAND :
      return FALSE;        
    default : { widgetCommandCallback(widget, command); break; }
    }    
    
  return TRUE;  
}                                 

gint widgetRClickCommandCallback(GtkWidget *widget, GdkEventButton *event,
                                 gint command)
{
  if (event->button != 3)
    return FALSE;

  switch (command)
    {
    case CMD_POPUP_MENU : 
      {
        gtk_menu_popup(GTK_MENU(activeWindow->main_menu),
                       NULL, NULL, NULL, NULL,
                       3, event->time);
        break;
      }  
    case CMD_DRAG_CAPTURE : shape_pressed(widget, event, activeWindow);
    case CMD_NO_COMMAND :
      return FALSE;
    default : { widgetCommandCallback(widget, command); break; }
    }    
    
  return TRUE;
}                                 

/***********************
 * Main initialization *
 ***********************/

int main(int argc, char *argv[])
{
  installSigHandler();
  gtk_init(&argc, &argv);
  gtk_rc_init();
  gdk_imlib_init();
  root_win = gdk_window_foreign_new(GDK_ROOT_WINDOW());    

  g_print("GMP3 Version %s\n", VERSION);     
  initializeMixer();
  initializePlaylist();  
  initializeAlbumnList(&albumnList);

  setupRCTokens();
  loadRCFile();
  
  parseThemeRC(&bigWindow,
               &smallWindow);
  activeWindow = &bigWindow;
  gtk_widget_show(activeWindow->main_window);
  loadCommandlineFiles(argc, argv);

  gtk_timeout_add(1000, (GtkFunction) adjustTimer, NULL);
  adjustTimer();
  
  gtk_timeout_add(100, (GtkFunction) playLoop, NULL);    
  
  gtk_main();  
  
  saveRCFile();
  
  destroyPlaylist();
  destroyAlbumnList(albumnList);
  closeMixer();
  
  return 0;
}
