/* This file is part of
 * ====================================================== 
 * 
 *           LyX, The Document Processor
 * 	 
 *	    Copyright 1995 Matthias Ettrich
 *          Copyright 1995-1999 The LyX Team.
 *
 * ====================================================== */

#include <config.h>
#include <time.h>
#include <locale.h>

#include <cstdlib>
#include <cctype>
#include <cstring>

#ifdef __GNUG__
#pragma implementation
#endif

#include "lyxlookup.h"
#include "kbmap.h"
#include "lyxfunc.h"
#include "bufferlist.h"
#include "lyxserver.h"
#include "lyx.h"
#include "intl.h"
#include "lyx_main.h"
#include "lyx_cb.h"
#include "LyXAction.h"
#if 0
#include "insets/insetlatex.h"
#endif
#include "insets/inseturl.h"
#include "insets/insetlatexaccent.h"
#include "insets/insettoc.h"
#include "insets/insetlof.h"
#include "insets/insetloa.h"
#include "insets/insetlot.h"
#include "insets/insetref.h"
#include "insets/insetparent.h"
#include "insets/insetindex.h"
#include "insets/insetinclude.h"
#include "insets/insetbib.h"
#include "mathed/formulamacro.h"
#include "toolbar.h"
#include "spellchecker.h" // RVDK_PATCH_5
#include "minibuffer.h"
#include "vspace.h"
#include "LyXView.h"
#include "filedlg.h"
#include "lyx_gui_misc.h"
#include "support/filetools.h"
#include "support/FileInfo.h"
#include "support/syscall.h"
#include "support/lstrings.h"
#include "support/path.h"
#include "lyxscreen.h"
#include "debug.h"
#include "lyxrc.h"
#include "lyxtext.h"
#include "gettext.h"
#include "trans_mgr.h"
#include "ImportLaTeX.h"
#include "ImportNoweb.h"
#include "layout.h"

extern bool cursor_follows_scrollbar;

extern void InsertAsciiFile(string const &, bool);
extern void math_insert_symbol(char const *);
extern Bool math_insert_greek(char const); // why "Bool"?
extern BufferList bufferlist;
extern LyXServer * lyxserver;
extern short greek_kb_flag;
extern FD_form_toc * fd_form_toc;
extern bool selection_possible;

extern kb_keymap * toplevel_keymap;

extern void MenuWrite(Buffer *);
extern void MenuWriteAs(Buffer *);
extern int  MenuRunLaTeX(Buffer *);
extern int  MenuBuildProg(Buffer *);
extern int  MenuRunChktex(Buffer *);
extern bool CreatePostscript(Buffer *, bool);
extern void MenuPrint(Buffer *);
extern void MenuSendto();
extern void QuitLyX();
extern void MenuFax(Buffer *);
extern void MenuExport(Buffer *, string const &);
extern void MenuPasteSelection(char at);
extern LyXAction lyxaction;
// (alkis)
extern tex_accent_struct get_accent(kb_action action);

extern void AutoSave();
extern void MenuSearch();
extern void SetUpdateTimer(float timer = 0.3);
extern void FreeUpdateTimer();
extern bool PreviewDVI(Buffer *);
extern bool PreviewPostscript(Buffer *);
extern void MenuInsertLabel(char const *);
extern void MenuInsertRef();
extern void MenuLayoutCharacter();
extern void MenuLayoutParagraph();
extern void MenuLayoutDocument();
extern void MenuLayoutPaper();
extern void MenuLayoutTable(int flag);
extern void MenuLayoutQuotes();
extern void MenuLayoutPreamble();
extern void MenuLayoutSave();
extern void bulletForm();

extern Buffer * NewLyxFile(string const &);
extern void LoadLyXFile(string const &);
extern void Reconfigure();

extern int current_layout;
extern int getISOCodeFromLaTeX(char *);

//extern int UnlockInset(UpdatableInset *);

extern void ShowLatexLog();

extern void UpdateInset(Inset * inset, bool mark_dirty = true);

/* === globals =========================================================== */

bool LyXFunc::show_sc = true;


LyXFunc::LyXFunc(LyXView * o)
	: owner(o)
{
	meta_fake_bit = 0;
	lyx_dead_action = LFUN_NOACTION;
	lyx_calling_dead_action = LFUN_NOACTION;
	setupLocalKeymap();
}


// I changed this func slightly. I commented out the ...FinishUndo(),
// this means that all places that used to have a moveCursorUpdate, now
// have a ...FinishUndo() as the preceeding statement. I have also added
// a moveCursorUpdate to some of the functions that updated the cursor, but
// that did not show its new position.
inline
void LyXFunc::moveCursorUpdate(bool selecting)
{
	if (selecting || owner->view()->text->mark_set) {
		owner->view()->text->SetSelection();
		owner->view()->getScreen()->ToggleToggle();
		owner->view()->update(0);
	} else {
		owner->view()->update(-2); // this IS necessary
		// (Matthias) 
	}

	owner->view()->getScreen()->ShowCursor();
	
	/* ---> Everytime the cursor is moved, show the current font state. */
	// should this too me moved out of this func?
	//owner->getMiniBuffer()->Set(CurrentState());
}


int LyXFunc::processKeyEvent(XEvent * ev)
{
	char s_r[10];
	string argument;
	XKeyEvent * keyevent = &ev->xkey;
	KeySym keysym_return;

	int num_bytes = LyXLookupString(ev, s_r, 10, &keysym_return);
	s_r[num_bytes] = '\0';

	if (lyxerr.debugging(Debug::KEY)) {
		char * tmp = XKeysymToString(keysym_return);
		string stm = (tmp ? tmp : "");
		lyxerr << "KeySym is "
		       << stm
		       << "["
		       << keysym_return << "]"
		       << " and num_bytes is "
		       << num_bytes
		       << " the string returned is \""
		       << s_r << '\"' << endl;
	}
	// Do nothing if we have nothing (JMarc)
	if (num_bytes == 0 && keysym_return == NoSymbol) {
		lyxerr[Debug::KEY] << "Empty kbd action (probably composing)"
				   << endl;
		return 0;
	}
	
	// this function should be used always [asierra060396]
	if (owner->view()->available() &&
	    owner->view()->the_locking_inset &&
	    keysym_return == XK_Escape) {
		owner->view()->unlockInset(owner->view()->the_locking_inset);
		owner->view()->text->CursorRight();
		return 0;
	}

	// Can we be sure that this will work for all X-Windows
	// implementations? (Lgb)
	// This code snippet makes lyx ignore some keys. Perhaps
	// all of them should be explictly mentioned?
	if((keysym_return >= XK_Shift_L && keysym_return <= XK_Hyper_R)
	   || keysym_return == XK_Mode_switch || keysym_return == 0x0)
		return 0;

	// Do a one-deep top-level lookup for
	// cancel and meta-fake keys. RVDK_PATCH_5
	cancel_meta_seq.reset();

	int action = cancel_meta_seq.addkey(keysym_return, keyevent->state
					    &(ShiftMask|ControlMask
					      |Mod1Mask)); 

	// When not cancel or meta-fake, do the normal lookup. 
	// Note how the meta_fake Mod1 bit is OR-ed in and reset afterwards.
	// Mostly, meta_fake_bit = 0. RVDK_PATCH_5.
	if ( (action != LFUN_CANCEL) && (action != LFUN_META_FAKE) ) {

		// remove Caps Lock and Mod2 as a modifiers
		action = keyseq.addkey(keysym_return,
				       (keyevent->state | meta_fake_bit)
				       &(ShiftMask|ControlMask
					 |Mod1Mask));      
	}
	// Dont remove this unless you know what you are doing.
	meta_fake_bit = 0;
		
	if (action == 0) action = LFUN_PREFIX;

	if (lyxerr.debugging(Debug::KEY)) {
		string buf;
		keyseq.print(buf);
		lyxerr << "Key ["
		       << action << "]["
		       << buf << "]["
		       << num_bytes << "]" << endl;
	}

	// already here we know if it any point in going further
	// why not return already here if action == -1 and
	// num_bytes == 0? (Lgb)

	if(keyseq.length > 1 || keyseq.length < -1){
		string buf;
		keyseq.print(buf);
		owner->getMiniBuffer()->Set(buf);
	}

	if (action == -1) {
		if (keyseq.length < -1) { // unknown key sequence...
			string buf;
			LyXBell();
			keyseq.print(buf);
			owner->getMiniBuffer()->Set(_("Unknown sequence:"), buf);
			return 0;
		}
	
		char isochar = keyseq.getiso();
		if (!(keyevent->state&ControlMask) &&
		    !(keyevent->state&Mod1Mask) &&
		    (isochar && keysym_return < 0xF000)) {
			argument += isochar;
		}
		if (argument.empty()) {
			lyxerr.debug() << "Empty argument!" << endl;
			// This can`t possibly be of any use
			// so we`ll skip the dispatch.
			return 0;
		}
	} else
		if (action == LFUN_SELFINSERT) {
			argument = s_r[0];
		}
    
        bool tmp_sc = show_sc;
	show_sc = false;
	Dispatch(action, argument.c_str());
	show_sc = tmp_sc;
	
	return 0;
} 


LyXFunc::func_status LyXFunc::getStatus(int ac) const
{
        kb_action action;
        func_status flag = LyXFunc::OK;
        string argument;
        Buffer * buf = owner->buffer();
	
 	if (lyxaction.isPseudoAction(ac)) 
		action = lyxaction.retrieveActionArg(ac, argument);
	else 
		action = static_cast<kb_action>(ac);
	
	if (action == LFUN_UNKNOWN_ACTION) {
		setErrorMessage(N_("Unknown action"));
		return LyXFunc::Unknown;
	} 
	
	// Check whether we need a buffer
	if (!lyxaction.funcHasFlag(action, LyXAction::NoBuffer)) {
		// Yes we need a buffer, do we have one?
		if (buf) {
			// yes
			// Can we use a readonly buffer?
			if (buf->isReadonly() && 
			    !lyxaction.funcHasFlag(action,
						   LyXAction::ReadOnly)) {
				// no
				setErrorMessage(N_("Document is read-only"));
				flag |= LyXFunc::Disabled;
			}
		} else {
			// no
			setErrorMessage(N_("Command not allowed with"
					   "out any document open"));
			flag |= LyXFunc::Disabled;
		}
	}

	if (flag & LyXFunc::Disabled)
		return flag;

	// I would really like to avoid having this switch and rather try to
	// encode this in the function itself.
        static bool noLaTeX = lyxrc->latex_command == "none";
        bool disable = false;
        switch (action) {
	case LFUN_PREVIEW:
		disable = noLaTeX || lyxrc->view_dvi_command == "none";
		break;	
	case LFUN_PREVIEWPS: 
		disable = noLaTeX || lyxrc->view_ps_command == "none";
		break;
	case LFUN_RUNLATEX:
	case LFUN_RUNDVIPS:
		disable = noLaTeX;
		break;
	case LFUN_MENUPRINT:
		disable = noLaTeX || lyxrc->print_command == "none";
		break;
	case LFUN_FAX:
		disable = noLaTeX || lyxrc->fax_command == "none"; 
		break;
	case LFUN_IMPORT:
		if (argument == "latex")
			disable = lyxrc->relyx_command == "none";
		break;
	case LFUN_EXPORT:
		if (argument == "dvi" || argument == "postscript")
			disable = noLaTeX;
		if (argument == "html")
			disable = lyxrc->html_command == "none";
		break;
	case LFUN_UNDO:
		disable = buf->undostack.empty();
		break;
	case LFUN_REDO:
		disable = buf->redostack.empty();
		break;
	case LFUN_SPELLCHECK:
		disable = lyxrc->isp_command == "none";
		break;
	case LFUN_RUNCHKTEX:
		disable = lyxrc->chktex_command == "none";
		break;
	case LFUN_LAYOUT_TABLE:
		disable = ! owner->view()->text->cursor.par->table;
		break;
	default:
		break;
        }
        if (disable)
                flag |= LyXFunc::Disabled;

	if (buf) {
		func_status box = LyXFunc::ToggleOff;
		LyXFont font = owner->view()->text->real_current_font;
		switch (action) {
		case LFUN_EMPH:
			if (font.emph() == LyXFont::ON)
				box = LyXFunc::ToggleOn;
			break;
		case LFUN_NOUN:
			if (font.noun() == LyXFont::ON)
				box = LyXFunc::ToggleOn;
			break;
		case LFUN_BOLD:
			if (font.series() == LyXFont::BOLD_SERIES)
				box = LyXFunc::ToggleOn;
			break;
		case LFUN_TEX:
			if (font.latex() == LyXFont::ON)
				box = LyXFunc::ToggleOn;
			break;
		default:
			box = LyXFunc::OK;
			break;
		}
		flag |= box;
	}

	return flag;
}


string LyXFunc::Dispatch(string const & s) 
{
	// Split command string into command and argument
	string cmd, line = frontStrip(s);
	string arg = strip(frontStrip(split(line, cmd, ' ')));

	return Dispatch(lyxaction.LookupFunc(cmd.c_str()), arg.c_str());
}


string LyXFunc::Dispatch(int ac,
			 char const * do_not_use_this_arg)
{
	string argument;
	kb_action action;
        
	FL_OBJECT * ob = 0;  // This will disapear soon
    
        // we have not done anything wrong yet.
        errorstat = false;
	dispatch_buffer.clear();
	
	// if action is a pseudo-action, we need the real action
	if (lyxaction.isPseudoAction(ac)) {
		string tmparg;
		action = static_cast<kb_action>
			(lyxaction.retrieveActionArg(ac, tmparg));
		if (!tmparg.empty())
			argument = tmparg;
	} else {
		action = static_cast<kb_action>(ac);
		if (do_not_use_this_arg)
			argument = do_not_use_this_arg; // except here
	}
    
	selection_possible = false;
	
	if (owner->view()->available() 
	    && owner->view()->getScreen())
		owner->view()->getScreen()->HideCursor();

	// We cannot use this function here
	if (getStatus(action) & Disabled)
		goto exit_with_message;

	commandshortcut.clear();
	
	if (lyxrc->display_shortcuts && show_sc) {
		if (action != LFUN_SELFINSERT) {
			// Put name of command and list of shortcuts
			// for it in minibuffer
			string comname = lyxaction.getActionName(action);

			int pseudoaction = action;
			bool argsadded = false;

			if (!argument.empty()) {
				// If we have the command with argument, 
				// this is better
				pseudoaction = 
					lyxaction.searchActionArg(action,
							  	  argument.c_str());

				if (pseudoaction == -1) {
					pseudoaction = action;
				} else {
					comname += " " + argument;
					argsadded = true;
				}
			}

			string shortcuts = toplevel_keymap->findbinding(pseudoaction);

			if (!shortcuts.empty()) {
				comname += ": " + shortcuts;
			} else if (!argsadded) {
				comname += " " + argument;
			}

			if (!comname.empty()) {
				comname = strip(comname);
				commandshortcut = "(" + comname + ')';
				owner->getMiniBuffer()->Set(commandshortcut);
				// Here we could even add a small pause,
				// to annoy the user and make him learn
				// the shortcuts.
				// No! That will just annoy, not teach
				// anything. The user will read the messages
				// if they are interested. (Asger)
			}
		}
        }

	// If in math mode pass the control to
	// the math inset [asierra060396]
	if (owner->view()->available() &&
	    owner->view()->the_locking_inset) {
		if (action > 1
		    || (action == LFUN_UNKNOWN_ACTION
			&& keyseq.length >= -1)) {
			if (action == LFUN_UNKNOWN_ACTION
			    && argument.empty()) {
				argument = keyseq.getiso();
			}
			// Undo/Redo pre 0.13 is a bit tricky for insets.
		        if (action == LFUN_UNDO) {
				int slx, sly;
				UpdatableInset * inset = 
					owner->view()->the_locking_inset;
				inset->GetCursorPos(slx, sly);
				owner->view()->unlockInset(inset);
				owner->view()->menuUndo();
				inset = static_cast<UpdatableInset*>(owner->view()->text->cursor.par->GetInset(owner->view()->text->cursor.pos));
				if (inset) 
					inset->Edit(slx, sly);
				return string();
			} else 
				if (action == LFUN_REDO) {
					int slx, sly;
					UpdatableInset * inset = owner->view()->the_locking_inset;
					inset->GetCursorPos(slx, sly);
					owner->view()->unlockInset(inset);
					owner->view()->menuRedo();
					inset = static_cast<UpdatableInset*>(owner->view()->text->cursor.par->GetInset(owner->view()->text->cursor.pos));
					if (inset)
						inset->Edit(slx, sly);
					return string();
				} else
					if (owner->view()->the_locking_inset->LocalDispatch(action, argument.c_str()))
						return string();
					else {
						setMessage(N_("Text mode"));
						if (action == LFUN_RIGHT || action == -1)
							owner->view()->text->CursorRight();
						if (action == LFUN_LEFT || action == LFUN_RIGHT)
							return string();
					}
		}
	}

	switch(action) {
		// --- Misc -------------------------------------------
	case LFUN_WORDFINDFORWARD  : 
	case LFUN_WORDFINDBACKWARD : {
		static string last_search;
		string searched_string;
	    
		if (!argument.empty()) {
			last_search = argument;
			searched_string = argument;
		} else {
			searched_string = last_search;
		}

		LyXText * ltCur = owner->view()->text ;

		if (!searched_string.empty() &&
		    ((action == LFUN_WORDFINDBACKWARD) ? 
		     ltCur->SearchBackward(searched_string.c_str()) :
		     ltCur->SearchForward(searched_string.c_str()))) {

			// ??? What is that ???
			owner->view()->update(-2);

			// ??? Needed ???
			// clear the selection (if there is any) 
			owner->view()->getScreen()->ToggleSelection();
			owner->view()->text->ClearSelection();

			// Move cursor so that successive C-s 's will not stand in place. 
			if( action == LFUN_WORDFINDFORWARD ) 
				owner->view()->text->CursorRightOneWord();
			owner->view()->text->FinishUndo();
			moveCursorUpdate(false);

			// ??? Needed ???
			// set the new selection 
			// SetSelectionOverLenChars(owner->view()->currentBuffer()->text, iLenSelected);
			owner->view()->getScreen()->ToggleSelection(false);
		} else 
			LyXBell();	
	 
		// REMOVED : if (owner->view()->getWorkArea()->focus)
		owner->view()->getScreen()->ShowCursor();
	}
	break;

	case LFUN_PREFIX:
	{
		if (owner->view()->available()
		    && owner->view()->getScreen()) {
			owner->view()->update(-2);
		}
		string buf;
		keyseq.print(buf, true);
		owner->getMiniBuffer()->Set(buf, string(), string(), 1);
	}
	break;

	// --- Misc -------------------------------------------
	case LFUN_EXEC_COMMAND:
		owner->getMiniBuffer()->ExecCommand(); 
		break;
		
	case LFUN_CANCEL:                   // RVDK_PATCH_5
		keyseq.reset();
		meta_fake_bit = 0;
		if(owner->view()->available())
			// cancel any selection
			Dispatch(LFUN_MARK_OFF, 0);
		setMessage(N_("Cancel"));
		break;

	case LFUN_META_FAKE:                                 // RVDK_PATCH_5
	{
		meta_fake_bit = Mod1Mask;
		string buf;
		keyseq.print(buf, true);
		string res = string("M-") + buf;
		setMessage(buf); // RVDK_PATCH_5
	}
	break;  

	case LFUN_READ_ONLY_TOGGLE:
		if (owner->buffer()->lyxvc.inUse()) {
			owner->buffer()->lyxvc.toggleReadOnly();
		} else {
			owner->buffer()->setReadonly(
				!owner->buffer()->isReadonly());
		}
		break;
		
	case LFUN_CENTER: // this is center and redraw.
		owner->view()->beforeChange();
		if (owner->view()->text->cursor.y >
		    owner->view()->getWorkArea()->h / 2)	{
			owner->view()->getScreen()->
				Draw(owner->view()->text->cursor.y -
				     owner->view()->getWorkArea()->h / 2);
		} else { // <= 
			owner->view()->getScreen()->
				Draw(0);
		}
		owner->view()->update(0);
		owner->view()->redraw();
		break;
		
	case LFUN_APPENDIX:
		if (owner->view()->available()) {
			owner->view()->text->toggleAppendix();
			owner->view()->update(1);
		}
		break;

		// --- Menus -----------------------------------------------
	case LFUN_MENUNEW:
		MenuNew(false);
		break;
		
	case LFUN_MENUNEWTMPLT:
		MenuNew(true);
		break;
		
	case LFUN_MENUOPEN:
		MenuOpen();
		break;
		
	case LFUN_CLOSEBUFFER:
		CloseBuffer();
		break;
		
	case LFUN_MENUWRITE:
		MenuWrite(owner->buffer());
		break;
		
	case LFUN_MENUWRITEAS:
		MenuWriteAs(owner->buffer());
		break;
		
	case LFUN_MENURELOAD:
		reloadBuffer();
		break;
		
	case LFUN_PREVIEW:
		PreviewDVI(owner->buffer());
		break;
			
	case LFUN_PREVIEWPS:
		PreviewPostscript(owner->buffer());
		break;
		
	case LFUN_RUNLATEX:
		MenuRunLaTeX(owner->buffer());
		break;
		
        case LFUN_BUILDPROG:
                MenuBuildProg(owner->buffer());
                break;
                
 	case LFUN_RUNCHKTEX:
		MenuRunChktex(owner->buffer());
		break;
		
	case LFUN_RUNDVIPS:
		CreatePostscript(owner->buffer(), false);
		break;
		
	case LFUN_MENUPRINT:
		MenuPrint(owner->buffer());
		break;
		
	case LFUN_FAX:
		MenuFax(owner->buffer());
		break;
			
	case LFUN_EXPORT:
		MenuExport(owner->buffer(), argument);
		break;

	case LFUN_IMPORT:
	{
		//needs argument as string
		string imtyp = argument;
		
		// latex
		if (imtyp == "latex") {
			doImportLaTeX(false);
		}
		// ascii
		else if (imtyp == "ascii") {
			doImportASCII(false);
		} else if (imtyp == "asciiparagraph") {
			doImportASCII(true);
			// noweb
		} else if (imtyp == "noweb") {
			doImportLaTeX(true);
		} else {
			setErrorMessage(string(N_("Unknown import type: "))
					+ imtyp);
		}
		break;
	}
		
	case LFUN_QUIT:
		QuitLyX();
		break;
		
	case LFUN_TOCVIEW:
		TocUpdateCB(ob, 0);
		if (fd_form_toc->form_toc->visible) {
			fl_raise_form(fd_form_toc->form_toc);
		} else {
			static int ow = -1, oh;
			fl_show_form(fd_form_toc->form_toc,
				     FL_PLACE_MOUSE |
				     FL_FREE_SIZE, FL_FULLBORDER,
				     _("Table of Contents"));
			if (ow < 0) {
				ow = fd_form_toc->form_toc->w;
				oh = fd_form_toc->form_toc->h;
			}
			fl_set_form_minsize(fd_form_toc->form_toc, ow, oh);
		}
		break;
		
	case LFUN_TOC_INSERT:
	{
		Inset * new_inset = new InsetTOC(owner->buffer());
		owner->view()->insertInset(new_inset, "Standard", true);
		break;
	}
	
	case LFUN_LOF_INSERT:
	{
		Inset * new_inset = new InsetLOF(owner->buffer());
		owner->view()->insertInset(new_inset, "Standard", true);
		break;
	}
	
	case LFUN_LOA_INSERT:
	{
		Inset * new_inset = new InsetLOA(owner->buffer());
		owner->view()->insertInset(new_inset, "Standard", true);
		break;
	}

	case LFUN_LOT_INSERT:
	{
		Inset * new_inset = new InsetLOT(owner->buffer());
		owner->view()->insertInset(new_inset, "Standard", true);
		break;
	}
		
	case LFUN_TABLE:
		TableCB(ob, 0);
		break;
		
	case LFUN_FIGURE:
		FigureCB(ob, 0);
		break;
		
	case LFUN_AUTOSAVE:
		AutoSave();
		break;
		
	case LFUN_UNDO:
		owner->view()->menuUndo();
		break;
		
	case LFUN_REDO:
		owner->view()->menuRedo();
		break;
		
	case LFUN_MENUSEARCH:
		MenuSearch();
		break;
		
	case LFUN_PASTE:
		owner->view()->paste();
		break;
		
	case LFUN_PASTESELECTION:
	{
	        bool asPara = false;
		if (argument == "paragraph") asPara = true;
		MenuPasteSelection(asPara);
		break;
	}

	case LFUN_CUT:
		owner->view()->cut();
		break;
		
	case LFUN_COPY:
		owner->view()->copy();
		break;
		
	case LFUN_LAYOUT_COPY:
		owner->view()->copyEnvironment();
		break;
		
	case LFUN_LAYOUT_PASTE:
		owner->view()->pasteEnvironment();
		break;
		
	case LFUN_GOTOERROR:
		owner->view()->gotoError();
		break;
		
	case LFUN_REMOVEERRORS:
		if (owner->view()->removeAutoInsets()) {
			owner->view()->redraw();
			owner->view()->fitCursor();
			owner->view()->updateScrollbar();
		}
		break;
		
	case LFUN_GOTONOTE:
		owner->view()->gotoNote();
		break;
		
	case LFUN_OPENSTUFF:
		owner->view()->openStuff();
		break;
		
	case LFUN_HYPHENATION:
		owner->view()->hyphenationPoint();
		break;
		
	case LFUN_LDOTS:
		owner->view()->ldots();
		break;
		
	case LFUN_END_OF_SENTENCE:
		owner->view()->endOfSentenceDot();
		break;

	case LFUN_MENU_SEPARATOR:
		owner->view()->menuSeparator();
		break;
		
	case LFUN_HFILL:
		owner->view()->hfill();
		break;
		
	case LFUN_DEPTH:
		DepthCB(ob, 0);
		break;
		
	case LFUN_DEPTH_MIN:
		DepthCB(ob, -1);
		break;
		
	case LFUN_DEPTH_PLUS:
		DepthCB(ob, 1);
		break;
		
	case LFUN_FREE:
		FreeCB();
		break;
		
	case LFUN_TEX:
		TexCB();
		break;
		
	case LFUN_MELT:
		MeltCB(ob, 0);
		break;
		
	case LFUN_RECONFIGURE:
		Reconfigure();
		break;

	case LFUN_FOOTMELT:
		if (owner->view()->available()
		    && !owner->view()->text->selection
		    && owner->view()->text->cursor.par->footnoteflag
		    != LyXParagraph::NO_FOOTNOTE)
			{ // only melt footnotes with FOOTMELT, not margins etc
				if(owner->view()->text->cursor.par->footnotekind == LyXParagraph::FOOTNOTE)
					MeltCB(ob, 0);
			}
		else
			FootCB(ob, 0); 
		break;

	case LFUN_MARGINMELT:
		if (owner->view()->available()
		    && !owner->view()->text->selection
		    && owner->view()->text->cursor.par->footnoteflag
		    != LyXParagraph::NO_FOOTNOTE) {
			// only melt margins
			if(owner->view()->text->cursor.par->footnotekind == LyXParagraph::MARGIN)
				MeltCB(ob, 0);
		}
		else
			MarginCB(ob, 0); 
		break;
		
		// --- version control -------------------------------
	case LFUN_VC_REGISTER:
	{
		if (!owner->buffer()->lyxvc.inUse())
			owner->buffer()->lyxvc.registrer();
	}
	break;
		
	case LFUN_VC_CHECKIN:
	{
		if (owner->buffer()->lyxvc.inUse()
		    && !owner->buffer()->isReadonly())
			owner->buffer()->lyxvc.checkIn();
	}
	break;
		
	case LFUN_VC_CHECKOUT:
	{
		if (owner->buffer()->lyxvc.inUse()
		    && owner->buffer()->isReadonly())
			owner->buffer()->lyxvc.checkOut();
	}
	break;
	
	case LFUN_VC_REVERT:
	{
		owner->buffer()->lyxvc.revert();
	}
	break;
		
	case LFUN_VC_UNDO:
	{
		owner->buffer()->lyxvc.undoLast();
	}
	break;
		
	case LFUN_VC_HISTORY:
	{
		owner->buffer()->lyxvc.showLog();
		break;
	}
	
	// --- buffers ----------------------------------------
	case LFUN_PREVBUFFER:
#ifdef WITH_WARNINGS
#warning fix this please
#endif
		// it is the LyXView or the BufferView that should
		// remember the previous buffer, not bufferlist.
// 			if (owner->view()->available()){	  
// 				owner->view()->beforeChange();
// 				owner->buffer()->update(-2);
// 			}
// 			owner->view()->setBuffer(bufferlist.prev());

// 			owner->view()->
// 				resizeCurrentBufferPseudoExpose();
		break;
			
	case LFUN_FILE_INSERT:
	{
		MenuInsertLyXFile(argument);
	}
	break;
	
	case LFUN_FILE_INSERT_ASCII:
	{
		bool asPara = (argument == "paragraph");
		InsertAsciiFile(string(), asPara);
	}
	break;
	
	case LFUN_FILE_NEW:
	{
		// servercmd: argument must be <file>:<template>
		Buffer * tmpbuf = NewLyxFile(argument);
		if (tmpbuf)
			owner->view()->buffer(tmpbuf);
	}
	break;
			
	case LFUN_FILE_OPEN:
		owner->view()->buffer(bufferlist.loadLyXFile(argument));
		break;

	case LFUN_LATEX_LOG:
		ShowLatexLog();
		break;
		
	case LFUN_LAYOUTNO:
	{
		lyxerr.debug() << "LFUN_LAYOUTNO: (arg) " << argument << endl;
		int sel = strToInt(argument);
		lyxerr.debug() << "LFUN_LAYOUTNO: (sel) "<< sel << endl;
		
		// Should this give a setMessage instead?
		if (sel == 0) 
			return string(); // illegal argument

		--sel; // sel 1..., but layout 0...

		// Pretend we got the name instead.
		Dispatch(int(LFUN_LAYOUT), 
			 textclasslist.NameOfLayout(owner->view()->
						    text->parameters->
						    textclass,
						    sel).c_str());
		return string();
	}
		
	case LFUN_LAYOUT:
	{
		lyxerr.debug() << "LFUN_LAYOUT: (arg) "
			       << argument << endl;
		
		// Derive layout number from given argument (string)
		// and current buffer's textclass (number). */    
		int layoutno = 
			textclasslist.NumberOfLayout(owner->
						     view()->
						     text->parameters->
						     textclass,
						     argument).second;

		// see if we found the layout number:
		if (layoutno == -1) {
			setErrorMessage(string(N_("Layout ")) + argument + 
					N_(" not known"));
			break;
		}
			
		if (current_layout != layoutno) {
			owner->view()->getScreen()->HideCursor();
			current_layout = layoutno;
			owner->view()->update(-2);
			owner->view()->text->
				SetLayout(layoutno);
			owner->getToolbar()->combox->
				select(owner->view()->
				       text->cursor.par->
				       GetLayout() + 1);
			owner->view()->update(1);
		}
	}
	break;

	case LFUN_LAYOUT_DOCUMENT:
		MenuLayoutDocument();
		break;
		
	case LFUN_LAYOUT_PARAGRAPH:
		MenuLayoutParagraph();
		break;
		
	case LFUN_LAYOUT_CHARACTER:
		MenuLayoutCharacter();
		break;
		
	case LFUN_LAYOUT_TABLE:
	{
	        int flag = 0;
	        if (argument == "true") flag = 1;
		MenuLayoutTable(flag);
	}
	break;
		
	case LFUN_LAYOUT_PAPER:
		MenuLayoutPaper();
		break;
		
	case LFUN_LAYOUT_QUOTES:
		MenuLayoutQuotes();
		break;
		
	case LFUN_LAYOUT_PREAMBLE:
		MenuLayoutPreamble();
		break;
		
	case LFUN_LAYOUT_SAVE_DEFAULT:
		MenuLayoutSave();
		break;
		
	case LFUN_DROP_LAYOUTS_CHOICE:
		owner->getToolbar()->combox->Show();
		break;

	case LFUN_EMPH:
		EmphCB();
		break;
		
	case LFUN_BOLD:
		BoldCB();
		break;
		
	case LFUN_NOUN:
		NounCB();
		break;
		
	case LFUN_CODE:
		CodeCB();
		break;
		
	case LFUN_SANS:
		SansCB();
		break;
		
	case LFUN_ROMAN:
		RomanCB();
		break;
		
	case LFUN_DEFAULT:
		StyleResetCB();
		break;
		
	case LFUN_UNDERLINE:
		UnderlineCB();
		break;
		
	case LFUN_FONT_SIZE:
		FontSizeCB(argument);
		break;
		
	case LFUN_FONT_STATE:
		setMessage(CurrentState());
		break;
		
	case LFUN_UPCASE_WORD:
		owner->view()->update(-2);
		FreeUpdateTimer();
		owner->view()->text->ChangeWordCase(LyXText::text_uppercase);
		owner->view()->update(1);
		SetUpdateTimer();
		break;
		
	case LFUN_LOWCASE_WORD:
		owner->view()->update(-2);
		FreeUpdateTimer();
		owner->view()->text->ChangeWordCase(LyXText::text_lowercase);
		owner->view()->update(1);
		SetUpdateTimer();
		break;
		
	case LFUN_CAPITALIZE_WORD:
		owner->view()->update(-2);
		FreeUpdateTimer();
		owner->view()->text->ChangeWordCase(LyXText::text_capitalization);
		owner->view()->update(1);
		SetUpdateTimer();
		break;
		
	case LFUN_INSERT_LABEL:
		MenuInsertLabel(argument.c_str());
		break;
		
	case LFUN_INSERT_REF:
		MenuInsertRef();
		break;
		
	case LFUN_REFTOGGLE:
	{
		InsetRef * inset = 
			static_cast<InsetRef*>(getInsetByCode(Inset::REF_CODE));
		if (inset) {
			if (inset->getFlag() == InsetRef::REF)
				inset->setFlag(InsetRef::PAGE_REF);
			else
				inset->setFlag(InsetRef::REF);
			UpdateInset(inset);
		} else {
			setErrorMessage(N_("No cross-reference to toggle"));
		}
	}
	break;
	
	case LFUN_REFBACK:
	{
		owner->view()->restorePosition();
	}
	break;

	case LFUN_REFGOTO:
	{
		string label(argument);
		if (label.empty()) {
			InsetRef * inset = 
				static_cast<InsetRef*>(getInsetByCode(Inset::REF_CODE));
			if (inset)
                                label = inset->getContents();
		}
		
		if (!label.empty()) {
			owner->view()->savePosition();
			owner->view()->gotoLabel(label.c_str());
		}
	}
	break;
		
	case LFUN_MENU_OPEN_BY_NAME:
		owner->getMenus()->openByName(argument);
		break; // RVDK_PATCH_5
		
	case LFUN_SPELLCHECK:
		if (lyxrc->isp_command != "none")
			ShowSpellChecker();
		break; // RVDK_PATCH_5
		
		// --- Cursor Movements -----------------------------
	case LFUN_RIGHT:
	{
		LyXText * tmptext = owner->view()->text;
		if(!tmptext->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		if (tmptext->cursor.pos < tmptext->cursor.par->Last()
		    && tmptext->cursor.par->GetChar(tmptext->cursor.pos)
		    == LyXParagraph::META_INSET
		    && tmptext->cursor.par->GetInset(tmptext->cursor.pos)
		    && tmptext->cursor.par->GetInset(tmptext->cursor.pos)->Editable() == 2){
			Inset * tmpinset = tmptext->cursor.par->GetInset(tmptext->cursor.pos);
			setMessage(tmpinset->EditMessage());
			tmpinset->Edit(0, 0);
			break;
		}
		tmptext->CursorRight();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
	}
	break;
		
	case LFUN_LEFT:
	{
		// This is soooo ugly. Isn`t it possible to make
		// it simpler? (Lgb)
		LyXText * txt = owner->view()->text;
		if(!txt->mark_set) owner->view()->beforeChange();
		owner->view()->update(-2);
		txt->CursorLeft();
		if (txt->cursor.pos < txt->cursor.par->Last()
		    && txt->cursor.par->GetChar(txt->cursor.pos)
		    == LyXParagraph::META_INSET
		    && txt->cursor.par->GetInset(txt->cursor.pos)
		    && txt->cursor.par->GetInset(txt->cursor.pos)->Editable() == 2) {
			Inset * tmpinset = txt->cursor.par->GetInset(txt->cursor.pos);
			setMessage(tmpinset->EditMessage());
			tmpinset->Edit(tmpinset->Width(txt->GetFont(txt->cursor.par,
								    txt->cursor.pos)), 0);
			break;
		}
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
	}
	break;
		
	case LFUN_UP:
		if(!owner->view()->text->mark_set) owner->view()->beforeChange();
		owner->view()->update(-3);
		owner->view()->text->CursorUp();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_DOWN:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-3);
		owner->view()->text->CursorDown();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;

	case LFUN_UP_PARAGRAPH:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-3);
		owner->view()->text->CursorUpParagraph();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_DOWN_PARAGRAPH:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-3);
		owner->view()->text->CursorDownParagraph();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_PRIOR:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-3);
		owner->view()->cursorPrevious();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_NEXT:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-3);
		owner->view()->cursorNext();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_HOME:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		owner->view()->text->CursorHome();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_END:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		owner->view()->text->CursorEnd();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_TAB:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		owner->view()->text->CursorTab();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_WORDRIGHT:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		owner->view()->text->CursorRightOneWord();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_WORDLEFT:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		owner->view()->text->CursorLeftOneWord();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_BEGINNINGBUF:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		owner->view()->text->CursorTop();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_ENDBUF:
		if(!owner->view()->text->mark_set)
			owner->view()->beforeChange();
		owner->view()->update(-2);
		owner->view()->text->CursorBottom();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(false);
		owner->getMiniBuffer()->Set(CurrentState());
		break;

      
		/* cursor selection ---------------------------- */
	case LFUN_RIGHTSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorRight();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_LEFTSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorLeft();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_UPSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorUp();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_DOWNSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorDown();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;

	case LFUN_UP_PARAGRAPHSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorUpParagraph();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_DOWN_PARAGRAPHSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorDownParagraph();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_PRIORSEL:
		owner->view()->update(-2);
		owner->view()->cursorPrevious();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_NEXTSEL:
		owner->view()->update(-2);
		owner->view()->cursorNext();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_HOMESEL:
		owner->view()->update(-2);
		owner->view()->text->CursorHome();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_ENDSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorEnd();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_WORDRIGHTSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorRightOneWord();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_WORDLEFTSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorLeftOneWord();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_BEGINNINGBUFSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorTop();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;
		
	case LFUN_ENDBUFSEL:
		owner->view()->update(-2);
		owner->view()->text->CursorBottom();
		owner->view()->text->FinishUndo();
		moveCursorUpdate(true);
		owner->getMiniBuffer()->Set(CurrentState());
		break;

		// --- text changing commands ------------------------
	case LFUN_BREAKLINE:
		owner->view()->beforeChange();
		owner->view()->text->InsertChar(LyXParagraph::META_NEWLINE);
		owner->view()->smallUpdate(1);
		SetUpdateTimer(0.01);
		moveCursorUpdate(false);
		break;
		
	case LFUN_PROTECTEDSPACE:
		owner->view()->beforeChange();
		owner->view()->text->
			InsertChar(LyXParagraph::META_PROTECTED_SEPARATOR);
		owner->view()->smallUpdate(1);
		SetUpdateTimer();
                moveCursorUpdate(false);
		break;
		
	case LFUN_SETMARK:
		if(owner->view()->text->mark_set) {
			owner->view()->beforeChange();
			owner->view()->update(0);
			setMessage(N_("Mark removed"));
		} else {
			owner->view()->beforeChange();
			owner->view()->text->mark_set = 1;
			owner->view()->update(0);
			setMessage(N_("Mark set"));
		}
		owner->view()->text->sel_cursor = 
			owner->view()->text->cursor;
		break;
		
	case LFUN_DELETE:
		FreeUpdateTimer();
		if (!owner->view()->text->selection) {
			owner->view()->text->Delete();
			owner->view()->text->sel_cursor = 
				owner->view()->text->cursor;
			owner->view()->smallUpdate(1);
			// It is possible to make it a lot faster still
			// just comment out the lone below...
			owner->view()->getScreen()->ShowCursor();
		} else {
			owner->view()->cut();
		}
		SetUpdateTimer();
		break;

	case LFUN_DELETE_SKIP:
	{
		// Reverse the effect of LFUN_BREAKPARAGRAPH_SKIP.
		
		LyXCursor cursor = owner->view()->text->cursor;

		FreeUpdateTimer();
		if (!owner->view()->text->selection) {
			if (cursor.pos == cursor.par->Last()) {
				owner->view()->text->CursorRight();
				cursor = owner->view()->text->cursor;
				if (cursor.pos == 0
				    && !(cursor.par->added_space_top 
					 == VSpace (VSpace::NONE))) {
					owner->view()->text->SetParagraph
						(cursor.par->line_top,
						 cursor.par->line_bottom,
						 cursor.par->pagebreak_top, 
						 cursor.par->pagebreak_bottom,
						 VSpace(VSpace::NONE), 
						 cursor.par->added_space_bottom,
						 cursor.par->align, 
						 cursor.par->labelwidthstring, 0);
					owner->view()->text->CursorLeft();
					owner->view()->update (1);
				} else {
					owner->view()->text->CursorLeft();
					owner->view()->text->Delete();
					owner->view()->text->sel_cursor = 
						owner->view()->text->cursor;
					owner->view()->smallUpdate(1);
				}
			} else {
				owner->view()->text->Delete();
				owner->view()->text->sel_cursor = 
					owner->view()->text->cursor;
				owner->view()->smallUpdate(1);
			}
		} else {
			owner->view()->cut();
		}
		SetUpdateTimer();
	}
	break;

	/* -------> Delete word forward. */
	case LFUN_DELETE_WORD_FORWARD:
		owner->view()->update(-2);
		FreeUpdateTimer();
		owner->view()->text->DeleteWordForward();
		owner->view()->update( 1 );
		SetUpdateTimer();
		moveCursorUpdate(false);
		break;

		/* -------> Delete word backward. */
	case LFUN_DELETE_WORD_BACKWARD:
		owner->view()->update(-2);
		FreeUpdateTimer();
		owner->view()->text->DeleteWordBackward();
		owner->view()->update( 1 );
		SetUpdateTimer();
		moveCursorUpdate(false);
		break;
		
		/* -------> Kill to end of line. */
	case LFUN_DELETE_LINE_FORWARD:
		FreeUpdateTimer();
		owner->view()->update(-2);
		owner->view()->text->DeleteLineForward();
		owner->view()->update( 1 );
		SetUpdateTimer();
		moveCursorUpdate(false);
		break;
		
		/* -------> Set mark off. */
	case LFUN_MARK_OFF:
		owner->view()->beforeChange();
		owner->view()->update(0);
		owner->view()->text->sel_cursor = 
			owner->view()->text->cursor;
		setMessage(N_("Mark off"));
		break;

		/* -------> Set mark on. */
	case LFUN_MARK_ON:
		owner->view()->beforeChange();
		owner->view()->text->mark_set = 1;
		owner->view()->update( 0 );
		owner->view()->text->sel_cursor = 
			owner->view()->text->cursor;
		setMessage(N_("Mark on"));
		break;
		
	case LFUN_BACKSPACE:
	{
		FreeUpdateTimer();
		if (!owner->view()->text->selection) {
			if (owner->getIntl()->getTrans()->backspace()) {
				owner->view()->text->Backspace();
				owner->view()->text->sel_cursor = 
					owner->view()->text->cursor;
				owner->view()->smallUpdate(1);
				// It is possible to make it a lot faster still
				// just comment out the lone below...
				owner->view()->getScreen()->ShowCursor();
			}
		} else {
			owner->view()->cut();
		}
		SetUpdateTimer();
	}
	break;

	case LFUN_BACKSPACE_SKIP:
	{
		// Reverse the effect of LFUN_BREAKPARAGRAPH_SKIP.
		
		LyXCursor cursor = owner->view()->text->cursor;
		
		FreeUpdateTimer();
		if (!owner->view()->text->selection) {
			if (cursor.pos == 0 
			    && !(cursor.par->added_space_top 
				 == VSpace (VSpace::NONE))) {
				owner->view()->text->SetParagraph 
					(cursor.par->line_top,      
					 cursor.par->line_bottom,
					 cursor.par->pagebreak_top, 
					 cursor.par->pagebreak_bottom,
					 VSpace(VSpace::NONE), cursor.par->added_space_bottom,
					 cursor.par->align, 
					 cursor.par->labelwidthstring, 0);
				owner->view()->update (1);
			} else {
				owner->view()->text->Backspace();
				owner->view()->text->sel_cursor 
					= cursor;
				owner->view()->smallUpdate (1);
			}
		} else
			owner->view()->cut();
		SetUpdateTimer();
	}
	break;

	case LFUN_BREAKPARAGRAPH:
	{
		owner->view()->beforeChange();
		owner->view()->text->BreakParagraph(0);
		owner->view()->smallUpdate(1);
		SetUpdateTimer(0.01);
		owner->view()->text->sel_cursor = 
			owner->view()->text->cursor;
		break;
	}

	case LFUN_BREAKPARAGRAPHKEEPLAYOUT:
	{
		owner->view()->beforeChange();
		owner->view()->text->BreakParagraph(1);
		owner->view()->smallUpdate(1);
		SetUpdateTimer(0.01);
		owner->view()->text->sel_cursor = 
			owner->view()->text->cursor;
		break;
	}
	
	case LFUN_BREAKPARAGRAPH_SKIP:
	{
		// When at the beginning of a paragraph, remove
		// indentation and add a "defskip" at the top.
		// Otherwise, do the same as LFUN_BREAKPARAGRAPH.
		
		LyXCursor cursor = owner->view()->text->cursor;
		
		owner->view()->beforeChange();
		if (cursor.pos == 0) {
			if (cursor.par->added_space_top == VSpace(VSpace::NONE)) {
				owner->view()->text->SetParagraph
					(cursor.par->line_top,      
					 cursor.par->line_bottom,
					 cursor.par->pagebreak_top, 
					 cursor.par->pagebreak_bottom,
					 VSpace(VSpace::DEFSKIP), cursor.par->added_space_bottom,
					 cursor.par->align, 
					 cursor.par->labelwidthstring, 1);
				owner->view()->update(1);
			} 
		}
		else {
			owner->view()->text->BreakParagraph(0);
			owner->view()->smallUpdate(1);
		}
		SetUpdateTimer(0.01);
		owner->view()->text->sel_cursor = cursor;
	}
	break;
	
	case LFUN_QUOTE:
		owner->view()->beforeChange();
		owner->view()->text->InsertChar('\"');  // This " matches the single quote in the code
		owner->view()->smallUpdate(1);
		SetUpdateTimer();
                moveCursorUpdate(false);
		break;

	case LFUN_HTMLURL:
	case LFUN_URL:
	{
		InsetCommand * new_inset;
		if (action == LFUN_HTMLURL)
			new_inset = new InsetUrl("htmlurl", "", "");
		else
			new_inset = new InsetUrl("url", "", "");
		owner->view()->insertInset(new_inset);
		new_inset->Edit(0, 0);
	}
	break;

	// --- lyxserver commands ----------------------------

	case LFUN_CHARATCURSOR:
	{
		LyXParagraph::size_type pos = 
			owner->view()->text->cursor.pos;
		if(pos < owner->view()->text->cursor.par->size())
			dispatch_buffer = owner->view()->text->
				cursor.par->text[pos];
		else
			dispatch_buffer = "EOF";
	}
	break;
	
	case LFUN_GETXY:
		dispatch_buffer = 
			tostr(owner->view()->text->cursor.x) + ' '
			+ tostr(owner->view()->text->cursor.y);
		break;
		
	case LFUN_SETXY:
	{
		int  x;
		long y;
		sscanf(argument.c_str(), " %d %ld", &x, &y);
		owner->view()->text->SetCursorFromCoordinates(x, y);
	}
	break;
	
	case LFUN_GETLAYOUT:
		dispatch_buffer =  
			tostr(owner->view()->text->cursor.par->layout);
		break;
			
	case LFUN_GETFONT:
	{
		LyXFont * font = &(owner->view()->text->current_font);
                if(font->shape() == LyXFont::ITALIC_SHAPE)
			dispatch_buffer = 'E';
                else if(font->shape() == LyXFont::SMALLCAPS_SHAPE)
			dispatch_buffer = 'N';
                else
			dispatch_buffer = '0';

	}
	break;

	case LFUN_GETLATEX:
	{
		LyXFont * font = &(owner->view()->text->current_font);
                if(font->latex() == LyXFont::ON)
			dispatch_buffer = 'L';
                else
			dispatch_buffer = '0';
	}
	break;

	case LFUN_GETNAME:
		setMessage(owner->buffer()->fileName());
		lyxerr.debug() << "FNAME["
			       << owner->buffer()->fileName()
			       << "] " << endl;
		break;
		
	case LFUN_NOTIFY:
	{
		string buf;
		keyseq.print(buf);
		dispatch_buffer = buf;
		lyxserver->notifyClient(dispatch_buffer);
	}
	break;

	case LFUN_GOTOFILEROW:
	{
	        char file_name[100];
		int  row;
		sscanf(argument.c_str(), " %s %d", file_name, &row);

		// Must replace extension of the file to be .lyx and get full path
		string s = ChangeExtension(string(file_name), ".lyx", false);

		// Either change buffer or load the file
		if (bufferlist.exists(s))
		        owner->view()->buffer(bufferlist.getBuffer(s));
		else
		        owner->view()->buffer(bufferlist.loadLyXFile(s));

		// Set the cursor  
		owner->view()->setCursorFromRow(row);

		// Recenter screen
		owner->view()->beforeChange();
		if (owner->view()->text->cursor.y >
		    owner->view()->getWorkArea()->h / 2)	{
			owner->view()->getScreen()->
				Draw(owner->view()->text->cursor.y -
				     owner->view()->getWorkArea()->h/2);
		} else { // <= 
			owner->view()->getScreen()->
				Draw(0);
		}
		owner->view()->update(0);
		owner->view()->redraw();
	}
	break;

	case LFUN_APROPOS:
	case LFUN_GETTIP:
	{
		int qa = lyxaction.LookupFunc(argument.c_str());
		setMessage(lyxaction.helpText(static_cast<kb_action>(qa)));
	}
	break;

	// --- accented characters ---------------------------
		
	case LFUN_UMLAUT:
	case LFUN_CIRCUMFLEX:
	case LFUN_GRAVE:
	case LFUN_ACUTE:
	case LFUN_TILDE:
	case LFUN_CEDILLA:
	case LFUN_MACRON:
	case LFUN_DOT:
	case LFUN_UNDERDOT:
	case LFUN_UNDERBAR:
	case LFUN_CARON:
	case LFUN_SPECIAL_CARON:
	case LFUN_BREVE:
	case LFUN_TIE:
	case LFUN_HUNG_UMLAUT:
	case LFUN_CIRCLE:
	case LFUN_OGONEK:
	{
		char c;
		
		if (keyseq.length == -1 && keyseq.getiso() != 0) 
			c = keyseq.getiso();
		else
			c = 0;
		
		owner->getIntl()->getTrans()->
			deadkey(c, get_accent(action).accent, 
				owner->view()->text);
		
		// Need to reset, in case the minibuffer calls these
		// actions
		keyseq.reset();
		keyseq.length = 0;
		
		// copied verbatim from do_accent_char
		owner->view()->smallUpdate(1);
		SetUpdateTimer();
		owner->view()->text->sel_cursor = 
			owner->view()->text->cursor;
	}   
	break;
	
	// --- toolbar ----------------------------------
	case LFUN_PUSH_TOOLBAR:
	{
		int nth = strToInt(argument);
		if (lyxerr.debugging(Debug::TOOLBAR)) {
			lyxerr << "LFUN_PUSH_TOOLBAR: argument = `"
			       << argument << "'\n"
			       << "LFUN_PUSH_TOOLBAR: nth = `"
			       << nth << "'" << endl;
		}
		
		if (nth <= 0) {
			LyXBell();
			setErrorMessage(N_("Push-toolbar needs argument > 0"));
		} else {
			owner->getToolbar()->push(nth);
		}
	}
	break;
	
	case LFUN_ADD_TO_TOOLBAR:
	{
		if (lyxerr.debugging(Debug::TOOLBAR)) {
			lyxerr << "LFUN_ADD_TO_TOOLBAR:"
				"argument = `" << argument << '\'' << endl;
		}
		string tmp(argument);
		//lyxerr <<string("Argument: ") + argument);
		//lyxerr <<string("Tmp     : ") + tmp);
		if (tmp.empty()) {
			LyXBell();
			setErrorMessage(N_("Usage: toolbar-add-to <LyX command>"));
		} else {
			owner->getToolbar()->add(argument, false);
			owner->getToolbar()->set();
		}
	}
	break;
	
	// --- insert characters ----------------------------------------
#if 0
	case LFUN_INSERT_INSET_LATEX:
	{
		Inset * new_inset = new InsetLatex(argument);
		owner->buffer()->insertInset(new_inset);
	}
	break;
#endif
	// ---  Mathed stuff. If we are here, there is no locked inset yet.
	
	// Greek mode     
	case LFUN_GREEK:
	{
		if (!greek_kb_flag) {
			greek_kb_flag = 1;
			setMessage(N_("Math greek mode on"));
		} else
			greek_kb_flag = 0;
	}  
	break;
      
	// Greek keyboard      
	case LFUN_GREEK_TOGGLE:
	{
		greek_kb_flag = greek_kb_flag ? 0 : 2;
		if (greek_kb_flag) {
			setMessage(N_("Math greek keyboard on"));
		} else {
			setMessage(N_("Math greek keyboard off"));
		}
	}
	break;
	
	case LFUN_MATH_DELIM:     
	case LFUN_INSERT_MATRIX:
	{ 	   
		if (owner->view()->available()) { 
			owner->view()->
				open_new_inset(new InsetFormula(false));
			owner->view()->
				the_locking_inset->LocalDispatch(action, argument.c_str());
		}
	}	   
	break;
	       
	case LFUN_INSERT_MATH:
	{
		math_insert_symbol(argument.c_str());
	}
	break;
	
	case LFUN_MATH_DISPLAY:
	{
		if (owner->view()->available())
			owner->view()->open_new_inset(new InsetFormula(true));
		break;
	}
		    
	case LFUN_MATH_MACRO:
	{
		if (owner->view()->available()) {
			string s(argument);
		        if (s.empty())
				setErrorMessage(N_("Missing argument"));
		        else {
				string s1 = token(s, ' ', 1);
				int na = s1.empty() ? 0: atoi(s1.c_str());
				owner->view()->
					open_new_inset(new InsetFormulaMacro(token(s, ' ', 0), na));
			}
		}
	}
	break;

	case LFUN_MATH_MODE:   // Open or create a math inset
	{
		
		if (owner->view()->available())
			owner->view()->open_new_inset(new InsetFormula);
		setMessage(N_("Math editor mode"));
	}
	break;
	  
	case LFUN_MATH_NUMBER:
	case LFUN_MATH_LIMITS:
	{
		setErrorMessage(N_("This is only allowed in math mode!"));
	}
	break;
	
	case LFUN_INSERT_CITATION:
	{   
		InsetCitation * new_inset = new InsetCitation();
		// ale970405
		// The note, if any, must be after the key, delimited
		// by a | so both key and remark can have spaces.
		if (!argument.empty()) {
			string lsarg(argument);
			if (contains(lsarg, "|")) {
				new_inset->setContents(token(lsarg, '|', 0));
				new_inset->setOptions(token(lsarg, '|', 1));
			} else
				new_inset->setContents(lsarg);
			owner->view()->insertInset(new_inset);
		} else {
			owner->view()->insertInset(new_inset);
			new_inset->Edit(0, 0);
		}
	}
	break;
		    
	case LFUN_INSERT_BIBTEX:
	{   
		// ale970405+lasgoutt970425
		// The argument can be up to two tokens separated 
		// by a space. The first one is the bibstyle.
		string lsarg(argument);
		string bibstyle = token(lsarg, ' ', 1);
		if (bibstyle.empty())
			bibstyle = "plain";
		InsetBibtex * new_inset 
			= new InsetBibtex(token(lsarg, ' ', 0),
					  bibstyle,
					  owner->buffer());
		
		owner->view()->insertInset(new_inset);
		if (lsarg.empty()) {
			new_inset->Edit(0, 0);
		}
	}
	break;
		
	// BibTeX data bases
	case LFUN_BIBDB_ADD:
	{
		InsetBibtex * inset = 
			static_cast<InsetBibtex*>(getInsetByCode(Inset::BIBTEX_CODE));
		if (inset) {
			inset->addDatabase(argument);
		}
	}
	break;
		    
	case LFUN_BIBDB_DEL:
	{
		InsetBibtex * inset = 
			static_cast<InsetBibtex*>(getInsetByCode(Inset::BIBTEX_CODE));
		if (inset) {
			inset->delDatabase(argument);
		}
	}
	break;
	
	case LFUN_BIBTEX_STYLE:
	{
		InsetBibtex * inset = 
			static_cast<InsetBibtex*>(getInsetByCode(Inset::BIBTEX_CODE));
		if (inset) {
			inset->setOptions(argument);
		}
	}
	break;
		
	case LFUN_INDEX_INSERT:
	case LFUN_INDEX_INSERT_LAST:
	{
	  	// Can't do that at the beginning of a paragraph.
	  	if (owner->view()->text->cursor.pos - 1 < 0)
			break;

		InsetIndex * new_inset = new InsetIndex();
		if (!argument.empty()) {
  			string lsarg(argument);
			new_inset->setContents(lsarg);
			owner->view()->insertInset(new_inset);
		} else {
			//reh 98/09/21
			//get the current word for an argument
			LyXParagraph::size_type lastpos = 
				owner->view()->text->cursor.pos - 1;
			// Get the current word. note that this must be done
			// before inserting the inset, or the inset will
			// break the word
			string curstring(owner->view()
					 ->text->cursor.par->GetWord(lastpos));

			//make the new inset and write the current word into it
			InsetIndex * new_inset = new InsetIndex();

			new_inset->setContents(curstring);

			//don't edit it if the call was to INSERT_LAST
			if(action != LFUN_INDEX_INSERT_LAST) {
				new_inset->Edit(0, 0);
			} else {
				//it looks blank on the screen unless
				//we do  something.  put it here.

				// move the cursor to the returned value of lastpos
				// but only for the auto-insert
				owner->view()->text->cursor.pos = lastpos;
			}

			//put the new inset into the buffer.
			// there should be some way of knowing the user
			//cancelled & avoiding this, but i don't know how
			owner->view()->insertInset(new_inset);
		}
	}
	break;

	case LFUN_INDEX_PRINT:
	{
		Inset * new_inset = new InsetPrintIndex(owner->buffer());
		owner->view()->insertInset(new_inset, "Standard", true);
	}
	break;

	case LFUN_PARENTINSERT:
	{
		lyxerr << "arg " << argument << endl;
		Inset * new_inset = new InsetParent(argument, owner->buffer());
		owner->view()->insertInset(new_inset, "Standard", true);
	}
	break;

	case LFUN_CHILDINSERT:
	{
		Inset * new_inset = new InsetInclude(argument,
						     owner->buffer());
		owner->view()->insertInset(new_inset, "Standard", true);
		new_inset->Edit(0, 0);
	}
	break;

	case LFUN_CHILDOPEN:
	{
		string filename =
			MakeAbsPath(argument, 
				    OnlyPath(owner->buffer()->fileName()));
		setMessage(N_("Opening child document ") +
			   MakeDisplayPath(filename) + "...");
		owner->view()->savePosition();
		if (bufferlist.exists(filename))
			owner->view()->buffer(bufferlist.getBuffer(filename));
		else
			owner->view()->buffer(bufferlist.loadLyXFile(filename));
	}
	break;

	case LFUN_INSERT_NOTE:
		owner->view()->insertNote();
		break;
		
	case LFUN_INSERTFOOTNOTE: 
	{
		LyXParagraph::footnote_kind kind;
		if (argument == "footnote")
			{ kind = LyXParagraph::FOOTNOTE; }
		else if (argument == "margin")
			{ kind = LyXParagraph::MARGIN; }
		else if (argument == "figure")
			{ kind = LyXParagraph::FIG; }
		else if (argument == "table")
			{ kind = LyXParagraph::TAB; }
		else if (argument == "wide-fig")
			{ kind = LyXParagraph::WIDE_FIG; }
		else if (argument == "wide-tab")
			{ kind = LyXParagraph::WIDE_TAB; }
		else if (argument == "algorithm")
			{ kind = LyXParagraph::ALGORITHM; }
		else {
			setErrorMessage(N_("Unknown kind of footnote"));
			break;
		}
		owner->view()->text->InsertFootnoteEnvironment(kind);
		owner->view()->update(1);
	}
	break;
	
	case LFUN_BUFFERBULLETSSELECT:
		bulletForm();
		break;
		
	case LFUN_TOGGLECURSORFOLLOW:
		cursor_follows_scrollbar = !cursor_follows_scrollbar;
		break;
		
	case LFUN_KMAP_OFF:		// keymap off
		owner->getIntl()->KeyMapOn(false);
		break;
		
	case LFUN_KMAP_PRIM:	// primary keymap
		owner->getIntl()->KeyMapPrim();
		break;
		
	case LFUN_KMAP_SEC:		// secondary keymap
		owner->getIntl()->KeyMapSec();
		break;
		
	case LFUN_KMAP_TOGGLE:	// toggle keymap
		owner->getIntl()->ToggleKeyMap();
		break;

	case LFUN_SELFINSERT:
	{
		for (string::size_type i = 0; i < argument.length(); ++i) {
			owner->view()->text->InsertChar(argument[i]);
			// This needs to be in the loop, or else we
			// won't break lines correctly. (Asger)
			owner->view()->smallUpdate(1);
		}
		SetUpdateTimer();
		owner->view()->text->sel_cursor = 
			owner->view()->text->cursor;
		moveCursorUpdate(false);
	}
	break;

	case LFUN_SEQUENCE: 
	{
		// argument contains ';'-terminated commands
		while (argument.find(';') != string::npos) {
			string first;
			argument = split(argument, first, ';');
			Dispatch(first);
		}
	}
	break;

	case LFUN_DATE_INSERT:  // jdblair: date-insert cmd
	{
		char datetmp[32];
		int datetmp_len;
		time_t now_time_t;
		struct tm *now_tm;
		static string arg;
		
		now_time_t = time(NULL);
		now_tm = localtime(&now_time_t);
		(void)setlocale(LC_TIME, "");
		if (!argument.empty())
			arg = argument;
		else if (arg.empty())
			arg = lyxrc->date_insert_format;
		datetmp_len = (int) strftime(datetmp, 32, arg.c_str(), now_tm);
		for (int i = 0; i < datetmp_len; i++) {
			owner->view()->text->InsertChar(datetmp[i]);
			owner->view()->smallUpdate(1);
		}
		SetUpdateTimer();
		owner->view()->text->sel_cursor = owner->view()->text->cursor;
		moveCursorUpdate(false);
	}
	break;

	case LFUN_SAVEPREFERENCES:
	{
		Path p(user_lyxdir);
		lyxrc->write("preferences");
	}
	break;
	
	case LFUN_UNKNOWN_ACTION:
	{
		if(!owner->buffer()) {
			LyXBell();
			setErrorMessage(N_("No document open"));
			break;
		}

		if (owner->buffer()->isReadonly()) {
			LyXBell();
			setErrorMessage(N_("Document is read only"));
			break;
		}
			 
		if (!argument.empty()) {
			
			/* Automatically delete the currently selected
			 * text and replace it with what is being
			 * typed in now. Depends on lyxrc settings
			 * "auto_region_delete", which defaults to
			 * true (on). */
		
			if ( lyxrc->auto_region_delete ) {
				if (owner->view()->text->selection){
					owner->view()->text->CutSelection(false);
					owner->view()->update(-1);
				}
			}
			
			owner->view()->beforeChange();
			for (string::size_type i = 0;
			     i < argument.length(); ++i) {
				if (greek_kb_flag) {
					if (!math_insert_greek(argument[i]))
						owner->getIntl()->getTrans()->TranslateAndInsert(argument[i], owner->view()->text);
				} else
					owner->getIntl()->getTrans()->TranslateAndInsert(argument[i], owner->view()->text);
			}
			
			owner->view()->smallUpdate(1);
			SetUpdateTimer();

			owner->view()->text->sel_cursor = 
				owner->view()->text->cursor;
			moveCursorUpdate(false);
			return string();
		} else {
			// why is an "Unknown action" with empty
			// argument even dispatched in the first
			// place? I`ll probably change that. (Lgb)
			LyXBell();
			setErrorMessage(N_("Unknown action"));
		}
		break;
	default:
		lyxerr << "A truly unknown func!" << endl;
		break;
	}
	} // end of switch
  exit_with_message:

	string res = getMessage();

	if (res.empty()) {
		if (!commandshortcut.empty()) {
			string newbuf = owner->getMiniBuffer()->GetText();
			if (newbuf != commandshortcut) {
				owner->getMiniBuffer()->Set(newbuf
							    + " " +
							    commandshortcut);
			}
		}
	} else {
		owner->getMiniBuffer()->Set(string(_(res.c_str()))
					    + " " + commandshortcut);
	}

	return res;
}


void LyXFunc::setupLocalKeymap()
{
	keyseq.stdmap = keyseq.curmap = toplevel_keymap;
	cancel_meta_seq.stdmap = cancel_meta_seq.curmap = toplevel_keymap;
}


void LyXFunc::MenuNew(bool fromTemplate)
{
	string fname, initpath = lyxrc->document_path;
	LyXFileDlg fileDlg;

	if (owner->view()->available()) {
		string trypath = owner->buffer()->filepath;
		// If directory is writeable, use this as default.
		if (IsDirWriteable(trypath) == 1)
			initpath = trypath;
	}

	ProhibitInput();
	fileDlg.SetButton(0, _("Documents"), lyxrc->document_path);
	fileDlg.SetButton(1, _("Templates"), lyxrc->template_path);
	fname = fileDlg.Select(_("Enter Filename for new document"), 
			       initpath, "*.lyx", _("newfile"));
 	AllowInput();
	
	if (fname.empty()) {
		owner->getMiniBuffer()->Set(_("Canceled."));
		lyxerr.debug() << "New Document Cancelled." << endl;
		return;
	}
        
	// get absolute path of file and make sure the filename ends
	// with .lyx
	string s = MakeAbsPath(fname);
	if (!IsLyXFilename(s))
		s += ".lyx";

	// Check if the document already is open
	if (bufferlist.exists(s)){
		switch(AskConfirmation(_("Document is already open:"), 
				       MakeDisplayPath(s, 50),
				       _("Do you want to close that document now?\n"
					 "('No' will just switch to the open version)")))
			{
			case 1: // Yes: close the document
				if (!bufferlist.close(bufferlist.getBuffer(s)))
				// If close is canceled, we cancel here too.
					return;
				break;
			case 2: // No: switch to the open document
				owner->view()->buffer(bufferlist.getBuffer(s));
				return;
			case 3: // Cancel: Do nothing
				owner->getMiniBuffer()->Set(_("Canceled."));
				return;
			}
	}
        
	// Check whether the file already exists
	if (IsLyXFilename(s)) {
		FileInfo fi(s);
		if (fi.readable() &&
		    AskQuestion(_("File already exists:"), 
				MakeDisplayPath(s, 50),
				_("Do you want to open the document?"))) {
			// loads document
			owner->getMiniBuffer()->Set(_("Opening document"), 
						    MakeDisplayPath(s), "...");
			XFlush(fl_display);
			owner->view()->buffer(
				bufferlist.loadLyXFile(s));
			owner->getMiniBuffer()->Set(_("Document"),
						    MakeDisplayPath(s),
						    _("opened."));
			return;
		}
	}

	// The template stuff
	string templname;
	if (fromTemplate) {
		ProhibitInput();
		fname = fileDlg.Select(_("Choose template"),
				       lyxrc->template_path,
				       "*.lyx");
                templname = fname;
		AllowInput();
	}
  
	// find a free buffer
	lyxerr.debug() << "Find a free buffer." << endl;
	owner->view()->buffer(bufferlist.newFile(s, templname));
}


void LyXFunc::MenuOpen()
{
	string initpath = lyxrc->document_path;
	LyXFileDlg fileDlg;
  
	if (owner->view()->available()) {
		string trypath = owner->buffer()->filepath;
		// If directory is writeable, use this as default.
		if (IsDirWriteable(trypath) == 1)
			initpath = trypath;
	}

	// launches dialog
	ProhibitInput();
	fileDlg.SetButton(0, _("Documents"), lyxrc->document_path);
	fileDlg.SetButton(1, _("Examples"), 
			  AddPath(system_lyxdir, "examples"));
	string filename = fileDlg.Select(_("Select Document to Open"),
					 initpath, "*.lyx");
	AllowInput();
 
	// check selected filename
	if (filename.empty()) {
		owner->getMiniBuffer()->Set(_("Canceled."));
		return;
	}

	// get absolute path of file and make sure the filename ends
	// with .lyx
	filename = MakeAbsPath(filename);
	if (!IsLyXFilename(filename))
		filename += ".lyx";

	// loads document
	owner->getMiniBuffer()->Set(_("Opening document"),
				    MakeDisplayPath(filename), "...");
	Buffer * openbuf = bufferlist.loadLyXFile(filename);
	if (openbuf) {
		owner->view()->buffer(openbuf);
		owner->getMiniBuffer()->Set(_("Document"),
					    MakeDisplayPath(filename),
					    _("opened."));
	} else {
		owner->getMiniBuffer()->Set(_("Could not open document"),
					    MakeDisplayPath(filename));
	}
}


void LyXFunc::doImportASCII(bool linorpar)
{
	string initpath = lyxrc->document_path;
	LyXFileDlg fileDlg;
  
	if (owner->view()->available()) {
		string trypath = owner->buffer()->filepath;
		// If directory is writeable, use this as default.
		if (IsDirWriteable(trypath) == 1)
			initpath = trypath;
	}

	// launches dialog
	ProhibitInput();
	fileDlg.SetButton(0, _("Documents"), lyxrc->document_path);
	fileDlg.SetButton(1, _("Examples"), 
			  AddPath(system_lyxdir, "examples"));
	string filename = fileDlg.Select(_("Select ASCII file to Import"),
					 initpath, "*.txt");
	AllowInput();
 
	// check selected filename
	if (filename.empty()) {
		owner->getMiniBuffer()->Set(_("Canceled."));
		return;
	}

	// get absolute path of file
	filename = MakeAbsPath(filename);

	string s = ChangeExtension(filename, ".lyx", false);

	// Check if the document already is open
	if (bufferlist.exists(s)) {
		switch(AskConfirmation(_("Document is already open:"), 
				       MakeDisplayPath(s, 50),
				       _("Do you want to close that document now?\n"
					 "('No' will just switch to the open version)")))
			{
			case 1: // Yes: close the document
				if (!bufferlist.close(bufferlist.getBuffer(s)))
				// If close is canceled, we cancel here too.
					return;
				break;
			case 2: // No: switch to the open document
				owner->view()->buffer(bufferlist.getBuffer(s));
				return;
			case 3: // Cancel: Do nothing
				owner->getMiniBuffer()->Set(_("Canceled."));
				return;
			}
	}

	// Check if a LyX document by the same root exists in filesystem
	FileInfo f(s, true);
	if (f.exist() && !AskQuestion(_("A document by the name"), 
				      MakeDisplayPath(s),
				      _("already exists. Overwrite?"))) {
		owner->getMiniBuffer()->Set(_("Canceled."));
		return;
	}

	owner->view()->buffer(bufferlist.newFile(s, string()));
	owner->getMiniBuffer()->Set(_("Importing ASCII file"),
				    MakeDisplayPath(filename), "...");
	// Insert ASCII file
	InsertAsciiFile(filename, linorpar);
	owner->getMiniBuffer()->Set(_("ASCII file "),
				    MakeDisplayPath(filename),
				    _("imported."));
}


void LyXFunc::doImportLaTeX(bool isnoweb)
{
	string initpath = lyxrc->document_path;
	LyXFileDlg fileDlg;
  
	if (owner->view()->available()) {
		string trypath = owner->buffer()->filepath;
		// If directory is writeable, use this as default.
		if (IsDirWriteable(trypath) == 1)
			initpath = trypath;
	}

	// launches dialog
	ProhibitInput();
	fileDlg.SetButton(0, _("Documents"), lyxrc->document_path);
	fileDlg.SetButton(1, _("Examples"), 
			  AddPath(system_lyxdir, "examples"));
	string filename;
	if (isnoweb) {
		filename = fileDlg.Select(_("Select Noweb file to Import"),
					  initpath, "*.nw");
	} else {
		filename = fileDlg.Select(_("Select LaTeX file to Import"),
					  initpath, "*.tex");
	}
	
	AllowInput();
 
	// check selected filename
	if (filename.empty()) {
		owner->getMiniBuffer()->Set(_("Canceled."));
		return;
	}

	// get absolute path of file
	filename = MakeAbsPath(filename);

	// Check if the document already is open
	string LyXfilename = ChangeExtension(filename, ".lyx", false);
	if (bufferlist.exists(LyXfilename)){
		switch(AskConfirmation(_("Document is already open:"), 
				       MakeDisplayPath(LyXfilename, 50),
				       _("Do you want to close that document now?\n"
					 "('No' will just switch to the open version)")))
			{
			case 1: // Yes: close the document
				if (!bufferlist.close(bufferlist.getBuffer(LyXfilename)))
				// If close is canceled, we cancel here too.
					return;
				break;
			case 2: // No: switch to the open document
				owner->view()->buffer(
					bufferlist.getBuffer(LyXfilename));
				return;
			case 3: // Cancel: Do nothing
				owner->getMiniBuffer()->Set(_("Canceled."));
				return;
			}
	}

	// Check if a LyX document by the same root exists in filesystem
	FileInfo f(LyXfilename, true);
	if (f.exist() && !AskQuestion(_("A document by the name"), 
				      MakeDisplayPath(LyXfilename),
				      _("already exists. Overwrite?"))) {
		owner->getMiniBuffer()->Set(_("Canceled."));
		return;
	}

	// loads document
	Buffer * openbuf;
	if (!isnoweb) {
		owner->getMiniBuffer()->Set(_("Importing LaTeX file"),
					    MakeDisplayPath(filename), "...");
		ImportLaTeX myImport(filename);
		openbuf = myImport.run();
	} else {
		owner->getMiniBuffer()->Set(_("Importing Noweb file"),
					    MakeDisplayPath(filename), "...");
		ImportNoweb myImport(filename);
		openbuf = myImport.run();
	}
	if (openbuf) {
		owner->view()->buffer(openbuf);
		owner->getMiniBuffer()->Set(isnoweb ?
					    _("Noweb file ") : _("LateX file "),
					    MakeDisplayPath(filename),
					    _("imported."));
	} else {
		owner->getMiniBuffer()->Set(isnoweb ?
					    _("Could not import Noweb file") :
					    _("Could not import LaTeX file"),
					    MakeDisplayPath(filename));
	}
}


void LyXFunc::MenuInsertLyXFile(string const & filen)
{
	string filename = filen;

	if (filename.empty()) {
		// Launch a file browser
		string initpath = lyxrc->document_path;
		LyXFileDlg fileDlg;

		if (owner->view()->available()) {
			string trypath = owner->buffer()->filepath;
			// If directory is writeable, use this as default.
			if (IsDirWriteable(trypath) == 1)
				initpath = trypath;
		}

		// launches dialog
		ProhibitInput();
		fileDlg.SetButton(0, _("Documents"), lyxrc->document_path);
		fileDlg.SetButton(1, _("Examples"), 
				  AddPath(system_lyxdir, "examples"));
		filename = fileDlg.Select(_("Select Document to Insert"),
					  initpath, "*.lyx");
		AllowInput();

		// check selected filename
		if (filename.empty()) {
			owner->getMiniBuffer()->Set(_("Canceled."));
			return;
		}
	} 

	// get absolute path of file and make sure the filename ends
	// with .lyx
	filename = MakeAbsPath(filename);
	if (!IsLyXFilename(filename))
		filename += ".lyx";

	// Inserts document
	owner->getMiniBuffer()->Set(_("Inserting document"),
				    MakeDisplayPath(filename), "...");
	bool res = owner->view()->insertLyXFile(filename);
	if (res) {
		owner->getMiniBuffer()->Set(_("Document"),
					    MakeDisplayPath(filename),
					    _("inserted."));
	} else {
		owner->getMiniBuffer()->Set(_("Could not insert document"),
					    MakeDisplayPath(filename));
	}
}


void LyXFunc::reloadBuffer()
{
	string fn = owner->buffer()->fileName();
	if (bufferlist.close(owner->buffer()))
		owner->view()->buffer(bufferlist.loadLyXFile(fn));
}


void LyXFunc::CloseBuffer()
{
	if (bufferlist.close(owner->buffer()) && !quitting) {
		if (bufferlist.empty()) {
			// need this otherwise SEGV may occur while trying to
			// set variables that don't exist
			// since there's no current buffer
			CloseAllBufferRelatedPopups();
		}
		else {
			owner->view()->buffer(bufferlist.first());
		}
	}
}


Inset * LyXFunc::getInsetByCode(Inset::Code code)
{
	bool found = false;
	Inset * inset = 0;
	LyXCursor cursor = owner->view()->text->cursor;
	LyXParagraph::size_type pos = cursor.pos;
	LyXParagraph * par = cursor.par;
	
	while (par && !found) {
		while ((inset = par->ReturnNextInsetPointer(pos))){
			if (inset->LyxCode() == code) {
				found = true;
				break;
			}
			++pos;
		} 
		par = par->next;
	}
	return found ? inset : 0;
}


// Each "owner" should have it's own message method. lyxview and
// the minibuffer would use the minibuffer, but lyxserver would
// send an ERROR signal to its client.  Alejandro 970603
// This func is bit problematic when it comes to NLS, to make the
// lyx servers client be language indepenent we must not translate
// strings sent to this func.
void LyXFunc::setErrorMessage(string const & m) const
{
	dispatch_buffer = m;
	errorstat = true;
}


void LyXFunc::setMessage(string const & m)
{
	dispatch_buffer = m;
}
