// vs_subtree.h  (this is -*-c++-*-)
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Subtrees for vs_trees  [ my header files are getting out of hand :P ]

#ifndef VS_SUBTREE_H
#define VS_SUBTREE_H

#include <list.h>
#include "vs_treeitem.h"
#include "config/keybindings.h"

class vs_tree;

template<class childtype>
class cmp_policy
{
 public:
  virtual int compare(const childtype *item1, const childtype *item2) const=0;
  bool operator()(const childtype *item1, const childtype *item2) const
    {
      int i=compare(item1,item2)<0;
      return i;
    }
};

template<class childtype,class sorter=cmp_policy<childtype> >
class vs_subtree:virtual public vs_treeitem
// A tree-item which can contain other tree items.  Still abstract -- the
// display routines have to be filled in, and you may want to add more behavior
// on keypresses -- but we're getting there :)
{
  bool expanded;

  typedef list<childtype *> child_list;
  typedef child_list::iterator child_iterator;

  child_list children;

protected:

  class levelref:public vs_tree_levelref
  {
    child_iterator realitem;

    child_list *parent_list;
  public:
    levelref(const levelref &x)
      :vs_tree_levelref(x), realitem(x.realitem), parent_list(x.parent_list) {}
    levelref(const child_iterator &_realitem, child_list *_parent_list)
      :realitem(_realitem), parent_list(_parent_list)
    {
    }

    vs_treeitem *get_item() {return *realitem;}
    virtual void advance_next() {realitem++;}
    virtual void return_prev() {realitem--;}
    bool is_begin() {return realitem==parent_list->begin();}
    bool is_end() {return realitem==parent_list->end();}
    levelref *clone() const {return new levelref(*this);}
  };
public:
  typedef sorter sort_policy;
  // compare should have the semantics expected by STL sorting routines

  typedef vs_treeiterator iterator;

  vs_subtree(bool _expanded):vs_treeitem(),expanded(_expanded) {}

  bool get_expanded() {return expanded;}

  void expand() {expanded=true;}
  // Hmm, should I be exporting this?  Why not everything else? ...

  void display(vs_tree *win, int y, string str, int depth_shift=2)
  {
    int width, height;
    int basex=depth_shift*get_depth();
    win->getmaxyx(height,width);

    win->move(y,0);
    for(int i=0; i<basex && i<width; i++)
      win->addch(' ');

    if(basex>width)
      return;

    win->addnstr(get_expanded()?"[+] ":"[-] ", width-basex-1);

    if(basex+4>width)
      return;

    win->addnstr(str.c_str(), width-(basex+4)-1);

    int newy,newx;
    win->getyx(newy,newx);

    for( ; newx<width; newx++)
      win->addch(' ');
  }

  void set_depth(int _depth)
  {
    for(child_iterator i=children.begin(); i!=children.end(); i++)
      (*i)->set_depth(_depth+1);

    vs_treeitem::set_depth(_depth);
  }

  void add_child(childtype *newchild)
  {
    newchild->set_depth(get_depth()+1);

    children.push_back(newchild);
  }

  // Adds a new child item at an unspecified location -- you should call sort()
  // after adding children or the tree will have an undetermined order.  (yes,
  // you can deduce the real order.  Don't.

  void sort(const sort_policy &sort_method)
  {
    bool sorted=true;
    for(child_iterator i=children.begin(); i!=children.end(); i++)
      (*i)->sort();

    for(child_iterator i=children.begin(); i!=children.end(); i++)
      {
	child_iterator j=i;
	j++;
	if(j!=children.end() && sort_method.compare(*i, *j)>0)
	  {
	    sorted=false;
	    break;
	  }
      }
    // That's just a quick hack to avoid overhead in the already-sorted case

    if(sorted)
      return;

    children.sort(sort_method);
  }

  virtual bool dispatch_char(chtype ch)
  {
    if(global_bindings.key_matches(ch, "ToggleExpanded"))
      {
	expanded=!expanded;
	return true;
      }
    else if(global_bindings.key_matches(ch, "ExpandTree"))
      {
	expanded=true;
	return true;
      }
    else if(global_bindings.key_matches(ch, "CollapseTree"))
      {
	expanded=false;
	return true;
      }
    return false;
  }
  // The default action is to expand or shrink the tree when Enter is pressed.
  // FIXME: should I use '+' and '-' here?  That would appear to conflict with
  // the other keybindings I need.  Hm.

  virtual levelref *begin() {return new levelref(children.begin(), &children);}
  virtual levelref *end() {return new levelref(children.end(), &children);}

  bool has_visible_children() {return expanded && children.size()>0;}
  bool has_children() {return children.size()>0;}

  virtual ~vs_subtree()
  {
    child_iterator i,j;
    for(i=children.begin(); i!=children.end(); i=j)
      {
	j=i;
	j++;
	delete *i;
      }
  }
};

class tag_sort_policy:public cmp_policy<vs_treeitem>
{
 public:
  int compare(const vs_treeitem *item1, const vs_treeitem *item2) const
    {return strcmp(item1->tag(), item2->tag());}
};

class vs_subtree_generic:public vs_subtree<vs_treeitem,tag_sort_policy>
{
 public:
  vs_subtree_generic(int _expanded):vs_subtree<vs_treeitem,tag_sort_policy>(_expanded) {}

  void sort()
    {
      vs_subtree<vs_treeitem,tag_sort_policy>::sort(tag_sort_policy());
    }
};

#endif
