#include <stdio.h>
#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/fd.h>
#include "../floppy_split.h"
#include "dbootstrap.h"
#include "lang.h"

#ifdef _TESTING_
  #define TEMPFILE "/tmp/buff"
#else
  #define TEMPFILE "/target/tmp/buff"
#endif

#define    HEADER_SIZE    512
static const char    magic[] = "Floppy split";

// static const char bad_floppy_message[] = _("Some important data was not read from the floppy disks. You should\nresolve this problem and re-install using valid floppy disks. If you\nare installing from a CD-ROM, there are floppy-disk \"image\" files on\nthe CD, which you can write to a blank floppy using DOS, Windows, OS/2,\nUnix or Linux. They should be in a directory on the CD called \"dos\"\nor \"disks\".\n\nOther things to try if you just can't get the floppies to install are\nreformatting the floppies before writing them, and writing them using\na different system. Also, if you got the \"image\" files from a BBS or\nInternet system and they are not working, download them again, and make\nsure to use \"binary\" mode. If you are writing the floppies from Unix\nor Linux, be sure to run the \"sync\" command before you remove each\nfloppy disk from the drive.\n");

static int
readFloppy(int fd_in, int fd_out, struct floppy_header * h)
{
    long            bytesInThisFloppy = atol(h->bytesInThisFloppy);
    long            bytesToRead = bytesInThisFloppy;
    long            bytesRead = 0;
    unsigned long   originalChecksum = strtoul(h->checksum, 0, 0x10);
    unsigned long   checksum = 0;
    int             fd_tmp,byte = 0;
    long     status=0;
    char    buf[20 * 1024];

    DEBUGMSG("tempfile is %s", TEMPFILE);

    if ( (fd_tmp = creat(TEMPFILE, S_IRUSR | S_IWUSR )) < 0 ) {
        perrorBox(_("Couldn't open tempfile for writing"));
        return (-2);
    }
    
    scaleBox(prtbuf, _("Reading Disk"), (long long)bytesInThisFloppy, 0);

    while ( bytesToRead > 0 ) {
        long    chunk = sizeof(buf);

        if ( bytesToRead < chunk )
            chunk = bytesToRead;

        errno = 0;
        if ( (status = read(fd_in, buf, chunk)) < chunk ) {
            int     old_errno = errno;
            if ( old_errno == 0 && read(fd_in, buf, 1) != 1 )
                old_errno = errno;
            if ( old_errno == 0 )
                problemBox(_("Floppy read error: the floppy unexpectedly ended."), _("Floppy Error!"));
            else
                perrorBox(_("Floppy read error"));
	    status=-1;
	    break;
        }

        bytesRead += chunk;
        bytesToRead -= chunk;

        scaleBox(NULL, NULL, (long long)bytesRead, 1);

        for ( byte = 0; byte < chunk; byte++ )
            checksum += (buf[byte] & 0xff);

        if ( (status = write(fd_tmp, buf, chunk)) != chunk ) {
            perrorBox(_("Error writing to tempfile"));
	    status = -1;
	    break;
        }
    }
    scaleBox(NULL, NULL, 0, 2);
    close(fd_tmp);
    if (status==-1) {
        unlink(TEMPFILE);
        return (-1);
    }
    if ( checksum != originalChecksum ) {
        problemBox(_("The floppy disk checksum does not match the checksum made when it was written."),_("Floppy Error!"));
        unlink(TEMPFILE);
        return (-1);
    }

    if ( (fd_tmp = open(TEMPFILE, O_RDONLY)) < 0 ) {
        perrorBox(_("Couldn't open tempfile for reading"));
        return (-2);
    }
    while ( status > 0 ) {
        if ( (status = read(fd_tmp, buf, sizeof(buf))) < 0) {
            perrorBox(_("Error reading from tempfile"));
	    status=-2;
	    break;
        }
        if ( write(fd_out, buf, status) != status ) {
            perrorBox(_("Error writing to output file"));
            status=-2;
            break;
        }
    }
    close(fd_tmp);
    unlink(TEMPFILE);
    return(status);
}

/* Return Values:
 *   0           successful
 *   1           Cannot create local filename (presumely on /target)
 *  DLG_CANCEL   Cancel (10)
 */
int getFloppies(const char * message, const char * device, const char * filename, char **basenames, const char *series) {
    int		expectedDisk = 1;
    int		totalFloppies = 0;
    char	name[1024];
    int		fd_in=-1, fd_out;

    if (filename) {
        struct stat statbuf;

        if (NAME_ISREG (filename, &statbuf)) {
            unlink (filename);
        }

        if ( (fd_out = creat(filename, S_IRUSR | S_IWUSR )) < 0 ) {
            perrorBox(filename);
	    return 1;
        }
    } else 
        fd_out=1;

    for ( ; ; ) {
        union header_block    block;
        int	diskNumber = 0;
        int	diskTotalFloppies;
        int	status;

        if (fd_in > 0) close(fd_in);

        sprintf(prtbuf, message,
#ifdef SCSI_FLOPPY
        (device[5]=='s')?"SCSI":
#endif
          ((device[7]=='0')?_("first"):_("second")),expectedDisk);
	if (twoButtonBox(prtbuf,_("Insert disk"),_("Continue"),_("Cancel"))==DLG_NO) {
	    close(fd_out);
	    return DLG_CANCEL;
	}

        if ( (fd_in = open(device, O_RDONLY)) < 0 ) {
            eject_floppy(device);
            perrorBox(device);
            continue;
        }

#ifdef SCSI_FLOPPY
    if (strncmp(device,"/dev/sfd",8)) /* not scsi floppy */
#endif
#ifndef _TESTING_
        if ( ioctl(fd_in, FDFLUSH, 0) != 0 )
            perrorBox(_("ioctl fdflush failed (not fatal)"));
#endif

        if ( (status = read(fd_in,(char*)&block,HEADER_SIZE)) != HEADER_SIZE) {
            eject_floppy(device);
            if ( status > 0 )
                problemBox(_("Could not read floppy: Unexpected end-of-file."),_("Floppy Error!"));
            else
                perrorBox(_("Could not read floppy"));
            continue;
        }
        if ( strncmp(block.header.magic, magic, sizeof(magic) - 1) != 0 ) {
            eject_floppy(device);
            problemBox(_("That doesn't look like the right floppy."),_("Wrong Floppy!"));
            continue;
        }
        
        diskNumber = atoi(block.header.floppyNumber);
        diskTotalFloppies = atoi(block.header.totalFloppies);
        sprintf(prtbuf,_("This is disk %d of %d in the %s series of %s.\n"),
          diskNumber, diskTotalFloppies, block.header.name, block.header.date);
	status=strlen(prtbuf);
        if ( diskNumber != expectedDisk ) {
            eject_floppy(device);
            sprintf(prtbuf+status,_("Wrong disk. I was looking for disk %d.")
              ,expectedDisk);
	    problemBox(prtbuf,_("Wrong disk!"));
            continue;
        }
        if ( diskNumber == 1 ) {
	    int i;
	    int wrong_disk = 1;

	    for (i=0; basenames[i]; i++)
		if (!strcmp (basenames[i], block.header.name))
		    wrong_disk = 0;
	    if (wrong_disk) {
		sprintf(prtbuf+status,_("Wrong disk. This is from series %s.\nYou need disk %d of series %s."),
			block.header.name, expectedDisk, series);
		problemBox(prtbuf,_("Wrong disk!"));
		continue;
	    }

            strcpy(name, block.header.name);
            totalFloppies = diskTotalFloppies;
        }
        else if ( strcmp(name, block.header.name) != 0 ) {
            eject_floppy(device);
            sprintf(prtbuf+status,_("Wrong disk. This is from series %s.\nYou need disk %d of series %s."),
              block.header.name, expectedDisk, name);
	    problemBox(prtbuf,_("Wrong disk!"));
            continue;
        }
        status=readFloppy(fd_in, fd_out, &block.header);
#ifdef SCSI_FLOPPY
    if (strncmp(device,"/dev/sfd",8)) /* not scsi floppy */
#endif
        ioctl(fd_in, FDFLUSH, 0);
        close(fd_in);
        eject_floppy(device);
        fd_in=-1;
	if (status==0) {
	  expectedDisk++;
	} else if (status==-2) {
	  break;
	}
        if (expectedDisk > totalFloppies) break;
    }
    close(fd_out);
    return 0;
}

#ifdef _TESTING_
/*
 * To test, compile using: make floppy_merge_test
 *
 * And run: losetup /dev/loop0 /home/project/Debian/CVS/boot-floppies/drv14-1.bin
 */
void main(void) {
    int status;
    char *base_filenames[] = { "base12", "base14", NULL };

    LOAD_TRMFILE("test.trm");

    get_kver();
    boxInit();

    status = getFloppies(_("The system is being installed from the %s floppy drive.\n\nPlease insert base disk %d."),
			 "/dev/loop1","/tmp/testing.tgz", base_filenames, _("the base series"));

    boxFinished();

    if (status == 10)
	printf ("Cancelled.\n");
    else
	printf ("Status of getFloppies: %d\n", status);
}
#endif

/*
 * Local variables:
 *   c-basic-offset: 4
 *   indent-tabs-mode: t
 * End:
 */
