/*
 Copyright (C) 1999 The Software in the Public Interest (SPI)
 Written by Michael Sobolev <mss@transas.com>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 This program uses parts of iconv_prog.c file from glibc-2.1.2
*/

#include "Python.h"

#include <iconv.h>
#include <errno.h>
#include <stdarg.h>

static PyObject *__error = NULL;

typedef struct PyIconv {
    PyObject_VAR_HEAD
    iconv_t engine;
} PyIconv;

staticforward PyTypeObject PyIconvType;
staticforward PyMethodDef PyIconvMethods[];

#define CHUNK_SIZE  1024

static void
set_error (const char *fmt, ...)
{
    va_list args;

    va_start (args, fmt);

    {
        char *buffer = (char *)malloc (CHUNK_SIZE); /* I want to hope this is sufficient for the most errors */

        vsprintf (buffer, fmt, args);

        PyErr_SetString (__error, buffer);

        free (buffer);
    }

    va_end (args);
}

static PyObject *
do_convert (iconv_t engine, const char *what)
{
    PyObject *result = NULL;
    char *current = (char *)what;
    char *buffer = (char *)malloc (CHUNK_SIZE), *out;
    size_t pending = strlen (what), buffer_len = CHUNK_SIZE;

    for (out = buffer ; pending > 0 && buffer != NULL ;)
    {
        size_t code, avail = buffer_len - (out - buffer);

        code = iconv (engine, &current, &pending, &out, &avail);

        if (code != -1)     /* Finished? */
            break;

        if (errno != E2BIG) 
            switch (errno)
            {
                case EILSEQ:
                    set_error ("illegal input sequence at position %ld", current - what);
                    free (buffer);
                    buffer = NULL;
                    break;
                case EINVAL:
                    set_error ("incomplete character or shift sequence at end of buffer");
                    free (buffer);
                    buffer = NULL;
                    break;
                case EBADF:
                    set_error ("internal error (illegal descriptor)");
                    free (buffer);
                    buffer = NULL;
                    break;
                default:
                    set_error ("unknown iconv() error %d", errno);
                    free (buffer);
                    buffer = NULL;
                    break;
            }
        else    /* Output buffer is insufficient */
        {
            char *tempo;
            buffer_len += CHUNK_SIZE;
            tempo = (char *)realloc (buffer, buffer_len);

            if (tempo == NULL)
            {
                set_error ("unable to allocate a buffer");
                free (buffer);
                buffer = NULL;
            }
            else
                buffer = tempo;
        }
    }

    if (buffer != NULL)
        result = PyString_FromStringAndSize (buffer, out - buffer);

    return result;
}

static char __convert_doc[] = "convert (from, to, what) -- convert string `what' from charset `from' into charset `to'";

static PyObject *
__convert (PyObject *self, PyObject *args)
{
    iconv_t engine;
    char *to, *from, *what;
    PyObject *result = NULL;

    if (PyArg_ParseTuple (args, "sss", &from, &to, &what))
    {
        if ((engine = iconv_open (to, from)) != (iconv_t)-1)
            result = do_convert (engine, what);
        else
            set_error ("error initializing conversion engine");
    }

    return result;
}

static PyObject *
PyIconvNew (const char *from, const char *to)
{
    iconv_t engine;
    PyIconv *result = NULL;

    if ((engine = iconv_open (to, from)) != (iconv_t)-1 && (result = PyObject_NEW (PyIconv, &PyIconvType)) != NULL)
        result->engine = engine;
    else
        set_error ("error initializing conversion engine");

    return (PyObject *)result;
}

static void
PyIconvFree (PyObject *self)
{
    iconv_close (((PyIconv *)self)->engine);

    PyMem_DEL (self);
}

static PyObject *
PyIconvGetAttr (PyObject *self, char *name)
{
    return Py_FindMethod (PyIconvMethods, self, name);
}

static char __engine_convert_doc[] = "";

static PyObject *
__engine_convert (PyObject *self, PyObject *args)
{
    char *what;
    PyObject *result;

    if (PyArg_ParseTuple (args, "s", &what))
        result = do_convert (((PyIconv *)self)->engine, what);

    return result;
}

statichere PyTypeObject PyIconvType = {
    PyObject_HEAD_INIT(&PyType_Type)
    0,                                  /*ob_size*/
    "engine",                           /*tp_name*/
    sizeof (PyIconv),                   /*tp_size*/
    sizeof (iconv_t),                   /*tp_itemsize*/
    PyIconvFree,                        /*tp_dealloc*/
    0,                                  /*tp_print*/
    PyIconvGetAttr,                     /*tp_getattr*/
    0,                                  /*tp_setattr*/
    0,                                  /*tp_compare*/
    0,                                  /*tp_repr*/
    0,                                  /*tp_as_number*/
    0,                                  /*tp_as_sequence*/
    0,                                  /*tp_as_mapping*/
    0,                                  /*tp_hash*/
    0,                                  /*tp_call*/
    0,                                  /*tp_str*/
    0,                                  /*tp_getattro*/
    0,                                  /*tp_setattro*/
};

statichere PyMethodDef PyIconvMethods[] = {
    { "convert", __engine_convert, 1, __engine_convert_doc},
    { NULL, NULL }
};

static char __engine_doc[] = "";

static PyObject *
__engine (PyObject *self, PyObject *args)
{
    PyObject *result = NULL;
    char *to, *from;

    if (PyArg_ParseTuple (args, "ss", &from, &to))
        result = PyIconvNew (from, to);

    return result;
}

static char __iconv_doc[] = "Convert encoding of given string from one encoding to another";

static struct PyMethodDef __iconv_methods[] = {
    { "convert", __convert, 1, __convert_doc},
    { "engine", __engine, 1, __engine_doc},
    { NULL, NULL }  /* End of table */
};

void
initiconv (void)
{
    PyObject *module, *dict;

    module = Py_InitModule4 ("iconv", __iconv_methods, __iconv_doc, 0, PYTHON_API_VERSION);

    dict = PyModule_GetDict (module);

    __error = PyString_FromString ("iconv.error");

    PyDict_SetItemString (dict, "error", __error);
}
