/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "acquirestatus.h"
#include <unistd.h>
#include <apt-pkg/acquire-item.h>
#include <apt-pkg/acquire-worker.h>
#include <stdio.h>
#include <apt-pkg/strutl.h>

gboolean 
GAcqStatus::close_cb(GtkWidget* dialog, gpointer data)
{
  GAcqStatus* as = static_cast<GAcqStatus*>(data);

  as->cancelled_ = true;

  return TRUE; /* block the close - we'll close in the destructor after the next iteration of the acquire */
}

GAcqStatus::GAcqStatus()
  : cancelled_(false), dialog_(0), itembar_(0), overallbar_(0), label_(0)
{
  dialog_ = gnome_dialog_new(_("Download Progress"),
                             GNOME_STOCK_BUTTON_CANCEL, 
                             NULL);
  
  gnome_apt_setup_dialog(dialog_);  
  gtk_window_set_modal(GTK_WINDOW(dialog_), TRUE);
  gtk_window_set_policy(GTK_WINDOW(dialog_), TRUE, TRUE, FALSE);
  gnome_dialog_set_close(GNOME_DIALOG(dialog_), TRUE);
  gtk_signal_connect(GTK_OBJECT(dialog_), "close",
                     GTK_SIGNAL_FUNC(close_cb), this);

  gtk_widget_set_usize(dialog_, 450, 450);

  itembar_    = gtk_progress_bar_new();

  gtk_progress_set_format_string(GTK_PROGRESS(itembar_), _("Active download(s) %P%%"));
  gtk_progress_set_show_text(GTK_PROGRESS(itembar_), TRUE);

  overallbar_ = gtk_progress_bar_new();

  gtk_progress_set_format_string(GTK_PROGRESS(overallbar_), _("Overall %P%%"));
  gtk_progress_set_show_text(GTK_PROGRESS(overallbar_), TRUE);
  gtk_progress_bar_set_activity_blocks(GTK_PROGRESS_BAR(overallbar_), 10);

  label_ = gtk_label_new("");
  etalabel_ = gtk_label_new("");

  GtkWidget* frame = gtk_frame_new(_("Error Log"));

  errors_ = gtk_text_new(0,0);
  gtk_widget_set_usize(errors_, -1, 100);

  gtk_container_add(GTK_CONTAINER(frame), errors_);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     label_, 
                     TRUE, TRUE, GNOME_PAD);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     etalabel_, 
                     TRUE, TRUE, GNOME_PAD);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     frame, 
                     FALSE, FALSE, GNOME_PAD_SMALL);  

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     itembar_, 
                     FALSE, FALSE, GNOME_PAD_SMALL);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     overallbar_, 
                     FALSE, FALSE, GNOME_PAD_SMALL);

   // flush
  while (gtk_events_pending())
    gtk_main_iteration();
}

void
GAcqStatus::Start() 
{
   pkgAcquireStatus::Start(); 
   ID = 1;
   cancelled_ = false;

   cerr << "Start" << endl;

   gtk_progress_set_percentage(GTK_PROGRESS(itembar_), 0.0);
   gtk_progress_set_percentage(GTK_PROGRESS(overallbar_), 0.0);

   gtk_editable_delete_text(GTK_EDITABLE(errors_), 
                            0, gtk_text_get_length(GTK_TEXT(errors_)));

   gtk_widget_show_all(dialog_);
};

bool 
GAcqStatus::MediaChange(string Media,string Drive)
{
  
  gchar buf[256];

  g_snprintf(buf,255,_("Please insert:\n'%s'\nin drive %s"),
             Media.c_str(), Drive.c_str());

  GtkWidget* subdialog = gnome_ok_dialog(buf);
  gnome_dialog_set_parent(GNOME_DIALOG(subdialog), GTK_WINDOW(dialog_));

  // block until clicked
  gnome_dialog_run(GNOME_DIALOG(subdialog));
   
  // Hope apt-pkg pops the dialog again if the media isn't really
  //  there?

  Update = true;
  return true;
}

// This means the file hasn't changed, If Modified Since.
void
GAcqStatus::IMSHit(pkgAcquire::ItemDesc &Itm)
{
  gchar buf[256];

  cerr << "Hit " << Itm.Description << endl;

  int n = g_snprintf(buf, 255, _("Already have %s "), Itm.Description.c_str());
  if (n > 0) {
    if (Itm.Owner->FileSize != 0)
      g_snprintf(&buf[n], 255-n, _("[%s]"), SizeToStr(Itm.Owner->FileSize).c_str());
  }
  
  gtk_label_set(GTK_LABEL(label_), buf);
  
  // flush
  while (gtk_events_pending())
    gtk_main_iteration();

  // ??
  Update = true;
};

void
GAcqStatus::Fetch(pkgAcquire::ItemDesc &Itm)
{
  Update = true;
  
  cerr << "Fetch " << Itm.Description << endl;

  if (Itm.Owner->Complete == true)
    return;
  
  Itm.Owner->ID = ID++;
  
  gchar buf[256];

#if 0
  int n = g_snprintf(buf, 255, _("%s    %%P%%"), Itm.Description.c_str());
  if (n > 0) {
    if (Itm.Owner->FileSize != 0)
      g_snprintf(&buf[n], 255-n, _("/%s"), SizeToStr(Itm.Owner->FileSize).c_str());
  }

   gtk_progress_set_format_string(GTK_PROGRESS(itembar_), buf);
   gtk_progress_set_show_text(GTK_PROGRESS(itembar_), TRUE);
#endif

   g_snprintf(buf, 255, _("Get: %ld %s [%s]"),
              Itm.Owner->ID, Itm.Description.c_str(),
              SizeToStr(Itm.Owner->FileSize).c_str());

   gtk_label_set(GTK_LABEL(label_), buf);

   // flush
   while (gtk_events_pending())
     gtk_main_iteration();
};

void
GAcqStatus::Done(pkgAcquire::ItemDesc &Itm)
{
  cerr << "Done " << Itm.Description << endl;

  gchar buf[256];
  g_snprintf(buf, 255, _("Finished %s"), 
             Itm.Description.c_str());

  gtk_label_set(GTK_LABEL(label_), buf);

   // flush
  while (gtk_events_pending())
    gtk_main_iteration();

  Update = true;
};


void
GAcqStatus::Fail(pkgAcquire::ItemDesc &Itm)
{   
  gchar buf[256];

  cerr << "Fail " << Itm.Description << endl;
  
  g_snprintf(buf, 255, _("%s\n%s"), 
             Itm.Description.c_str(), 
             Itm.Owner->ErrorText.c_str());

  gtk_label_set(GTK_LABEL(label_), buf);

  gint pos = gtk_text_get_length(GTK_TEXT(errors_));

  gtk_editable_insert_text(GTK_EDITABLE(errors_),
                           buf,
                           strlen(buf),
                           &pos);

#if 0
  gtk_progress_set_show_text(GTK_PROGRESS(itembar_), FALSE);
  gtk_progress_set_percentage(GTK_PROGRESS(itembar_), 0.0);
#endif

   // flush
  while (gtk_events_pending())
    gtk_main_iteration();

  Update = true;
};

void
GAcqStatus::Stop()
{
  pkgAcquireStatus::Stop();

  cerr << "Stop" << endl;

  gchar buf[256];

  if (FetchedBytes != 0)  
    g_snprintf(buf, 255, _("Fetched %s in %s  (%s/s)"),
               SizeToStr(FetchedBytes).c_str(),
               TimeToStr(ElapsedTime).c_str(), 
               SizeToStr(CurrentCPS).c_str());
  else
    g_snprintf(buf, 255, _("Fetched no data."));

  gtk_label_set(GTK_LABEL(label_), buf);
  
  // flush
  while (gtk_events_pending())
    gtk_main_iteration();

  // The beep is supposed to notify that the download finished,
  //  so if it finished via cancel we don't want it.
  if (!cancelled_)
    gdk_beep();
}

bool
GAcqStatus::Pulse(pkgAcquire *Owner)
{   
   bool Ret = pkgAcquireStatus::Pulse(Owner);

   // This should never happen, but if it could we'd do this.
   if (Ret == false)
     return false;

   bool Shown = false;

   vector<string> workers;
   
   gulong workers_current = 0;
   gulong workers_total   = 0;
   
   pkgAcquire::Worker *i = Owner->WorkersBegin(); 
   while (i != 0) {
     // There is no item running 
     if (i->CurrentItem == 0)
       {
         if ( ! i->Status.empty())
           {
             workers.push_back(i->Status);
             Shown = true;
           }
         i = Owner->WorkerStep(i);
         continue;
       }

     Shown = true;
      
     string s;
      
     gchar buf[512];

      // Add in the short description
     if (i->CurrentItem->Owner->ID != 0)
       g_snprintf(buf,511, _("Item %lx, %s"),
                  i->CurrentItem->Owner->ID,
                  i->CurrentItem->ShortDesc.c_str());
     else
       g_snprintf(buf,511,"%s",i->CurrentItem->ShortDesc.c_str());

     s += buf;

     // Show the short mode string (What is this?)
     if (i->CurrentItem->Owner->Mode != 0)
       {
         g_snprintf(buf,511," %s",i->CurrentItem->Owner->Mode);
         s += buf;
       }
            
     // Add the current progress
     g_snprintf(buf,511," %s",SizeToStr(i->CurrentSize).c_str());

     s += buf;
      
      // Add the total size and percent
     if (i->TotalSize > 0 && i->CurrentItem->Owner->Complete == false)
       {
         g_snprintf(buf,511,
                    "/%s %lu%%",
                    SizeToStr(i->TotalSize).c_str(),
                    long(double(i->CurrentSize*100.0)/double(i->TotalSize)));
         s += buf;
       }      

     workers_current += i->CurrentSize;
     workers_total   += i->TotalSize;

     workers.push_back(s);
      
     i = Owner->WorkerStep(i);
   }

   gtk_progress_set_activity_mode(GTK_PROGRESS(overallbar_), !Shown);
   gtk_progress_set_activity_mode(GTK_PROGRESS(itembar_), !Shown);

   if (!Shown) {
     gfloat new_val;
     GtkAdjustment *adj;
     
     adj = GTK_PROGRESS(overallbar_)->adjustment;
     
     new_val = adj->value + 1;
     if (new_val > adj->upper)
       new_val = adj->lower;

     gtk_progress_set_value (GTK_PROGRESS (overallbar_), new_val);

     gtk_label_set(GTK_LABEL(label_), _("No active downloads"));
   }
   else {
     // Set label
     string list_of_workers = _("Active download(s):\n");
     vector<string>::iterator q = workers.begin();
     while (q != workers.end()) {
       list_of_workers += *q;
       list_of_workers += "\n";
       ++q;
     }
     gtk_label_set(GTK_LABEL(label_), list_of_workers.c_str());
     
     float workerpercent = 0.0;
     if (workers_total != 0) 
       workerpercent = float(workers_current)/float(workers_total);
     else 
       workerpercent = 0.0;

     // Put in the percent done, clamping to valid values.
     gtk_progress_set_percentage(GTK_PROGRESS(itembar_), 
                                 CLAMP(workerpercent, 0.0, 1.0));
     gtk_progress_set_percentage(GTK_PROGRESS(overallbar_), 
                                 CLAMP(float(CurrentBytes + CurrentItems)/float(TotalBytes+TotalItems), 0.0, 1.0));
   }
      
   // Put in the ETA and cps meter
   if (CurrentCPS != 0)
     {      
       char Tmp[300];
       unsigned long ETA = (unsigned long)((TotalBytes - CurrentBytes)/CurrentCPS);
       g_snprintf(Tmp,299,_("%s remaining at %s/s"),
                  TimeToStr(ETA).c_str(), 
                  SizeToStr(CurrentCPS).c_str());
       gtk_label_set(GTK_LABEL(etalabel_), Tmp);
     }
   else 
     gtk_label_set(GTK_LABEL(etalabel_), _("Stalled"));
 
   // flush
   while (gtk_events_pending())
     gtk_main_iteration();
   
   Update = false;

   return !cancelled_;
}
