/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "filterbar.h"
#include "tooltips.h"

static GtkWidget* 
arrow_item(const char* name)
{
  GtkWidget* item       = gtk_menu_item_new();
  GtkWidget* hbox       = gtk_hbox_new(FALSE, 5);
  GtkWidget* arrow      = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
  GtkWidget* label      = gtk_label_new(name);
  
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), arrow, TRUE, TRUE, 0);

  gtk_container_add (GTK_CONTAINER(item), hbox);

  gtk_widget_show_all(item);

  return item;
}

GAptFilterBar::GAptFilterBar(Filter* filter)
  : filter_(filter), changing_state_(false)
{
  g_assert(filter_);

  filter_->add_view(this);

  bar_ = gtk_menu_bar_new();

  // Filters

  GtkWidget* filters      = gtk_menu_item_new_with_label(_("Filters:             "));

  gtk_menu_bar_append(GTK_MENU_BAR(bar_), filters);

  GtkWidget* menu       = gtk_menu_new();

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(filters), menu);

  GtkWidget* item       = gtk_menu_item_new_with_label(_("Clear all"));

  gtk_menu_append(GTK_MENU(menu), item);

  gtk_signal_connect(GTK_OBJECT(item), "activate",
                     GTK_SIGNAL_FUNC(::all_clear),
                     this);

  item                  = gtk_menu_item_new_with_label(_("Reverse all"));

  gtk_menu_append(GTK_MENU(menu), item);

  gtk_signal_connect(GTK_OBJECT(item), "activate",
                     GTK_SIGNAL_FUNC(::all_reverse),
                     this);

  // Status

  item       = arrow_item(_("Status"));
  menu       = gtk_menu_new();  

  gtk_menu_set_title(GTK_MENU(menu), _("Status Filter"));

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);
  gtk_menu_bar_append(GTK_MENU_BAR(bar_), item);

  item = gtk_tearoff_menu_item_new();
  gtk_menu_append(GTK_MENU(menu), item);

  item = gtk_menu_item_new_with_label(_("Clear"));
  gtk_menu_append(GTK_MENU(menu), item);

  gtk_signal_connect(GTK_OBJECT(item), "activate",
                     GTK_SIGNAL_FUNC(::status_clear),
                     this);

  item = gtk_menu_item_new_with_label(_("Reverse"));
  gtk_menu_append(GTK_MENU(menu), item);

  gtk_signal_connect(GTK_OBJECT(item), "activate",
                     GTK_SIGNAL_FUNC(::status_reverse),
                     this);

  int i = 0;
  while (i < Util::StatusTypeEnd)
    {
      Util::StatusType status = static_cast<Util::StatusType>(i);
      const char* s = Util::status_string(status);
      GtkWidget* mi = gtk_check_menu_item_new_with_label(s);
      // why you'd ever not do this is beyond me
      gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (mi), TRUE);
      gtk_check_menu_item_set_state       (GTK_CHECK_MENU_ITEM (mi), 
                                           filter_->get_status_filter(status));

      statuses_[mi] = status;

      gtk_signal_connect(GTK_OBJECT(mi), "toggled",
                         GTK_SIGNAL_FUNC(::status_toggled),
                         this);

      gtk_menu_append(GTK_MENU(menu), mi);

      //gtk_widget_show(mi);

      ++i;
    }

  // Priority

  item       = arrow_item(_("Priority"));
  menu       = gtk_menu_new();  

  gtk_menu_set_title(GTK_MENU(menu), _("Priority Filter"));

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);
  gtk_menu_bar_append(GTK_MENU_BAR(bar_), item);

  item = gtk_tearoff_menu_item_new();
  gtk_menu_append(GTK_MENU(menu), item);

  item = gtk_menu_item_new_with_label(_("Clear"));
  gtk_menu_append(GTK_MENU(menu), item);

  gtk_signal_connect(GTK_OBJECT(item), "activate",
                     GTK_SIGNAL_FUNC(::priority_clear),
                     this);

  item = gtk_menu_item_new_with_label(_("Reverse"));
  gtk_menu_append(GTK_MENU(menu), item);

  gtk_signal_connect(GTK_OBJECT(item), "activate",
                     GTK_SIGNAL_FUNC(::priority_reverse),
                     this);

  // don't want to depend on int mapping for this,
  //  since it's outside my control
  pkgCache::State::VerPriority iter_array[] = {
    pkgCache::State::Important,
    pkgCache::State::Required,
    pkgCache::State::Standard,
    pkgCache::State::Optional,
    pkgCache::State::Extra
  };

  // apt-pkg has no static function to convert to string,
  //  plus they wouldn't be capitalized
  // Don't i18n this until apt-pkg is; these aren't really 
  //  translatable anyway
  const char* pri_names[] = {
    "Important",
    "Required",
    "Standard",
    "Optional",
    "Extra"
  };

  i = 0;
  while (i < int(sizeof(iter_array)/sizeof(iter_array[0])))
    {
      pkgCache::State::VerPriority priority = iter_array[i];

      const char* s = pri_names[i];
      GtkWidget* mi = gtk_check_menu_item_new_with_label(s);
      // why you'd ever not do this is beyond me
      gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (mi), TRUE);
      gtk_check_menu_item_set_state       (GTK_CHECK_MENU_ITEM (mi), 
                                           filter_->get_priority_filter(priority));

      priorities_[mi] = priority;

      gtk_signal_connect(GTK_OBJECT(mi), "toggled",
                         GTK_SIGNAL_FUNC(::priority_toggled),
                         this);

      gtk_menu_append(GTK_MENU(menu), mi);

      //gtk_widget_show(mi);

      ++i;
    }
}

GAptFilterBar::~GAptFilterBar()
{
  filter_->remove_view(this);
}

void
GAptFilterBar::filter_changed()
{
  if (changing_state_) return;

  changing_state_ = true;

  map<GtkWidget*,Util::StatusType>::iterator i = statuses_.begin();
  while (i != statuses_.end()) 
    {
      gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->first),
                                    filter_->get_status_filter(i->second));

      ++i;
    }

  map<GtkWidget*,pkgCache::State::VerPriority>::iterator j = priorities_.begin();
  while (j != priorities_.end())
    {
      gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(j->first),
                                    filter_->get_priority_filter(j->second));

      ++j;
    }

  changing_state_ = false;
}

void
status_toggled(GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->status_toggled(w);
}


void
priority_toggled(GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->priority_toggled(w);
}


void
priority_reverse(GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->priority_reverse(w);
}

void
priority_clear  (GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->priority_clear(w);
}

void
status_reverse  (GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->status_reverse(w);
}

void
status_clear    (GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->status_clear(w);
}

void
all_reverse     (GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->all_reverse(w);
}

void
all_clear       (GtkWidget* w, gpointer data)
{
  GAptFilterBar* fb = static_cast<GAptFilterBar*>(data);
  g_return_if_fail(fb != 0);
  fb->all_clear(w);
}


void
GAptFilterBar::status_toggled(GtkWidget* w)
{
  if (changing_state_) return;

  map<GtkWidget*,Util::StatusType>::const_iterator i = statuses_.find(w);
  g_return_if_fail (i != statuses_.end());

  Util::StatusType status = i->second;

  changing_state_ = true;
  filter_->set_status_filter(status, 
                             GTK_CHECK_MENU_ITEM(w)->active);
  changing_state_ = false;
}


void
GAptFilterBar::priority_toggled(GtkWidget* w)
{
  if (changing_state_) return;

  map<GtkWidget*,pkgCache::State::VerPriority>::const_iterator i = priorities_.find(w);
  g_return_if_fail (i != priorities_.end());
  
  pkgCache::State::VerPriority priority = i->second;
  
  changing_state_ = true;
  filter_->set_priority_filter(priority, 
                               GTK_CHECK_MENU_ITEM(w)->active);
  changing_state_ = false;
}


void
GAptFilterBar::status_reverse  (GtkWidget* w)
{
  filter_->reverse_status_filter();
}

void
GAptFilterBar::status_clear    (GtkWidget* w)
{
  filter_->clear_status_filter();
}

void
GAptFilterBar::priority_reverse(GtkWidget* w)
{
  filter_->reverse_priority_filter();
}

void
GAptFilterBar::priority_clear  (GtkWidget* w)
{
  filter_->clear_priority_filter();
}

void
GAptFilterBar::all_reverse     (GtkWidget* w)
{
  filter_->reverse_entire_filter();
}

void
GAptFilterBar::all_clear       (GtkWidget* w)
{
  filter_->clear_entire_filter();
}


GtkWidget* 
GAptFilterBar::widget()
{
  return bar_;
}


