/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "pkgtree.h"

#include "pkgutil.h"

#include <algorithm>
#include <map>

#include <apt-pkg/strutl.h>

#include <gdk/gdkkeysyms.h>

////////////////////////
/// some utility gibberish

#include "pixmaps/checkoff.xpm"
#include "pixmaps/checkon.xpm"
#include "pixmaps/conflicts.xpm"
#include "pixmaps/depends.xpm"
#include "pixmaps/downgrade.xpm"
#include "pixmaps/minus.xpm"
#include "pixmaps/package.xpm"
#include "pixmaps/plus.xpm"
#include "pixmaps/radiooff.xpm"
#include "pixmaps/radioon.xpm"
#include "pixmaps/recommends.xpm"
#include "pixmaps/section.xpm"
#include "pixmaps/suggests.xpm"
#include "pixmaps/upgrade.xpm"
#include "pixmaps/replaces.xpm"

const char**
GAptPkgTree::pixmap_data(PixmapType p) 
{
  switch (p) {
    
  case ConflictsPixmap:
    return const_cast<const char**>(conflicts_xpm);
    break;
    
  case DependsPixmap:
    return const_cast<const char**>(depends_xpm);
    break;
    
  case RecommendsPixmap:
    return const_cast<const char**>(recommends_xpm);
    break;
    
  case SuggestsPixmap:
    return const_cast<const char**>(suggests_xpm);
    break;
      
  case PackagePixmap:
    return const_cast<const char**>(package_xpm);
    break;

  case FolderPixmap:
    return const_cast<const char**>(section_xpm);
    break;

  case ReplacesPixmap:
    return const_cast<const char**>(replaces_xpm);
    break;

  default:
    g_warning("unknown PixmapType, %s", __FUNCTION__);
    return 0;
    break;
  }
}


// These are for loading and saving config files

static const char*
_column_names[] = {
  "Delete",
  "Keep",
  "Install",
  "PackageName",
  "InstalledVersion",
  "AvailableVersion",
  "Status",
  "Section",
  "Priority",
  "Description"
};

static int _n_column_names = sizeof(_column_names)/sizeof(_column_names[0]);

static GAptPkgTree::ColumnType
string_to_column(const char* s)
{
  int i = 0;
  while (i < _n_column_names)
    {
      if (strcmp(s, _column_names[i]) == 0)
        {
          return static_cast<GAptPkgTree::ColumnType>(i);
        }

      ++i;
    }

  return GAptPkgTree::ColumnTypeEnd;
}


static const char*
column_to_string(GAptPkgTree::ColumnType ct)
{
  gint i = static_cast<int>(ct);

  g_return_val_if_fail(i < static_cast<int>(GAptPkgTree::ColumnTypeEnd), 0);
  g_return_val_if_fail(i < _n_column_names, 0);
  
  return _column_names[i];
}

static void
load_column_order(vector<GAptPkgTree::ColumnType> & columns)
{
  gpointer config_iterator;
  guint loaded = 0;

  config_iterator = gnome_config_init_iterator("/gnome-apt/ColumnOrder");

  if (config_iterator != 0)
    {
      gchar * col, * pos;
      columns.reserve(GAptPkgTree::ColumnTypeEnd);
      
      loaded = 0;
      while ((config_iterator = 
              gnome_config_iterator_next(config_iterator, 
                                         &col, &pos))) 
        {
          // shouldn't happen, but I'm paranoid
          if (pos == 0 || col == 0)
            {
              if (pos) g_free(pos);
              if (col) g_free(col);
              continue;
            }
          
          GAptPkgTree::ColumnType ct = string_to_column(col);
          
          gint index = atoi(pos);

          g_free(pos); pos = 0;
          g_free(col); col = 0;
          
          // the user could mangle the config file to make this happen
          if (static_cast<guint>(index) >= columns.size()) continue;
          
          columns[index] = ct;
          
          ++loaded;
        }
    }

  if (loaded != static_cast<guint>(GAptPkgTree::ColumnTypeEnd))
    {
      // Either there was no saved order, or something is busted - use
      // default order
      columns.clear();
      
      int i = 0;
      while (i < GAptPkgTree::ColumnTypeEnd)
        {
          columns.push_back(static_cast<GAptPkgTree::ColumnType>(i));
          ++i;
        }

      // Clean the section - otherwise an old entry could 
      //  remain forever and keep screwing us up in the future.
      gnome_config_clean_section("/gnome-apt/ColumnOrder");
      gnome_config_sync();
    }
  
  g_return_if_fail(columns.size() == static_cast<guint>(GAptPkgTree::ColumnTypeEnd));
}

static void
save_column_order(const vector<GAptPkgTree::ColumnType> & columns)
{
  g_return_if_fail(columns.size() == static_cast<guint>(GAptPkgTree::ColumnTypeEnd));

  int position = 0;
  vector<GAptPkgTree::ColumnType>::const_iterator i = columns.begin();
  while (i != columns.end())
    {
      gchar key[256];
      g_snprintf(key, 255, "/gnome-apt/ColumnOrder/%s", column_to_string(*i));
      gchar val[30];
      g_snprintf(val, 29, "%d", position);
      gnome_config_set_string(key, val);
      
      ++position;
      ++i;
    }

  gnome_config_sync();
}

static void
load_column_widths(map<GAptPkgTree::ColumnType,gint> & columns)
{
  gpointer config_iterator;
  guint loaded = 0;

  config_iterator = gnome_config_init_iterator("/gnome-apt/ColumnWidths");

  if (config_iterator != 0)
    {
      gchar * col, * width;
      
      loaded = 0;
      while ((config_iterator = 
              gnome_config_iterator_next(config_iterator, 
                                         &col, &width))) 
        {
          // shouldn't happen, but I'm paranoid
          if (width == 0 || col == 0) 
            {
              if (width) g_free(width);
              if (col)   g_free(col);
              continue;
            }          

          GAptPkgTree::ColumnType ct = string_to_column(col);
          
          gint wid = atoi(width);
          
          g_free(width); width = 0;
          g_free(col); col = 0;

          columns[ct] = wid;
          
          ++loaded;
        }
    }

  
  if (loaded != static_cast<guint>(GAptPkgTree::ColumnTypeEnd))
    {
      // Maybe the user just changed gnome-apt versions, or we are otherwise
      //  weirded up.
      // Clean the section - otherwise an old entry could 
      //  remain forever and keep screwing us up in the future.
      gnome_config_clean_section("/gnome-apt/ColumnWidths");
      gnome_config_sync();

      // We'll fill in some defaults on empty entries.
      int i = 0;
      while (i < GAptPkgTree::ColumnTypeEnd)
        {
          GAptPkgTree::ColumnType ct = static_cast<GAptPkgTree::ColumnType>(i);

          map<GAptPkgTree::ColumnType,gint>::iterator j = columns.find(ct);

          if (j == columns.end())
            {
              gint def = 0;

              switch (ct) {
              case GAptPkgTree::ColumnDelete:
              case GAptPkgTree::ColumnKeep:
              case GAptPkgTree::ColumnInstall:
                def = 18;
                break;
              case GAptPkgTree::ColumnName:
                def = 140;
                break;
              default:
                def = 50;
                break;         
              }

              columns[ct] = def;
            }

          ++i;
        }
    }
  
  g_return_if_fail(columns.size() == static_cast<guint>(GAptPkgTree::ColumnTypeEnd));  
}

static void
save_column_widths(const vector<GAptPkgTree::ColumnType> & columns,
                   const vector<gint> & widths)
{
  g_return_if_fail(columns.size() == widths.size());

  vector<GAptPkgTree::ColumnType>::const_iterator i = columns.begin();
  vector<gint>::const_iterator j = widths.begin();
  while (i != columns.end())
    {
      gchar key[256];
      g_snprintf(key, 255, "/gnome-apt/ColumnWidths/%s", column_to_string(*i));
      gchar val[30];
      g_snprintf(val, 29, "%d", *j);
      gnome_config_set_string(key, val);

#ifdef GNOME_ENABLE_DEBUG_ANNOYING
      ga_debug("Saved %s - %s (%d)\n", key, val, *j);
#endif

      ++i;
      ++j;
    }

  gnome_config_sync();
}


static void
load_column_visibility(map<GAptPkgTree::ColumnType,bool> & vis)
{
  gpointer config_iterator;
  guint loaded = 0;

  config_iterator = gnome_config_init_iterator("/gnome-apt/ColumnVisibility");

  if (config_iterator != 0)
    {
      gchar * col, * visibility;
      
      loaded = 0;
      while ((config_iterator = 
              gnome_config_iterator_next(config_iterator, 
                                         &col, &visibility))) 
        {
          // shouldn't happen, but I'm paranoid
          if (visibility == 0 || col == 0) 
            {
              if (visibility) g_free(visibility);
              if (col)        g_free(col);
              continue;
            }          

          GAptPkgTree::ColumnType ct = string_to_column(col);
          
          bool is_visible = true;
          
          const gchar* stripped = g_strstrip(visibility); // nuke leading/trailing
          if (g_strcasecmp(stripped,"false") == 0)
            {
              is_visible = false;
            }
          
          g_free(visibility); visibility = 0;
          g_free(col); col = 0;

          vis[ct] = is_visible;
          
          ++loaded;
        }
    }

  
  if (loaded != static_cast<guint>(GAptPkgTree::ColumnTypeEnd))
    {
      // Maybe the user just changed gnome-apt versions, or we are otherwise
      //  weirded up.
      // Clean the section - otherwise an old entry could 
      //  remain forever and keep screwing us up in the future.
      gnome_config_clean_section("/gnome-apt/ColumnVisibility");
      gnome_config_sync();

      // We'll default to true on empty entries.
      int i = 0;
      while (i < GAptPkgTree::ColumnTypeEnd)
        {
          GAptPkgTree::ColumnType ct = static_cast<GAptPkgTree::ColumnType>(i);

          map<GAptPkgTree::ColumnType,bool>::iterator j = vis.find(ct);

          if (j == vis.end())
            {
              vis[ct] = true;
            }

          ++i;
        }
    }
  
  g_return_if_fail(vis.size() == static_cast<guint>(GAptPkgTree::ColumnTypeEnd));    
}

static void 
save_column_visibility(const map<GAptPkgTree::ColumnType,bool> & vis)
{
  map<GAptPkgTree::ColumnType,bool>::const_iterator i = vis.begin();
  while (i != vis.end())
    {
      gchar key[256];
      g_snprintf(key, 255, "/gnome-apt/ColumnVisibility/%s", column_to_string(i->first));
      gnome_config_set_bool(key, i->second);

      ++i;
    }

  gnome_config_sync();
}

void 
GAptPkgTree::set_column_order(const vector<ColumnType> & cts)
{
  g_return_if_fail(cts.size() == columns_.size());

  // First we need to save the current widths,
  //  the DrawTree order matches columns_
  //  but column_widths_ is not kept in sync
  vector<gint> widths;
  tree_->get_column_widths(widths);

  vector<GAptPkgTree::ColumnType>::const_iterator i = columns_.begin();
  vector<gint>::const_iterator j = widths.begin();
  while (i != columns_.end())
    {
      column_widths_[*i] = *j;

      ++i;
      ++j;
    }

  columns_ = cts;

  reorder_columns();

  save_column_order(columns_);
}

gushort    
GAptPkgTree::type_column(ColumnType col) const 
{
  gushort i = 0;
  while (i < columns_.size()) {
    if (columns_[i] == col) return i;
    ++i;
  }
  g_return_val_if_fail(0,0); // warn and return 0
}

static const char* 
internal_column_name(GAptPkgTree::ColumnType ct, bool small)
{
  switch (ct) {
  case GAptPkgTree::ColumnDelete:
    if (small) return _("D");
    else return _("Delete");
    break;
    
  case GAptPkgTree::ColumnKeep:
    if (small) return _("K");
    else return _("Keep");
    break;
    
  case GAptPkgTree::ColumnInstall:
    if (small) return _("I");
    else return _("Install");
    break;
    
  case GAptPkgTree::ColumnName:
    return _("Package Name");
    break;
    
  case GAptPkgTree::ColumnCurrent:
    if (small) return _("Inst. Vers.");
    else return _("Installed Version");
    break;
    
  case GAptPkgTree::ColumnAvailable:
    if (small) return _("Avail. Vers.");
    else return _("Available Version");
    break;

  case GAptPkgTree::ColumnStatus:
    return _("Status");
    break;
    
  case GAptPkgTree::ColumnSection:
    return _("Section");
    break;
    
  case GAptPkgTree::ColumnPriority:
    return _("Priority");
    break;

  case GAptPkgTree::ColumnDescription:
    return _("Description");
    break;

  default:
    return "";
  }
}

const char* 
GAptPkgTree::column_name(GAptPkgTree::ColumnType ct)
{
  return internal_column_name(ct, true);
}

const char* 
GAptPkgTree::long_column_name(GAptPkgTree::ColumnType ct)
{
  return internal_column_name(ct, false);
}


///////////////////////////////////////////////
// Sort objects

typedef GAptPkgTree::Item * pred_target;

class NamePredicate  {
public:  
  bool operator() (pred_target a, pred_target b) {
    return (strcmp(a->name(),b->name()) < 0);
  }
    
} name_predicate;

class SectionPredicate  {
public:  
  bool operator() (pred_target a, pred_target b) {
    // apparently section can be 0 - we're going to say that 0 is greater 
    //  than any real string.
    const char* as = a->section();
    const char* bs = b->section();
    if (as && (bs == 0)) return true;// a < b 
    else if (as == 0) return false;      // a is >= b
    else if (bs == 0) return true;      // b > a
    else return (strcmp(as, bs) < 0);
  }

} section_predicate;

class PriorityPredicate  {
public:  
  bool operator() (pred_target a, pred_target b) {
    // This could be done with the enum, but it isn't quite as nice
    //  and the speed benefit is not user visible
    return (strcmp(a->priority(), b->priority()) < 0);
  }
} priority_predicate;


class StatusPredicate  {
public:  
  bool operator() (pred_target a, pred_target b) {
    return (a->status() < b->status());
  }
} status_predicate;

//// 

class FilterPredicate {
public:
  FilterPredicate(Filter* f) : filter_(f) {}

  bool operator() (pred_target a) {
    return a->filter(filter_);
  }  

private:
  Filter* filter_;

};


///////////////////// 
// GAptPkgTree

static const double _row_height = 25.0;
static const double _row_gap    = 2.0;
static const double _level_indent = 18.0; // each depth indents this much

GAptPkgTree::GAptPkgTree()
  : cache_(0), 
    sort_(SortNone),
    category_(CategoryNone),
    tree_(0),
    category_list_(0),
    broken_gc_(0),
    columns_(ColumnTypeEnd),
    status_(0),
    filter_(0),
    change_notify_queued_(false)
{
  tree_ = new DrawTree;

  int i = 0;
  while (i < PixmapTypeEnd) {
    pixmaps_[i] = 0;
    masks_[i] = 0;
    ++i;
  }

  load_column_order(columns_);

  category_list_ = new CategoryList(this);

  tree_->set_nodelist(category_list_);

  gdk_color_parse("red", &broken_color_);

  load_column_widths(column_widths_);
  load_column_visibility(column_vis_);

#ifdef GNOME_ENABLE_DEBUG
  if (column_widths_.size() != columns_.size())
    {
      g_warning("%u widths, %u columns!",
                column_widths_.size(), columns_.size());
    }
#endif

  reorder_columns();
}

GAptPkgTree::~GAptPkgTree()
{
  if (tree_ != 0)
    {
#ifdef GNOME_ENABLE_DEBUG
      ga_debug("Saving column widths\n");
#endif
      vector<gint> widths;
      tree_->get_column_widths(widths);

      ga_debug("First width: %d\n", widths[0]);

      save_column_widths(columns_,widths);
    }        

  save_column_visibility(column_vis_);

  int k = 0;
  while ( k < PixmapTypeEnd ) {
    if (pixmaps_[k]) {
      gdk_pixmap_unref(pixmaps_[k]);
      pixmaps_[k] = 0;
    }
    if (masks_[k]) {
      gdk_pixmap_unref(masks_[k]);
      masks_[k] = 0;
    }
    ++k;
  }

  if (filter_) filter_->remove_view(this);

  if (tree_) {
    delete tree_;
    tree_ = 0;
  }

  delete category_list_;
}

void 
GAptPkgTree::set_cache(GAptCache* cache)
{
  category_list_->clear_nodes();

  cache_ = cache;

  if (cache_ == 0) return;

  update_status();

  create_category(category_);
}

void 
GAptPkgTree::filter_changed()
{  
  CategoryList::iterator j = category_list_->begin();
  while (j != category_list_->end()) {
    Item* i = static_cast<Item*>(*j);
    if (i->filter(filter_) == false) i->hide();
    else i->show();
    ++j;
  }
  
  tree_->queue_recalc_rows();
  tree_->queue_display_update();
}

void 
GAptPkgTree::set_filter(Filter* f)
{
  if (filter_) {
    filter_->remove_view(this);
  }
  filter_ = f;
  if (filter_) {
    filter_->add_view(this);
  }
  filter_changed();
}

void 
GAptPkgTree::set_sort(SortType st)
{
  g_return_if_fail(st < SortTypeEnd);

  if (st == sort_) return;

  sort_ = st;

  CategoryList::iterator j = category_list_->begin();
  while (j != category_list_->end()) {
    Item* i = static_cast<Item*>(*j);
    i->sort(sort_);
    ++j;
  }
  
  tree_->queue_recalc_rows();
  tree_->queue_display_update();
}

void 
GAptPkgTree::set_category(CategoryType ct)
{
  g_return_if_fail(ct < CategoryTypeEnd);

  if (ct == category_) return;

  category_list_->clear_nodes();
  
  category_ = ct;
  
  if (cache_) create_category(ct);
}

void 
GAptPkgTree::set_visible(ColumnType ct, bool visibility)
{
  g_return_if_fail(tree_);

  column_vis_[ct] = visibility;

  // sync DrawTree
  gushort col = type_column(ct);
  
  tree_->set_visible(col, visibility);
}

bool 
GAptPkgTree::is_visible (ColumnType ct) const
{
  map<ColumnType,bool>::const_iterator x = column_vis_.find(ct);

  g_return_val_if_fail(x != column_vis_.end(), true);

#ifdef GNOME_ENABLE_DEBUG
  gushort col = type_column(ct);
  g_return_val_if_fail(x->second == tree_->is_visible(col), true);
#endif  

  return x->second;
}

void 
GAptPkgTree::set_width  (ColumnType ct, double width)
{
  g_warning(__FUNCTION__);
}

void 
GAptPkgTree::move_selection(pkgCache::Package* p)
{
  Item* node = 0;

  CategoryList::iterator i = category_list_->begin();
  while ((node == 0) && (i != category_list_->end()))
    {
      DrawTree::Node::iterator j = (*i)->begin();
      while ((node == 0) && (j != (*i)->end()))
        {
          Pkg* pkg = reinterpret_cast<Pkg*>(*j);
          if (pkg->match(p)) 
            {
              node = pkg;
              g_return_if_fail(node != 0);
              if (node->hidden()) return;
              if ((*i)->expanded() == false)
                (*i)->expand(); // expand category if it isn't.
            }
          ++j;
        }
      ++i;
    }

  g_return_if_fail(node != 0);

  tree_->change_selection(node);
  tree_->focus_node(node);
}

void 
GAptPkgTree::reorder_columns()
{
  vector<DrawTree::ColumnStyle> styles;
  vector<string> names;
  vector<guint> shortcuts1;
  vector<guint> shortcuts2;

  vector<ColumnType>::iterator i = columns_.begin();
  while (i != columns_.end()) {
    switch (*i) {
    case ColumnDelete:
      shortcuts1.push_back('-');
      //      shortcuts2.push_back('d');
      styles.push_back(DrawTree::BoolCol);
      break;
    case ColumnKeep:
      shortcuts1.push_back('=');
      //      shortcuts2.push_back('k');
      styles.push_back(DrawTree::BoolCol);
      break;
    case ColumnInstall:
      shortcuts1.push_back('+');
      //      shortcuts2.push_back('i');
      styles.push_back(DrawTree::BoolCol);
      break;
    case ColumnName:
      styles.push_back(DrawTree::TreeCol);
      break;
    default:
      styles.push_back(DrawTree::DrawCol);
      break;
    }

    names.push_back(string(column_name(*i)));

    ++i;
  }

  tree_->set_columns(styles, names, shortcuts1, shortcuts2); 

  vector<gint> widths;

  vector<ColumnType>::iterator j = columns_.begin();
  guint count = 0;
  while (j != columns_.end())
    {
      // Collect widths in a vector
      map<ColumnType,gint>::iterator f = column_widths_.find(*j);

      if (f != column_widths_.end())
        {
          widths.push_back(f->second);
        }
      else 
        {
          g_warning("Column width not found: should not happen");
          widths.push_back(12);
        }

      // Set visibilities
      map<ColumnType,bool>::iterator v = column_vis_.find(*j);
      
      if (v != column_vis_.end())
        {
          tree_->set_visible(count, v->second);
        }
      else 
        {
          g_warning("Column visibility not found: should not happen");
          tree_->set_visible(count, true);
        }

      ++j;
      ++count;
    }

  tree_->set_column_widths(widths);
}


// Someone please break this function up!
void 
GAptPkgTree::create_category(CategoryType ct)
{
  g_assert(cache_);

  switch (ct) {
  case CategoryAlpha:
    {
      map<char,Category*> letters;
      
      pkgCache::PkgIterator i = cache_->PkgBegin();
      
      while (!i.end()) {
        const char* pkgname = i.Name();
        char first = pkgname[0];
        if (first == '\0') {
          ++i;
          continue;
        }
        
        char upper = isalpha(first) ? toupper(first) : first;
        
        map<char,Category*>::iterator cat = letters.find(upper);
        
        Category * c = 0;
        if (cat == letters.end()) {
          char buf[2];
          buf[0] = upper;
          buf[1] = '\0';
          c = new Category(buf, this);
          letters[upper] = c;           // temporary quick tree
          category_list_->add_node(c);
        }
        else {
          c = cat->second;
        }
        
        Item* item = new Pkg(Item::PackageItem, i, this, c);
        
        c->add(item);
        
        ++i;
      } 

      std::stable_sort((vector<Item*>::iterator)category_list_->begin(), 
                       (vector<Item*>::iterator)category_list_->end(), 
                       name_predicate);
    }
    break; // Alpha end

  case CategorySection:
    {
      map<string,Category*> sections;
      Category* nosection_bucket = 0;

      pkgCache::PkgIterator i = cache_->PkgBegin();

      while (!i.end()) {
        Category * c = 0;
        const char* section = i.Section();
        
        if (section == 0) {           // apparently this can happen.
          if (nosection_bucket == 0) {
            nosection_bucket = new Category(_("no section"), this);
            category_list_->add_node(nosection_bucket);
          }
          c = nosection_bucket;
        }        
        else {
          map<string,Category*>::iterator sec = sections.find(string(section));
          
          if (sec == sections.end()) {
            c = new Category(section, this);
            sections[string(section)] = c;           // temporary quick tree
            category_list_->add_node(c);
          }
          else {
            c = sec->second;
          }
        }
        
        g_assert(c != 0);
        
        Item* item = new Pkg(Item::PackageItem, i, this, c);
        
        c->add(item);
        
        ++i;
      }

      std::stable_sort((vector<Item*>::iterator)category_list_->begin(), 
                       (vector<Item*>::iterator)category_list_->end(), 
                       section_predicate);
    }
    break; // Section end
    
    // It's stupid to do priority this way since we have 
    //  only a few categories, but I am lazy for now and 
    //  pasted the section code
  case CategoryPriority:
    {
      map<pkgCache::State::VerPriority,Category*> priorities;
      
      pkgCache::PkgIterator i = cache_->PkgBegin();
      
      while (!i.end()) {
        Category * c = 0;

        pkgCache::State::VerPriority priority;        
        const char* pri_string = 0;

        pri_string = Util::priority_string(i, cache_, &priority);
        
        map<pkgCache::State::VerPriority,Category*>::iterator pri = priorities.find(priority);
          
        if (pri == priorities.end()) {
          c = new Category(pri_string, this);
          priorities[priority] = c;           // temporary quick tree
          category_list_->add_node(c);
        }
        else {
          c = pri->second;
        }

        g_assert(c != 0);
        
        Item* item = new Pkg(Item::PackageItem, i, this, c);
        
        c->add(item);
        
        ++i;
      }
      std::stable_sort((vector<Item*>::iterator)category_list_->begin(), 
                       (vector<Item*>::iterator)category_list_->end(), 
                       priority_predicate);
    }
    break; // Priority end

    // Look, we can do this one a stupid way too!
  case CategoryStatus:
    {
      map<Util::StatusType,Category*> statuses;

      pkgCache::PkgIterator i = cache_->PkgBegin();

      while (!i.end()) {
        Category * c = 0;
        Util::StatusType status = Util::status_type(i,cache_); 

        map<Util::StatusType,Category*>::iterator sec = statuses.find(status);
        
        if (sec == statuses.end()) {
          c = new Category(Util::status_string(status), this);
          statuses[status] = c;           // temporary quick tree
          category_list_->add_node(c);
        }
        else {
          c = sec->second;
        }
        
        g_assert(c != 0);
        
        Item* item = new Pkg(Item::PackageItem, i, this, c);
        
        c->add(item);
        
        ++i;
      }
      std::stable_sort((vector<Item*>::iterator)category_list_->begin(), 
                       (vector<Item*>::iterator)category_list_->end(), 
                       status_predicate);
    }
    break; // Status end

  case CategoryNone: // FALL THRU
  default:
    {
      Category * c = new Category(_("All Packages"), this);
      category_list_->add_node(c);
      
      pkgCache::PkgIterator i = cache_->PkgBegin();
      
      while (!i.end()) {
        Item* item = new Pkg(Item::PackageItem, i, this, c);
        
        c->add(item);
        
        ++i;
      } 
    }
    break;
  };

  // Now sort the category using current sort
  vector<DrawTree::Node*>::iterator j = category_list_->begin();
  while (j != category_list_->end()) {
    Item* it = static_cast<Item*>(*j);
    it->sort(sort_);
    ++j;
  }      

  // and filter
  filter_changed();
}

void 
GAptPkgTree::pixmap(PixmapType p, GdkPixmap** pixmap, GdkPixmap ** mask)
{
  g_return_if_fail(pixmap);
  g_return_if_fail(mask);

  if (pixmaps_[p] == 0) {
    if (tree_ == 0) return;
    if (!GTK_WIDGET_REALIZED(GTK_WIDGET(tree_->widget()))) return;
 
    const char** data = pixmap_data(static_cast<PixmapType>(p));
    
    if (data) {
      GdkPixmap * pix = 0;
      GdkPixmap * m = 0;
#if 0
      GdkImlibImage* im = 0;
      static GdkImlibColor shape_color = { 0xff, 0, 0xff, 0 };

      im = gdk_imlib_create_image_from_xpm_data (data);
      
      if (im != 0) 
        {
          gdk_imlib_set_image_shape (im, &shape_color);
          
          /* Render the image, return it */
          gdk_imlib_render (im, im->rgb_width, im->rgb_height);
          
          pix = gdk_imlib_move_image (im);
          m   = gdk_imlib_move_mask  (im);
          
          gdk_imlib_destroy_image (im);
          im = 0;
        }
#else
      pix = gdk_pixmap_create_from_xpm_d(tree_->widget()->window,
                                         &m,
                                         &tree_->widget()->style->bg[GTK_STATE_NORMAL],
                                         const_cast<gchar**>(data));
#endif     
 
      pixmaps_[p] = pix;
      masks_[p] = m;

      if (!pixmaps_[p]) g_warning("Pixmap creation failed, %s", __FUNCTION__);
      if (!masks_[p]) g_warning("Mask creation failed, %s", __FUNCTION__);
    }
  }
  *pixmap = pixmaps_[p];
  *mask = masks_[p];
}

void 
GAptPkgTree::set_broken_color(GdkColor* c)
{
  g_return_if_fail(c != 0);

  broken_color_ = *c;

  update_broken_gc();

  tree_->queue_display_update();
}

void
GAptPkgTree::update_broken_gc()
{
  if (tree_ == 0) return;
  if (!GTK_WIDGET_REALIZED(GTK_WIDGET(tree_->widget()))) return;
  
  if (broken_gc_ == 0)
    {
      broken_gc_ = gdk_gc_new(GTK_WIDGET(tree_->widget())->window);
    }  

  gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(tree_->widget())), 
                  &broken_color_);
  gdk_gc_set_foreground(broken_gc_, &broken_color_);
}

GdkGC* 
GAptPkgTree::broken_gc()
{
  if (broken_gc_ != 0) return broken_gc_;
  else 
    {
      update_broken_gc();
      return broken_gc_;
    }
}

//////////////////////////////////////////////// Item

GAptPkgTree::Item::Item(ItemRelationshipType rel,
                        GAptPkgTree* t)
  : tree_(t),
    relation_(rel)
{

}

GAptPkgTree::Item::~Item()
{

}

void 
GAptPkgTree::Item::sort(GAptPkgTree::SortType st)
{
  if (children_.empty()) return; // minor optimization

  vector<Item*>::iterator b = reinterpret_cast<vector<Item*>::iterator>(children_.begin());
  vector<Item*>::iterator e = reinterpret_cast<vector<Item*>::iterator>(children_.end());

  switch (st) {
  case SortAlpha:
    std::stable_sort(b, e, name_predicate);
    break;

  case SortSection:
    std::stable_sort(b, e, section_predicate);
    break;

  case SortPriority:
    std::stable_sort(b, e, priority_predicate);
    break;

  case SortStatus:
    std::stable_sort(b, e, status_predicate);
    break;

  case SortNone: // FALL THRU
  default:
    g_warning(__FUNCTION__);
    break;
  }

  // should we do it recursively? Why not.
  vector<Node*>::iterator i = children_.begin();
  while (i != children_.end()) {
    Item* it = static_cast<Item*>(*i);
    it->sort(st);
    ++i;
  }
}


gint 
package_changes_idle(gpointer data)
{
  GAptPkgTree* tree = static_cast<GAptPkgTree*>(data);
  g_return_val_if_fail(tree != 0, FALSE);

  tree->do_package_changes();
  return FALSE; // remove idle
}

void 
GAptPkgTree::package_change_notify()
{
  if (!change_notify_queued_) { 
    gtk_idle_add(package_changes_idle, this); 
    tree_->queue_recalc_rows(); // since shown dependencies might change
    tree_->queue_display_update();
    change_notify_queued_ = true;
  }
}

void 
GAptPkgTree::selection_change_notify(pkgCache::Package* p)
{
  if (status_) status_->set_selection(p);
}

void 
GAptPkgTree::do_package_changes()
{
  change_notify_queued_ = false;

  update_status();
}

void 
GAptPkgTree::update_status()
{
  if (status_ && cache_) {
    string s;
    char buf[100];
    g_snprintf(buf,100,_("%lu to install; "),
               cache_->InstCount());
    s += buf;
    g_snprintf(buf,100,_("%lu to delete; "),cache_->DelCount());
    s += buf;
    if (cache_->UsrSize() >= 0)
      g_snprintf(buf,100,_("%s will be used."), SizeToStr(cache_->UsrSize()).c_str());
    else
      g_snprintf(buf,100,_("%s will be freed."), SizeToStr(-1*cache_->UsrSize()).c_str());
    s += buf;

    if (cache_->BrokenCount() != 0)
      {
        g_snprintf(buf,100,_("  *** %lu broken packages ***"), cache_->BrokenCount());
        s += buf;
      }
        
    status_->set_status(s.c_str());
  }
}

///////////////////////////////////////// Category


GAptPkgTree::Category::Category(const char* name, GAptPkgTree* t)
  : Item(CategoryItem, t), name_(name), expanded_(false)
{

}

GAptPkgTree::Category::~Category()
{

}

void 
GAptPkgTree::Category::add   (Item* child)
{
  children_.push_back(child);
}

void 
GAptPkgTree::Category::remove(Item* child)
{
  vector<Node*>::iterator i = find(children_.begin(), children_.end(), child);
  g_return_if_fail(i != children_.end());
  children_.erase(i);
}

void
GAptPkgTree::Category::expand()
{
  // we keep children around always
  expanded_ = true;
}

void
GAptPkgTree::Category::collapse()
{
  // we keep children around always
  expanded_ = false;
}

bool 
GAptPkgTree::Category::get_bool(gushort column)
{
  // for now
  return false;
}

void
GAptPkgTree::Category::set_bool(gushort column, bool state)
{
  // nothing for now.
}


bool
GAptPkgTree::Category::display_bool(gushort column)
{
  return false;
}

bool 
GAptPkgTree::Category::filter(Filter* filter)
{  
  if (filter == 0) return true;

  vector<Item*>::iterator b = 
    reinterpret_cast<vector<Item*>::iterator>(children_.begin());
  vector<Item*>::iterator e = 
    reinterpret_cast<vector<Item*>::iterator>(children_.end());
  
  FilterPredicate fp(filter);
  
  bool one_shown = false;

  while (b != e) {
    if (fp(*b)){
      (*b)->show();
      one_shown = true;
    }
    else (*b)->hide();
    ++b;
  }

  return one_shown;
}

void
GAptPkgTree::Category::draw_column(gushort column, gushort remaining,
                                   GtkStateType state,
                                   GdkDrawable* drawable, 
                                   GdkRectangle* rect)
{
  GdkPixmap* pixmap = 0;
  GdkPixmap* mask   = 0;
  gint ourx = rect->x;

  switch (tree()->column_type(column)) {
  case ColumnDelete:
  case ColumnKeep:
  case ColumnInstall:
    g_warning("Boolean column being drawn!");
    break;

  case ColumnName:
    tree()->pixmap(GAptPkgTree::FolderPixmap, &pixmap, &mask);

    if (pixmap) {
      // The folder pixmap is 18 wide, unlike the others
      static const gint pixw = 18, pixh = 16;
      gint x = ourx; gint y = static_cast<gint>(rect->y + 2);
      GdkGC* gc = tree()->tree()->widget()->style->black_gc;

      gdk_gc_set_clip_mask(gc, mask);
      gdk_gc_set_clip_origin (gc, x, y);

      gdk_draw_pixmap(drawable, 
                      gc,
                      pixmap,
                      0, 0,
                      x,
                      y,
                      pixw, 
                      pixh);


      gdk_gc_set_clip_mask(gc, 0); // unset the mask
      gdk_gc_set_clip_origin (gc, 0, 0);

      ourx += pixw + 2;
    }

    gtk_paint_string(tree()->tree()->widget()->style,
                     drawable,
                     state,
                     rect,
                     tree()->tree()->widget(),
                     "apt_category_column",
                     ourx,
                     static_cast<gint>(rect->y + (rect->height/1.6)),
                     name_.c_str());
    break;
  default:
    // category doesn't draw on the other columns
    break;
  }
}

void
GAptPkgTree::Category::select()
{
  tree()->selection_change_notify(0);
}

void
GAptPkgTree::Category::unselect()
{
  tree()->selection_change_notify(0);
}

////////////////////////////////////////////////// Pkg

static GAptPkgTree::Item::ItemRelationshipType
DepType_2_ItemRelationshipType(pkgCache::Dep::DepType dt)
{
  // This could maybe sped up with an array, but not as safe.
  //  compiler should make a jump table out of the switch
  switch (dt) {
  case pkgCache::Dep::Depends:
    return GAptPkgTree::Item::DependencyItem;
    break;
  case pkgCache::Dep::PreDepends:    
    return GAptPkgTree::Item::PreDependencyItem;
    break;
  case pkgCache::Dep::Suggests:
    return GAptPkgTree::Item::SuggestedItem;
    break;
  case pkgCache::Dep::Recommends:
    return GAptPkgTree::Item::RecommendedItem;
    break;
  case pkgCache::Dep::Conflicts:
    return GAptPkgTree::Item::ConflictingItem;
    break;
  case pkgCache::Dep::Replaces:
    return GAptPkgTree::Item::ReplacedItem;
    break;
  default:
    g_warning("Bad DepType %s", __FUNCTION__);
    return GAptPkgTree::Item::InvalidItem;
  }
}

GAptPkgTree::Pkg::Pkg(ItemRelationshipType rel,
                      pkgCache::Package* pkg, 
                      GAptPkgTree* t, 
                      Item* p)
  : Item(rel, t), pkg_(pkg), parent_(p)
{
  g_assert(tree_);
  g_assert(tree_->cache());
  
}

GAptPkgTree::Pkg::~Pkg()
{

}


void
GAptPkgTree::Pkg::expand()
{
  // Create our children
  
  // Avoid infinite travel around dependency graph
  // It would be much cleaner to save a flag, but we 
  //  are worried about RAM 
  if (parent_ &&           // have at least category...
      parent_->parent()) return;  // we appear to be a dependency...
  // need to allow another level for virtual packages provides

  pkgCache::PkgIterator pi = package(*(tree_->cache()));
  g_assert(!pi.end());

  pkgCache::DepIterator di = Util::depends(pi, tree_->cache());
    
  while (!di.end()) {
    ItemRelationshipType irt = 
      DepType_2_ItemRelationshipType(static_cast<pkgCache::Dep::DepType>(di->Type));
    Pkg* np = new Pkg(irt, di.TargetPkg(), tree_, this);
    children_.push_back(np);
    ++di;
  }

  // sort children
  sort(tree_->sort());
  
  // FIXME, add provides also?
}

void
GAptPkgTree::Pkg::collapse()
{
  // Nuke children
  vector<Node*>::iterator i = children_.begin();
  while (i != children_.end()) {
    delete *i;
    ++i;
  }
  children_.clear();
}


bool
GAptPkgTree::Pkg::expandable()
{
  // Eventually we'll expand to a another level (provides)
  if (parent_ &&           // have at least category...
      parent_->parent()) return false; // we appear to be a dependency...

  pkgCache::PkgIterator pi = package(*(tree_->cache()));
  pkgCache::DepIterator di = Util::depends(pi, tree_->cache());

  return !di.end();
}

// The problem is that if the installed version changes, we have to
// change the dependency list. This is majorly slowing down our life,
// but not much help for it.
void 
GAptPkgTree::Pkg::refresh_expansion()
{
  if (!children_.empty())
    {
      collapse();
      expand();
    }
}

const char*
GAptPkgTree::Pkg::priority()
{
  pkgCache::PkgIterator i = package(*(tree_->cache()));
  pkgCache::VerIterator vi = i.CurrentVer();
  if (!vi.end()) {
    return vi.PriorityType();
  }
  else return _("No current version");
}


Util::StatusType
GAptPkgTree::Pkg::status()
{
  pkgCache::PkgIterator i = package(*(tree_->cache()));
  return Util::status_type(i, tree_->cache());
}

bool 
GAptPkgTree::Pkg::get_bool(gushort column)
{
  // I hope this isn't expensive; I really have no idea.
  pkgCache::PkgIterator i = package(*(tree_->cache()));


  switch (tree()->column_type(column)) {
  case ColumnDelete:
    {
      return Util::removed(i,tree_);
    }
    break;

  case ColumnKeep:
    {
      return Util::kept(i,tree_);
    }
    break;

  case ColumnInstall:
    {
      return Util::installed(i,tree_);
    }
    break;
    
  default:
    g_warning("Setting bool on non-bool column!");
    break;
  }
  return false;
}

void
GAptPkgTree::Pkg::set_bool(gushort column, bool state)
{
  switch (tree()->column_type(column)) {
  case ColumnDelete:
    {
      if (state)
        Util::remove(pkg_,tree_);
      else 
        Util::unremove(pkg_,tree_);
    }
    break;

  case ColumnKeep:
    {
      if (state)
        Util::keep(pkg_,tree_);
      else 
        Util::unkeep(pkg_,tree_);
    }
    break;

  case ColumnInstall:
    {
      if (state)
        Util::install(pkg_,tree_);
      else
        Util::uninstall(pkg_,tree_);
    }
    break;
    
  default:
    g_warning("Setting bool on non-bool column!");
    return;
    break;
  }
}


bool
GAptPkgTree::Pkg::display_bool(gushort column)
{
  pkgCache::PkgIterator i = package(*(tree_->cache()));

  switch (tree()->column_type(column)) {
  case ColumnDelete:
    {
      return Util::can_change_remove(i,tree_);
    }
    break;

  case ColumnKeep:
    {
      return Util::can_change_keep(i,tree_);
    }
    break;

  case ColumnInstall:
    {
      return Util::can_change_install(i,tree_);
    }
    break;
    
  default:
    g_warning("Asking whether to display bool on non-bool column!");
    return false;
    break;
  }
  
  return false; // not reached
}

void
GAptPkgTree::Pkg::draw_column(gushort column, gushort remaining,
                              GtkStateType state,
                              GdkDrawable* drawable, 
                              GdkRectangle* rect)
{
  gint ourx = rect->x;

  pkgCache::PkgIterator i = package(*(tree_->cache()));

  bool virtual_package = (i->ProvidesList != 0);

  const char* s = 0;  
  char* alloced_s = 0;

  switch (tree()->column_type(column)) {
  case ColumnDelete:
  case ColumnKeep:
  case ColumnInstall:
    g_warning("Boolean column being drawn!");
    break;

  case ColumnName:
    {
      s = i.Name();

      GdkPixmap* pixmap = 0;
      GdkPixmap* mask = 0;

      switch (relation_) {
      case GAptPkgTree::Item::CategoryItem:
        tree()->pixmap(GAptPkgTree::FolderPixmap, &pixmap, &mask);
        break;
        
      case GAptPkgTree::Item::PackageItem:
        tree()->pixmap(GAptPkgTree::PackagePixmap, &pixmap, &mask);
        break;
        
      case GAptPkgTree::Item::DependencyItem:
        tree()->pixmap(GAptPkgTree::DependsPixmap, &pixmap, &mask);
        break;
        
      case GAptPkgTree::Item::PreDependencyItem:
        tree()->pixmap(GAptPkgTree::DependsPixmap, &pixmap, &mask);
        break;
        
      case GAptPkgTree::Item::RecommendedItem:
        tree()->pixmap(GAptPkgTree::RecommendsPixmap, &pixmap, &mask);
        break;
        
      case GAptPkgTree::Item::SuggestedItem:
        tree()->pixmap(GAptPkgTree::SuggestsPixmap, &pixmap, &mask);
        break;
        
      case GAptPkgTree::Item::ConflictingItem:
        tree()->pixmap(GAptPkgTree::ConflictsPixmap, &pixmap, &mask);
        break;
        
      case GAptPkgTree::Item::ReplacedItem:
        tree()->pixmap(GAptPkgTree::ReplacesPixmap, &pixmap, &mask);
        break;
        
      default:
        pixmap = 0;
        break;
      }

      if (pixmap) {
        // I just happen to know all these pixmaps are 16x16; 
        //  this is a pure hack to save querying them each time.
        static const gint pixw = 16, pixh = 16;
        GdkGC* gc = tree()->tree()->widget()->style->white_gc;
        gint x = ourx; gint y = static_cast<gint>(rect->y + 2);

        gdk_gc_set_clip_mask(gc, mask);
        gdk_gc_set_clip_origin (gc, x, y);

        gdk_draw_pixmap(drawable, 
                        gc,
                        pixmap,
                        0, 0,
                        x,
                        y,
                        pixw, 
                        pixh);

        gdk_gc_set_clip_mask(gc, 0); // unset mask
        gdk_gc_set_clip_origin(gc, 0, 0);

        ourx += pixw + 2;
      }
    }
    break;

  case ColumnCurrent:
    {
      if (virtual_package) 
        {
          s = ""; // virtual package, field doesn't apply
        }
      else 
        {
          pkgCache::VerIterator vi = i.CurrentVer();
          if (!vi.end()) s = vi.VerStr();
          else s = _("None");
        }
    }
    break;

  case ColumnAvailable:  
    {
      if (virtual_package)
        {
          s = ""; // n/a
        }
      else 
        {
          pkgDepCache::StateCache & state = (*tree()->cache())[i];
          pkgCache::VerIterator vi = state.CandidateVerIter(*tree()->cache()); //  i.TargetVer();
      
          if (!vi.end())
            s = vi.VerStr();
          else 
            s = _("None");
        }
    }
    break;

  case ColumnSection:
    s = i.Section();
    break;

  case ColumnPriority:
    {
      if (virtual_package)
        {
          s = "";
        }
      else 
        {
          s = Util::priority_string(i, tree()->cache(), 0);
        }
    }
    break;

  case ColumnStatus:
    {
      if (virtual_package)
        {
          s = _("Virtual package");
        }
      else
        {
          s = Util::status_string(Util::status_type(i,tree_->cache()));
        }
    }
    break;

  case ColumnDescription:
    {
      pkgRecords::Parser* p = Util::pkg_parser(i,tree_->cache());
      if (p != 0)
        {
          string tmp = p->ShortDesc();
          
          // this is fairly annoying
          if (tmp.empty()) s = 0;
          else 
            {
              s = alloced_s = g_strdup(tmp.c_str());
            }
        }
    }
    break;

  default:
    g_warning("Unknown column type");
    return;
    break;
  }

  if (s) {
    Util::StatusType status = Util::status_type(i,tree_->cache());

    gint x = ourx;
    gint y = static_cast<gint>(rect->y + (rect->height/1.45));

    if (status == Util::StatusNowBroken ||
        status == Util::StatusInstBroken)
      {
        gdk_gc_set_clip_rectangle(tree()->broken_gc(), 
                                  rect);
        gdk_draw_string(drawable,
                        tree()->tree()->widget()->style->font,
                        tree()->broken_gc(),
                        x, 
                        y,
                        s);

        gdk_gc_set_clip_rectangle(tree()->broken_gc(), 
                                  0);
      }
    else 
      {
        gtk_paint_string(tree()->tree()->widget()->style,
                         drawable,
                         state,
                         rect,
                         tree()->tree()->widget(),
                         "apt_package_column",
                         x,
                         y,
                         s);
      }
  }
  if (alloced_s) g_free(alloced_s);
}


void
GAptPkgTree::Pkg::select()
{
  tree()->selection_change_notify(pkg_);
}

void
GAptPkgTree::Pkg::unselect()
{
  tree()->selection_change_notify(0);
}

bool 
GAptPkgTree::Pkg::filter(Filter* filter)
{
  if (filter == 0) return true;

  pkgCache::PkgIterator i = package(*(tree_->cache()));
  // if we aren't included, return false
  if (filter->include_package(i, tree_->cache()) == false) return false;
  else return true;
}


/////////////////////// Category list

GAptPkgTree::CategoryList::CategoryList(GAptPkgTree* tree)
  : tree_(tree)
{

}

GAptPkgTree::CategoryList::~CategoryList()
{
  vector<Item*>::iterator j = nodes_.begin();
  while (j != nodes_.end()) {
    delete *j;
    ++j;
  }
}
 
void
GAptPkgTree::CategoryList::add_node(Item* node)
{
  g_return_if_fail(node != 0);
  nodes_.push_back(node);
  tree_->tree()->queue_recalc_rows();
  tree_->tree()->queue_display_update();
}

void 
GAptPkgTree::CategoryList::add_nodes(vector<Item*> & nodes)
{
  vector<Item*>::iterator si = nodes.begin();
  while (si != nodes.end()) {
    nodes_.push_back(*si);
    ++si;
  }
  tree_->tree()->queue_recalc_rows();
  tree_->tree()->queue_display_update();
}

void
GAptPkgTree::CategoryList::remove_node(Item* node)
{
  g_return_if_fail(node != 0);
  vector<Item*>::iterator i = find(nodes_.begin(), nodes_.end(), node);

  g_return_if_fail(i != nodes_.end());
  
  if (node == tree_->tree()->selected_node()) 
    tree_->tree()->change_selection(0);

  delete node;

  nodes_.erase(i);

  tree_->tree()->queue_recalc_rows();

  tree_->tree()->queue_display_update();
}

void
GAptPkgTree::CategoryList::clear_nodes()
{
  tree_->tree()->change_selection(0);
  vector<Item*>::iterator j = nodes_.begin();
  while (j != nodes_.end()) {
    delete *j;
    ++j;
  }
  nodes_.clear();
  tree_->tree()->queue_recalc_rows();
  tree_->tree()->queue_display_update();
}
