/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <signal.h>
#include <asm/page.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "details.h"
#include "filter.h"
#include "gps.h"
#include "msgbox.h"
#include "renice.h"
#include "tabledata.h"
#include "diefast.h"
#include "history.h"

#include "xray-tux.xpm"

/* defined in details.cc */
#define POSIXSIG 19

static char *posignames[]={"SIGHUP", "SIGINT", "SIGQUIT","SIGILL", "SIGABRT","SIGFPE",
                  "SIGKILL","SIGSEGV","SIGPIPE","SIGALRM","SIGTERM","SIGUSR1",
		  "SIGUSR2","SIGCHLD","SIGCONT","SIGSTOP","SIGTSTP","SIGTTIN",
		  "SIGTTOU"};

static gint posigvalues[]={SIGHUP,SIGINT,SIGQUIT,SIGILL,SIGABRT,SIGFPE,
                  SIGKILL,SIGSEGV,SIGPIPE,SIGALRM,SIGTERM,SIGUSR1,
                  SIGUSR2,SIGCHLD,SIGCONT,SIGSTOP,SIGTSTP,SIGTTIN,
                  SIGTTOU};

GtkWidget *psl,*show_long_names_item,*item2,*refb,*kb[2],*sc[4];

gint rows_on_list=0;
tabledata *mainlist=NULL;

gint show_long_names=FALSE;
gint continuum=FALSE;
gint mem_unit=1; // 0=bytes, 1=kbytes, 2=mbytes (3=giga?)

gint to_tag=-1;

gint pid_sel=-1;
gint waysort[10]={0,0,0,0,0,0,0,0,0,0}; /* 0=asc 1=des */
gint last_sort=0;

gchar col_title[10][10]={"PID","Name","Owner","State","%CPU","Size",
			 "RSS","Nice","Pri","Start"};

time_t btime=0;
gint have_filter=FALSE;

struct filter_info fi_work={0,0,"0"},fi_stru;

int main (int argc, char *argv[])
{
  int i,j;
  gint dashc=FALSE,dashl=FALSE;

  for(i=1;i<argc;i++) {
    if (strcmp(argv[i],"-c")==0) {
      dashc=TRUE;
      continue;
    }
    if (strcmp(argv[i],"-l")==0) {
      dashl=TRUE;
      continue;
    }
    if ((i+1)<argc)
      if (strcmp(argv[i],"-sc")==0) {
	j=atoi(argv[i+1]);
	i++;
	if (j>9) j=0;
	last_sort=j;
	continue;
      }
    if (strcmp(argv[i],"-sd")==0) {
      for(j=0;j<10;j++)
	waysort[j]=1;
      continue;
    }
    if (strcmp(argv[i],"-sa")==0) {
      for(j=0;j<10;j++)
	waysort[j]=0;
      continue;
    }

    if ((strcmp(argv[i],"--version")==0)||(strcmp(argv[i],"-v")==0)) {
      printf("gPS version %s  (c)1999 Felipe Bergo <bergo@seul.org>\n",
	     GPS_RELEASE);
      printf("Distributed under the terms of the GNU General Public License\n\n");
      return(0);
    }

    fprintf(stderr,"** gps: unrecognized command line argument %s, exiting.\n\n",
	    argv[i]);
    return(2);
  }
    
  gtk_init (&argc, &argv);

  if (dashc)
    continuum=TRUE;

  if (dashl)
    show_long_names=TRUE;

  create_device_list(); /* gets info from /proc/devices */
  init_main_window ();
  nice(8);
  refresh();
  gdk_rgb_init(); /* for history gauges */
  gtk_main ();
  return 0;
}

void init_main_window() {
  GtkWidget *mainw,*v1,*h1,*pl,*b1,*b2,*b3,*b4,*sw,*h4,*h5;
  GtkWidget *mb,*mi[40],*mj[20],*m,*n,*o,*sep,*se2;
  GtkStyle *style;
  GdkBitmap *mask;
  GdkPixmap *xray;
  gint startup_width[10]={50,90,50,35,35,45,45,30,30,150}; // good guesses
  int i;
  char *buffer,*bu2;

  buffer=(char *)g_malloc(512);
  bu2=(char *)g_malloc(512);
  gethostname(bu2,512);
  sprintf(buffer,"Process Information (%s)",bu2);

  /* MAIN WINDOW */
  mainw=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_usize (mainw, 620, 400);
  gtk_widget_set_uposition (mainw, 30, 30);
  gtk_window_set_title (GTK_WINDOW (mainw), buffer);
  gtk_widget_show (mainw);

  g_free(bu2);
  g_free(buffer);
  
  /* MENU */
  v1=gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(mainw),v1);  

  /* View */
  m=gtk_menu_new();
  mi[0]=gtk_check_menu_item_new_with_label("Show long process names");
  mi[1]=gtk_check_menu_item_new_with_label("Continuous Refresh");

  mi[2]=gtk_menu_item_new();
  gtk_container_add(GTK_CONTAINER(mi[2]),se2=gtk_hseparator_new());
  gtk_widget_show(se2);

  mi[3]=gtk_menu_item_new_with_label("CPU and memory usage...");

  mi[4]=gtk_menu_item_new();
  gtk_container_add(GTK_CONTAINER(mi[4]),sep=gtk_hseparator_new());

  mi[5]=gtk_menu_item_new_with_label("About gps...");
  mi[6]=gtk_menu_item_new_with_label("Close");
  gtk_widget_show(sep);

  show_long_names_item=mi[0];
  item2=mi[1];
  gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(mi[0]),TRUE);
  gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(mi[1]),TRUE);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(show_long_names_item),
				 show_long_names);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item2),
				 continuum);
  for(i=0;i<7;i++)
    gtk_menu_append(GTK_MENU(m),mi[i]);
  for(i=0;i<7;i++)
    gtk_widget_show(mi[i]);

  gtk_widget_set_sensitive(mi[2],FALSE);
  gtk_widget_set_sensitive(mi[4],FALSE);

  gtk_signal_connect (GTK_OBJECT (mi[0]), "activate",
		      GTK_SIGNAL_FUNC (toggle_long), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[1]), "activate",
		      GTK_SIGNAL_FUNC (toggle_cont), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[3]), "activate",
		      GTK_SIGNAL_FUNC (pop_history), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[5]), "activate",
		      GTK_SIGNAL_FUNC (about_gps), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[6]), "activate",
		      GTK_SIGNAL_FUNC (destroy), NULL);

  /* Filter */
  n=gtk_menu_new();
  mi[5]=gtk_menu_item_new_with_label("Set filter...");
  mi[6]=gtk_menu_item_new_with_label("Clear filter");

  for(i=5;i<7;i++)
    gtk_menu_append(GTK_MENU(n),mi[i]);
  for(i=5;i<7;i++)
    gtk_widget_show(mi[i]);

  gtk_signal_connect (GTK_OBJECT (mi[5]), "activate",
		      GTK_SIGNAL_FUNC (set_filter_dialog), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[6]), "activate",
		      GTK_SIGNAL_FUNC (clear_filter), NULL);

  /* Action */
  o=gtk_menu_new();
  mi[7]=gtk_menu_item_new_with_label("Send Hang Up Signal (SIGHUP)");
  mi[8]=gtk_menu_item_new_with_label("Send Kill Signal (SIGKILL)");
  mi[9]=gtk_menu_item_new_with_label("Send POSIX Signal");
  mi[10]=gtk_menu_item_new();
  gtk_container_add(GTK_CONTAINER(mi[10]),sep=gtk_hseparator_new());
  gtk_widget_show(sep);
  mi[11]=gtk_menu_item_new_with_label("Renice...");
  mi[12]=gtk_menu_item_new();
  gtk_container_add(GTK_CONTAINER(mi[12]),sep=gtk_hseparator_new());
  gtk_widget_show(sep);

  mi[13]=gtk_menu_item_new_with_label("Details...");

  sc[0]=mi[7];
  sc[1]=mi[8];
  sc[2]=mi[11];

  /* POSIX critters */
  mj[0]=gtk_menu_new();

  buffer=(char *)g_malloc(128);
  
  for(i=0;i<POSIXSIG;i++) {
    sprintf(buffer,"%s (%d)",posignames[i],posigvalues[i]);
    mj[i+1]=gtk_menu_item_new_with_label(buffer);
    gtk_menu_append(GTK_MENU(mj[0]),mj[i+1]);
    gtk_signal_connect(GTK_OBJECT(mj[i+1]),"activate",
		       (GtkSignalFunc)PosixKill,&posigvalues[i]);
    gtk_widget_show(mj[i+1]);
  }

  g_free(buffer);

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[9]),mj[0]);
  gtk_menu_item_configure(GTK_MENU_ITEM(mi[9]),FALSE,TRUE);

  for(i=7;i<14;i++)
    gtk_menu_append(GTK_MENU(o),mi[i]);
  for(i=7;i<14;i++)
    gtk_widget_show(mi[i]);

  gtk_widget_set_sensitive(mi[10],FALSE);
  gtk_widget_set_sensitive(mi[12],FALSE);

  gtk_signal_connect (GTK_OBJECT (mi[7]), "activate",
		      GTK_SIGNAL_FUNC (kill_with_sighup), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[8]), "activate",
		      GTK_SIGNAL_FUNC (kill_with_sigkill), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[11]), "activate",
		      GTK_SIGNAL_FUNC (renice_dialog), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[13]), "activate",
		      GTK_SIGNAL_FUNC (show_details), NULL);

  /* All */
  mb=gtk_menu_bar_new();
  gtk_box_pack_start(GTK_BOX(v1),mb,FALSE,TRUE,0);
  gtk_widget_show(mb);

  mi[0]=gtk_menu_item_new_with_label("View");
  mi[1]=gtk_menu_item_new_with_label("Filter");
  mi[2]=gtk_menu_item_new_with_label("Action");
  sc[3]=mi[2];
  gtk_widget_show(mi[0]);
  gtk_widget_show(mi[1]);
  gtk_widget_show(mi[2]);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[0]),m);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[1]),n);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[2]),o);
  gtk_menu_bar_append(GTK_MENU_BAR(mb),mi[0]);
  gtk_menu_bar_append(GTK_MENU_BAR(mb),mi[1]);
  gtk_menu_bar_append(GTK_MENU_BAR(mb),mi[2]);

  /* LIST */

  h5=gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(v1),h5,TRUE,TRUE,4);
  sw=gtk_scrolled_window_new(NULL,NULL);
  gtk_box_pack_start(GTK_BOX(h5),sw,TRUE,TRUE,4);
  gtk_widget_show(sw);
  gtk_container_set_border_width(GTK_CONTAINER(sw),0);
  
  pl=gtk_clist_new(10);
  gtk_clist_set_shadow_type(GTK_CLIST(pl),GTK_SHADOW_IN);
  gtk_clist_set_selection_mode(GTK_CLIST(pl),GTK_SELECTION_SINGLE);
  for(i=0;i<10;i++)
    gtk_clist_set_column_title(GTK_CLIST(pl),i,(gchar *)(col_title[i]));
  gtk_clist_column_titles_passive(GTK_CLIST(pl));
  gtk_clist_column_titles_show(GTK_CLIST(pl));

  for(i=4;i<=8;i++)
    gtk_clist_set_column_justification(GTK_CLIST(pl),i,GTK_JUSTIFY_RIGHT);

  for(i=0;i<10;i++)
    gtk_clist_set_column_width(GTK_CLIST(pl),i,startup_width[i]);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(sw),pl);
  gtk_widget_show(pl);

  refb=b1=gtk_button_new_with_label("Refresh");
  kb[0]=b2=gtk_button_new_with_label("Hang Up (SIGHUP)");
  kb[1]=b4=gtk_button_new_with_label("Kill (SIGKILL)");
  b3=gtk_button_new_with_label("Close");

  h1=gtk_hbox_new(TRUE,4);
  gtk_box_pack_start(GTK_BOX(v1),h1,FALSE,TRUE,0);

  gtk_box_pack_start(GTK_BOX(h1),b1,FALSE,TRUE,6);
  gtk_box_pack_start(GTK_BOX(h1),b2,FALSE,TRUE,6);
  gtk_box_pack_start(GTK_BOX(h1),b4,FALSE,TRUE,6);
  gtk_box_pack_start(GTK_BOX(h1),b3,FALSE,TRUE,6);

  gtk_widget_show(h5);
  gtk_widget_show(b1);
  gtk_widget_show(b2);
  gtk_widget_show(b4);
  gtk_widget_show(b3);
  gtk_widget_show(h1);
  gtk_widget_show(v1);

  gtk_widget_set_sensitive(kb[0],FALSE);
  gtk_widget_set_sensitive(kb[1],FALSE);

  gtk_widget_set_sensitive(sc[0],FALSE);
  gtk_widget_set_sensitive(sc[1],FALSE);
  gtk_widget_set_sensitive(sc[2],FALSE);
  gtk_widget_set_sensitive(sc[3],FALSE);

  gtk_container_set_border_width(GTK_CONTAINER(h1),6);

  /* set icon */
  style = gtk_widget_get_style (mainw);
  xray = gdk_pixmap_create_from_xpm_d (mainw->window, &mask,
				       &style->bg[GTK_STATE_NORMAL],
				       (gchar **) xray_tux_xpm);
  gdk_window_set_icon (mainw->window, NULL, xray, mask);
  gdk_window_set_icon_name(mainw->window,"Processes");

  /* signal plumbing */
  gtk_signal_connect (GTK_OBJECT (mainw), "delete_event",
		      GTK_SIGNAL_FUNC (kill_main), NULL);
  gtk_signal_connect (GTK_OBJECT (mainw), "destroy",
		      GTK_SIGNAL_FUNC (destroy), NULL);
  gtk_signal_connect (GTK_OBJECT (b1), "clicked",
		      GTK_SIGNAL_FUNC (refresh_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (b2), "clicked",
		      GTK_SIGNAL_FUNC (kill_with_sighup), NULL);
  gtk_signal_connect (GTK_OBJECT (b4), "clicked",
		      GTK_SIGNAL_FUNC (kill_with_sigkill), NULL);
  gtk_signal_connect (GTK_OBJECT (b3), "clicked",
		      GTK_SIGNAL_FUNC (destroy), NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"select_row",
		      GTK_SIGNAL_FUNC(selection_callback),NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"unselect_row",
		      GTK_SIGNAL_FUNC(deselection_callback),NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"click_column",
		      GTK_SIGNAL_FUNC(sort_callback),NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"key_press_event",
		      GTK_SIGNAL_FUNC(userkey),NULL);

  psl=pl;

  if (continuum) {
    to_tag=gtk_timeout_add(999,refresh_to,NULL);
    gtk_widget_set_sensitive(refb,FALSE);
  }
}

gint kill_main (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  return FALSE;
}

void destroy (GtkWidget * widget, gpointer data)
{
  if (to_tag!=-1) { /* kill refresh timer */
    gtk_timeout_remove(to_tag);
    to_tag=-1;
  }
  while(!enter_mutex()) usleep(140); // hope this doesn't lock
  gtk_main_quit();
}

void refresh() {
  // grasp info from /proc fs
  FILE *f,*g;
  char buffer[1024],buffercopy[1024],b2[128],b3[512];
  char *p1,*p2;
  int lastpid,i,j,yav;
  struct stat dv;
  struct passwd *ident;
  unsigned long rtime,smem[2];
  unsigned long ulv,dlv,rib;
  double cpufrac;
  time_t ptime;
  gint rowcount,row2sel=-1;
  gint do_append;
  tabledata *aux;
  DIR *slashproc;
  struct dirent *dentry;

  static gchar *pcoll[10];
  static char pid[10],name[256],owner[128],state[5],
    size[16],cpu[10],start[32],nicety[16],rss[16],priority[16];

  if (mainlist!=NULL)
    delete mainlist;
  mainlist=NULL;

  f=fopen("/proc/stat","r");
  while(1) {
    if (fgets(buffer,1024,f)==NULL)
      break;
    if (strstr(buffer,"btime ")!=NULL) {
      strtok(buffer," \n\t");
      btime=atol(strtok(NULL," \n\t"));
    }
    if (strstr(buffer,"processes ")!=NULL) {
      strtok(buffer," \n\t");
      lastpid=atoi(strtok(NULL," \n\t"));
      break;
    }
  } 
  fclose(f);

  slashproc=opendir("/proc");
  if (slashproc==NULL) {
    fprintf(stderr,"** gps: unable to open /proc, cannot read process table\n");
    return;
  }

  gtk_clist_freeze(GTK_CLIST(psl));

  pcoll[0]=pid;
  pcoll[1]=name;
  pcoll[2]=owner;
  pcoll[3]=state;
  pcoll[4]=cpu;
  pcoll[5]=size;
  pcoll[6]=rss;
  pcoll[7]=nicety;
  pcoll[8]=priority;
  pcoll[9]=start;

  dentry=readdir(slashproc);
  rowcount=0;

  while(dentry!=NULL) {
    i=which_number(dentry->d_name); /* is process dir ? */
    if (i<0) {
      dentry=readdir(slashproc);
      continue;
    }
    
    sprintf(buffer,"/proc/%d/stat",i);
    f=fopen(buffer,"r");
    if (f==NULL)
      continue;

    /* get owner */
    sprintf(buffer,"/proc/%d/status",i);
    g=fopen(buffer,"r");
    if (g==NULL)
      strcpy(owner,"???");
    else {
      while(1) {
	fgets(buffer,1024,g);
	if (strstr(buffer,"Uid:")==NULL)
	  continue;
	strtok(buffer," \t");
	j=atoi(strtok(NULL," \t"));
	fclose(g);
	ident=getpwuid(j);
	strcpy(owner,ident->pw_name);
	break;
      }
    }

    /* PID */
    sprintf(pid,"%d",i);
    fgets(buffer,1024,f);
    strcpy(buffercopy,buffer);
    strtok(buffer," \t\n");

    /* NAME */
    p1=strchr(buffercopy,'(');
    p2=strrchr(buffercopy,')'); /* pattern matching, the hard way */

    memset(b2,0,128);
    memcpy(b2,p1+1,(p2-p1)-1);
    strcpy(name,b2);

    /* STATE */
    strcpy(buffer,p2+1);
    strcpy(state,strtok(buffer," \t\n"));

    for(j=0;j<10;j++)
      strtok(NULL," \t\n"); /* fix later */

    /* CPU */
    ulv=(unsigned long)atol(strtok(NULL," \t\n"));
    ulv+=(unsigned long)atol(strtok(NULL," \t\n"));
    dlv=(unsigned long)atol(strtok(NULL," \t\n"));
    dlv+=(unsigned long)atol(strtok(NULL," \t\n"));

    /* PRIORITY & NICE */
    rib=atoi(strtok(NULL," \t\n"));
    sprintf(priority,"%d",rib);
    rib=atoi(strtok(NULL," \t\n"));
    sprintf(nicety,"%d",rib);

    for(j=0;j<2;j++)
      strtok(NULL," \t\n");

    rtime=(unsigned long)atol(strtok(NULL," \t\n"));
    ptime=btime+(time_t)(rtime/100UL);

    strcpy(b3,ctime(&ptime));
    memset(b2,0,128);
    memcpy(b2,b3+11,5);
    strcat(b2," ");
    memcpy(b2+6,b3,11);
    memcpy(b2+17,b3+20,4); // how cute
    strcpy(start,b2);

    //  000000000011111111112222
    //  012345678901234567890123
    // "Wed Jun 30 21:49:08 1993\n"
    // to 
    // "21:49 Wed Jun 30 1993"

    dlv=time(NULL)-ptime;
    if (dlv==0)
      dlv=1;
    cpufrac=((double)ulv)/((double)dlv);
    sprintf(cpu,"%.1f",cpufrac);

    /* MEM (VSIZE) */
    smem[0]=(unsigned long)atol(strtok(NULL," \t\n"));
    mem_unit=0;
    while(smem[0]>(99UL<<10)) {
      ++mem_unit;
      smem[0]>>=10;
    }
    sprintf(b3,"%lu",smem[0]);
    strcat(b3,power_of_two_suffix(mem_unit));
    strcpy(size,b3);

    /* RSS */
    smem[1]=(unsigned long)atol(strtok(NULL," \t\n"));

    /* state */
    if (smem[1]==0)
      if (rib>0) /* NICED */
	strcat(state,"WN");
      else
	strcat(state,"W");
    else
      if (rib>0)
	strcat(state," N");

    smem[1]<<=PAGE_SHIFT;
    mem_unit=0;
    while(smem[1]>(99UL<<10)) {
      ++mem_unit;
      smem[1]>>=10;
    }
    sprintf(b3,"%lu",smem[1]);
    strcat(b3,power_of_two_suffix(mem_unit));
    strcpy(rss,b3);

    fclose(f);

    if (show_long_names) {
      sprintf(buffer,"/proc/%d/cmdline",i);
      f=fopen(buffer,"r");
      if (f!=NULL) {
	memset(buffer,0,1024);
	yav=fread(buffer,1,1024,f);
	fclose(f);
	for(int yayav=0;yayav<(yav-1);yayav++)
	  if (buffer[yayav]==0)
	    buffer[yayav]=0x20;
	if (strlen(buffer)>0)
	  strcpy(name,buffer);
      }
    }
    if (have_filter)
      if (power_cmp(fi_work.field,pcoll[fi_work.field],fi_work.value)==fi_work.cmp)
	do_append=TRUE;
      else
	do_append=FALSE;
    else
      do_append=TRUE;
    if (do_append) {
      aux=new tabledata();
      for(int paprika=0;paprika<10;paprika++)
	aux->setv(paprika,pcoll[paprika]);
      if (mainlist==NULL)
	mainlist=aux;
      else
	mainlist->add(aux);
      if (i==pid_sel)
	row2sel=rowcount;
      ++rowcount;
    }
    dentry=readdir(slashproc); /* read next entry */
  }

  closedir(slashproc);
  if (row2sel<0)
    pid_sel=-1;
  
  pre_sort();
  update_list_widget();
  gtk_clist_thaw(GTK_CLIST(psl));
}

void toggle_long (GtkWidget * widget, gpointer data) {
  if (show_long_names)
    show_long_names=FALSE;
  else
    show_long_names=TRUE;
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(show_long_names_item),
				 show_long_names);
  refresh();
}

void toggle_cont (GtkWidget * widget, gpointer data) {
  if (continuum)
    continuum=FALSE;
  else
    continuum=TRUE;
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item2),
				 continuum);
  if (continuum) {
    gtk_widget_set_sensitive(refb,FALSE);
    to_tag=gtk_timeout_add(999,refresh_to,NULL);
  } else {
    gtk_timeout_remove(to_tag);
    to_tag=-1;
    gtk_widget_set_sensitive(refb,TRUE);
  }
}

void refresh_cb (GtkWidget * widget, gpointer data) {
  refresh();
}

gint refresh_to (gpointer data) {

  if (!enter_mutex())
    return FALSE;
  refresh();
  exit_mutex();
  return TRUE;
}

void selection_callback(GtkWidget *widget,gint row,gint column,
			GdkEventButton *event,gpointer data) {
  gchar *pp;
  int np;
  //  GTK_CLIST(psl)->focus_row=-1;
  gtk_clist_get_text(GTK_CLIST(psl),row,0,&pp);
  np=atoi(pp);
  pid_sel=np;
  gtk_widget_set_sensitive(kb[0],TRUE);
  gtk_widget_set_sensitive(kb[1],TRUE);
  gtk_widget_set_sensitive(sc[0],TRUE);
  gtk_widget_set_sensitive(sc[1],TRUE);
  gtk_widget_set_sensitive(sc[2],TRUE);
  gtk_widget_set_sensitive(sc[3],TRUE);

  if (event!=NULL)
    if (event->type==GDK_2BUTTON_PRESS)
      open_details(pid_sel);
}

void deselection_callback(GtkWidget *widget,gint row,gint column,
			GdkEventButton *event,gpointer data) {
  gchar *pp;
  pid_sel=-1;
  gtk_widget_set_sensitive(kb[0],FALSE);
  gtk_widget_set_sensitive(kb[1],FALSE);
  gtk_widget_set_sensitive(sc[0],FALSE);
  gtk_widget_set_sensitive(sc[1],FALSE);
  gtk_widget_set_sensitive(sc[2],FALSE);
  gtk_widget_set_sensitive(sc[3],FALSE);
  //  GTK_CLIST(psl)->focus_row=-1;
}

void sort_callback(GtkCList *clist,gint column,gpointer data) {
  if (last_sort==column)
    waysort[column]=(++waysort[column])%2;
  last_sort=column;
  pre_sort();
  update_list_widget();
}

gint hyper_cmp(const gpointer a,const gpointer b) {
  tabledata *p1,*p2;
  p1=(tabledata *)a;
  p2=(tabledata *)b;
  return(power_cmp(last_sort,p1->getv(last_sort),p2->getv(last_sort)));
}

/*

 ABOUT DIALOG 

 */

void about_gps(GtkWidget *,gpointer) {
  static GtkWidget *dlg;
  GtkStyle *style;
  GdkPixmap *xray;
  GdkBitmap *mask;
  GtkWidget *xrayp;
  char about[1024];
  int i;

  GtkWidget *v,*h,*h2,*p,*text,*b;

  dlg=gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_widget_realize(dlg);
  gtk_window_set_policy(GTK_WINDOW(dlg),TRUE,TRUE,TRUE); /* 0.3.6 */
  gtk_window_set_title (GTK_WINDOW (dlg), "About gps");

  style=gtk_widget_get_style(dlg);
  xray=gdk_pixmap_create_from_xpm_d(dlg->window,&mask,
				    &style->bg[GTK_STATE_NORMAL],
				    (gchar **)xray_tux_xpm);
  xrayp=gtk_pixmap_new(xray,mask);

  v=gtk_vbox_new(FALSE,2);
  gtk_container_add(GTK_CONTAINER(dlg),v);

  h=gtk_hbox_new(FALSE,2);
  gtk_box_pack_start(GTK_BOX(v),h,FALSE,TRUE,2);
  gtk_box_pack_start(GTK_BOX(h),xrayp,FALSE,TRUE,6);

  strcpy(about,"gPS - Graphical Process Statistics,  version ");
  strcat(about,GPS_RELEASE);
  strcat(about,"\nCopyright (C) 1999 Felipe Paulo G. Bergo\ne-mail: bergo@seul.org\n\n");
  strcat(about,"gPS comes with ABSOLUTELY NO WARRANTY;\n");
  strcat(about,"This is free software, and you are welcome to redistribute\n");
  strcat(about,"it under certain conditions. Read the file COPYING,\n");
  strcat(about,"located in ");
  strcat(about,WDOC);
  strcat(about,"/gps, for details.\n");
  text=gtk_label_new(about);
  gtk_box_pack_start(GTK_BOX(h),text,FALSE,TRUE,2);
  gtk_label_set_justify(GTK_LABEL(text),GTK_JUSTIFY_LEFT);

  h2=gtk_hbox_new(TRUE,2);
  gtk_box_pack_start(GTK_BOX(v),h2,FALSE,TRUE,2);

  for(i=0;i<3;i++) {
    p=gtk_label_new(" ");
    gtk_box_pack_start(GTK_BOX(h2),p,FALSE,FALSE,2);
    gtk_widget_show(p);
  }

  b=gtk_button_new_with_label("Dismiss");
  gtk_box_pack_start(GTK_BOX(h2),b,FALSE,TRUE,4);

  gtk_container_set_border_width(GTK_CONTAINER(dlg),6);

  gtk_widget_show(b);
  gtk_widget_show(text);
  gtk_widget_show(xrayp);
  gtk_widget_show(h2);
  gtk_widget_show(h);
  gtk_widget_show(v);
  gtk_widget_show(dlg);
  gtk_signal_connect(GTK_OBJECT(b),"clicked",
		     GTK_SIGNAL_FUNC(dismiss_about),(gpointer)(&dlg));
  gtk_grab_add(dlg);
}

void dismiss_about(GtkWidget *wid,gpointer data) {
  GtkWidget **dlg;
  dlg=(GtkWidget **)(data);
  gtk_grab_remove(*dlg);
  gtk_widget_destroy(*dlg);
}

/* process signaling */

void kill_with_sighup (GtkWidget * widget, gpointer data) {
  deal_with_kill(SIGHUP);
}

void kill_with_sigkill (GtkWidget * widget, gpointer data) {
  deal_with_kill(SIGKILL);
}

void PosixKill(GtkWidget *widget,gpointer data) {
  int r,sig;
  char *emsg;

  sig=*((int *)data);

  if (pid_sel==-1)
    return;
  r=kill(pid_sel,sig);
  if (r==0) {
    if (!continuum)
      refresh();
    return;
  }
  switch(errno) {
  case EINVAL:
    emsg=ERR_KILL_INS2;
    break;
  case ESRCH:
    emsg=ERR_KILL_NOS2;
    break;
  case EPERM:
    emsg=ERR_KILL_PER2;
    break;
  default:
    emsg=ERR_BAD_MAN_PAGES;
  }
  message_box_ok(emsg,"Cannot send signal",MSGBOX_ICON_ERROR);
}

void deal_with_kill(gint sig) {
  int r;
  char *emsg;
  if (pid_sel==-1)
    return;
  r=kill(pid_sel,sig);
  if (r==0) {
    if (!continuum)
      refresh();
    return;
  }
  switch(errno) {
  case EINVAL:
    emsg=ERR_KILL_INSG;
    break;
  case ESRCH:
    emsg=ERR_KILL_NOSP;
    break;
  case EPERM:
    emsg=ERR_KILL_PERM;
    break;
  default:
    emsg=ERR_BAD_MAN_PAGES;
  }
  message_box_ok(emsg,"Unable to kill",MSGBOX_ICON_ERROR);
}


gint power_cmp(gint column,char *k,char *m) {
  int i,j;
  double c,d;
  long l1,l2;
  int values[128];
  char b1[128],b2[128];

  if (k==NULL)
    return -1;
  if (m==NULL)
    return 1;
  switch(column) {
  case 1: // text cases
  case 2:
  case 3:
    return(strcmp(k,m));
  case 0: // simple numbers
  case 7:
  case 8:
    i=atoi(k);
    j=atoi(m);
    if (i<j) return -1;
    if (i>j) return 1;
    return 0;
  case 4: // fractionary
    c=atof(k);
    d=atof(m);
    if (c<d) return -1;
    if (c>d) return 1;
    return 0;
  case 5: // memory
  case 6:
    for(i=0;i<128;i++)
      values[i]=0;
    values['K']=1;
    values['M']=2;
    values['G']=3;
    values['T']=4;
    values['P']=5;
    if (k[strlen(k)-1]!=m[strlen(m)-1]) {
      if (values[k[strlen(k)-1]]>values[m[strlen(m)-1]])
	return 1;
      else
	return -1;
    }
    strcpy(b1,k);
    strcpy(b2,m);
    b1[strlen(b1)-1]=0;
    b2[strlen(b2)-1]=0;
    l1=atol(b1);
    l2=atol(b2);
    if (l1>l2)
      return 1;
    if (l1<l2)
      return -1;
    return 0;
  case 9:
    if ((strlen(k)<21)||(strlen(m)<21))
      return 0;
    memset(b1,0,128);
    memcpy(b1,k+17,4);

    i=k[10]+k[11]+k[12];
    strcat(b1,month_letter(i));
    memcpy(b1+5,k+14,2);
    memcpy(b1+7,k,5);

    memset(b2,0,128);
    memcpy(b2,m+17,4);

    i=m[10]+m[11]+m[12];
    strcat(b2,month_letter(i));
    memcpy(b2+5,m+14,2);
    memcpy(b2+7,m,5);
    return(strcmp(b1,b2));
  default:
    return 0;
  }
}

char * month_letter(int sol) {
    switch(sol) {
    case 'J'+'a'+'n':
	return("A"); break;
    case 'F'+'e'+'b':
	return("B"); break;
    case 'M'+'a'+'r':
	return("C"); break;
    case 'A'+'p'+'r':
	return("D"); break;
    case 'M'+'a'+'y':
	return("E"); break;
    case 'J'+'u'+'n':
	return("F"); break;
    case 'J'+'u'+'l':
	return("G"); break;
    case 'A'+'u'+'g':
	return("H"); break;
    case 'S'+'e'+'p':
	return("I"); break;
    case 'O'+'c'+'t':
	return("J"); break;
    case 'N'+'o'+'v':
	return("K"); break;
    case 'D'+'e'+'c':
	return("L"); break;
    default:
      return("M");
    }
}

void update_list_widget() {
  gint i,j,k,rts;
  static gchar *dp[10];
  tabledata *p;

  gtk_clist_freeze(GTK_CLIST(psl));

  for(i=0;i<10;i++)
    dp[i]=col_title[i];

  deselection_callback(NULL,0,0,NULL,NULL);
  gtk_clist_unselect_all(GTK_CLIST(psl));

  if (mainlist==NULL) {
    gtk_clist_clear(GTK_CLIST(psl));
    gtk_clist_thaw(GTK_CLIST(psl));
    rows_on_list=0;
    return;
  }

  i=mainlist->count();

  if (i<rows_on_list)
    for(j=rows_on_list-i;j>0;j--)
      gtk_clist_remove(GTK_CLIST(psl),0);

  if (i>rows_on_list)
    for(j=i-rows_on_list;j>0;j--)
      gtk_clist_append(GTK_CLIST(psl),dp);

  rows_on_list=i;
  p=mainlist;
  rts=-1;
  for(j=0;j<rows_on_list;j++,p=p->getNext()) {
    if (p->selected())
      rts=j;
    for(i=0;i<10;i++)
      gtk_clist_set_text(GTK_CLIST(psl),j,i,p->getv(i));
  }

  if (rts>=0)
    gtk_clist_select_row(GTK_CLIST(psl),rts,-1);

  gtk_clist_thaw(GTK_CLIST(psl));
}

char * power_of_two_suffix(gint expon) {
  static char sarray[16]={'K',0,'M',0,'G',0,'T',0,'P',0,'<','!','>',0};
  gint i;

  i=expon;
  if (i==0)
    return(&sarray[1]);
  i=((i%7)-1)<<1;
  return(&sarray[i]);
}

void show_details(GtkWidget *a,gpointer b) {
  open_details(pid_sel);
}

int which_number(char *s) {
  gint sl,i;
  sl=strlen(s);
  for(i=0;i<sl;i++)
    if ((s[i]<'0')||(s[i]>'9'))
      return -1;
  return(atoi(s));
}

gboolean userkey(GtkWidget *widget, GdkEventKey *gek,gpointer data) {
  if (gek==NULL)
    return FALSE;
  if (gek->keyval==GDK_Return)
    if (pid_sel!=-1)
      open_details(pid_sel);
}

void pre_sort() {
  GList *gl,*v;
  tabledata *p,*q;
  gint wts;

  gl=NULL;

  for(p=mainlist;p!=NULL;p=p->getNext())
    gl=g_list_prepend(gl,(gpointer)p);

  if (gl==NULL)
    return;

  /* dirty trick to keep equal rows straight */
  if (last_sort>0) {
    wts=last_sort;
    last_sort=0;
    gl=g_list_sort(gl,(GCompareFunc)hyper_cmp);
    last_sort=wts;
  }
  gl=g_list_sort(gl,(GCompareFunc)hyper_cmp);
 
  if (waysort[last_sort])
    gl=g_list_reverse(gl);
  
  q=(tabledata *)gl->data;
  p=q;
  for(v=g_list_next(gl);v!=NULL;v=g_list_next(v)) {
    p->setNext((tabledata *)v->data);

    if (pid_sel!=-1)
      if (atoi(p->getv(0))==pid_sel)
	p->select_item();

    p=p->getNext();
  }
  p->setNext(NULL);
  mainlist=q;
  g_list_free(gl); /* 0.3.7 */
}
