/* #Specification: VIEWITEMS / principle
	The VIEWITEMS object is used to parse a configuration file with
	comments and continuation line. The special character starting
	comments is configurable. The default is the # character.

	All comments before statements and at the end of the statements
	are preserved. Application using that object will generally scan
	the VIEWITEM (VIEWITEMS is derived from ARRAY) and extract various
	configuration statement they can handle. While doing so, they generally
	remember the VIEWITEM holding that statement. This allows application
	to make correction to the file and preserve comment as much as possible.

	This preserve also the original ordering of the file. New statement
	are generally append at the end though.

	This could be used for simple module. The first client for this code
	is the wuftpd module.
*/
#pragma implementation
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "fviews.h"

// To force this at link time (called by misc.cc)
void fviews_required(){}

PUBLIC VIEWITEM::VIEWITEM (
	const SSTRING &_comment,
	const char *_line,
	const SSTRING &_endcomment)
{
	line.setfrom (_line);
	comment.setfrom (_comment);
	endcomment.setfrom (_endcomment);
}

PUBLIC VIEWITEM::VIEWITEM (
	const char *_line)
{
	line.setfrom (_line);
}


PUBLIC VIEWITEM *VIEWITEMS::getitem (int no) const
{
	return (VIEWITEM*)ARRAY::getitem(no);
}

/*
	Locate a VIEWITEM starting with the keyword key
*/
PUBLIC VIEWITEM* VIEWITEMS::locate (const char *key)
{
	VIEWITEM *ret = NULL;
	int n = getnb();
	for (int i=0; i<n; i++){
		VIEWITEM *it = getitem (i);
		char word[200];
		str_copyword (word,it->line.get(),sizeof(word));
		if (strcmp(word,key)==0){
			ret = it;
			break;
		}
	}
	return ret;
}
/*
	Locate a VIEWITEM starting with the keyword key1 and key2
*/
PUBLIC VIEWITEM* VIEWITEMS::locate (const char *key1, const char *key2)
{
	VIEWITEM *ret = NULL;
	int n = getnb();
	for (int i=0; i<n; i++){
		VIEWITEM *it = getitem (i);
		char word1[200],word2[200];
		const char *pt = str_copyword (word1,it->line.get(),sizeof(word1));
		str_copyword (word2,pt,sizeof(word2));
		if (strcmp(word1,key1)==0 && strcmp(word2,key2)==0){
			ret = it;
			break;
		}
	}
	return ret;
}
/*
	Locate a VIEWITEM which is an assignment of the variable var
	(var=value)
*/
PUBLIC VIEWITEM* VIEWITEMS::locateassign (const char *var)
{
	/* #Specification: VIEWITEMS / various file format
		VIEWITEMS may be used to managed to types of configuration files.
		You have normal keyed configuration file of the form:

		#
		keyword value ...
		#

		and you have the shell type ones of the form:

		#
		#!/bin/sh
		VAR1=value
		VAR2=value
		#

		Functions like VIEWITEMS::locate() and VIEWITEMS::locateassign()
		are used to dig into the VIEWITEMS object.
	*/
	int lenvar = strlen(var);
	VIEWITEM *ret = NULL;
	int n = getnb();
	for (int i=0; i<n; i++){
		VIEWITEM *it = getitem (i);
		const char *pt = it->line.get();
		pt = str_skip(pt);
		if (strncmp(pt,var,lenvar)==0 && pt[lenvar] == '='){
			ret = it;
			break;
		}
	}
	return ret;
}

/*
	Locate all VIEWITEMs starting with the keyword key
	(full or partial match)
*/
PRIVATE int VIEWITEMS::locate_gen (
	const char *key,
	VIEWITEMS &items,
	bool fullmatch)		// key match completly the first word on the
						// line or not
{
	int ret = 0;
	int n = getnb();
	items.neverdelete();
	int len = strlen (key);
	for (int i=0; i<n; i++){
		VIEWITEM *it = getitem (i);
		char word[200];
		str_copyword (word,it->line.get(),sizeof(word));
		if (fullmatch){
			if (strcmp(word,key)==0){
				ret++;
				items.add (it);
			}
		}else{
			if (strncmp(word,key,len)==0){
				ret++;
				items.add (it);
			}
		}
	}
	return ret;
}

/*
	Locate all VIEWITEMs starting with the keyword key (full match)
*/
PUBLIC int VIEWITEMS::locate (const char *key, VIEWITEMS &items)
{
	return locate_gen (key,items,true);
}

/*
	Locate all VIEWITEMs starting with the keyword key (partial match)
*/
PUBLIC int VIEWITEMS::locate_prefix (const char *key, VIEWITEMS &items)
{
	return locate_gen (key,items,false);
}


/*
	Locate all VIEWITEMs starting with the keyword key
*/
PUBLIC int VIEWITEMS::locate (
	const char *key1,
	const char *key2,
	VIEWITEMS &items)
{
	int ret = 0;
	int n = getnb();
	items.neverdelete();
	for (int i=0; i<n; i++){
		VIEWITEM *it = getitem (i);
		char word1[200],word2[200];
		const char *pt = str_copyword (word1,it->line.get(),sizeof(word1));
		str_copyword (word2,pt,sizeof(word2));
		if (strcmp(word1,key1)==0 && strcmp(word2,key2)==0){
			ret++;
			items.add (it);
		}
	}
	return ret;
}

PUBLIC VIEWITEMS::VIEWITEMS ()
{
	comcar = '#';
}

/*
	Define the comment character used in the config file. This
	is the # sign by default
*/
PUBLIC void VIEWITEMS::setcomcar ( char com)
{
	comcar = com;
}


/*
	Read a configuration file in memory
*/
PUBLIC int VIEWITEMS::read (CONFIG_FILE &fconf)
{
	int ret = -1;
	FILE *fin = fconf.fopen ("r");
	if (fin != NULL){
		SSTRING comments;
		char buf[2000];
		while (fgets_comments (buf,sizeof(buf)-1,fin,comments,comcar)!=NULL){
			char *pt = strchr(buf,comcar);
			SSTRING endcom;
			if (pt != NULL){
				// Preserve the spaces beforce the comcar
				while (pt > buf && isspace(pt[-1])) pt--;
				endcom.setfrom (pt);
				*pt = '\0';
			}
			if (getnb()==0){
				firstcomment.setfrom (comments);
				comments.setfrom ("");
			}
			add (new VIEWITEM (comments,buf,endcom));
		}
		lastcomment.setfrom (comments);
		ret = fclose (fin);
	}
	return ret;
}
/*
	Write back a configuration file (overwrite).
	Return -1 if any error
*/
PUBLIC int VIEWITEMS::write (CONFIG_FILE &fconf, PRIVILEGE *priv)
{
	int ret = -1;
	FILE *fout = fconf.fopen (priv,"w");
	if (fout != NULL){
		int n = getnb();
		comment_write (firstcomment,fout);
		for (int i=0; i<n; i++){
			VIEWITEM *it = getitem (i);
			comment_write (it->comment,fout);
			fprintf (fout,"%s%s\n",it->line.get(),it->endcomment.get());
		}
		comment_write (lastcomment,fout);
		ret = fclose (fout);
	}
	return ret;
}

/*
	Locate the value of a variable.
	Return NULL if the variable is not defined

	The quote surrounding the value are removed.
*/
PUBLIC const char *VIEWITEMS::locateval (
	const char *var,
	char tmp[1000])		// Work buffer, ret might point to it
{
	const char *ret = NULL;
	VIEWITEM *it = locateassign(var);
	if (it != NULL){
		const char *pt = it->line.strchr('=');
		if (pt != NULL){
			ret = pt+1;
			if (ret[0] == '"'){
				ret++;
				strcpy (tmp,ret);
				strip_end (tmp);
				int len = strlen(tmp)-1;
				if (len >= 0 && tmp[len] == '"') tmp[len] = '\0';
				ret = tmp;
			}
		}
	}
	return ret;
}

/*
	Locate a boolean (yes/no) value
*/
PUBLIC int VIEWITEMS::locatebval (
	const char *var)
{
	int ret = 0;
	char tmp[1000];
	const char *val = locateval (var,tmp);
	if (val != NULL){
		ret = stricmp(val,"yes")==0 || stricmp(val,"true")==0;
	}
	return ret;
}	
/*
	Locate an hexadecimal value
*/
PUBLIC int VIEWITEMS::locatehval (
	const char *var)
{
	int ret = 0;
	char tmp[1000];
	const char *val = locateval (var,tmp);
	if (val != NULL && isxdigit(val[0])){
		sscanf (val,"%x",&ret);
	}
	return ret;
}	
/*
	Locate a numeric value
*/
PUBLIC int VIEWITEMS::locatenval (
	const char *var)
{
	int ret = 0;
	char tmp[1000];
	const char *val = locateval (var,tmp);
	if (val != NULL){
		ret = atoi (val);
	}
	return ret;
}	


/*
	Update or add an assignement of the form var="val"
*/
PUBLIC void VIEWITEMS::update (
	const char *var,
	const char *val)
{
	/* #Specification: VIEWITEMS / update strategy
		When updating a config file using the VIEWITEMS object, the code
		generally goes like this.

		#
		CONFIG_FILE cf (...);
		VIEWITEMS items;
		items.read (cf);
		.
		items.update (var,value);
		.
		items.write(cf,(PRIVILEGE*)NULL);
		#

		There are several VIEWITEMS::update() function to support various
		data type. For some project, you may be better to write your own
		update function. An update function always looks like the following
		pseudo-code

		#
		// Used one of the VIEWITEMS::locate or write your own
		VIEWITEM *it = items.locate (key);
		if (it == NULL){
			// No record found, add a new empty one
			// It will go at the end of the config file
			it = new VIEWITEM;
			items.add (it);
		}
		// formats the updated value of the record and stores it
		// in the VIEWITEM object.
		char line[1000];
		snprintf (line,sizeof(line)-1,"....",....);
		it->line.setfrom (line);
		#
	*/
	VIEWITEM *it = locateassign(var);
	if (it == NULL){
		it = new VIEWITEM("");
		add (it);
	}
	char buf[1000];
	snprintf (buf,sizeof(buf)-1,"%s=\"%s\"",var,val);
	it->line.setfrom (buf);
}

/*
	Update or add an assignement of the form var="val"
*/
PUBLIC void VIEWITEMS::update (
	const char *var,
	const SSTRING &val)
{
	update (var,val.get());
}

/*
	Update or add an assignement of the form var="val"
*/
PUBLIC void VIEWITEMS::updatehval (
	const char *var,
	int val)
{
	char buf[20];
	sprintf (buf,"%x",val);
	update (var,buf);
}
/*
	Update or add an assignement of the form var="val"
*/
PUBLIC void VIEWITEMS::update (
	const char *var,
	int val)
{
	char buf[20];
	sprintf (buf,"%d",val);
	update (var,buf);
}
/*
	Update or add an assignement of the form var="yes" or var="no"
*/
PUBLIC void VIEWITEMS::updatebval (
	const char *var,
	bool val)
{
	update (var,val ? "yes" : "no");
}

