// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: x11dev.cc,v 1.7 1998/09/15 05:57:27 jgg Exp $
/* ######################################################################
   
   X11Dev - X Windows Device Interface
   
   ###################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#ifdef __GNUG__
#pragma implementation "deity/x11dev.h"
#endif  
#include <iostream.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#include <deity/widget.h>
#include <deity/widget-thread.h>
#include <deity/window.h>
#include <deity/utils.h>

#include <deity/x11dev.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>
									/*}}}*/

XWindowsGC *XWindowsGC::GC = 0;

// XWindowsGC::XWindowsGC - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
XWindowsGC::XWindowsGC()
{
   // Connect to the X server
   XDisplay = XOpenDisplay(0);
   if (XDisplay == 0)
      return;
   CurGC = 0;
   CurWin = 0;
   XFont = 0;
   ResizeBlock = 0;

   // Create a generic GC
   XGCValues gcv;
   CurGC = XCreateGC(*this,(Window)DefaultRootWindow(XDisplay),0,&gcv);
   
   // Determine if this is a paletted visual (video mode)
   Paletted = false;
   int Vis = DefaultVisual(XDisplay,DefaultScreen(XDisplay))->c_class;
   if (Vis == GrayScale || Vis == PseudoColor)
      Paletted = true;
   
   // We use a private colormap if this is a paletted mode.
   if (Paletted == true)
   {
      PrivColors = XCreateColormap(XDisplay,(Window)DefaultRootWindow(XDisplay),
				   DefaultVisual(XDisplay,DefaultScreen(XDisplay)),
				   AllocAll);
   
      // Transfer the default colormap
      for (int I = 0; I != 255; I++)
      {
	 CMapValues[I].pixel = I;
	 CMapUsed[I] = false;
      }

      XQueryColors(*this,DefaultColormap(XDisplay,DefaultScreen(XDisplay)),CMapValues,255);
      XStoreColors(*this,PrivColors,CMapValues,255);
   }
   else
      PrivColors = DefaultColormap(XDisplay,DefaultScreen(XDisplay));
   
   for (int I = 0; I != 10; I++)
      FontCache[I].XFont = 0;
   for (int I = 0; I != 20; I++)
      ColorCache[I].Clr = Wc_None;
   LastColorInsert = 0;
}
									/*}}}*/
// XWindowsGC::~XWindowsGC - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
XWindowsGC::~XWindowsGC()
{
   if (XDisplay != 0)
      XCloseDisplay(XDisplay);
}
									/*}}}*/
// XWindowsGC::CreateRegion - Creates a region for a window		/*{{{*/
// ---------------------------------------------------------------------
/* */
void *XWindowsGC::CreateRegion(Widget *For,void *Parent)
{
   // This is the root window
   if (For->Parent == 0)
      return (void *)DefaultRootWindow(XDisplay);

   // Set the window attributes
   XSetWindowAttributes At;
   if (For->IsFlag(Widget::Window) == true)
      At.override_redirect = False;
   else
      At.override_redirect = True;
   At.event_mask = ExposureMask | StructureNotifyMask |
                   ButtonPressMask | KeyPressMask | ButtonMotionMask | 
                   ButtonReleaseMask;

   // Determine the widget location relative to the parent region
   Rect Size = For->Size();
   Widget *I = For->Parent;
   for (; I != 0 && I->IsFlag(Widget::Region) == false; I = I->Parent)
   {
      Size.x += I->Pos.x + I->BorderX;
      Size.y += I->Pos.y + I->BorderY;
   }
   if (I != 0)
   {
      Size.x += I->BorderX;
      Size.y += I->BorderY;
   }
   
   if (Size.w == 0 || Size.h == 0)
      clog << "Note, widget " << (void *)For << " has 0 size, X will die now" << endl;
   
   At.colormap = PrivColors;
   Window Win = XCreateWindow(*this,(Window)Parent,
			      Size.x,Size.y,
			      Size.w,Size.h,0,
			      CopyFromParent,CopyFromParent,CopyFromParent,
			      CWOverrideRedirect | CWEventMask | CWColormap,&At);
   XMapWindow(*this,Win);

   // Set the close notify atom
   if (For->IsFlag(Widget::Window) == true)
   {
      // Enable delete window protocol
      Atom Tmp = XInternAtom(*this,"WM_DELETE_WINDOW",False);
      XSetWMProtocols(*this,Win,&Tmp,1);
   }
   
   return (void *)Win;
}
									/*}}}*/
// XWindowsGC::AttachRegion - Makes the region the current draw source	/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::AttachRegion(void *Region)
{
   // See if we need to construct a new GC
   if (CurGC == 0)
   {
      XGCValues gcv;
      CurGC = XCreateGC(*this,(Window)Region,0,&gcv);
   }   
   CurWin = (Window)Region;
}
									/*}}}*/
// XWindowsGC::DeleteRegion - Deletes the region			/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::DeleteRegion(void *Region)
{
   XDestroyWindow(*this,(Window)Region);
}
									/*}}}*/
// XWindowsGC::ResizeRegion - Changes the size of a region		/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::ResizeRegion(Widget *For,void *Region,Rect Size)
{
   if (Region == 0 || Region == ResizeBlock)
      return;

   For->ActionFlags |= Widget::RedrawPending;
   XMoveResizeWindow(*this,(Window)Region,Size.x,Size.y,Size.w,Size.h);
}
									/*}}}*/
// XWindowsGC::SetTitle - Changes the window title			/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::SetTitle(void *Region,const char *Title,const char *Icon)
{
   if (Title != 0)
      XChangeProperty(*this,(Window)Region,XA_WM_NAME,XA_STRING,8,
		      PropModeReplace,(unsigned char *)Title,strlen(Title)+1);
   if (Icon != 0)
      XChangeProperty(*this,(Window)Region,XA_WM_ICON_NAME,XA_STRING,8,
		      PropModeReplace,(unsigned char *)Icon,strlen(Icon)+1);
}
									/*}}}*/
// XWindowsGC::GrabMouse - Caputure all mouse input			/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::GrabMouse(void *Region)
{
   if (Region == 0)
   {
      XUngrabPointer(*this,CurrentTime);
      return;
   }
   
   XGrabPointer(*this,(Window)Region,false,ButtonPressMask | 
		ButtonMotionMask | ButtonReleaseMask,
		GrabModeAsync,GrabModeSync,None,None,CurrentTime);
}
									/*}}}*/
// XWindowsGC::ClipRect - Set the clipping rectangle			/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::ClipRect(Rect Win)
{
   XRectangle Clip;
   Clip.x = 0;
   Clip.y = 0;
   Clip.width = Win.w;
   Clip.height = Win.h;
   this->Clip = Win;
   XSetClipRectangles(*this,CurGC,Win.x,Win.y,&Clip,1,Unsorted);
}
									/*}}}*/
// XWindowsGC::MatchColor - Converts a colour value into a pixel value	/*{{{*/
// ---------------------------------------------------------------------
/* This finds a pixel number for a given colour triplet */
unsigned long XWindowsGC::MatchColor(Color C)
{
   // Examine the cache
   for (int I = 0; I != 20; I++)
   {
      if (ColorCache[I].Clr.operator ==(C))
	 return ColorCache[I].Pix;
   }

   XColor xcol;
   xcol.pixel = WhitePixel(XDisplay,DefaultScreen(XDisplay));
   
   // X uses 16 bit colour values   
   xcol.red = C.RedLevel*0xFFFF/0xFF;
   xcol.green = C.GreenLevel*0xFFFF/0xFF;
   xcol.blue = C.BlueLevel*0xFFFF/0xFF;
 
   // Unpaletted we just get X to do it.
   if (Paletted == false)
   {         
      /* Allocate a colour cell, we assume this is smart and compresses..
         (it isn't in paletted modes but the palette handler is used in those
         cases anyhow */
      if (XAllocColor(*this,PrivColors,
		      &xcol) == 0)
      {
	 /* We should try the colour indicated by the Text parameter
	  too */
	 clog << "Unable to allocate a colour cell for the color R:" << 
	    xcol.red << " G:" << xcol.green << " B:" << xcol.blue <<
	    " Using white instead" << endl;
      }      
   }
   else
   {
      // Attempt to allocate a global colour cell
      if (XAllocColor(*this,DefaultColormap(XDisplay,DefaultScreen(XDisplay)),
		      &xcol) == 0)
      {
	 // Okay, we closest match against the color data
	 unsigned long Max = (unsigned long)(-1);
	 long Hit = 0;
	 for (int I = 0; I != 255; I++)
	 {
	    // Compute the squared distance on the colour cube
	    unsigned long Distance = ((CMapValues[I].red - xcol.red)>>8)*((CMapValues[I].red - xcol.red)>>8) +
	       ((CMapValues[I].blue - xcol.blue)>>8)*((CMapValues[I].blue - xcol.blue)>>8) +
	       ((CMapValues[I].green - xcol.green)>>8)*((CMapValues[I].green - xcol.green)>>8);
	    
	    // It is used and it is very close (3.4 units)
	    if (CMapUsed[I] == true && Distance < 3*4)
	    {
	       xcol.pixel = I;
	       Max = 0;
	       break;
	    }
	    
	    // Check if it is maxed
	    if (CMapUsed[I] == false && Distance < Max) 
	    {
	       Max = Distance;
	       Hit = I;
	    }
	 }
	 xcol.pixel = Hit;
      }
      
      // Store the pixel into our private colour map
      if (CMapUsed[xcol.pixel] == false)
      {
	 CMapValues[xcol.pixel].red = xcol.red;
	 CMapValues[xcol.pixel].green = xcol.green;
	 CMapValues[xcol.pixel].blue = xcol.blue;
	 XStoreColor(*this,PrivColors,&CMapValues[xcol.pixel]);
      }
      CMapUsed[xcol.pixel] = true;
   }
   
   ColorCache[LastColorInsert].Clr = C;
   ColorCache[LastColorInsert].Pix = xcol.pixel;
   LastColorInsert = (LastColorInsert + 1) % 20;
   
   return xcol.pixel;
}
									/*}}}*/
// XWindowsGC::SetColor - Set the foreground colour			/*{{{*/
// ---------------------------------------------------------------------
/* The forground colour is used for almost all operations */
void XWindowsGC::SetColor(Color C)
{
   XSetForeground(*this,CurGC,ForeColor = MatchColor(C));
}
									/*}}}*/
// XWindowsGC::Background - Sets the background colour			/*{{{*/
// ---------------------------------------------------------------------
/* The background colour is used for the background of the DrawText
   character cells. */
void XWindowsGC::Background(Color C)
{
   XSetBackground(*this,CurGC,BackColor = MatchColor(C));
}
									/*}}}*/
// XWindowsGC::Flush - Sends the draw buffer to the X Server		/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::Flush()
{
   XFlush(*this);
}
									/*}}}*/
// XWindowsGC::BFill - Fill an area with the Background Color		/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::BFill(Rect Area)
{
   XSetForeground(*this,CurGC,BackColor);
   XFillRectangle(*this,CurWin,CurGC,Area.x+Clip.x,Area.y+Clip.y,Area.w,Area.h);
   XSetForeground(*this,CurGC,ForeColor);
}
									/*}}}*/
// XWindowsGC::Fill - Fill an area with the Color			/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::Fill(Rect Area)
{
   XFillRectangle(*this,CurWin,CurGC,Area.x+Clip.x,Area.y+Clip.y,Area.w,Area.h);
}
									/*}}}*/
// XWindowsGC::Box3d - Draw a 3d box					/*{{{*/
// ---------------------------------------------------------------------
/* A 3d box is one that has something like a white upper left border
   with a dark grey lower left border. It looks pushed out (light
   source from UL) Inverting the colours looks pushed in. This is
   done so often it warrents it's own draw routine! */
void XWindowsGC::Box3d(Rect Box,Color Ul,Color Lr,Color Fill,unsigned long Width)
{
   Box.x += Clip.x;
   Box.y += Clip.y;
   Box.w--;
   Box.h--;
   SetColor(Ul);

   for (unsigned long I = 0; I != Width; I++)
   {
      XDrawLine(*this,CurWin,CurGC,Box.x + I,Box.y + I,
		Box.x + I,Box.y+Box.h - I);
      XDrawLine(*this,CurWin,CurGC,Box.x + I,Box.y + I,
		Box.x+Box.w - I,Box.y + I);
   }
   
   SetColor(Lr);
   for (unsigned long I = 0; I != Width; I++)
   {
      XDrawLine(*this,CurWin,CurGC,Box.x + I,Box.y+Box.h - I,
		Box.x+Box.w - I,Box.y+Box.h - I);
      XDrawLine(*this,CurWin,CurGC,Box.x+Box.w - I,Box.y+Box.h - I,
		Box.x+Box.w - I,Box.y + I);
   }
   
   // Color().Text = Color::None which cant work because of Xs #define None
   if (Fill.Text != Color().Text)
   {
      SetColor(Fill);
      XFillRectangle(*this,CurWin,CurGC,Box.x + Width,Box.y + Width,
		     Box.w - 2*Width + 1,Box.h - 2*Width + 1);
   }   
}
									/*}}}*/
// XWindowsGC::Line - Draw a line					/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::Line(Point Start,Point Stop)
{
   XDrawLine(*this,CurWin,CurGC,Start.x + Clip.x,Start.y + Clip.y,
	     Stop.x + Clip.x,Stop.y + Clip.y);
}
									/*}}}*/
// XWindowsGC::SetFont - Set a font					/*{{{*/
// ---------------------------------------------------------------------
/* Because there is a wide variety of fonts this routine will try being
   less specific untill it finds a suitable font. The idea is that it
   is more important to display something than to display a perfect 
   image. If the user has the usual font packs then it will show
   properly. Use xfontsel to play with the matching routine X 
   uses for it's fonts */
bool XWindowsGC::SetFont(SimpleFont Font)
{
   char S[900];

   // Range check the parameters
   if (Font.Slant > 2)
      Font.Slant = 0;
   if (Font.Weight > 5)
      Font.Weight = 0;

   // Transform maps
   const char *WeightMap[] = {"*","bold","medium","regular","demibold","black"};
   const char *SlantMap[] = {"*","r","i"};

   // Check our current font for a match
   if (CurFont == Font)
   {
      if (XFont == 0)
	 return false;
      return true;
   }
   CurFont = Font;

   // Check the font cache for a match
   int I;
   for (I = 0; I != 10; I++)
      if (FontCache[I].XFont != 0 && FontCache[I].Font == Font)
	 break;
   
   if (I != 10)
   {
      XFont = FontCache[I].XFont;
      XSetFont(*this,CurGC,XFont->fid);
      return true;
   }
   
   XFont = 0;
   while (XFont == 0)
   {
      sprintf(S,"-*-%s-%s-%s-*-*-*-%lu-*-*-*-*-*-*",Font.Face.c_str(),
	      WeightMap[Font.Weight],SlantMap[Font.Slant],Font.Points);
      clog << "Trying load of " << S << endl;
      XFont = XLoadQueryFont(*this,S);
      
      // Hmm, try harder
      if (XFont == 0)
      {
	 // No font!
	 if (Font.Face == "fixed")
	 {
	    clog << "Loading attempt failed for fixed font" << endl;
	    return false;
	 }
	 
	 // Accept any slant
	 if (Font.Slant != 0)
	 {
	    Font.Slant = 0;
	    continue;
	 }

	 // Accept any weight
	 if (Font.Weight != 0)
	 {
	    Font.Weight = 0;
	    continue;
	 }

	 // The fixed font is pretty standard
	 Font.Face = "fixed";
      }      
   }

   // Store this font in the font cache
   for (I = 0; I != 10; I++)
      if (FontCache[I].XFont == 0)
      {
	 FontCache[I].XFont = XFont;
	 FontCache[I].Font = CurFont;
	 break;
      }

   // Discard the last font
   if (I == 10)
   {
      // Dump the old font
      XFreeFont(*this,FontCache[9].XFont);
      FontCache[9].XFont = XFont;
      FontCache[9].Font = CurFont;
   }

   XSetFont(*this,CurGC,XFont->fid);
   return true;
}
									/*}}}*/
// XWindowsGC::ExtentText - Extent a text string			/*{{{*/
// ---------------------------------------------------------------------
/* */
Rect XWindowsGC::ExtentText(const char *Text,unsigned int Len)
{
   if (XFont == 0)
      return Rect(0,0,10,10);

   int Dir;
   int Ascent;
   int Descent;
   XCharStruct S;
   XTextExtents(XFont,Text,Len,&Dir,&Ascent,&Descent,&S);
   if (XFont != 0)
      return Rect(S.lbearing,XFont->max_bounds.ascent,S.width,
		  XFont->max_bounds.ascent+XFont->max_bounds.descent);
   else
      return Rect(S.lbearing,S.ascent,S.width,Ascent+Descent);
}
									/*}}}*/
// XWindowsGC::TextTranslate - Translate helper for DrawText		/*{{{*/
// ---------------------------------------------------------------------
/* This converts the P point into a X compatable Point that specifies
   the start of the string. */
Rect XWindowsGC::TextTranslate(Point &P,const char *Text,unsigned int Len,
			      unsigned long Flags)
{
   int Dir;
   int Ascent;
   int Descent;
   XCharStruct S;
   XTextExtents(XFont,Text,Len,&Dir,&Ascent,&Descent,&S);

   Rect Res = Rect(P.x,P.y,S.width,Ascent + Descent);
   FlagPosition(Res,Flags);
   P.x = Res.x;
   P.y = Res.y;
   return Res;
}
									/*}}}*/
// XWindowsGC::DrawTextT - Draw a transparent text string		/*{{{*/
// ---------------------------------------------------------------------
/* This draws a text string with a clear background. */
void XWindowsGC::DrawTextT(Point Pos,const char *Text,unsigned int Len,
			  unsigned long Flags)
{
   TextTranslate(Pos,Text,Len,Flags);
   XDrawString(*this,CurWin,CurGC,Clip.x + Pos.x,
	       Clip.y + Pos.y + XFont->ascent,Text,Len);

}
									/*}}}*/
// XWindowsGC::DrawText - Draw a text string				/*{{{*/
// ---------------------------------------------------------------------
/* This is the primary text drawing routine for the widget library. It is
   needed to achieve flicker free drawing. Unfortunately X doesn't provide
   this routine as an internal function so we need to fake it. */
void XWindowsGC::DrawText(Rect Loc,Point Pos,const char *Text,unsigned int Len,
			  unsigned long Flags)
{
   // Compute a new clip rectangle
   Rect Tmp = Loc;
   Tmp.Clip(Clip);

   Rect TR = TextTranslate(Pos,Text,Len,Flags);

   // Apply a clipping region if the text goes outside the boundry
   if (TR.Inside(Loc) == false)
   {
      XRectangle CRect;
      CRect.x = Tmp.x;
      CRect.y = Tmp.y;
      CRect.width = Tmp.w;
      CRect.height = Tmp.h;
      XSetClipRectangles(*this,CurGC,Clip.x,Clip.y,&CRect,1,Unsorted);
   }

   // Draw the string
   XDrawImageString(*this,CurWin,CurGC,Clip.x + Pos.x + Loc.x,
		    Clip.y + Pos.y + Loc.y + XFont->ascent,Text,Len);
   
   // Restore the clipping rectangle
   if (TR.Inside(Loc) == false)
   {
      XRectangle CRect;
      CRect.x = 0;
      CRect.y = 0;
      CRect.width = Clip.w;
      CRect.height = Clip.h;
      XSetClipRectangles(*this,CurGC,Clip.x,Clip.y,&CRect,1,Unsorted);   
   }
   
   // Compensate for clipping of one of the coord pairs.
   TR.x += Loc.x - Tmp.x;
   TR.y += Loc.y - Tmp.y;
   XSetForeground(*this,CurGC,BackColor);
   MaskFill(Tmp,TR);
   XSetForeground(*this,CurGC,ForeColor);
}
									/*}}}*/
// XWindowsGC::DrawText - Draw a text string				/*{{{*/
// ---------------------------------------------------------------------
/* This draws a text string with a background fill colour */
void XWindowsGC::DrawText(Point Pos,const char *Text,unsigned int Len,
			  unsigned long Flags)
{
   TextTranslate(Pos,Text,Len,Flags);   
   XDrawImageString(*this,CurWin,CurGC,Clip.x + Pos.x,
	       Clip.y + Pos.y + XFont->ascent,Text,Len);
}
									/*}}}*/
// XWindowsGC::WrapText - Return the char before max pixels		/*{{{*/
// ---------------------------------------------------------------------
/* This is used to determine where to wrap a text string. Max is the
   number of pixels to wrap at. The return value will be the 
   number of chars into Text that can be drawn without clipping. */
unsigned int XWindowsGC::WrapText(const char *Start,unsigned int Len,long Max)
{
   if (XFont == 0)
   {
      clog << "Text extenting failed" << endl;
      return 0;
   }
   
   long Count = 0;
   const char *I;
   for (I = Start; I != Start + Len; I++)
   {
      if ((unsigned)*I < XFont->min_char_or_byte2 ||
	  (unsigned)*I > XFont->max_char_or_byte2)
	 continue;
      
      Count += XFont->per_char[*I - XFont->min_char_or_byte2].width;
      if (Count >= Max)
	 return I - Start - 1;
   }   
   return I - Start;
}
									/*}}}*/

// XWindowsFd::XWindowsFd - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* This connects to the X Server */
XWindowsFd::XWindowsFd(Widget *Root) : SelectLoop::Fd(-1,true), Root(Root)
{
   // Associate us with the connection
   FileDesc = XConnectionNumber(*XWindowsGC::GC);
   
   WMDeleteWindow = XInternAtom(*XWindowsGC::GC,"WM_DELETE_WINDOW",False);   
}
									/*}}}*/
// XWindowsFd::~XWindowsFd - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* This connects to the X Server */
XWindowsFd::~XWindowsFd()
{
   delete XWindowsGC::GC;
   GenGC::GC = GraphicGC::GC = XWindowsGC::GC = 0;
}
									/*}}}*/
// XWindowsFd::Read - Read events from the X Server			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool XWindowsFd::Read(SelectLoop &)
{
   XEvent ev;

   double C = GetCurrentTime();
   Widget::Lock Lock;

   while (1)
   {
      /* Flushing can cause more events to be enqued. The only safe way is
         to test if the queue is empty after flushing. This is not a race
	 condition to the select loop because xlib internally queues events
	 when processing commands which is what we are checking for here. */
      if (XPending(*XWindowsGC::GC) == 0)
      {
	 Root->DoActions();
	 GenGC::GC->Flush();
	 if (XPending(*XWindowsGC::GC) == 0)
	    break;
      }
      
      XNextEvent(*XWindowsGC::GC,&ev);
      
      switch (ev.type)
      {
	 // ICCCM?
	 case ClientMessage:
	 {
	    // Check if it is a WM_DELETE_WINDOW notification
	    if (!((unsigned)ev.xclient.data.l[0] == WMDeleteWindow &&
		ev.xclient.format == 32))
	       break;
	    
	    // Locate the window the notification was for
	    Widget *Target = LocateWidget((void *)ev.xclient.window);
	    if (Target == 0)
	       break;
	    
	    if (Target->Trigger(Nt_CloseReq,0) == true)
	       exit(0);
	    
	    break;
	 }
	 
	 // Remove the building bit on the Widget
	 case MapNotify:
	 {
	    Widget *Target = LocateWidget((void *)ev.xmap.window);
	    if (Target == 0)
	       break;
	    Target->Flag(0,Widget::Building);
	    break;
	 }
	 
	 // Expose event
	 case Expose:
	 case GraphicsExpose:
	 {
	    // We only do full redraws...
	    if (ev.xexpose.count != 0)
	       break;
	    
	    Widget *Target = LocateWidget((void *)ev.xexpose.window);
	    if (Target == 0)
	       break;
//	    Target->Draw();
	    Target->Damage();
	    break;
	 }
	 
	 // Notification of region size change (usually only top level)
	 case ConfigureNotify:
	 {
	    Widget *Target = LocateWidget((void *)ev.xconfigure.window);
	    if (Target == 0)
	       break;

	    /* We must go up an generate an offset value for any 
	       Widgets between */
	    Point Pos;
	    for (Widget *I = Target->Parent; I != 0 && I->IsFlag(Widget::Region) == false; 
		 I = I->Parent)
	    {
	       Pos.x += I->Pos.x + I->BorderX;
	       Pos.y += I->Pos.y + I->BorderY;
	    }

	    // Generate an absolute rectangle for top level windows
	    XWindowsGC::GC->ResizeBlock = Target->RegionData;
	    Window Jnk;
	    if (Target->Parent == Root)
	    {
	       int x,y;
	       XTranslateCoordinates(*XWindowsGC::GC,ev.xconfigure.window,
				     DefaultRootWindow(XWindowsGC::GC->XDisplay),
				     0,0,&x,&y,&Jnk);
	       Target->Resize(Rect(x,y,
				   ev.xconfigure.width,ev.xconfigure.height));
	    }
	    else
	       Target->Resize(Rect(ev.xconfigure.x+Pos.x,ev.xconfigure.y+Pos.y,
				   ev.xconfigure.width,ev.xconfigure.height));
	    XWindowsGC::GC->ResizeBlock = 0;
	    break;
	 }

	 // Keyboard input
	 case KeyPress:
	 TranslateKey(ev);
	 break;
	 
	 // Mouse Button input
	 case ButtonPress:
	 case ButtonRelease:
	 case MotionNotify:
	 TranslateMouse(ev);
	 break;
      }      
   }

   double New = GetCurrentTime();
   if (New - C > 0.2)
      clog << "Time " << (New - C) << endl;
   return true;
}
									/*}}}*/
// XWindowsFd::Setup - Setup the root widget				/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsFd::Setup(SelectLoop &Loop)
{
   {      
      Widget::Lock Lock;
   
      Root->RealizeFamily();
      GenGC::GC->Flush();
   }
   
   // There may be some events that were internally queued before we got here
   Read(Loop);   
}
									/*}}}*/
// XWindowsFd::LocateWidget - Find the widget for a region		/*{{{*/
// ---------------------------------------------------------------------
/* Just sweep the widget tree for the a widget with the
   matching region. (Should really use a hash table) */
Widget *XWindowsFd::LocateWidget(void *Region)
{
   for (Widget *I = Root; I != 0;)
   {
      if (I->IsFlag(Widget::Region) == true && Region == I->RegionData)
	 return I;
      if (I->Child != 0)
      {
	 I = I->Child;
	 continue;
      }
      if (I->Brother != 0)
      {
	 I = I->Brother;
	 continue;
      }
      for (I = I->Parent; I != 0 && I->Brother == 0; I = I->Parent);
      if (I == 0)
	 break;
      I = I->Brother;
   } 
   clog << "Couldn't find anything!" << endl;
   return 0;
}
									/*}}}*/
// XWindowsFd::TranslateKey - Convert to non-X keystrokes		/*{{{*/
// ---------------------------------------------------------------------
/* This converts the X windows key events into our less powerfull 
   simple format. Unfortunately X doesn't seem to directly support unicode
   :< */
void XWindowsFd::TranslateKey(const XEvent &ev)
{
   // Decode the keystroke
   char S[30];
   KeySym Sym;
   XLookupString((XKeyEvent *)&ev.xkey,S,sizeof(S),&Sym,0);

   // See if its just a letter
   if (S[0] != 0)
   {
      KeyEvent Key;
      Key.Key = S[0];
      Root->RouteKey(Key);
      return;
   }
   
   // Translate the keysym
   KeyEvent Key;
   switch (Sym)
   {
      // Basic translations
      case XK_KP_Up:
      case XK_Up: Key.Extended = KeyEvent::Up; break;
      case XK_KP_Down:
      case XK_Down: Key.Extended = KeyEvent::Down; break;
      case XK_Return: Key.Extended = KeyEvent::Enter; break;
      case XK_KP_Left:
      case XK_Left: Key.Extended = KeyEvent::Left; break;
      case XK_KP_Right:
      case XK_Right: Key.Extended = KeyEvent::Right; break;
      case XK_KP_Page_Up:
      case XK_Page_Up: Key.Extended = KeyEvent::PgUp; break;
      case XK_KP_Page_Down:
      case XK_Page_Down: Key.Extended = KeyEvent::PgDown; break;
      case XK_KP_Home:
      case XK_Home: Key.Extended = KeyEvent::Home; break;
      case XK_KP_End:
      case XK_End: Key.Extended = KeyEvent::End; break;
      case XK_KP_Delete:
      case XK_Delete: Key.Extended = KeyEvent::Delete; break;
      case XK_BackSpace: Key.Extended = KeyEvent::Backspace; break;
      default:
      return;
   }
   
   // Translate the state
   if ((ev.xkey.state & ShiftMask) != 0)
      Key.Modifiers |= KeyEvent::LShift;
   if ((ev.xkey.state & ControlMask) != 0)
      Key.Modifiers |= KeyEvent::LControl;
   if ((ev.xkey.state & Mod1Mask) != 0)
      Key.Modifiers |= KeyEvent::LAlt;
   
   Root->RouteKey(Key);
}
									/*}}}*/
// XWindowsFd::TranslateMouse - Convert mouse events			/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsFd::TranslateMouse(const XEvent &ev)
{
   // Recode the mouse event
   MouseEvent E;
   memset(&E,0,sizeof(E));
   if (ev.type == MotionNotify)
      E.Type = MouseEvent::Motion;
   if (ev.type == ButtonPress)
      E.Type = MouseEvent::ButtonDn;
   if (ev.type == ButtonRelease)
      E.Type = MouseEvent::ButtonUp;

   // Recode the position
   E.Pos = Point(ev.xbutton.x,ev.xbutton.y);
   Point CurPos = Point(ev.xbutton.x_root,ev.xbutton.y_root);
   
   // Click check
   if (abs(CurPos.x - LastPos.x) > 4 || abs(CurPos.y - LastPos.y) > 4)
   {
      Click = false;
      LastPos = Point(-100,-100);
   }
   
   // Recode the button Press
   if (ev.type == ButtonRelease || ev.type == ButtonPress)
   {
      if (ev.xbutton.button == Button1)
	 E.Button = MouseEvent::Select;
      if (ev.xbutton.button == Button2)
	 E.Button = MouseEvent::Menu;
      if (ev.xbutton.button == Button3)
	 E.Button = MouseEvent::Aux;

      if (ev.type == ButtonPress)
	 Click = true;
      
      if (Click ==true && ev.type == ButtonRelease)
	 E.Type |= MouseEvent::Click;
      
      LastPos = CurPos;
      E.Clicks = 1;
   }
   
   // Recode the button state
   if ((ev.xbutton.state & Button1Mask) == Button1Mask)
      E.ButtonState |= MouseEvent::Select;
   if ((ev.xbutton.state & Button2Mask) == Button2Mask)
      E.ButtonState |= MouseEvent::Menu;
   if ((ev.xbutton.state & Button3Mask) == Button3Mask)
      E.ButtonState |= MouseEvent::Aux;

   if (E.ButtonState != 0 && Click == false)
      E.Type |= MouseEvent::Drag;
   
   Widget *Target = LocateWidget((void *)ev.xbutton.window);

   if (Target != 0)
      Target->RouteMouse(E);
}
									/*}}}*/

// XAttachDisplay - Attempt to attach to the X server			/*{{{*/
// ---------------------------------------------------------------------
/* */
Widget *XAttachDisplay(SelectLoop &Loop)
{
   // Construct the X GC and check if it found an X server
   XWindowsGC *GC = new XWindowsGC;
   if (GC->XDisplay == 0)
   {
      delete GC;
      return 0;
   }

   GenGC::GC = GraphicGC::GC = XWindowsGC::GC = GC;
   TextGC::GC = 0;
   
   // Construct the Root widget (matches the root window)
   Widget *Root = new Widget;
   Root->Flag(Widget::Region);
   Screen *Sc = XDefaultScreenOfDisplay(XWindowsGC::GC->XDisplay);
   Root->Resize(Rect(0,0,XWidthOfScreen(Sc),XHeightOfScreen(Sc)));
		
   // Construct the X input fd.
   Loop.Add(new XWindowsFd(Root));

   return Root;
}
									/*}}}*/
