/*
 *             Automatically Tuned Linear Algebra Software v3.0Beta
 *                    (C) Copyright 1998 R. Clint Whaley                     
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions, and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. The name of the University, the ATLAS group, or the names of its 
 *      contributers may not be used to endorse or promote products derived
 *      from this software without specific written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE. 
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include <string.h>

char *fmake="Make.top";
char ARCH[128], TOPdir[256], LIBdir[256], ATLASlib[256], RBLASlib[256];
char CBLASlib[256];
char TBLASlib[256];
int L1SIZE=(-1), L2SIZE=4194304;
char F2CDEFS[256];
char F77[128], CC[128], MCC[128], XCC[128];
char F77FLAGS[512], CCFLAGS[512], MMFLAGS[512], XCCFLAGS[512];
char FLINKER[128], CLINKER[128], ARCHIVER[128], RANLIB[128];
char CLINKFLAGS[512], FLINKFLAGS[512], ARFLAGS[256];
char BLASlib[256], LIBS[256];
char sATLres[128], dATLres[128], cATLres[128], zATLres[128];
char INSTflag[16];

char *machnam[28] = {"21264", "21164", "21064", "HPPA8K", "HP9K735", "ThinPwr2",
                     "WidePwr2", "POWER", "POWER3",
                     "P5", "P5MMX", "PPRO", "PII", "XEON", "ATHLON",
                     "SGIIP28", "SGIIP27", "SGIIP32", "SGIIP22", "SGIIP30",
                     "SunMS", "SunSS", "SunUS1", "SunUS2", "SunUSX",
                     "PPC604e", "PPC604", "UNKNOWN"};
enum MACHTYPE {Dec21264, Dec21164, Dec21064, HPPA8K, HP9K735, IbmTPwr2, 
               IbmWPwr2, IbmPwr, IbmPwr3,
               IntP5, IntP5MMX, IntPPRO, IntPII, IntXeon, AmdAthlon,
               SgiIP28, SgiIP27, SgiIP32, SgiIP22, SgiIP30, 
               SunMS, SunSS, SunUS1, SunUS2, SunUSX, 
               PPC604e, PPC604, MACHOther};
char *osnam[10] = {"Linux", "SunOS", "SunOS4", "OSF1", "IRIX", "AIX", 
                  "Win9x", "WinNT", "HP-UX", "Other/UNKNOWN"};
enum OSTYPE {OSLinux=0, OSSunOS, OSSunOS4, OSOSF1, OSIRIX, OSAIX, 
             OSWin9x, OSWinNT, OSHPUX, OSOther};

enum F2CNAME {f2c_Add_=0, f2c_Add__, f2c_NoChange, f2c_UpCase, f2c_NamErr};
enum F2CINT {FintCint=0, FintClong, FintCshort, f2c_IntErr};
enum F2CSTRING {fstrSun=0, fstrCray, fstrStructVal, fstrStructPtr, f2c_StrErr};

int XCOMP=0, ALPHA=0, INTEL=0, PowPC=0;
char TARGNAM[512];
enum MACHTYPE mach=MACHOther;

#define my_join(pre, nam) pre ## nam
#define Mstr2(m) # m
#define Mstr(m) Mstr2(m)

int QUERY=0;
FILE *fpI, *fparch;

void PrintBanner(char *fnam, int START, int sec, int subsec, int subsubsec);
#define ATL_Cassert(cond_, exp_, logfile_) \
{\
if (!(cond_)) \
{ \
   FILE *fPeRoR; \
   if (logfile_) \
      fprintf(stderr, \
              "ERROR %d DURING %s!!  CHECK %s FOR DETAILS.\n", \
              __LINE__, (exp_), (logfile_)); \
   else \
      fprintf(stderr, "ERROR %d DURING %s!!\n", __LINE__, (exp_), (logfile_)); \
   fprintf(stderr, \
      "If you can't figure it out, mail all logfiles and a description\n"); \
   fprintf(stderr, \
      "of your problem (including this error message) to atlas@cs.utk.edu.\n");\
   PrintBanner("INSTALL_LOG/ERROR.LOG", 1, 0, 0, 0); \
   fPeRoR = fopen("INSTALL_LOG/ERROR.LOG", "a"); \
   if (logfile_) \
      fprintf(fPeRoR, \
              "ERROR %d DURING %s!!  CHECK %s FOR DETAILS.\n", \
              __LINE__, (exp_), (logfile_)); \
   else \
      fprintf(fPeRoR, "ERROR %d DURING %s!!\n", __LINE__, (exp_), (logfile_)); \
   fprintf(fPeRoR, \
      "If you can't figure it out, mail all logfiles and a description\n"); \
   fprintf(fPeRoR, \
      "of your problem (including this error message) to atlas@cs.utk.edu.\n");\
      PrintBanner("INSTALL_LOG/ERROR.LOG", 1, 0, 0, 0); \
   fclose(fPeRoR); \
   exit(-1); \
}\
}

#define Mciswspace(C) ( (((C) > 8) && ((C) < 14)) || ((C) == 32) )
#define Mlowcase(C) ( ((C) > 64 && (C) < 91) ? (C) | 32 : (C) )

#include <time.h>
void GetDate(int *month, int *day, int *year, int *hour, int *min)
{
   time_t tv;
   struct tm *tp;

   tv = time(NULL);
   tp = localtime(&tv);
   *month = tp->tm_mon + 1;
   *day = tp->tm_mday;
   *year = tp->tm_year + 1900;
   *hour = tp->tm_hour;
   *min = tp->tm_min;
}
long GetInt(FILE *fpin, long Default, char *spc, char *expstr)
/*
 * Gets a signed integral type from fpin.  If nothing or garbage is entered,
 * Default is returned.
 */
{
   char str[64];
   long iin;
   if (expstr) fprintf(stdout, "%sEnter %s [%d]: ", spc, expstr, Default);
   if (fgets(str, 64, fpin) == NULL) return(Default);
   if (sscanf(str, " %ld ", &iin) != 1) return(Default);
   return(iin);
}

long GetIntRange(long Default, long Min, long Max, char *spc, char *expstr)
{
   long i;
   int keepOn=0;
   do
   {
      i = GetInt(stdin, Default, spc, expstr);
      if (i > Max)
      {
         keepOn = 1;
         fprintf(stderr, "\n%d larger than max value of %d.  Try again.\n\n",
                 i, Max);
      }
      else if (i < Min)
      {
         keepOn = 1;
         fprintf(stderr, "\n%d smaller than min value of %d.  Try again.\n\n",
                 i, Min);
      }
      else keepOn = 0;
   }
   while (keepOn);
   return(i);
}

long GetIntVer(long Default, long Min, long Max, char *spc, char *expstr)
{
   long i, j;

   do 
   {
      i = GetIntRange(Default, Min, Max, spc, expstr);
      fprintf(stdout, "%s   You entered: %d\n", spc, i);
      j = GetIntRange(0, 0, 1, spc, "1 to reenter, 0 accepts");
   }
   while(j);
   return(i);
}


void GetString(FILE *fpin, char *Default, char *spc, char *expstr, 
               int len, char *str0)
/*
 * Get a string of length len, not including NULL terminator; pads
 * any extra len with NULLs
 */
{
   char str[512], *sp;
   int i;

   assert(len+1 <= 512);
   if (expstr)
   {
      if (Default) fprintf(stdout, "%sEnter %s [%s]: ", spc, expstr, Default);
      else fprintf(stdout, "%sEnter %s:", spc, expstr);
   }
   sp = fgets(str, 512, fpin);
   if ( (sp == NULL) || (str[0] == '\0') || (str[0] == '\n') )
   {
      if (Default) strcpy(str0, Default);
      else str0[0] = '\0';
      return;
   }
   str[len] = '\0';
   for (i=0; str0[i] = str[i]; i++);
   if (i) i--;
   while (Mciswspace(str0[i])) i--;
   while (++i < len) str0[i] = '\0';
   str0[i] = '\0';
}

void GetStrVer(char *def, char *spc, char *expstr, int len, char *str)
{
   int i;

   do
   {
      GetString(stdin, def, spc, expstr, len, str);
      fprintf(stdout, "%sYou have entered '%s'\n", spc, str);
      i = GetIntRange(0, 0, 1, spc, "1 to reenter, 0 to accept");
   }
   while(i);
}
int IsYes(char def, char *spc, char *expstr)
{
   char ch, ln[256];
   fprintf(stdout, "%s%s [%c]: ", spc, expstr, def);
   if (fgets(ln, 256, stdin) == NULL) ch=def;
   else if (ln[0] == '\0' || ln[0] == '\n') ch=def;
   else ch = ln[0];
   return( ((ch == 'y') || (ch == 'Y')) );
}

char GetChar(char def, char *spc, char *expstr)
{
   char ch, ln[256];
   fprintf(stdout, "%s%s [%c]:\n", spc, expstr, def);
   if (fgets(ln, 256, stdin) == NULL) ch=def;
   else if (ln[0] = '\0' || ln[0] == '\n') ch=def;
   else ch = ln[0];
   return(ch);
}

int CmndOneLine(char *targ, char *cmnd, char *ln)
/*
 * executes a system call with contents of cmnd, returns the output in ln;
 * Returns value returned by system call
 * if targ is set, we rsh to that machine
 */
{
   char ln2[512];
   int i;
   FILE *fp;
   static char tnam[128];
   static int FirstTime=1;

   if (FirstTime)
   {
      FirstTime = 0;
      assert(tmpnam(tnam));
   }
   if (targ) sprintf(ln2, "rsh %s \"%s\" > %s\n", targ, cmnd, tnam);
   else sprintf(ln2, "%s > %s\n", cmnd, tnam);
   i = system(ln2);
   if (i == 0)
   {
      fp = fopen(tnam, "r");
      assert(fp);
      if (!fgets(ln, 512, fp)) ln[0] = '\0';
      fclose(fp);
   }
   else ln[0] = '\0';
   return(i);
}

int DisplayFile(char *fnam, FILE *fpout)
{
   FILE *fp;
   char ln[256];
   int i;

   fp = fopen(fnam, "r");
   if (fp == NULL)
   {
      fprintf(stderr, "Unable to open file '%s', continuing without display.\n",
              fnam);
      return(-1);
   }
   while (fgets(ln, 256, fp)) fprintf(fpout, "%s", ln);
   i = ferror(fp);
   fclose(fp);
   return(i);
}

int FileIsThere(char *nam)
{
   FILE *fp;

   fp = fopen(nam, "r");
   if (fp == NULL) return(0);
   fclose(fp);
   return(1);
}

int FoundInFile(char *fnam, char *str)
{
   FILE *fp;
   int found=0;
   char ln[256];

   fp = fopen(fnam, "r");
   assert(fp);
   while (fgets(ln, 256, fp))
   {
      if (strstr(ln, str))
      {
         found=1;
         break;
      }
   }
   fclose(fp);
   return(found);
}

int FindFiles(char *dir, char *fnam, char ***files0)
{
   int i, j;
   FILE *fp;
   char ln[256];
   static int FirstTime=1;
   static char *fpp[32], files[32][256], tnam[128], fnd[128];

   if (FirstTime)
   {
      if (FileIsThere("/usr/bin/find")) strcpy(fnd, "/usr/bin/find");
      else if (FileIsThere("/bin/find")) strcpy(fnd, "/bin/find");
      else if (FileIsThere("/usr/local/bin/find"))
         strcpy(fnd, "/usr/local/bin/find");
      else strcpy(fnd, "find");
      for (i=0; i < 32; i++) fpp[i] = files[i];
      assert(tmpnam(tnam));
      FirstTime=0;
   }
   *files0 = fpp;
   sprintf(ln, "%s %s/ -name \'%s\' 2> /dev/null > %s\n", fnd, dir, fnam, tnam);
   i = system(ln);
   fp = fopen(tnam, "r");
   if (fp)
   {
      for (i=0; i < 32; i++)
      {
         if (fgets(files[i], 256, fp) == NULL) break;
         for (j=0; files[i][j]; j++) if (files[i][j] == '\n') files[i][j] = ' ';
      }
      fclose(fp);
   }
   else
   {
      fprintf(stderr, "Find not working, error code %d\n", i);
      return(0);
   }
   remove(tnam);
   return(i);
}

int FindFirstInPath(char *nam, char *fnd)
/*
 * finds first instance of nam in you path, returns 0 if not found, 1 else
 */
{
   char ln[256], *sp, *cp;
   FILE *fp;

   sp = getenv("PATH");
   cp = ln;
   do
   {
      while (*sp && *sp != ':') *cp++ = *sp++;
      if (*sp) sp++;
      if (cp[-1] != '/') *cp++ = '/';
      strcpy(cp, nam);
      fp = fopen(ln, "r");
      if (fp != NULL)
      {
         fclose(fp);
         strcpy(fnd, ln);
         return(1);
      }
      cp = ln;
   }
   while(*sp);
   return(0);
}
void FindAllGccs(FILE *fpout, FILE *fplog, char *gcc, char *egcs, char *pgcc)
{
   char *tmpc = "CONFIG/tst.c";
   char *tnam = "CONFIG/tmp.t";
   char **files;
   char ln[256];
   #define ND 2
   char *dirs[ND] = {"/usr/bin", "/usr/local"};
   int i, d, n;
   FILE *fp;

   remove(tnam);
   *pgcc = *egcs = *gcc = '\0';
   for (d=0; d < ND; d++)
   {
      n = FindFiles(dirs[d], "gcc", &files);
      for (i=0; i < n; i++)
      {
         sprintf(ln, "%s -v -c %s > %s 2>&1\n", files[i], tmpc, tnam);
         if (system(ln) == 0)
         {
            if (FoundInFile(tnam, "pgcc")) strcpy(pgcc, files[i]);
            else if (FoundInFile(tnam, "egcs")) strcpy(egcs, files[i]);
            else strcpy(gcc, files[i]);
         }
         remove(tnam);
         if (*gcc && *egcs && *pgcc) break;
      }
   }
   if (*gcc)
   {
      if (fpout) fprintf(fpout, "   REGULAR GCC : %s\n", gcc);
      if (fplog) fprintf(fplog, "   REGULAR GCC : %s\n", gcc);
   }
   else
   {
      if (fpout) fprintf(fpout, "   REGULAR GCC : not found\n");
      if (fplog) fprintf(fplog, "   REGULAR GCC : not found\n");
   }
   if (*egcs)
   {
      if (fpout) fprintf(fpout, "   EGCS        : %s\n", egcs);
      if (fplog) fprintf(fplog, "   EGCS        : %s\n", egcs);
   }
   else
   {
      if (fpout) fprintf(fpout, "   EGCS        : not found\n");
      if (fplog) fprintf(fplog, "   EGCS        : not found\n");
   }
   if (*pgcc)
   {
      if (fpout) fprintf(fpout, "   PGCC        : %s\n", pgcc);
      if (fplog) fprintf(fplog, "   PGCC        : %s\n", pgcc);
   }
   else
   {
      if (fpout) fprintf(fpout, "   PGCC        : not found\n");
      if (fplog) fprintf(fplog, "   PGCC        : not found\n");
   }
   if (fpout) fprintf(fpout, "\n");
   if (fplog) fprintf(fplog, "\n");
}
#undef ND

int Wstrfndsub(char *fnd, char *sub, char *in, char *out)
/*
 * copies IN to OUT, replacing the first occurrence of FND with SUB
 * 1 if FND was found, 0 otherwise
 */
{
   char *sp;
   int i, j;

   sp = strstr(in, fnd);
   if (sp)
   {
      for (i=0; &in[i] != sp; i++) out[i] = in[i];
      for (j=i; out[j] = sub[j-i]; j++);
      for (i += strlen(fnd); out[j] = in[i]; j++, i++);
   }
   else strcpy(out, in);
   return(sp != NULL);
}

#define F2C_assert(cond_) \
{ \
   if (!(cond_)) \
   { \
      fprintf(stderr, "Assertion %s failed, line %d of %s\n", \
              Mstr(cond_), __LINE__, __FILE__); \
      fprintf(stderr, "Unable to figure F2C data\n\n"); \
      return; \
   } \
}

int findSolErr(FILE *fplog, char *redir, char *TOPdir, char *CC, char *CCFLAGS,
               char *MCC, char *MMFLAGS, char *XCC, char *XCCFLAGS)
{
   char *sp, *tnam = "CONFIG/config.tmp";
   char compdef[512], ln[640];
   int i;
   FILE *fp;

   sprintf(ln, "cd CONFIG ; make IGetSunVers %s\n", redir);
   system(ln);
   fp = fopen(tnam, "r");
   fgets(ln, 256, fp);
   fclose(fp);
   remove(tnam);
   for (i=0; !isdigit(ln[i]) && ln[i]; i++);
   if (isdigit(ln[i]))
   {
      fprintf(stdout, "   cc major version number: %c\n", ln[i]);
      if (ln[i] == '4')
      {
         sp = strstr(MMFLAGS, "-xO2");
         if (sp) sp[3] = '5';
      }
   }
   else fprintf(stdout, "   unable to find version in \'%s\'\n", ln);
   DisplayFile("CONFIG/SolErr.txt", stdout);
   return(IsYes('y', "", "Use gcc for Level 3 BLAS compilation?"));
}

void findF2C(char *redir, char *TOPdir, char *F77, char *F77FLAGS, char *FLINK0,
             char *FLFLAGS0, char *CC, char *CCFLAGS, enum F2CNAME *f2cnam,
             enum F2CINT *f2cint, enum F2CSTRING *f2cstr)
{
   char *namdef[4] = {"Add_", "Add__", "NoChange", "UpCase"};
   char *intdef[4]={"F77_INTEGER=int", "F77_INTEGER=long", "F77_INTEGER=short"};
   char *strdef[4] = {"SunStyle", "CrayStyle", "StructVal", "StructPtr"};
   char compdef[512], ln[640];
   char FLINK[512], FLFLAGS[512];
   char *tnam = "CONFIG/config.tmp";
   int i;
   FILE *fp;

   *f2cnam = f2c_NamErr;
   *f2cint = f2c_IntErr;
   *f2cstr = f2c_StrErr;
   Wstrfndsub("$(F77)", F77, FLINK0, FLINK);
   Wstrfndsub("$(F77FLAGS)", F77FLAGS, FLFLAGS0, FLFLAGS);
/*
 * Find naming strategy
 */
   sprintf(compdef, "F77=%s F77FLAGS=\"%s\" FLINK=%s FLFLAGS=\"%s\" CC=%s CCFLAGS=\"%s\" mydir=%s/CONFIG tnam=%s",
           F77, F77FLAGS, FLINK, FLFLAGS, CC, CCFLAGS, TOPdir, tnam);
   sprintf(ln, "cd CONFIG ; make IRunName %s %s\n", compdef, redir);
   F2C_assert(system(ln) == 0);
   fp = fopen(tnam, "r");
   F2C_assert(fp);
   fgets(ln, 256, fp);
   fclose(fp);
   remove(tnam);
   if (strstr(ln, "Add__")) *f2cnam = f2c_Add__;
   else if (strstr(ln, "Add_")) *f2cnam = f2c_Add_;
   else if (strstr(ln, "NoChange")) *f2cnam = f2c_NoChange;
   else if (strstr(ln, "UpCase")) *f2cnam = f2c_UpCase;
   else return;
/*
 * Finding integer correspondence
 */
   sprintf(compdef, "F77=%s F77FLAGS=\"%s\" FLINK=%s FLFLAGS=\"%s\" CC=%s CCFLAGS=\"%s -D%s\" mydir=%s/CONFIG tnam=%s",
           F77, F77FLAGS, FLINK, FLFLAGS, CC, CCFLAGS, namdef[*f2cnam], 
           TOPdir, tnam);
   sprintf(ln, "cd CONFIG ; make IRunInt %s %s\n", compdef, redir);
   F2C_assert(system(ln) == 0);
   fp = fopen(tnam, "r");
   F2C_assert(fp);
   fgets(ln, 256, fp);
   fclose(fp);
   remove(tnam);
   if (strstr(ln, "short")) *f2cint = FintCshort;
   else if (strstr(ln, "int")) *f2cint = FintCint;
   else if (strstr(ln, "long")) *f2cint = FintClong;
   else return;
/*
 * Finding string handling
 */
   sprintf(compdef, 
"F77=%s F77FLAGS=\"%s\" FLINK=%s FLFLAGS=\"%s\" CC=%s mydir=%s/CONFIG tnam=%s",
           F77, F77FLAGS, FLINK, FLFLAGS, CC, TOPdir, tnam);
   for (i=0; i < 5; i++)
   {
      sprintf(ln, 
              "cd CONFIG ; make IRunStr %s CCFLAGS=\"%s -D%s -D%s -D%s\" %s\n", 
              compdef, CCFLAGS, strdef[i], intdef[*f2cint], namdef[*f2cnam],
              redir);
      if (system(ln) == 0)
      {
         fp = fopen(tnam, "r");
         F2C_assert(fp);
         fgets(ln, 256, fp);
         fclose(fp);
         remove(tnam);
         if (strstr(ln, strdef[i]))
         {
            *f2cstr = i;
            return;
         }
      }
   }
}

#undef F2C_assert

#define ND 3
void FindBlas(FILE *fpout, FILE *fplog, char *redir, char *F77, char *F77FLAGS,
              char *BLASlib)
{
   char **files;
   char *dirs[ND] = {"/usr/lib", "/usr/local", ""};
   char ln[256];
   int i, j, n;

   if (fpout) fprintf(fpout, "Looking for BLAS:\n");
   if (fplog) fprintf(fplog, "Looking for BLAS:\n");
   if (*BLASlib)
   {
      sprintf(ln, 
      "cd CONFIG ; make IBlasLink F77=\"%s\" F77FLAGS=\"%s\" BLASlib=\"%s\" %s\n",
              F77, F77FLAGS, BLASlib, redir);
      if (system(ln) == 0)
      {
         if (fpout) fprintf(fpout, "BLASlib set to %s.\n", BLASlib);
         if (fplog) fprintf(fplog, "BLASlib set to %s.\n", BLASlib);
         return;
      }
      if (fpout) fprintf(fpout, "   Link failed, %s rejected\n", BLASlib);
      if (fplog) fprintf(fplog, "   Link failed, %s rejected\n", BLASlib);
   }
   dirs[ND-1] = getenv("HOME");
   for (j=0; j < ND; j++)
   {
      n = FindFiles(dirs[j], "lib*blas*.a", &files);
      if (n)
      {
         for (i=0; i < n; i++)
         {
            sprintf(ln, 
      "cd CONFIG ; make IBlasLink F77=\"%s\" F77FLAGS=\"%s\" BLASlib=\"%s\" %s\n",
                    F77, F77FLAGS, files[i], redir);
            if (system(ln) == 0)
            {
               strcpy(BLASlib, files[i]);
               if (fpout) fprintf(fpout, "BLASlib set to %s.\n", BLASlib);
               if (fplog) fprintf(fplog, "BLASlib set to %s.\n", BLASlib);
               return;
            }
            if (fpout) 
               fprintf(fpout, "   Link failed, %s rejected\n", files[i]);
            if (fplog)
               fprintf(fplog, "   Link failed, %s rejected\n", files[i]);
         }
      }
   }
   if (fpout)
      fprintf(fpout, "Unable to find usable BLAS, BLASlib left blank.\n");
   if (fplog)
      fprintf(fplog, "Unable to find usable BLAS, BLASlib left blank.\n");
   BLASlib[0] = '\0';
}
#undef ND

char *FindUname()
{
   static int FirstTime=1;
   static char unam[64];
   if (FirstTime)
   {
      if (FileIsThere("/bin/uname")) strcpy(unam, "/bin/uname");
      else if (FileIsThere("/usr/bin/uname")) strcpy(unam, "/usr/bin/uname");
      else strcpy(unam, "uname");
      FirstTime = 0;
   }
   return(unam);
}

enum OSTYPE GetOS(FILE *fpout, FILE *fplog, char *targ)
{
   int ierr;
   char ln[512], ln2[128];
   enum OSTYPE i, OS;
   char *unam = FindUname();

   if (fpout) 
      fprintf(fpout, "Probing to make operating system determination:\n");
   if (fplog)
      fprintf(fplog , "Probing to make operating system determination:\n");

   sprintf(ln2, "%s -s", unam);
   ierr = CmndOneLine(targ, ln2, ln);
   if (ierr == 0)
   {
      if(strstr(ln, "Linux")) OS = OSLinux;
      else if(strstr(ln, "SunOS")) 
      {
         sprintf(ln2, "%s -r", unam);
         CmndOneLine(targ, ln2, ln);
         if (ln[0] == '4') OS = OSSunOS4;
         else OS = OSSunOS;
      }
      else if(strstr(ln, "OSF1")) OS = OSOSF1;
      else if(strstr(ln, "IRIX")) OS = OSIRIX;
      else if(strstr(ln, "AIX")) OS = OSAIX;
      else if(strstr(ln, "WIN"))
      {
         if (strstr(ln, "95") || strstr(ln, "98")) OS = OSWin9x;
         else if (strstr(ln, "NT")) OS = OSWinNT;  /* check this */
      }
      else if (strstr(ln, "HP-UX")) OS = OSHPUX;
      else ierr = 1;
   }
   if (ierr)
   {
      fprintf(stdout, 
         "%s -s does not appear to work.\n", unam);
      for (i=0; i <= OSOther; i++) fprintf(stdout, " %3d. %s\n", i+1, osnam[i]);
      OS = (enum OSTYPE) GetIntRange(OSOther+1, 1, OSOther+1, "", 
                                     "the number of your operating system") - 1;
   }

   if (fpout) 
      fprintf(stdout, "Operating system configured as %s\n\n", osnam[OS]);
   if (fplog) 
      fprintf(fplog , "Operating system configured as %s\n\n", osnam[OS]);

   return(OS);
}

char *TryComp(FILE *fpout, FILE *fplog, char *redir, char lang, int np,
              char **paths, char *comp, char *flags)
{
   static char ln2[512];
   char **matches, ln[512];
   int p, n, i;

/*
 * Try user's path first
 */
   if (FindFirstInPath(comp, ln2));
   {
      sprintf(ln, "cd CONFIG ; make %cTryComp COMP=\"%s\" FLAGS=\"%s\" %s\n",
              lang, ln2, flags, redir);
      if (system(ln) == 0) return(ln2);
      if (fpout) fprintf(fpout, "   %s %s rejected\n", ln2, flags);
      if (fplog) fprintf(fplog, "   %s %s rejected\n", ln2, flags);
   }
/*
 * A separate find on each directory, in order of precedence
 */
   for (p=0 ; p < np; p++)
   {
/*
 *    Try every compiler that is found, search backwards so bigger version
 *    numbers will be tried first
 */
      n = FindFiles(paths[p], comp, &matches);
      for (i=n-1; i >= 0; i--)
      {
         sprintf(ln, "cd CONFIG ; make %cTryComp COMP=\"%s\" FLAGS=\"%s\" %s\n",
                 lang, matches[i], flags, redir);
         if (system(ln) == 0) return(matches[i]);
         if (fpout) fprintf(fpout, "   %s rejected\n", matches[i]);
         if (fplog) fprintf(fplog, "   %s rejected\n", matches[i]);
      }
   }
   return(NULL);
}

void GetCompInfo(FILE *fpout, FILE *fplog, enum OSTYPE OS, enum MACHTYPE mach,
                 char *targ, char *redir,
                 char *F77, char *F77FLAGS, char *FLINK, char *FLFLAGS,
                 char *CC, char *CCFLAGS, char *CLINK, char *CLFLAGS, 
                 char *MCC, char *MMFLAGS, char *BLASlib)
/*
 * Sets up good compiler flags for various OSs, and searches for the correct
 * compilers.  The compiler search should be improved so that it always takes
 * the newest release;  right now it simply takes the first compiler that
 * works with the desired flags.
 * CC & CCFLAGS are required;  other options may be set * to NULL, and
 * thus not probed.
 */
{
   int MCisCC=1;
   int i, np=2;
   char *paths[16];
   char *CCG=NULL, *MCCG=NULL, *F77G=NULL;
   char *sp, ln[512];
   char gcc[256], egcs[256], pgcc[256];
   char *unam = FindUname();
/*
 * normal defaults 
 */
   paths[0] = "/usr/bin/";
   paths[1] = "/usr/local/";
   strcpy(CC, "cc");
   strcpy(CCFLAGS, "-O");
   if (CLINKER) strcpy(CLINKER, "$(CC)");
   if (CLINKFLAGS) strcpy(CLINKFLAGS, "$(CCFLAGS)");

   if (MCC) strcpy(MCC, CC);
   if (MMFLAGS) strcpy(MMFLAGS, CCFLAGS);

   if (F77) strcpy(F77, "f77");
   if (F77FLAGS) strcpy(F77FLAGS, "-O");
   if (FLINKER) strcpy(FLINKER, "$(F77)");
   if (FLINKFLAGS) strcpy(FLINKFLAGS, "$(F77FLAGS)");
   if (BLASlib) BLASlib[0] = '\0';

   switch(OS)
   {
   case OSLinux:
      if (F77) strcpy(F77, "g77");
      strcpy(CC, "gcc");
      if (F77FLAGS) strcpy(F77FLAGS, "-funroll-all-loops -O3");
      strcpy(CCFLAGS, "-fomit-frame-pointer -O3 -funroll-all-loops");
      if (MMFLAGS) strcpy(MMFLAGS, "-fomit-frame-pointer -O");
      if (mach == Dec21164 || mach == Dec21264)
      {
         if (MMFLAGS) strcpy(MMFLAGS, 
         "-O1 -fschedule-insns -fschedule-insns2 -fno-expensive-optimizations");
      }
      if (mach == PPC604 || mach == PPC604e)  /* avoid unroll loops bug */
      {
         if (F77FLAGS) strcpy(F77FLAGS, "-O3");
         strcpy(CCFLAGS, "-fomit-frame-pointer -O3");
      }
      if (mach == SunUS1 || mach == SunUS2 || mach == SunUSX)
      {
         strcpy(CCFLAGS, 
         "-fomit-frame-pointer -O3 -funroll-all-loops -mcpu=v9 -Wa,-Av8plusa");
         if (MMFLAGS) 
            strcpy(MMFLAGS, "-fomit-frame-pointer -O -mcpu=v9 -Wa,-Av8plusa");
      }
      break;
   case OSSunOS:
      np = 3;
      paths[0] = "/opt/";
      paths[1] = "/usr/bin/";
      paths[2] = "/usr/local/";
      if (BLASlib) strcpy(BLASlib, "-xlic_lib=sunperf");
      if (F77) strcpy(F77, "f77");
      strcpy(CC, "cc");
      if (F77FLAGS) strcpy(F77FLAGS, "-dalign -native -xO5");
      strcpy(CCFLAGS, "-dalign -fsingle -xO5 -native");
      if (MMFLAGS) strcpy(MMFLAGS, "-dalign -fsingle -xO2 -native");
      if (mach == SunUS1 || mach == SunUS2 || mach == SunUSX)
      {
            if (F77FLAGS) strcpy(F77FLAGS, "-dalign -native -xarch=v8plusa -xO5");
            strcpy(CCFLAGS, "-dalign -fsingle -xO5 -native -xarch=v8plusa");
            if (MMFLAGS) strcpy(MMFLAGS, 
         "-dalign -fsingle -xO2 -native -xarch=v8plusa -fsimple=1 -xsafe=mem");
      }
      break;
   case OSSunOS4:
      if (F77) strcpy(F77, "f77");
      if (F77FLAGS) strcpy(F77FLAGS, "-dalign -O4 -fast");
      strcpy(CC, "gcc");
      strcpy(CCFLAGS, "-fomit-frame-pointer -O3 -funroll-all-loops");
      if (MMFLAGS) strcpy(MMFLAGS, "-fomit-frame-pointer -O");
      break;
   case OSOSF1 :
      strcpy(BLASlib, "-ldxml");
      if (F77) strcpy(F77, "f77");
      if (F77FLAGS) strcpy(F77FLAGS, "-O5 -arch host -tune host");
      strcpy(CC, "cc");
      strcpy(CCFLAGS, "-arch host -tune host -std -O5");
      MCisCC = 0;
      if (MCC) strcpy(MCC, "gcc");
      if (MMFLAGS) strcpy(MMFLAGS, 
         "-O1 -fschedule-insns -fschedule-insns2 -fno-expensive-optimizations");
      break;
   case OSIRIX :
      if (mach == SgiIP22) 
      {
         if (F77FLAGS) strcpy(F77FLAGS, "-O2 -mips2 -Olimit 15000");
         strcpy(CCFLAGS, "-O2 -mips2 -Olimit 15000");
      }
      else if (mach == SgiIP32)
      {
         if (F77FLAGS) sprintf(F77FLAGS, 
      "-O3 -n32 -mips4 -OPT:Olimit=15000 -TARG:platform=ip32_5k -LNO:blocking=OFF");
         sprintf(CCFLAGS, "-O2 -n32 -mips4 -OPT:Olimit=15000 -TARG:platform=ip32_5k -LNO:blocking=OFF -LOPT:aliased=typed");
      }
      else
      {
         sprintf(egcs, "%s -m", unam);
         assert(CmndOneLine(targ, egcs, ln) == 0);
         sp = strstr(ln, "IP");
         for (i=2; isdigit(sp[i]); i++);
         sp[i] = '\0';
         if (F77FLAGS) sprintf(F77FLAGS, 
            "-O3 -64 -OPT:Olimit=15000 -TARG:platform=%s -LNO:blocking=OFF", 
                 sp);
         sprintf(CCFLAGS, "%s -LOPT:alias=typed", F77FLAGS);
      }
      strcpy(MMFLAGS, CCFLAGS);
      strcpy(BLASlib, "-lblas");
      if (F77) strcpy(F77, "f77");
      strcpy(CC, "cc");
      break;
   case OSAIX  :
      strcpy(CC, "xlc");
      strcpy(CCFLAGS, "-O3 -qmaxmem=-1 -qfloat=hsflt");
      if (F77) strcpy(F77, "xlf");
      if (F77FLAGS) strcpy(F77FLAGS, CCFLAGS);
      if (F77FLAGS) strcpy(CCFLAGS, F77FLAGS);
      if (MMFLAGS) strcpy(MMFLAGS, "$(CCFLAGS) -O");
      if (mach == IbmPwr)
      {
         strcpy(CCFLAGS, "-O3 -qarch=pwr -qtune=pwr -qmaxmem=-1 -qfloat=hsflt");
      }
      else if (mach == IbmTPwr2)
      {
         strcpy(CCFLAGS, 
                "-O3 -qarch=pwr2 -qtune=pwr2 -qmaxmem=-1 -qfloat=hsflt");
         if (MMFLAGS) sprintf(MMFLAGS,
  "-qarch=pwr2 -qtune=pwr2 -qmaxmem=-1 -qfloat=hsflt -qansialias -qfold -O");
         mach = IbmTPwr2;
         strcpy(BLASlib, "-lesslp2");
      }
      else if (mach == IbmPwr3)
      {
         strcpy(CCFLAGS, 
                "-qtune=pwr3 -qarch=pwr3 -O3 -qmaxmem=-1 -qfloat=hsflt");
         if (MMFLAGS) sprintf(MMFLAGS, 
         "-qtune=pwr3 -qarch=pwr3 -O3 -qmaxmem=-1 -qfloat=hsflt -qalias=allp");
         mach = IbmPwr3;
         strcpy(BLASlib, "-lessl");
      }
      else if (mach == PPC604)
      {
         strcpy(CCFLAGS, "-O3 -qarch=ppc -qtune=604 -qmaxmem=-1 -qfloat=hsflt");
         if (MMFLAGS) strcpy(MMFLAGS, 
"-O -qarch=ppc -qtune=604 -qmaxmem=-1 -qfloat=hsflt -qproto -qansialias -qfold");
      }
      else if (mach == PPC604e)
         strcpy(CCFLAGS, "-O3 -qarch=ppc -qtune=604 -qmaxmem=-1 -qfloat=hsflt");
      if (F77FLAGS) strcpy(F77FLAGS, CCFLAGS);
      break;
      break;
   case OSWin9x:
   case OSWinNT:
      if (F77) strcpy(F77, "g77");
      strcpy(CC, "gcc");
      if (F77FLAGS) strcpy(F77FLAGS, "-funroll-all-loops -O3");
      strcpy(CCFLAGS, "-fomit-frame-pointer -O3 -funroll-all-loops -DUseClock");
      if (MMFLAGS) strcpy(MMFLAGS, "-fomit-frame-pointer -O");
      break;
   case OSHPUX :
      if (F77) strcpy(F77, "f77");
      strcpy(CC, "cc");
      if (F77FLAGS) strcpy(F77FLAGS, "+O4");
      strcpy(CCFLAGS, "-D_INCLUDE_POSIX_SOURCE -DUseTimes -Aa +O4");
      if (MMFLAGS) strcpy(MMFLAGS, "-Aa +O2");
      if (mach == HPPA8K)
         if (MMFLAGS) strcpy(MMFLAGS, 
                "-Aa +O2 +Onoinitcheck +Odataprefetch +Optrs_strongly_typed");
      if (mach == HP9K735)
      {
         if (CLINKFLAGS) strcpy(CLINKFLAGS, "-Aa");
         if (FLINKFLAGS) strcpy(FLINKFLAGS, "-Aa");
      }
      break;
   case OSOther:
      break;
   }
   if (fpout) fprintf(fpout, "Looking for compilers:\n");
   if (fplog) fprintf(fplog, "Looking for compilers:\n");
   if (mach == Dec21164 || mach == Dec21264 || OS == OSLinux)
   {
      FindAllGccs(fpout, fplog, gcc, egcs, pgcc);
/*
 * ATTENTION:  Need to make this guy print warnings when inferior gcc is chosen
 */
      if (mach == PPC604 || mach == PPC604e)  /* egcs best compiler for PPC */
      {
         if (*egcs) CCG = MCCG = egcs;
         else if (*gcc) CCG = MCCG = gcc;
         strcpy(CC, CCG);
         strcpy(MCC, MCCG);
      }
      else if (OS == OSOSF1)
      {
         if ( (*gcc || *egcs) && MCC && strstr(MCC, "gcc") )
         {
            if (*gcc) MCCG = gcc;
            else MCCG = egcs;
            if (MMFLAGS)
               strcpy(MMFLAGS, "-O1 -fschedule-insns -fschedule-insns2");
            if (MCCG && MCC) strcpy(MCC, MCCG);
         }
      }
      else
      {
         if (*gcc)  CCG = MCCG = gcc;
         else if (*pgcc) CCG = MCCG = pgcc;
         else CCG = MCCG = egcs;
         strcpy(MCC, MCCG);
         strcpy(CC, CCG);
      }
   }
   if (CCG == NULL)
      CCG = TryComp(fpout, fplog, redir, 'c', np, paths, CC, CCFLAGS);
   if (CCG != NULL)
   {
      strcpy(CC, CCG);
      if (MCCG == NULL && MCC)
      {
         strcpy(MCC, CCG);
         MCCG = CCG;
      }
   }
   if (!MCisCC && MCCG == NULL && MCC && MMFLAGS)
   {
      MCCG = TryComp(fpout, fplog, redir, 'c', np, paths, MCC, MMFLAGS);
   }
   if (F77G == NULL && F77)
   {
      F77G = TryComp(fpout, fplog, redir, 'f', np, paths, F77, F77FLAGS);
      if (F77G) strcpy(F77, F77G);
   }
   if (F77) if (fpout) fprintf(fpout, "F77 = %s %s\n", F77, F77FLAGS);
   if (fpout) fprintf(fpout, "CC = %s %s\n", CC, CCFLAGS);
   if (MCC) if (fpout) fprintf(fpout, "MCC = %s %s\n\n", MCC, MMFLAGS);
   if (F77) if (fplog) fprintf(fplog, "F77 = %s %s\n", F77, F77FLAGS);
   if (fplog) fprintf(fplog, "CC = %s %s\n", CC, CCFLAGS);
   if (MCC) if (fplog) fprintf(fplog, "MCC = %s %s\n\n", MCC, MMFLAGS);

   if (F77 && BLASlib) FindBlas(fpout, fplog, redir, F77, F77FLAGS, BLASlib);
}

int GetCacheSize(enum OSTYPE OS,  enum MACHTYPE arch, char *targ, 
                 int lvl, int *AmSure)
{
/*
 * Returns size of requested level of cache.  If *AmSure is 0, this is a maximum,
 * otherwise that machine only comes with that size of cache.  If lvl is < 0,
 * give me a safe size to flush (handling associativity and so forth)
 */
   int l1, l2, s1, s2;
   int lf1, lf2;
   char ln[512], *sp;

/*
 * See if we can figure out the size of L2 cache using hinv
 */
   if (OS == OSIRIX)  /* get L2 & L1 size */
   {
      l2 = -1; s2 = 0;
      if(CmndOneLine(targ, "hinv | fgrep Secondary | fgrep \"cache size\"", ln) == 0)
      {
         if (sp = strstr(ln, " Mbyte"))
         {
            s2 = 1;
            sp--;
            while(isdigit(*sp)) sp--;
            sp++;
            sscanf(sp, "%d", &l2);
            l2 *= 1024;
            lf2 = l2;
            fprintf(stdout, "\n   hinv reports L2SIZE=%d KB\n", l2);
         }
         else if (sp = strstr(ln, " Kbyte"))
         {
            s2 = 1;
            sp--;
            while(isdigit(*sp)) sp--;
            sp++;
            sscanf(sp, "%d", &l2);
            fprintf(stdout, "\n   hinv reports L2SIZE=%d KB\n", l2);
            lf2 = l2;
         }
      }
   }
   switch(mach)
   {
   case MACHOther:
      l1 = l2 = s1 = s2 = 0;
      lf2 = 4096;
      break;
   case SunUS2:
      lf1 = l1 = 16;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   case SunUS1:
      lf1 = l1 = 16;
      lf2 = l2 = 1024;
      s1 = 1;
      s2 = 0;
      break;
   case SunSS:
      lf1 = l1 = 32;
      lf2 = l2 = 1024;
      s1 = 1;
      s2 = 0;
      break;
   case SunMS:
      lf1 = l1 = 8;
      lf2 = l2 = l1;
      s1 = 1;
      s2 = 1;
      break;
   case SgiIP22:    /* R4600 */
      if (l2 == -1) lf2 = l2 = 128;
      lf1 = l1 = 16;
      s1 = 1;
      break;
   case SgiIP32:   /* R5K */
      if (l2 == -1) lf2 = l2 = 1024;
      l1 = 32;
      lf1 = 2*l1;
      s1 = 1;
      break;
   case SgiIP27:
   case SgiIP28:
   case SgiIP30:
      if (l2 == -1) lf2 = l2 = 4096;
      l1 = 32;
      lf1 = 2*l1;
      s1 = 1;
      break;
   case AmdAthlon:
      lf1 = l1 = 64;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   case IntXeon:
      lf1 = l1 = 16;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   case IntPII:
      lf1 = l1 = 16;
      lf2 = l2 = 512;
      s1 = 1;
      s2 = 0;
      break;
   case IntPPRO:
      lf1 = l1 = 8;
      lf2 = l2 = 1024;
      s1 = 1;
      s2 = 0;
      break;
   case IntP5MMX:
      lf1 = l1 = 16;
      lf2 = l2 = 1024;
      s1 = 1;
      s2 = 0;
      break;
   case IntP5:
      lf1 = l1 = 8;
      lf2 = l2 = 1024;
      s1 = 1;
      s2 = 0;
      break;
   case PPC604e:
      l1 = 32;
      lf1 = 4*l1;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   case PPC604:
      l1 = 16;
      lf1 = 4*l1;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   case IbmPwr:
      l1 = 64;
      l2 = 64;
      lf2 = lf1 = 4*l1;
      s1 = 0;
      s2 = 0;
      break;
   case IbmWPwr2:
      l1 = 256;
      l2 = 256;
      lf2 = lf1 = 4*l1;
      s1 = 1;
      s2 = 1;
      break;
   case IbmTPwr2:
      l1 = 128;
      l2 = 128;
      lf2 = lf1 = 4*l1;
      s1 = 1;
      s2 = 1;
      break;
   case IbmPwr3:
      l1 = 64;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   case HP9K735:
      lf1 = l1 = 256;
      l2 = 256;
      lf2 = 4*l2;
      s1 = 1;
      s2 = 1;
      break;
   case HPPA8K:
      lf1 = l1 = 1024;
      lf2 = l2 = 1024;
      lf2 = 4*l2;
      s1 = 0;
      s2 = 0;
      break;
   case Dec21064:
      lf1 = l1 = 16;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   case Dec21164:
      lf1 = l1 = 8;
      lf2 = l2 = 4096;
      s1 = 1;
      s2 = 0;
      break;
   default:
      l1 = l2 = s1 = s2 = 0;
      lf2 = 4096;
   }
   if (lvl == 1) 
   {
      if (AmSure) *AmSure = s1;
      return(l1);
   }
   else if (lvl == -1)
   {
      if (AmSure) *AmSure = s1;
      return(lf1);
   }
   else if (lvl == -2)
   {
      if (AmSure) *AmSure = s2;
      return(lf2);
   }
   else
   {
      if (AmSure) *AmSure = s2;
      return(l2);
   }
}

enum MACHTYPE GetArch(FILE *fpout, FILE *fplog, enum OSTYPE OS, char *targ)
{
   char ln[512], ln2[512];
   int i, j, ierr;
   enum MACHTYPE mach=MACHOther;
   char *unam=FindUname();

   fprintf(fpout, "Probing for architecture:\n");
   fprintf(fplog, "Probing for architecture:\n");
   strcpy(MCC, "$(CC)");
   strcpy(MMFLAGS, "$(CCFLAGS)");
   strcpy(XCC, "$(CC)");
   strcpy(XCCFLAGS, "$(CCFLAGS)");
   sprintf(ln2, "%s -m", unam);
   switch(OS)
   {
   case OSLinux:
      mach = MACHOther;
      if (CmndOneLine(targ, ln2, ln)) mach=MACHOther;
      else
      {
         ln2[0] = '\0';
         ierr = CmndOneLine(targ, "cat /proc/cpuinfo | fgrep \"model name\"", 
                            ln2);
         if (ierr || ln2[0] == '\0')
            ierr = CmndOneLine(targ, "cat /proc/cpuinfo | fgrep model", ln2);
         j = 0;
         if (strstr(ln, "ppc"))
         {
            mach = MACHOther;
            PowPC = 1;
            if (CmndOneLine(targ, "cat /proc/cpuinfo | fgrep cpu", ln) == 0)
            {
               if (strstr(ln, "604e")) mach = PPC604e;
               else if (strstr(ln, "604 ")) mach = PPC604;
            }
            if (mach == MACHOther)
            {
               fprintf(stdout, "Enter your machine type:\n");
               fprintf(stdout, "   1. PowerPC 604e\n");
               fprintf(stdout, "   2. PowerPC 604\n");
               fprintf(stdout, "   3. Other/UNKNOWN\n");
               i = GetIntRange(3, 1, 3, "", "machine number");
               if (i == 1) mach = PPC604e;
               else if (i == 2) mach = PPC604;
               else mach = MACHOther;
            }
         }
         else if (strstr(ln, "sparc"))
         {
            mach = MACHOther;
            if (CmndOneLine(targ, "cat /proc/cpuinfo | fgrep cpu", ln) == 0)
            {
               if (strstr(ln, "UltraSparc II")) mach = SunUS2;
               else if (strstr(ln, "UltraSparc I ")) mach = SunUS1;
               else if (strstr(ln, "UltraSparc")) mach = SunUSX;
            }
            if (mach == MACHOther)
            {
               fprintf(stdout, "Enter your machine type:\n");
               fprintf(stdout, "   1. UltraSparc I\n");
               fprintf(stdout, "   2. UltraSparc II\n");
               fprintf(stdout, "   3. Other UltraSparc\n");
               fprintf(stdout, "   4. Other/UNKNOWN\n");
               i = GetIntRange(4, 1, 4, "", "machine number");
               if (i == 1) mach = SunUS1;
               else if (i == 2) mach = SunUS2;
               else if (i == 3) mach = SunUSX;
               else mach = MACHOther;
            }
         }
         else if (strstr(ln, "alpha"))
         {
            mach = MACHOther;
            if (!ierr)
            {
               if (strstr(ln2, "EV5")) mach = Dec21164;
               else if (strstr(ln2, "EV4")) mach = Dec21064;
            }
            if (mach == MACHOther)
            {
               fprintf(stdout, "Enter your machine type:\n");
               fprintf(stdout, "   1. DEC ALPHA 21064\n");
               fprintf(stdout, "   2. DEC ALPHA 21164 (ev5/ev56)\n");
               fprintf(stdout, "   3. DEC ALPHA 21264 (ev6)\n");
               fprintf(stdout, "   4. Other/UNKNOWN\n");
               i = GetIntRange(4, 1, 4, "", "machine number");
               if (i == 1) mach = Dec21064;
               else if (i == 2) mach = Dec21164;
               else if (i == 3) mach = Dec21264;
               else mach = MACHOther;
            }
            ALPHA = 1;
         }
         else if (strstr(ln, "i686") || strstr(ln, "i586") || 
                  strstr(ln, "i486") || strstr(ln, "i386"))
         {
            INTEL = 1;
            mach = MACHOther;
            if (!ierr)
            {
               if (strstr(ln2, "Pentium II")) mach = IntPII;
               else if (strstr(ln2, "Pentium Pro")) mach = IntPPRO;
               else if (strstr(ln2, "AMD-K7")) mach = AmdAthlon;
               else if (strstr(ln2, "Athlon")) mach = AmdAthlon;
               else if (strstr(ln2, "Pentium MMX")) mach = IntP5MMX;
            }
            if (mach == MACHOther)
            {
               if (strstr(ln, "i686"))
               {
                  fprintf(stdout, "Enter your machine type:\n");
                  fprintf(stdout, "   1. Other/UNKNOWN\n");
                  fprintf(stdout, "   2. Pentium PRO\n");
                  fprintf(stdout, "   3. Pentium II\n");
                  fprintf(stdout, "   4. Xeon Pentium II\n");
                  fprintf(stdout, "   5. AMD Athlon\n");
                  j = 5;
               }
               else if (strstr(ln, "i586") || strstr(ln, "i486") || 
                        strstr(ln, "i386"))
               {
                  fprintf(stdout, "Enter your machine type:\n");
                  fprintf(stdout, "   1. Other/UNKNOWN\n\n");
                  fprintf(stdout, "   2. Pentium PRO\n");
                  fprintf(stdout, "   3. Pentium II\n");
                  fprintf(stdout, "   4. Xeon Pentium II\n");
                  fprintf(stdout, "   5. Pentium \n");
                  fprintf(stdout, "   6. Pentium MMX\n");
                  fprintf(stdout, "   7. AMD Athlon\n");
                  j = 7;
               }
               if (j > 0)
               {
                  i = GetIntRange(1, 1, j, "", "machine number");
                  if (i == 1) mach = MACHOther;
                  else if (i == 2) mach = IntPPRO;
                  else if (i == 3) mach = IntPII;
                  else if (i == 4) mach = IntXeon;
                  else if (i == 5) mach = IntP5;
                  else if (i == 6) mach = IntP5MMX;
                  else if (i == 7) mach = AmdAthlon;
                  else mach = MACHOther;
               }
            }
         }
      }
      break;
   case OSHPUX:
      if(CmndOneLine(targ, ln2, ln)) mach = MACHOther;
      else if (strstr(ln, "9000/735")) mach=HP9K735;
/*      else if (strstr(ln, "9000/715")) mach=HP9K715; */
      else mach = MACHOther;
      break;
   case OSSunOS4:
      mach = MACHOther;
      fprintf(stdout, "Running trusty old SunOS 4 (hey, is that a Nehru jacket you're wearing), I see.\n");
      if(CmndOneLine(targ, "sysinfo | fgrep CPU | fgrep -i MHz", ln))
         mach = MACHOther;
      else
      {
         if (strstr(ln, "microSPARC")) mach = SunMS;
         else if (strstr(ln, "UltraSPARC"))
         {
            if(CmndOneLine(targ, "sysinfo | fgrep Ultra", ln)) mach = MACHOther;
            else if (strstr(ln, "Ultra-2")) mach = SunUS2;
            else if (strstr(ln, "Ultra-1")) mach = SunUS1;
            else if (strstr(ln, "Ultra")) mach = SunUSX;
         }
         else if (strstr(ln, "Ultra-2")) mach = SunUS2;
         else if (strstr(ln, "Ultra-1")) mach = SunUS1;
         else if (strstr(ln, "Ultra")) mach = SunUSX;
         else mach = MACHOther;
      }
      if (mach == MACHOther)
      {
         fprintf(stdout, 
            "Since SunOS 4 is silent on the subject, what chip do you have:\n");
         fprintf(stdout, 
"   1. UltraSparc\n   2. SuperSparc 1/2\n   3. MicroSparc\n  4. Other/UNKNOWN\n");
         i = GetIntRange(4, 1, 4, "", "chip number");
         if (i == 1) mach = SunUS2;
         else if (i == 2) mach = SunSS;
         else if (i == 3) mach = SunMS;
         else mach = MACHOther;
      }
      break;
   case OSSunOS:
      sprintf(ln2, "%s -i", unam);
      if(CmndOneLine(targ, ln2, ln)) mach = MACHOther;
      else
      {
         if (strstr(ln, "SPARCstation-5")) mach = SunMS;
         else if (strstr(ln, "Ultra-1")) mach = SunUS1;
         else if (strstr(ln, "Ultra-2")) mach = SunUS2;
         else if (strstr(ln, "Ultra")) mach = SunUSX;
         else mach = MACHOther;
      }
      break;
   case OSOSF1:
      mach = MACHOther;
      if(CmndOneLine(targ, ln2, ln)) mach = MACHOther;
      else if (strstr(ln, "alpha"));
      {
         ALPHA = 1;
         ierr = CmndOneLine(targ, "/usr/sbin/pinfo -v | fgrep EV", ln);
         if (ierr)
         {
            ierr = CmndOneLine(targ, "pinfo -v | fgrep EV", ln);
            if (ierr)
            {
               ierr = CmndOneLine(targ, "/usr/sbin/psrinfo -v | fgrep EV", ln);
               if (ierr) ierr = CmndOneLine(targ, "psrinfo -v | fgrep EV", ln);
            }
         }
         if (!ierr)
         {
            if (strstr(ln, "EV6")) mach = Dec21264;
            else if (strstr(ln, "EV5")) mach = Dec21164;
            else if (strstr(ln, "EV4")) mach = Dec21064;
         }
         if (mach == MACHOther)
         {
            fprintf(stdout, "Looks like you are compiling for a DEC ALPHA.\n");
            fprintf(stdout, "Choose DEC alpha model:\n");
            fprintf(stdout, "   1. Alpha 21064\n");
            fprintf(stdout, "   2. Alpha 21164 (ev5/ev56)\n");
            fprintf(stdout, "   3. Alpha 21264 (ev6)\n");
            fprintf(stdout, "   4. Other/UNKNOWN\n\n");
            i = GetIntRange(4, 1, 4, "", "machine number");
            if (i == 1) mach = Dec21064;
            else if (i == 2) mach = Dec21164;
            else if (i == 3) mach = Dec21264;
            else mach = MACHOther;
         }
      }
      break;
   case OSIRIX:
      if(CmndOneLine(targ, ln2, ln)) mach = MACHOther;
      else
      {
         if (strstr(ln, "IP28")) mach = SgiIP28;
         else if (strstr(ln, "IP27")) mach = SgiIP27;
         else if (strstr(ln, "IP22")) mach = SgiIP22;
         else if (strstr(ln, "IP32")) mach = SgiIP32;
         else if (strstr(ln, "IP30")) mach = SgiIP30;
         else mach = MACHOther;
      }
      break;
   case OSAIX:  /* don't even think about translating IBM's wild-ass uname */
      mach = MACHOther;
      fprintf(stdout, "Here are the preconfigured ATLAS choices for AIX:\n");
      fprintf(stdout, "   1. IBM Power\n");
      fprintf(stdout, "   2. IBM Thin-node, Power2\n");
      fprintf(stdout, "   3. IBM PowerPC 604\n");
      fprintf(stdout, "   4. IBM PowerPC 604e (silver node)\n");
      fprintf(stdout, "   5. IBM Power3\n");
      fprintf(stdout, "   6. Other/UNKNOWN\n\n");
      i = GetIntRange(6, 1, 6, "", "machine number");
      if (i == 1) mach = IbmPwr;
      else if (i == 2) mach = IbmTPwr2;
      else if (i == 3) mach = PPC604;
      else if (i == 4) mach = PPC604e;
      else if (i == 5) mach = IbmPwr3;
      else mach = MACHOther;
   case OSWin9x:
   case OSWinNT:
      if (CmndOneLine(targ, ln2, ln) == 0)
      {
         j = 0;
         if (strstr(ln, "i686"))
         {
            INTEL = 1;
            fprintf(stdout, "Enter your machine type:\n");
            fprintf(stdout, "   1. Other/UNKNOWN\n");
            fprintf(stdout, "   2. Pentium PRO\n");
            fprintf(stdout, "   3. Pentium II\n");
            fprintf(stdout, "   4. Xeon Pentium II\n");
            fprintf(stdout, "   5. AMD Athlon\n");
            j = 5;
         }
         else if (strstr(ln, "i586") || strstr(ln, "i486") || 
                  strstr(ln, "i386"))
         {
            INTEL = 1;
            fprintf(stdout, "Enter your machine type:\n");
            fprintf(stdout, "   1. Other/UNKNOWN\n\n");
            fprintf(stdout, "   2. Pentium PRO\n");
            fprintf(stdout, "   3. Pentium II\n");
            fprintf(stdout, "   4. Xeon Pentium II\n");
            fprintf(stdout, "   5. Pentium \n");
            fprintf(stdout, "   6. Pentium MMX\n");
            fprintf(stdout, "   7. AMD Athlon\n");
            j = 7;
         }
         if (j > 0)
         {
            i = GetIntRange(1, 1, j, "", "machine number");
            if (i == 1) mach = MACHOther;
            else if (i == 2) mach = IntPPRO;
            else if (i == 3) mach = IntPII;
            else if (i == 4) mach = IntXeon;
            else if (i == 5) mach = IntP5;
            else if (i == 6) mach = IntP5MMX;
            else if (i == 7) mach = AmdAthlon;
            else mach = MACHOther;
         }
      }
      break;
   case OSOther:
      break;
   }
   fprintf(fpout, "Architecture is set to %s\n\n", machnam[mach]);
   fprintf(fplog , "Architecture is set to %s\n\n", machnam[mach]);
   return(mach);
}

char *GetXCompTarg(FILE *fpout0, FILE *fplog, char *targnam)
{
   char *targ=NULL;
   char ln[512];
   int ierr;
   FILE *fpout;

   fprintf(stdout, "\n\nI need to know if you are using a cross-compiler (i.e., you are compiling on\n");
   fprintf(stdout, 
      "a different architecture than you want the library built for).\n\n");
   targnam[0] = '\0';
   if (IsYes('n', "", "Are you using a cross-compiler?"))
   {
      DisplayFile("CONFIG/xcomp.txt", stdout);
      do
      {
         GetStrVer(targnam, "", "name of target machine", 511, targnam);
         fprintf(stdout, "\n\nTRYING TO rsh %s\n", targnam);
         sprintf(ln, "rsh %s \"ls\"", targnam);
         ierr = system(ln);
         if (ierr) DisplayFile("CONFIG/xcomp-err.txt", stdout);
         else fprintf(stdout, "\n\nrsh successful!\n\n");
         fprintf(fplog, "Using cross compiler, spawning runs to %s.\n\n",
                 targnam);
      }
      while (ierr);
      targ = targnam;
      fpout = fopen("CONFIG/Make.inc", "w");
      assert(fpout);
      fprintf(fpout, 
         "atlas_run:\n\trsh %s \"cd $(atldir) ; ./$(exe) $(args) > $(redir)\"\n",
              targnam);
   }
   else
   {
      fpout = fopen("CONFIG/Make.inc", "w");
      assert(fpout);
      fprintf(fpout, "atlas_run:\n\tcd $(atldir) ; ./$(exe) $(args) > $(redir)\n");
   }
   fclose(fpout);
   return(targ);
}

void GetAtlasParams(enum OSTYPE OS, enum MACHTYPE mach, char *INSTflag,
                    char *sATLres, char *dATLres, char *cATLres, char *zATLres)
{
   strcpy(INSTflag, "-d 0");
   switch(mach)
   {
   case Dec21164 :
      strcpy(dATLres, "pre=d muladd=0 ldc=28 nb=28 mu=4 nu=4 lat=4 ku=1");
      strcpy(sATLres, "pre=s muladd=0 ldc=40 nb=40 mu=4 nu=4 lat=4 ku=1");
      strcpy(cATLres, "pre=c muladd=0 ldc=40 nb=40 mu=4 nu=4 lat=4 ku=1");
      strcpy(zATLres, "pre=z muladd=0 ldc=28 nb=28 mu=4 nu=4 lat=4 ku=1");
      break;
   case Dec21064 : 
      strcpy(dATLres, "pre=d muladd=1 ldc=44 nb=44 mu=3 nu=2 lat=1 ku=44");
      strcpy(sATLres, "pre=s muladd=1 ldc=56 nb=56 mu=7 nu=1 lat=1 ku=56");
      strcpy(cATLres, "pre=c muladd=1 ldc=56 nb=56 mu=7 nu=1 lat=1 ku=56");
      strcpy(zATLres, "pre=z muladd=1 ldc=44 nb=44 mu=3 nu=2 lat=1 ku=44");
      break;
   case HPPA8K :
      strcpy(dATLres, "pre=d muladd=1 ldc=60 nb=60 mu=2 nu=5 lat=5 ku=60");
      strcpy(sATLres, "pre=s muladd=1 ldc=60 nb=60 mu=2 nu=5 lat=5 ku=60");
      strcpy(cATLres, "pre=c muladd=1 ldc=60 nb=60 mu=2 nu=5 lat=5 ku=60");
      strcpy(zATLres, "pre=z muladd=1 ldc=60 nb=60 mu=2 nu=5 lat=5 ku=60");
      break;
   HP9K735:
      strcpy(dATLres, "pre=d muladd=1 ldc=60 nb=60 mu=3 nu=3 lat=5 ku=1");
      strcpy(sATLres, "pre=s muladd=1 ldc=56 nb=56 mu=4 nu=4 lat=5 ku=1");
      strcpy(cATLres, "pre=c muladd=1 ldc=56 nb=56 mu=4 nu=4 lat=5 ku=1");
      strcpy(zATLres, "pre=z muladd=1 ldc=60 nb=60 mu=3 nu=3 lat=5 ku=1");
      break;
   case IbmTPwr2 :
      strcpy(dATLres, "pre=d muladd=1 ldc=60 nb=60 mu=6 nu=2 lat=6 ku=1 ");
      strcpy(sATLres, "pre=s muladd=1 ldc=64 nb=64 mu=10 nu=2 lat=0 ku=1");
      strcpy(cATLres, "pre=c muladd=1 ldc=64 nb=64 mu=10 nu=2 lat=0 ku=1");
      strcpy(zATLres, "pre=z muladd=1 ldc=60 nb=60 mu=6 nu=2 lat=6 ku=1 ");
      break;
   case IbmPwr :
      strcpy(INSTflag, "-d 1");
      break;
   case PPC604 :
      strcpy(dATLres, "pre=d muladd=1 ldc=36 nb=36 mu=6 nu=3 lat=5 ku=1 ");
      strcpy(sATLres, "pre=s muladd=1 ldc=56 nb=56 mu=8 nu=2 lat=2 ku=1 ");
      strcpy(cATLres, "pre=c muladd=1 ldc=56 nb=56 mu=8 nu=2 lat=2 ku=1 ");
      strcpy(zATLres, "pre=z muladd=1 ldc=36 nb=36 mu=6 nu=3 lat=2 ku=1 ");
      break;
   case IntP5 :
      strcpy(INSTflag, "-d 1");
      break;
   case IntP5MMX :
      strcpy(dATLres, "pre=d muladd=0 ldc=36 nb=36 mu=1 nu=2 lat=3 ku=36");
      strcpy(sATLres, "pre=s muladd=0 ldc=56 nb=56 mu=1 nu=2 lat=3 ku=56");
      strcpy(cATLres, "pre=c muladd=0 ldc=56 nb=56 mu=1 nu=2 lat=3 ku=56");
      strcpy(zATLres, "pre=z muladd=0 ldc=36 nb=36 mu=1 nu=2 lat=3 ku=36");
      break;
   case IntPPRO :
      strcpy(dATLres, "pre=d muladd=0 ldc=32 nb=32 mu=2 nu=1 lat=5 ku=32");
      strcpy(sATLres, "pre=s muladd=0 ldc=44 nb=44 mu=2 nu=1 lat=5 ku=44");
      strcpy(cATLres, "pre=c muladd=0 ldc=40 nb=40 mu=3 nu=1 lat=4 ku=40");
      strcpy(zATLres, "pre=z muladd=0 ldc=28 nb=28 mu=2 nu=1 lat=5 ku=28");
      break;
   case IntPII :
      strcpy(dATLres, "pre=d muladd=0 ldc=40 nb=40 mu=2 nu=1 lat=5 ku=40");
      strcpy(sATLres, "pre=s muladd=0 ldc=56 nb=56 mu=4 nu=1 lat=2 ku=56");
      strcpy(cATLres, "pre=c muladd=0 ldc=56 nb=56 mu=4 nu=1 lat=2 ku=56");
      strcpy(zATLres, "pre=z muladd=0 ldc=40 nb=40 mu=2 nu=1 lat=5 ku=40");
      break;
   case IntXeon :
      strcpy(INSTflag, "-d 1");
      break;
   case SgiIP28 :
   case SgiIP27 :
      strcpy(dATLres, "pre=d muladd=1 ldc=56 nb=56 mu=5 nu=4 lat=1 ku=1 ");
      strcpy(sATLres, "pre=s muladd=1 ldc=56 nb=56 mu=4 nu=4 lat=1 ku=1 ");
      strcpy(cATLres, "pre=c muladd=1 ldc=56 nb=56 mu=4 nu=4 lat=1 ku=1 ");
      strcpy(zATLres, "pre=z muladd=1 ldc=56 nb=56 mu=5 nu=4 lat=1 ku=1 ");
      break;
   case SgiIP32 :
      strcpy(dATLres, "pre=d muladd=1 ldc=56 nb=56 mu=1 nu=3 lat=3 ku=56 ");
      strcpy(sATLres, "pre=s muladd=1 ldc=64 nb=64 mu=2 nu=12 lat=3 ku=16");
      strcpy(cATLres, "pre=c muladd=1 ldc=64 nb=64 mu=2 nu=12 lat=3 ku=16");
      strcpy(zATLres, "pre=z muladd=1 ldc=56 nb=56 mu=1 nu=3 lat=3 ku=56 ");
      break;
   case SgiIP22 :
      strcpy(dATLres, "pre=d muladd=1 ldc=28 nb=28 mu=2 nu=2 lat=3 ku=28");
      strcpy(sATLres, "pre=s muladd=1 ldc=40 nb=40 mu=2 nu=2 lat=3 ku=40");
      strcpy(cATLres, "pre=c muladd=1 ldc=40 nb=40 mu=2 nu=2 lat=3 ku=40");
      strcpy(zATLres, "pre=z muladd=1 ldc=28 nb=28 mu=2 nu=2 lat=3 ku=28");
      break;
   case SunSS :
      strcpy(dATLres, "pre=d muladd=0 ldc=40 nb=40 mu=5 nu=1 lat=3 ku=40");
      strcpy(sATLres, "pre=s muladd=0 ldc=56 nb=56 mu=7 nu=2 lat=4 ku=56");
      strcpy(cATLres, "pre=c muladd=0 ldc=24 nb=24 mu=7 nu=2 lat=4 ku=24");
      strcpy(zATLres, "pre=z muladd=0 ldc=20 nb=20 mu=5 nu=1 lat=3 ku=20");
      break;
   case SunMS :
      strcpy(dATLres, "pre=d muladd=0 ldc=28 nb=28 mu=2 nu=2 lat=1 ku=28");
      strcpy(sATLres, "pre=s muladd=0 ldc=40 nb=40 mu=2 nu=2 lat=1 ku=40");
      strcpy(cATLres, "pre=c muladd=0 ldc=40 nb=40 mu=2 nu=2 lat=1 ku=40");
      strcpy(zATLres, "pre=z muladd=0 ldc=28 nb=28 mu=2 nu=2 lat=1 ku=28");
      break;
   case SunUS1 :
   case SunUS2 :
      strcpy(dATLres, "pre=d muladd=0 nb=44 ldc=44 mu=4 nu=4 lat=4 ku=1 ");
      strcpy(sATLres, "pre=s muladd=0 nb=56 ldc=56 mu=4 nu=4 lat=4 ku=1 ");
      strcpy(cATLres, "pre=c muladd=0 nb=56 ldc=56 mu=4 nu=4 lat=4 ku=1 ");
      strcpy(zATLres, "pre=z muladd=0 nb=44 ldc=44 mu=4 nu=4 lat=4 ku=1 ");
      break;
   case PPC604e :
      strcpy(INSTflag, "-d 1");
      break;
   case MACHOther :
   default:
      strcpy(INSTflag, "-d 1");
   }
   #ifdef ForceSearch
      strcpy(INSTflag, "-d 1");
   #endif
}

void GoToTown()
{
   char comp[64], cflg[512], ln[512], tnam[256], archdef[128];
   char BC[256], BCFLAGS[256];
   int GOGO=0, L2IsKnown=0;
   int ierr, i, j, SOLERR=0, np=2;
   char *paths[2] = {"/usr/bin/", "/usr/local/"};
   char *sp, *targ;
   enum OSTYPE OS=OSOther, XOS=OSOther;
   enum MACHTYPE Xmach=MACHOther;
   enum F2CNAME f2cnam;
   enum F2CINT f2cint;
   enum F2CSTRING f2cstr;
   FILE *fpout, *fplog, *fps[2];
   const char *lognam="ConfSummary.log", *dmpnam="ConfDump.log";
   char redir[128];
   sprintf(redir, ">> %s 2>&1", dmpnam);

   remove(dmpnam);
   fplog = fopen(lognam, "w");
   fps[0] = stdout;
   fps[1] = fplog;
   assert(fplog);
   assert(tmpnam(tnam));
   fprintf(stdout, "ATLAS configure started.\n");
   fprintf(fplog , "ATLAS configure started.\n\n");
   DisplayFile("CONFIG/init.txt", stdout);
   fprintf(stdout, "\n");
   if (!IsYes('y', "", "Are you ready to continue?")) exit(-1);
   targ = GetXCompTarg(stdout, fplog, TARGNAM);
   if (targ) XCOMP = 1;
/* 
 * Set up some good defaults
 */
   sp = getenv("PWD");
   if (sp) strcpy(TOPdir, sp);
   else 
   {
      i = CmndOneLine(targ, "pwd", ln);
      if (i || ln[0] == '\0') strcpy(TOPdir, "$(HOME)/ATLAS");
      else
      {
         for (i=0; TOPdir[i] = ln[i]; i++);
         i--;
         while(Mciswspace(TOPdir[i])) i--;
         TOPdir[i+1] = '\0';
      }
   }
   strcpy(LIBdir, "$(TOPdir)/lib/$(ARCH)");
   strcpy(ATLASlib, "libatlas.a");
   strcpy(RBLASlib, ATLASlib);
   strcpy(CBLASlib, "libcblas.a");
   strcpy(TBLASlib, "libatlas_pt.a");
   strcpy(ARCHIVER, "ar");
   strcpy(ARFLAGS, "r");
   strcpy(RANLIB, "echo");
   BLASlib[0] = '\0';
   strcpy(LIBS, "-lm");
   L2SIZE = 4194304;
   strcpy(dATLres, "None, figure them yourself");
   strcpy(sATLres, "None, figure them yourself");
   strcpy(cATLres, "None, figure them yourself");
   strcpy(zATLres, "None, figure them yourself");
   strcpy(INSTflag, "-d 0");

   OS = GetOS(stdout, fplog, targ);
   mach = GetArch(stdout, fplog, OS, targ);
   L1SIZE = GetCacheSize(OS,  mach, targ, 1, &i);
   if (!i) L1SIZE = -1;
/*
 * Linux paging may screw up L1 cache on 604e, so force detection
 */
   if (OS == OSLinux && mach == PPC604e) L1SIZE = -1;
   L2SIZE = GetCacheSize(OS,  mach, targ, -2, &L2IsKnown) * 1024;
   GetAtlasParams(OS, mach, INSTflag, sATLres, dATLres, cATLres, zATLres);
   GetCompInfo(stdout, fplog, OS, mach, targ, redir, F77, F77FLAGS, 
               FLINKER, FLINKFLAGS, CC, CCFLAGS, CLINKER, CLINKFLAGS, 
               MCC, MMFLAGS, BLASlib);
   if (XCOMP)
   {
      fprintf(stdout, "\n\n*******************************************************************************\n");
      fprintf(stdout, "****************  FINDING INFO ABOUT CROSS-COMPILING MACHINE  *****************\n");
      fprintf(stdout, "*******************************************************************************\n");
      fprintf(fplog,  "Finding info about cross-compiling machine:\n");
      XOS = GetOS(stdout, fplog, NULL);
      Xmach = GetArch(stdout, fplog, XOS, NULL);
      GetCompInfo(stdout, fplog, XOS, Xmach, NULL, redir, NULL, NULL, 
                  NULL, NULL, XCC, XCCFLAGS, NULL, NULL,
                  NULL, NULL, NULL);
      fprintf(stdout, "\n\n*******************************************************************************\n");
      fprintf(stdout, "****************  DONE FINDING INFO ABOUT CROSS-COMPILING MACHINE  ************\n");
      fprintf(stdout, "*******************************************************************************\n");
   }
   else
   {
      strcpy(XCC, CC);
      strcpy(XCCFLAGS, CCFLAGS);
   }
   if (OS == OSSunOS4) strcpy(RANLIB, "ranlib");


   if (OS != OSOther)
   {
      if (mach != MACHOther)
         fprintf(stdout, 
         "\n\nATLAS has default parameters for OS=\'%s\' and system=\'%s\'.\n",
                 osnam[OS], machnam[mach]);
      else
         fprintf(stdout, 
"\n\nAlthough your machine is not known, ATLAS has default parameters for OS=\'%s\'.\n",
                 osnam[OS]);
      fprintf(stdout,
"If you want to just trust these default values, you can use express setup,\n");
      fprintf(stdout,
"drastically reducing the amount of questions you are required to answer\n\n");
      GOGO = IsYes('y', "   ", "use express setup?");
      fprintf(stdout, "\n\n");
   }

   DisplayFile("CONFIG/arch.txt", stdout);
   if (OS == OSOther) strcpy(ARCH, "UNKNOWN_UNKNOWN");
   else if (mach == MACHOther) sprintf(ARCH, "%s_UNKNOWN", osnam[OS]);
   else sprintf(ARCH, "%s_%s", osnam[OS], machnam[mach]);
   do
   {
      GetString(stdin, ARCH, "   ", "Architecture name (ARCH)", 127, ARCH);
      sprintf(ln, "Make.%s", ARCH);
      ierr = FileIsThere(ln);
      if (ierr)
      {
         fprintf(stderr, "File Make.%s already exists!\n", ARCH);
         ierr = !IsYes('n', "", "overwrite it?");
      }
   }
   while (ierr);
   fprintf(fplog, "<arch> set to \'%s\'\n\n", ARCH);

   if (!L2IsKnown)
   {
      DisplayFile("CONFIG/l2size.txt", stdout);
      L2SIZE = GetIntRange(L2SIZE/1024, 64, 16*1024, "   ", 
                           "Maximum L2 cache size (KB)");
      L2SIZE *= 1024;
   }
   if (!GOGO)
   {
      GetString(stdin, TOPdir, "   ", "Top level ATLAS directory", 255, TOPdir);
      GetString(stdin, LIBdir, "   ", 
                "Directory to build libraries in", 255, LIBdir);

      GetString(stdin, ATLASlib, "   ", 
                "Library name for ATLAS matrix multiply", 255, ATLASlib);
      GetString(stdin, RBLASlib, "   ", "Library name for ATLAS Level 3 BLAS", 
                255, RBLASlib);
      GetString(stdin, CBLASlib, "   ", 
                "Library name for the C interface to BLAS", 255, CBLASlib);
      GetString(stdin, TBLASlib, "   ", 
                "Library name for the Posix threaded BLAS", 255, TBLASlib);
   }

   ierr = !GOGO;
   if (!GOGO) DisplayFile("CONFIG/f77exp.txt", stdout);
   do
   {
      if (ierr != 0)
      {
         if (GOGO) DisplayFile("CONFIG/f77exp.txt", stdout);
         GetString(stdin, F77, "   ", "f77 compiler", 127, F77);
         GetString(stdin, F77FLAGS, "   ", "F77 Flags", 511, F77FLAGS);
      }
      sprintf(ln, "%s -c CONFIG/tst.f %s\n", F77, redir);
      ierr = system(ln);
      if (ierr)
      {
         if (GOGO) DisplayFile("CONFIG/f77exp.txt", stdout);
         fprintf(stderr, "F77 = \'%s %s\' doesn't seem to work for me.\n", 
                 F77, F77FLAGS);
         ierr = GetIntRange(1, 0, 1, "   ", 
                         "1 to enter a different F77, 0 to continue with none");
      }
   }
   while (ierr);
   if (!GOGO)
   {
      GetString(stdin, FLINKER, "   ", "F77 Linker ", 127, FLINKER);
      GetString(stdin, FLINKFLAGS, "   ", "F77 Link Flags ", 511, FLINKFLAGS);
      fprintf(stdout, "\n\n");
   }
   fprintf(stdout, "F77 & FLAGS: %s %s\n", F77, F77FLAGS);
   fprintf(stdout, "FLINKER & FLAGS: %s %s\n\n", FLINKER, FLINKFLAGS);
   fprintf(fplog , "F77 & FLAGS: %s %s\n", F77, F77FLAGS);
   fprintf(fplog , "FLINKER & FLAGS: %s %s\n\n", FLINKER, FLINKFLAGS);

   ierr = !GOGO;
   if (!GOGO) DisplayFile("CONFIG/ccomp.txt", stdout);
   do
   {
      if (ierr != 0)
      {
         if (GOGO) DisplayFile("CONFIG/ccomp.txt", stdout);
         GetString(stdin, CC, "   ", "ANSI C compiler(CC)", 127, CC);
         GetString(stdin, CCFLAGS, "   ", "C Flags (CCFLAGS)", 511, CCFLAGS);
      }
      sprintf(ln, "%s %s -c CONFIG/tst.c %s\n", CC, CCFLAGS, redir);
      ierr = system(ln);
      if (ierr)
      {
         fprintf(stderr, 
                 "Compiling with = \'%s %s\' doesn't seem to work for me.\n",
                 CC, CCFLAGS);
         ierr = GetIntRange(1, 0, 1, "   ", 
                           "1 to try different CC/flags, 0 to continue anyway");
      }
   }
   while(ierr);
   fprintf(stdout, "CC & FLAGS: %s %s\n", CC, CCFLAGS);
   fprintf(fplog , "CC & FLAGS: %s %s\n", CC, CCFLAGS);

   ierr = !GOGO;
   if (ALPHA && !GOGO) DisplayFile("CONFIG/ccomp-alpha.txt", stdout);
   do
   {
      if (ierr != 0)
      {
         if (GOGO) DisplayFile("CONFIG/ccomp-alpha.txt", stdout);
         GetString(stdin, MCC, "   ", "C compiler for generated code (MCC)", 
                   127, MCC);
         GetString(stdin, MMFLAGS, "   ", "C FLAGS (MMFLAGS)", 511, MMFLAGS);
      }
      Wstrfndsub("$(CC)", CC, MCC, comp);
      Wstrfndsub("$(CCFLAGS)", CCFLAGS, MMFLAGS, ln);
      Wstrfndsub("$(CDEFS)", "", ln, cflg);
      sprintf(ln, "%s %s -c CONFIG/tst.c %s\n", comp, cflg, redir);
      ierr = system(ln);
      if (ierr)
      {
         fprintf(stderr, 
                 "Compiling with \'%s %s\' doesn't seem to work for me.\n", 
                 MCC, MMFLAGS);
         ierr = GetIntRange(1, 0, 1, "   ", 
                            "1 to re-enter, 0 to continue anyway");
      }
   }
   while(ierr);
   if (!GOGO)
   {
      GetString(stdin, CLINKER, "   ", "C Linker ", 127, CLINKER);
      GetString(stdin, CLINKFLAGS, "   ", "C Link Flags ", 511, CLINKFLAGS);
   }
   fprintf(stdout, "MCC & FLAGS: %s %s\n", MCC, MMFLAGS);
   fprintf(stdout, "CLINKER & FLAGS: %s %s\n\n", CLINKER, CLINKFLAGS);
   fprintf(fplog , "MCC & FLAGS: %s %s\n", MCC, MMFLAGS);
   fprintf(fplog , "CLINKER & FLAGS: %s %s\n\n", CLINKER, CLINKFLAGS);
   if (XCOMP)
   {
      ierr = !GOGO;
      do
      {
         if (ierr != 0)
         {
            GetString(stdin, XCC, "   ", 
                      "ANSI C compiler for cross-compiling machine (XCC)", 
                      127, XCC);
            GetString(stdin, XCCFLAGS, "   ", 
                      "C Flags for cross-compiling machine (XCCFLAGS)", 
                      511, XCCFLAGS);
         }
         Wstrfndsub("$(CC)", CC, XCC, comp);
         Wstrfndsub("$(CCFLAGS)", CCFLAGS, XCCFLAGS, ln);
         Wstrfndsub("$(CDEFS)", "", ln, cflg);
         sprintf(ln, "%s %s -c CONFIG/tst.c %s\n", comp, cflg, redir);
         ierr = system(ln);
         if (ierr)
         {
            fprintf(stderr, 
                    "Compiling with \'%s %s\' doesn't seem to work for me.\n", 
                    XCC, XCCFLAGS);
            ierr = GetIntRange(1, 0, 1, "   ", 
                               "1 to re-enter, 0 to continue anyway");
         }
      }
      while(ierr);
      fprintf(stdout, "XCC & FLAGS: %s %s\n\n", XCC, XCCFLAGS);
      fprintf(fplog , "XCC & FLAGS: %s %s\n\n", XCC, XCCFLAGS);
   }

   if (!GOGO)
   {
      GetString(stdin, ARCHIVER, "   ", "Archiver ", 127, ARCHIVER);
      GetString(stdin, ARFLAGS, "   ", "Archiver flags ", 255, ARFLAGS);
      GetString(stdin, RANLIB, "   ", "Ranlib ", 127, RANLIB);
      GetString(stdin, BLASlib, "   ", "BLAS library", 255, BLASlib);
      GetString(stdin, LIBS, "   ", "General and system libs", 255, LIBS);
   }

   fprintf(stdout, 
           "\nFinding F77 to C calling conventions (this may take a while):\n");
   findF2C(redir, TOPdir, F77, F77FLAGS, FLINKER, FLINKFLAGS, CC, CCFLAGS,
           &f2cnam, &f2cint, &f2cstr);
   fprintf(stdout, "\nCalculated F77/C interoperation conventions:\n");
   sp = F2CDEFS;
   if (f2cnam == f2c_Add_)
   {
      fprintf(stdout, "   Suffix F77 names with _\n");
      fprintf(fplog , "   Suffix F77 names with _\n");
      strcpy(sp, "-DAdd_ ");
      sp += 7;
   }
   else if (f2cnam == f2c_NoChange) 
   {
      fprintf(stdout, "   C & F77 share the same namespace\n");
      fprintf(fplog , "   C & F77 share the same namespace\n");
      strcpy(sp, "-DNoChange ");
      sp += 11;
   }
   else if (f2cnam == f2c_UpCase)
   {
      strcpy(sp, "-DUpCase ");
      sp += 9;
      fprintf(stdout, "   Make F77 names upper case\n");
      fprintf(fplog , "   Make F77 names upper case\n");
   }
   else if (f2cnam == f2c_Add__)
   {
      strcpy(sp, "-DAdd__ ");
      sp += 8;
      fprintf(stdout, "   Suffix F77 names with underscores with __\n");
      fprintf(fplog , "   Suffix F77 names with underscores with __\n");
   }
   else
   {
      fprintf(stdout, "   Unable to determine naming conventions\n");
      fprintf(fplog , "   Unable to determine naming conventions\n");
   }

   if (f2cint == FintCint)
   {
      fprintf(stdout, "   F77 INTEGER -> C int\n");
      fprintf(fplog , "   F77 INTEGER -> C int\n");
   }
   else if (f2cint == FintClong)
   {
      fprintf(stdout, "   F77 INTEGER -> C long\n");
      fprintf(fplog , "   F77 INTEGER -> C long\n");
      strcpy(sp, "-DF77_INTEGER=long ");
      sp += 19;
   }
   else if (f2cint == FintCshort)
   {
      fprintf(stdout,"   F77 INTEGER -> C short\n");
      fprintf(fplog ,"   F77 INTEGER -> C short\n");
      strcpy(sp, "-DF77_INTEGER=short ");
      sp += 20;
   }
   else
   {
      fprintf(stdout,"   Unable to determine F77/C integer correspondence\n");
      fprintf(fplog ,"   Unable to determine F77/C integer correspondence\n");
   }

   if (f2cstr == fstrSun)
   {
      strcpy(sp, "-DStringSunStyle");  
      fprintf(stdout, "   F77 strings handled via standard sun style\n");
      fprintf(fplog , "   F77 strings handled via standard sun style\n");
   }
   else if (f2cstr == fstrCray)
   {
      strcpy(sp, "-DCRAY");  
      fprintf(stdout, "   F77 strings handled using CRAY fcd\n");
      fprintf(fplog , "   F77 strings handled using CRAY fcd\n");
   }
   else if (f2cstr == fstrStructVal)
   {
      strcpy(sp, "-DStringStructVal");  
      fprintf(stdout, "   F77 strings handled by passing a struct\n");
      fprintf(fplog , "   F77 strings handled by passing a struct\n");
   }
   else if (f2cstr == fstrStructPtr)
   {
      strcpy(sp, "-DStringStructPtr");  
      fprintf(stdout,"   F77 strings handled by passing a pointer to struct\n");
      fprintf(fplog, "   F77 strings handled by passing a pointer to struct\n");
   }
   else
   {
      fprintf(stdout,"   Unable to determine F77/C string interoperation\n");
      fprintf(fplog, "   Unable to determine F77/C string interoperation\n");
   }
   sprintf(ln, "cd CONFIG/ARCHS/%s_%s %s", osnam[OS], machnam[mach], redir);
   if (system(ln) != 0)
   {
      sprintf(ln, "CONFIG/ARCHS/%s_%s.shar", osnam[OS], machnam[mach]);
      if (FileIsThere(ln))
      {
         fprintf(stdout, "\n\nUnpacking Architectural defaults . . . ");
         sprintf(ln, "cd CONFIG/ARCHS ; make ArchUnpack arch=%s_%s %s\n", 
                 osnam[OS], machnam[mach], redir);
         system(ln);  /* ignore return val; will work either way */
         fprintf(stdout, " done.");
      }
   }
   sprintf(ln, "cd %s/CONFIG/ARCHS/%s_%s", TOPdir, osnam[OS], 
           machnam[mach]);
   if (system(ln) == 0)
   {
      DisplayFile("CONFIG/NoSearch.txt", stdout);
      if (IsYes('y', "", "Use supplied default values for install?"))
      {
         sprintf(archdef, "$(TOPdir)/CONFIG/ARCHS/%s_%s", osnam[OS], 
                 machnam[mach]);
      }
      else archdef[0] = '\0';
   }

   fprintf(stdout, "\nCreating make include file Make.%s\n", ARCH);
   fprintf(fplog,  "\nCreating make include file Make.%s\n", ARCH);
   sprintf(ln, "Make.%s", ARCH);
   fpout = fopen(ln, "w");
   assert(fpout);
   fprintf(fpout, "#  ----------------------------------\n");
   fprintf(fpout, "#  Make sure we get the correct shell\n");
   fprintf(fpout, "#  ----------------------------------\n");
   fprintf(fpout, "   SHELL = /bin/sh\n\n");

   fprintf(fpout, "#  -------------------------------------------------\n");
   fprintf(fpout, "#  Name indicating the platform to configure BLAS to\n");
   fprintf(fpout, "#  -------------------------------------------------\n");
   fprintf(fpout, "   ARCH = %s\n\n", ARCH);

   fprintf(fpout, "#  -------------------\n");
   fprintf(fpout, "#  Various directories\n");
   fprintf(fpout, "#  -------------------\n");
   fprintf(fpout, "   TOPdir = %s\n", TOPdir);
   fprintf(fpout, "   INCdir = $(TOPdir)/include/$(ARCH)\n");
   fprintf(fpout, "   SYSdir = $(TOPdir)/tune/sysinfo/$(ARCH)\n");
   fprintf(fpout, "   GMMdir = $(TOPdir)/src/blas/gemm/$(ARCH)\n");
   fprintf(fpout, "   GMVdir = $(TOPdir)/src/blas/gemv/$(ARCH)\n");
   fprintf(fpout, "   GR1dir = $(TOPdir)/src/blas/ger/$(ARCH)\n");
   fprintf(fpout, "   L1Bdir = $(TOPdir)/src/blas/level1/$(ARCH)\n");
   fprintf(fpout, "   L2Bdir = $(TOPdir)/src/blas/level2/$(ARCH)\n");
   fprintf(fpout, "   L3Bdir = $(TOPdir)/src/blas/level3/$(ARCH)\n");
   fprintf(fpout, "   TSTdir = $(TOPdir)/src/testing/$(ARCH)\n");
   fprintf(fpout, "   AUXdir = $(TOPdir)/src/auxil/$(ARCH)\n");
   fprintf(fpout, "   CBLdir = $(TOPdir)/interfaces/blas/C/src/$(ARCH)\n");
   fprintf(fpout, "   FBLdir = $(TOPdir)/interfaces/blas/F77/src/$(ARCH)\n");
   fprintf(fpout, "   BINdir = $(TOPdir)/bin/$(ARCH)\n");
   fprintf(fpout, "   LIBdir = %s\n", LIBdir);
   fprintf(fpout, "   PTBdir = $(TOPdir)/tune/blas/ptblas/$(ARCH)\n");
   fprintf(fpout, "   MMTdir = $(TOPdir)/tune/blas/gemm/$(ARCH)\n");
   fprintf(fpout, "   MVTdir = $(TOPdir)/tune/blas/gemv/$(ARCH)\n");
   fprintf(fpout, "   R1Tdir = $(TOPdir)/tune/blas/ger/$(ARCH)\n");
   fprintf(fpout, "   L1Tdir = $(TOPdir)/tune/blas/level1/$(ARCH)\n");
   fprintf(fpout, "   L3Tdir = $(TOPdir)/tune/blas/level3/$(ARCH)\n\n");

   fprintf(fpout,
"#  ---------------------------------------------------------------------\n");
   fprintf(fpout,
"#  Name and location of shell script that runs executables during tuning\n");
   fprintf(fpout,
"#  ---------------------------------------------------------------------\n");
   fprintf(fpout, "   ATLRUN = $(BINdir)/ATLrun.sh\n\n");

   fprintf(fpout, "#  ---------------------\n");
   fprintf(fpout, "#  Libraries to be built\n");
   fprintf(fpout, "#  ---------------------\n");
   fprintf(fpout, "   ATLASlib = $(LIBdir)/%s\n", ATLASlib);
   fprintf(fpout, "   RBLASlib = $(LIBdir)/%s\n", RBLASlib);
   fprintf(fpout, "   CBLASlib = $(LIBdir)/%s\n", CBLASlib);
   fprintf(fpout, "   F77BLASlib = $(LIBdir)/libf77blas.a\n");
   fprintf(fpout, "   TESTlib = $(LIBdir)/libtstatlas.a\n");
   fprintf(fpout, "   TBLASlib = $(LIBdir)/%s\n", TBLASlib);
   fprintf(fpout, "   LAPACKlib = $(LIBdir)/liblapack.a\n\n");

   fprintf(fpout, "#  -------------------------------------------\n");
   fprintf(fpout, "#  Upper bound on largest cache size, in bytes\n");
   fprintf(fpout, "#  -------------------------------------------\n");
   fprintf(fpout, "   L2SIZE = -DL2SIZE=%d\n\n", L2SIZE);

   fprintf(fpout, "#  ---------------------------------------\n");
   fprintf(fpout, "#  Command setting up correct include path\n");
   fprintf(fpout, "#  ---------------------------------------\n");
   fprintf(fpout, 
      "   INCLUDES = -I$(TOPdir)/include -I$(TOPdir)/include/$(ARCH)\n\n");

   fprintf(fpout, "#  -------------------------------------------\n");
   fprintf(fpout, "#  Defines for setting up F77/C interoperation\n");
   fprintf(fpout, "#  -------------------------------------------\n");
   fprintf(fpout, "   F2CDEFS = %s\n\n", F2CDEFS);

   fprintf(fpout, 
   "#  -------------------------------------------------------------------\n");
   fprintf(fpout, 
   "#  NM is the flag required to name a compiled object/executable\n");
   fprintf(fpout, 
   "#  OJ is the flag required to compile to object rather than executable\n");
   fprintf(fpout, "#  These flags are used by all compilers.\n");
   fprintf(fpout, 
   "#  -------------------------------------------------------------------\n");
   fprintf(fpout, "   NM = -o\n");
   fprintf(fpout, "   OJ = -c\n\n");

   DisplayFile("CONFIG/f77make.txt", fpout);
   fprintf(fpout, "   F77 = %s\n", F77);
   fprintf(fpout, "   F77FLAGS = %s\n", F77FLAGS);
   fprintf(fpout, "   FLINKER = %s\n", FLINKER);
   fprintf(fpout, "   FLINKFLAGS = %s\n\n", FLINKFLAGS);

   DisplayFile("CONFIG/CCmake.txt", fpout);
   fprintf(fpout, "   CDEFS = $(L2SIZE) $(INCLUDES) $(F2CDEFS)\n");
   fprintf(fpout, "   CC = %s\n", CC);
   fprintf(fpout, "   CCFLAGS = $(CDEFS) %s\n", CCFLAGS);
   fprintf(fpout, "   MCC = %s\n", MCC);
   if (strstr(MMFLAGS, "CCFLAGS"))
      fprintf(fpout, "   MMFLAGS = %s\n", MMFLAGS);
   else
      fprintf(fpout, "   MMFLAGS = $(CDEFS) %s\n", MMFLAGS);
   fprintf(fpout, "   XCC = %s\n", XCC);
   if (strstr(XCCFLAGS, "CCFLAGS"))
      fprintf(fpout, "   XCCFLAGS = %s\n", XCCFLAGS);
   else fprintf(fpout, "   XCCFLAGS = $(CDEFS) %s\n", XCCFLAGS);
   fprintf(fpout, "   CLINKER = %s\n", CLINKER);
   fprintf(fpout, "   CLINKFLAGS = %s\n", CLINKFLAGS);
   if (SOLERR)
   {
      fprintf(fpout, "   BC = %s\n", BC);
      fprintf(fpout, "   BCFLAGS = $(CDEFS) %s\n", BCFLAGS);
   }
   else
   {
      fprintf(fpout, "   BC = $(CC)\n", CC);
      fprintf(fpout, "   BCFLAGS = $(CCFLAGS)\n");
   }
   fprintf(fpout, "   ARCHIVER = %s\n", ARCHIVER);
   fprintf(fpout, "   ARFLAGS  = %s\n", ARFLAGS);
   fprintf(fpout, "   RANLIB   = %s\n\n", RANLIB);

   fprintf(fpout, "#  ------------------------------------\n");
   fprintf(fpout, "#  Reference and system libraries\n");
   fprintf(fpout, "#  ------------------------------------\n");
   fprintf(fpout, "   BLASlib = %s\n", BLASlib);
   fprintf(fpout, "   FBLASlib = \n");
   fprintf(fpout, "   FLAPACKlib = \n");
   fprintf(fpout, "   LIBS = %s\n\n", LIBS);

   fprintf(fpout, "#  -------------------------------------------\n");
   fprintf(fpout, "#  Architecture default directory, if supplied\n");
   fprintf(fpout, "#  -------------------------------------------\n");
   fprintf(fpout, "   ARCHDEF = %s\n", archdef);
   fclose(fpout);

   fprintf(stdout, "Make.%s successfully created.\n\n", ARCH);
   fprintf(fplog,  "Make.%s successfully created.\n\n", ARCH);

   fprintf(stdout, "Creating subdirectories:\n");
   fprintf(stdout, "   Checking for already existing subdirectories  .....");
   fflush(stdout);
   sprintf(ln, "cd bin/%s %s", ARCH, redir);
   i = 1;
   if (system(ln) == 0)
   {
      fprintf(stdout, "...... found!\n");
      i = IsYes('y', "", "kill old subdirectores?");
   }
   else fprintf(stdout, "... no\n");
   if (i)
   {
      sprintf(ln, "make killall arch=%s %s\n", ARCH, redir);
      system(ln);
      sprintf(ln, "make startup arch=%s %s\n", ARCH, redir);
      if (system(ln))
      {
         fps[0] = stderr;
         fps[1] = fplog;
         for (i=0; i < 2; i++)
         {
            fprintf(fps[i], "UNABLE TO CREATE NECESSARY SUBDIRECTORIES:\n");
            fprintf(fps[i], 
       "  review Make.%s for accurracy, paying special attention to TOPdir.\n",
                    ARCH);
            fprintf(fps[i], "Examine %s and %s for specifics.\n", 
                    lognam, dmpnam);
            fprintf(fps[i], 
               "   for further help, see ATLAS/README/TroubleShoot.txt.\n");
            fprintf(fps[i],
                    "   when you have corrected the problem, finish the\n");
            fprintf(fps[i], "   configuration by typing :\n");
            fprintf(fps[i], "      make startup arch=%s\n", ARCH);
         }
         sprintf(ln, "make killall arch=%s %s\n", ARCH, redir);
         system(ln);
         fclose(fplog);
         exit(-1);
      }
   }
   fprintf(stdout, "Subdirectories successfully created.\n\n");
   fprintf(fplog,  "Subdirectories successfully created.\n\n");
   fprintf(stdout, "\n\nCreating ATLrun.sh\n\n", ARCH);
   fprintf(fplog,  "\n\nCreating ATLrun.sh\n\n", ARCH);
   sprintf(ln, "bin/%s/ATLrun.sh", ARCH);
   fpout = fopen(ln, "w");
   assert(fpout);
   fprintf(fpout, "atldir=$1\nshift\n");
   if (XCOMP)
      fprintf(fpout, "exe=$1\nshift\nrsh %s \"cd $atldir ; ./$exe $*\"", 
              TARGNAM);
   else fprintf(fpout, "$atldir/$*\n");
   fclose(fpout);
   if (L1SIZE != -1)
   {
      fprintf(stdout, "\nStoring L1 cache size of %dKB.\n\n", L1SIZE);
      fprintf(fplog , "\nStoring L1 cache size of %dKB.\n\n", L1SIZE);
      sprintf(ln, "make -f Make.top ISetL1 arch=%s L1Size=%d %s\n", 
              ARCH, L1SIZE, redir);
      if (system(ln))
      {
         fprintf(stderr, "Unable to store L1CacheSize, aborting.\n");
         fprintf(fplog,  "Unable to store L1CacheSize, aborting.\n");
         fclose(fplog);
         exit(-1);
      }
   }
   fprintf(fplog, "\n\nConfiguration completed successfully.\n\n");
   fclose(fplog);
   fprintf(stdout, 
           "\nMoving config logfiles %s and %s to bin/%s/INSTALL_LOG/\n",
           lognam, dmpnam, ARCH);
   sprintf(ln, "mv -f %s bin/%s/INSTALL_LOG/.\n", lognam, ARCH);
   system(ln);
   sprintf(ln, "mv -f %s bin/%s/INSTALL_LOG/.\n", dmpnam, ARCH);
   system(ln);

   fprintf(stdout, "\n\nConfiguration completed successfully.  You may want to examine the make include \n");
   fprintf(stdout, 
"file (Make.%s) for accuracy before starting the install with the command:\n",
           ARCH);
   fprintf(stdout, "   make install arch=%s\n\n", ARCH);
}

main(int nargs, char *args[])
{
   GoToTown();
   exit(0);
}
