/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1996, 1997, 1998, 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to perform shell-style file pattern matching
 */

#include <ac/errno.h>
#include <ac/stddef.h>
#include <ac/stdlib.h>
#include <ac/string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <ac/dirent.h>

#include <builtin/glob.h>
#include <error_intl.h>
#include <expr/position.h>
#include <mem.h>
#include <stracc.h>
#include <str_list.h>
#include <trace.h>


static	char		*original;
static	string_list_ty	*where;
static	stracc		temp;


/*
 * NAME
 *	gmatch - match entryname pattern
 *
 * SYNOPSIS
 *	int gmatch(char *formal, char *formal_end, char *actual);
 *
 * DESCRIPTION
 *	The formal strings is used as a template to match the given actual
 *	string against.
 *
 *	The pattern elements understood are
 *	*	match zero or more of any character
 *	?	match any single character
 *	[^xxx]	match any single character not in the set given.
 *	[xxx]	match any single character not in the set given.
 *	        The - character is understood to be a range indicator.
 *	        If the ] character is the first of the set it is considered
 *	        as part of the set, not the terminator.
 *
 * RETURNS
 *	the gmatch function returns zero if they do not match,
 *	and nonzero if they do.  Returns -1 on error.
 *
 * CAVEAT
 *	This is a limited set of the sh(1) patterns.
 *	Assumes that the `original' global variable has been initialized, it is
 *	used for error reporting.
 */

static int gmatch _((char *, char *, char *, const expr_position_ty *));

static int
gmatch(formal, formal_end, actual, pp)
	char		*formal;
	char		*formal_end;
	char		*actual;
	const expr_position_ty *pp;
{
	char		*cp;
	int		 result;
	sub_context_ty	*scp;

	trace(("gmatch(formal = %8.8lX, formal_end = %8.8lX, actual = %8.8lX)\n{\n"/*}*/, formal, formal_end, actual));
	while (formal < formal_end)
	{
		trace(("formal == \"%.*s\";\n", formal_end - formal, formal));
		trace(("actual = \"%s\";\n", actual));
		switch (*formal)
		{
		default:
			if (*actual++ != *formal++)
			{
				result = 0;
				goto ret;
			}
			break;

		case '?':
			if (!*actual++)
			{
				result = 0;
				goto ret;
			}
			++formal;
			break;

		case '*':
			++formal;

			/*
			 * Look for additional wild-cards or single
			 * character wilds, and accumulate them.  No
			 * sense recursing twice.  This takes care of
			 * weird patterns like "**" or "*?" etc, and
			 * allows us to use the trailing wild-card
			 * optimization.
			 */
			while (formal < formal_end)
			{
				if (*formal == '*')
					++formal;
				else if (*formal == '?')
				{
					++formal;
					if (!*actual++)
					{
						result = 0;
						goto ret;
					}
				}
				else
					break;
			}

			/*
			 * If we are at the end of the formal pattern,
			 * then we have a trailing wild-card.  This
			 * matches anything, so return success no matter
			 * how much actual is left.
			 */
			if (formal >= formal_end)
			{
				result = 1;
				goto ret;
			}

			/*
			 * Try to find the longest match possible for
			 * the wild-card, so start from the right hand
			 * end of the actual string.
			 */
			cp = actual + strlen(actual);
			for (;;)
			{
				if (gmatch(formal, formal_end, cp, pp))
				{
					result = 1;
					goto ret;
				}
				--cp;
				if (cp < actual)
				{
					result = 0;
					goto ret;
				}
			}

		case '[':
			++formal;
			if (*formal == '^')
			{
				++formal;
				for (;;)
				{
					if (formal >= formal_end)
					{
						no_close:
						scp = sub_context_new();
						sub_var_set
						(
							scp,
							"Name",
							"%s",
							original
						);
						error_with_position
						(
							pp,
							scp,
				   i18n("pattern \"$name\" missing closing ']'")
						);
						sub_context_delete(scp);
						result = -1;
						goto ret;
					}
					/* note: this allows leading ']' elegantly */
					if
					(
						formal_end >= formal + 3
					&&
						formal[1] == '-'
					&&
						formal[2] != ']'
					)
					{
						char	c1;
						char	c2;

						c1 = formal[0];
						c2 = formal[2];
						formal += 3;
						if
						(
							c1 <= c2
						?
							(c1 <= *actual && *actual <= c2)
						:
							(c2 <= *actual && *actual <= c1)
						)
						{
							result = 0;
							goto ret;
						}
					}
					else
					if (*actual == *formal++)
					{
						result = 0;
						goto ret;
					}
					if (*formal == ']')
						break;
				}
				++formal;
			}
			else
			{
				for (;;)
				{
					if (formal >= formal_end)
						goto no_close;
					/* note: this allows leading ']' elegantly */
					trace(("formal == \"%.*s\";\n", formal_end - formal, formal));
					trace(("actual = \"%s\";\n", actual));
					if
					(
						formal_end >= formal + 3
					&&
						formal[1] == '-'
					&&
						formal[2] != ']'
					)
					{
						char	c1;
						char	c2;

						c1 = formal[0];
						c2 = formal[2];
						formal += 3;
						if
						(
							c1 <= c2
						?
							(c1 <= *actual && *actual <= c2)
						:
							(c2 <= *actual && *actual <= c1)
						)
							break;
					}
					else
					if (*actual == *formal++)
						break;
					if (*formal == ']')
					{
						result = 0;
						goto ret;
					}
				}
				for (;;)
				{
					if (formal >= formal_end)
						goto no_close;
					trace(("formal == \"%.*s\";\n", formal_end - formal, formal));
					trace(("actual = \"%s\";\n", actual));
					if (*formal++ == ']')
						break;
				}
			}
			++actual;
			break;
		}
	}
	result = (*actual == 0);
	ret:
	trace(("return %d;\n", result));
	trace((/*{*/"}\n"));
	return result;
}


/*
 * NAME
 *	globber - file name expander
 *
 * SYNOPSIS
 *	int globber(char *formal);
 *
 * DESCRIPTION
 *	The globber function is used to generate a list of file names from the
 *	given `formal' pattern.  Results are appended to the word list pointed
 *	to the global `where' variable.
 *
 * RETURNS
 *	int; 0 on success, -1 on error.
 *
 * CAVEAT
 *	Assumes that the `where' global variable has been initialized.
 */

static int globber _((char *, const expr_position_ty *));

static int
globber(formal, pp)
	char		*formal;
	const expr_position_ty *pp;
{
	char		*formal_end;
	char		*cp;
	int		 retval;
	sub_context_ty	*scp;

	trace(("globber(formal = %8.8lX)\n{\n"/*}*/, formal));
	trace_string(formal);
	retval = 0;
	for (;;)
	{
		while (*formal == '/')
			sa_char(&temp, *formal++);
		formal_end = strchr(formal, '/');
		if (!formal_end)
			formal_end = formal + strlen(formal);
		for (cp = formal; cp < formal_end; ++cp)
			if (strchr("[?*", *cp))
				break;
		if (cp >= formal_end)
		{
			struct stat	st;
			size_t		n;

			/* nothing special */
			trace(("ordinary = \"%.*s\";\n", formal_end - formal, formal));
			for (cp = formal; cp < formal_end; ++cp)
				sa_char(&temp, *cp);
			n = sa_mark(&temp);
			sa_char(&temp, 0);
			trace(("generated = \"%s\"\n", temp.sa_buf));

			/*
			 * Need to confirm that it exists.  We are a
			 * file name matcher, not a file name generator.
			 */
			if (stat(temp.sa_buf, &st) < 0)
			{
				if (errno != ENOENT && errno != ENOTDIR)
				{
					scp = sub_context_new();
					sub_errno_set(scp);
					sub_var_set(scp, "File_Name", "%s", temp.sa_buf);
					error_with_position
					(
						pp,
						scp,
						i18n("stat $filename: $errno")
					);
					sub_context_delete(scp);
					retval = -1;
					goto ret;
				}

				/* fail the match */
				break;
			}
			sa_goto(&temp, n);

			if (!*cp)
			{
				string_ty *s;

				s = sa_close(&temp);
				temp.sa_inuse = 1; /* defeat assert protection */
				string_list_append(where, s);
				str_free(s);
				break;
			}
			formal = formal_end;
		}
		else
		{
			size_t		n;
			DIR		*dp;
			struct dirent	*dep;

			/* need to expand wild characters */
			trace(("expand = \"%.*s\";\n", formal_end - formal, formal));
			n = sa_mark(&temp);
			sa_char(&temp, 0);
			dp = opendir(n ? temp.sa_buf : ".");
			if (!dp)
			{
				if (errno == ENOTDIR)
					break;
				scp = sub_context_new();
				sub_errno_set(scp);
				sub_var_set(scp, "File_Name", "%s", temp.sa_buf);
				error_with_position
				(
					pp,
					scp,
					i18n("opendir $filename: $errno")
				);
				sub_context_delete(scp);
				retval = -1;
				goto ret;
			}
			sa_goto(&temp, n);
			for (;;)
			{
				char	*np;

				dep = readdir(dp);
				if (!dep)
					break;
				np = dep->d_name;
				if
				(
					np[0] == '.'
				&&
					(
						!np[1]
					||
						(np[1] == '.' && !np[2])
					)  
				)
					continue;
				trace(("filename = \"%s\"\n", np));
				switch (gmatch(formal, formal_end, np, pp))
				{
				case 0:
					continue;

				case -1:
					retval = -1;
					goto ret;
				}
				for (cp = np; *cp; ++cp)
					sa_char(&temp, *cp);
				if (!*formal_end)
				{
					string_ty *s;

					s = sa_close(&temp);
					temp.sa_inuse = 1; /* defeat assert protection */
					string_list_append(where, s);
					str_free(s);
				}
				else
				{
					sa_char(&temp, '/');
					if (globber(formal_end + 1, pp))
					{
						closedir(dp);
						retval = -1;
						goto ret;
					}
				}
				sa_goto(&temp, n);
				temp.sa_inuse = 1;
			}
			closedir(dp);
			break;
		}
	}
	ret:
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


/*
 * NAME
 *	cmp - compare strings
 *
 * SYNOPSIS
 *	int cmp(string_ty **, string_ty **);
 *
 * DESCRIPTION
 *	The cmp function is used to compare two strings.
 *
 * RETURNS
 *	int; <0 if a<b, 0 if a==b, >0 if a>b
 *
 * CAVEAT
 *	Intended for use by qsort.
 */

static int cmp _((const void *, const void *));

static int
cmp(va, vb)
	const void	*va;
	const void	*vb;
{
	string_ty	*a;
	string_ty	*b;

	a = *(string_ty **)va;
	b = *(string_ty **)vb;
	return strcmp(a->str_text, b->str_text);
}


/*
 * NAME
 *	builtin_glob - builtin function for expanding file names
 *
 * SYNOPSIS
 *	int builtin_glob(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	The builtin_glob function is used to implement the "glob" builtin
 *	function of cook to expand file name patterns.
 *
 * RETURNS
 *	int; 0 on success, -1 on any error
 *
 * CAVEAT
 *	This function is designed to be used as a "builtin" function.
 */

static int interpret _((string_list_ty *, const string_list_ty *,
	const expr_position_ty *, const struct opcode_context_ty *));

static int
interpret(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	size_t		j;
	size_t		start;
	int		retval;

	trace(("glob(result = %08X, args = %08X)\n{\n"/*}*/, result, args));
	retval = 0;
	where = result;
	for (j = 1; j < args->nstrings; ++j)
	{
		temp.sa_inuse = 0; /* bypass assert protection */
		sa_open(&temp);
		original = args->string[j]->str_text;
		start = result->nstrings;
		if (globber(original, pp))
		{
			retval = -1;
			break;
		}
		qsort
		(
			result->string + start,
			result->nstrings - start,
			sizeof(string_ty *),
			cmp
		);
	}
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


builtin_ty builtin_glob =
{
	"glob",
	interpret,
	interpret, /* script */
};


builtin_ty builtin_wildcard =
{
	"wildcard",
	interpret,
	interpret, /* script */
};
