(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                              constrtypes.ml                              *)
(****************************************************************************)


open Std;;
open Vectops;;
open Names;;
open Generic;;
open Pp;;
open Term;;

type discharge_recipe =
    {d_expand : section_path list;
     d_modify : (sorts oper * sorts oper modification) list;
     d_abstract : identifier list;
     d_from : section_path}
;;


type recipe =
    COOKED of constr
  | RECIPE of discharge_recipe
;;

type implicits_typ =
    IMPL_AUTO of (int list) | IMPL_MANUAL of (int list) | NO_IMPL;;

type constant_body =
    {cONSTKIND : path_kind;
     cONSTHYPS : type_judgement signature;
     mutable cONSTBODY : recipe ref option;
     cONSTTYPE : type_judgement;
     mutable cONSTOPAQUE : bool;
     mutable cONSTEVAL : (int * (int * constr) list * int * bool) option option;
     mutable cONSTIMPARGS : implicits_typ }
;;

type const_entry = section_path * constant_body;;
type constdecl_body = (path_kind,constant_body) Listmap.t;;

type recarg = 
    Param of int 
  | Norec 
  | Mrec of int 
(*| Imbr of (recarg list) array;; *)
  | Imbr of section_path*int*(recarg list);; 


type mutual_inductive_packet =
    {mINDCONSNAMES : identifier array; (* names of constructors *)
     mINDTYPENAME : identifier;        (* name of inductive type *)
     mINDLC : constr;
     mINDSTAMP : name;
     mINDARITY : type_judgement;        (* arity of inductive type *)
     mINDLAMEXP : constr option;
     mINDKD : sorts list;              (* dependent eliminations allowed *)
     mINDKN : sorts list;              (* non-dependent ''        '' *)
     mINDLISTREC : (recarg list) array;(* recursive arguments *)
     mINDFINITE : bool;                (* true -> Ind ; false -> CoInd type *)
     mINDIMPLICITS : implicits_typ;    (* implicit pos. in inductive type *)
     mINDCONSIMPLICITS : implicits_typ array (* the same for the construct. *)
     }

and mutual_inductive_body =
    {mINDKIND : path_kind;
     mINDNTYPES : int;
     mINDHYPS : type_judgement signature;
     mINDPACKETS : mutual_inductive_packet array;
     mINDSINGL : constr option;
     mINDNPARAMS : int}
;;

(* implicits *)

let is_impl_auto = function IMPL_AUTO _ -> true
                           | _ -> false;;
 
let list_of_implicits = function IMPL_AUTO l -> l
                               | IMPL_MANUAL l -> l
                               | NO_IMPL -> [];;

type mutual_ind_entry = section_path * mutual_inductive_body;;
type mutual_inddecl_body = (path_kind,mutual_inductive_body) Listmap.t;;


let index_pred p v = 
 let rec indrec n =
    if n = Array.length v then failwith "index_pred"
    else if p v.(n) then n
         else indrec (n+1)
 in indrec 0
    
;;
    
let mind_type_name_index mib id =
    let mipvect = mib.mINDPACKETS
    in index_pred (fun mip -> id = mip.mINDTYPENAME) mipvect
;;

let mind_nth_type_packet mib n = mib.mINDPACKETS.(n);;

let mind_singl mib = mib.mINDSINGL;;

let mind_hyps mib = mib.mINDHYPS;;

let first_result_i = fun f start -> 
 let rec aux n = function
    [] -> None
  | (h::t) -> try (Some (f n h))
              with Failure _ | UserError _ -> aux (n+1) t
 in aux start

;;

(* gives the first (f k v.k) which succedds *)

let first_result_vec_i f v = 
  let n = Array.length v in 
  let rec aux k = 
    if k = n then failwith "first_result_vec_i"
    else 
      try (f k v.(k))
      with Failure _ | UserError _ -> aux (k+1)
  in aux 0
;;

let mind_oper_of_id sp mib id =
   first_result_vec_i
   (fun tyi mip ->
	if id = mip.mINDTYPENAME then (MutInd(sp,tyi),mind_hyps mib)
	else first_result_vec_i
	(fun cidx cid ->
           if id = cid then (MutConstruct((sp,tyi),cidx+1),mind_hyps mib)
	   else failwith "caught")
	     mip.mINDCONSNAMES)
   mib.mINDPACKETS
;;

let mind_oper_of_id1 sp mib id =
   first_result_vec_i
   (fun tyi mip ->
	if id = mip.mINDTYPENAME then ((MutInd(sp,tyi),mind_hyps mib),
                                        list_of_implicits mip.mINDIMPLICITS)
	else first_result_vec_i
	(fun cidx cid ->
           if id = cid 
           then ((MutConstruct((sp,tyi),cidx+1),mind_hyps mib),
                 (list_of_implicits (mip.mINDCONSIMPLICITS.(cidx))))
           else failwith "caught")
	     mip.mINDCONSNAMES)
   mib.mINDPACKETS
;;

let specification_of_constant cmap =
    List.map (fun (pk,cb) ->
             (pk,
              if (not (cb.cONSTOPAQUE)) then cb
              else {cONSTKIND = cb.cONSTKIND;
                    cONSTHYPS = cb.cONSTHYPS;
                    cONSTBODY = None;
                    cONSTTYPE = cb.cONSTTYPE;
                    cONSTOPAQUE = true;
                    cONSTEVAL = None;
                    cONSTIMPARGS = NO_IMPL}))
    cmap
;;

let specification_of_minductive x = x;;

let mind_type_finite mib i  = mib.mINDPACKETS.(i).mINDFINITE;;

let is_recursive listind = 
    let rec one_is_rec rvec = 
       List.exists (function  Mrec(i)        -> List.mem i listind 
                            | Imbr(_,_,lvec) -> one_is_rec lvec
                            | Norec          -> false
                            | Param(_)       -> false) rvec
    in exists_vect one_is_rec;;


(* the closure at which the object will be discharged *)
type strength = DischargeAt of section_path | NeverDischarge;;

let make_strength = function
    [] -> NeverDischarge
  | l  -> DischargeAt (sp_of_wd l)
;;

(* $Id: constrtypes.ml,v 1.15 1999/11/30 19:25:43 mohring Exp $ *)
