(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                intset.ml                                 *)
(****************************************************************************)
(* Sets over ordered types *)



open Baltree;;

(* Sets are represented by AVL trees. *)

type t = int Baltree.t;;

let order = ( (-));;

let empty = Empty;;

let is_empty s =
  match s with Empty -> true | _ -> false;;

let mem x s =
  Baltree.contains (order x) s;;

let add x s =
  Baltree.add (order x) x s;;

let remove x s =
  Baltree.remove (order x) s;;

let union s1 s2 =
  let rec union = fun
    p_0 p_1 -> match p_0,p_1 with (Empty, t2) -> t2
  | (t1, Empty) -> t1
  | ((Node(l1, v1, r1, _)), t2) ->
      let (l2, _, r2) = Baltree.split (order v1) t2 in
      Baltree.join (union l1 l2) v1 (union r1 r2) in
  union s1 s2;;

let inter s1 s2 =
  let rec inter = fun
    p_0 p_1 -> match p_0,p_1 with (Empty, t2) -> Empty
  | (t1, Empty) -> Empty
  | ((Node(l1, v1, r1, _)), t2) ->
      match Baltree.split (order v1) t2 with
        (l2, None, r2) ->
          Baltree.concat (inter l1 l2) (inter r1 r2)
      | (l2, Some _, r2) ->
          Baltree.join (inter l1 l2) v1 (inter r1 r2) in
  inter s1 s2;;

let diff s1 s2 =
  let rec diff = fun
    p_0 p_1 -> match p_0,p_1 with (Empty, t2) -> Empty
  | (t1, Empty) -> t1
  | ((Node(l1, v1, r1, _)), t2) ->
      match Baltree.split (order v1) t2 with
        (l2, None, r2) ->
          Baltree.join (diff l1 l2) v1 (diff r1 r2)
      | (l2, Some _, r2) ->
          Baltree.concat (diff l1 l2) (diff r1 r2) in
  diff s1 s2;;

let compare s1 s2 =
  Baltree.compare order s1 s2;;

let equal s1 s2 =
  compare s1 s2 == 0;;

let iter f s =
  let rec iter = function
    Empty -> ()
  | Node(l, v, r, _) -> iter l; f v; iter r
  in iter s;;

let fold f s init =
  let rec fold accu = function
    Empty -> accu
  | Node(l, v, r, _) -> fold (f v (fold accu r)) l
  in fold init s;;

let elements s =
  let rec elements accu = function
    Empty -> accu
  | Node(l, v, r, _) -> elements (v :: elements accu r) l
  in elements [] s;;

(* $Id: intset.ml,v 1.5 1999/06/29 07:47:17 loiseleu Exp $ *)
