(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                  lib.ml                                  *)
(****************************************************************************)


open Std;;
open Names;;
open Pp;;

type node =
    LEAF of Libobject.obj
  | OpenDir of string * open_dir_contents
  | ClosedDir of
      	  string * open_dir_contents * closed_dir_contents * library_segment
  | Import of string * section_path * bool

and open_dir_contents = {module_p : bool;
                         imports : section_path list}

and closed_dir_contents = {exports : section_path list;
                           mutable export_fn : string option}

and library_segment = (section_path * node) list;;
type library_entry = section_path * node;;
type frozen_t = library_segment;;

let lib_stk = ref ([] : (section_path * node) list);;
let lib_tab = (Mhm.create 17 : (section_path,node) Mhm.t);;
let path_prefix = ref ([] : string list);;

let init () =
    lib_stk := [];
    Mhm.empty lib_tab;
    path_prefix := []
;;

let make_path kind str = Names.make_path !path_prefix str kind;;

let recalc_path_prefix () = 
 let rec recalcrec = function
    (sp,OpenDir _)::_ -> let (pl,id,_) = repr_path sp
                         in (string_of_id id)::pl
  | _::t -> recalcrec t
  | []   -> []
 in (path_prefix := recalcrec !lib_stk)
  
;;

(* an object SP is in a closed directory when the path-part of the SP
 * is not a (possibly improper) prefix of the current path.
 *)

let add_entry (sp,obj) =
    if Mhm.in_dom lib_tab sp then
        errorlabstrm "lib__add_entry"
        [< 'sTR"Cannot add the object " ; 'sTR(string_of_path sp) ; 'sTR": an";
      	   'sPC ; 'sTR"identically named object already exists" >];
    lib_stk := (sp,obj):: !lib_stk;
    Mhm.add lib_tab (sp,obj);
    (sp,obj)
;;

let add_leaf_entry(spopt,obj) = add_entry(spopt,LEAF obj);;

let add ((id,pk),obj) =
  (match obj with
      ClosedDir _ | OpenDir _ -> anomaly "lib__add"
    | _ -> ());
  let sp = make_path pk id
  in add_entry(sp,obj)
;;

let add_leaf (id,obj) = add(id,LEAF obj);;

let split_lib sp = 
 let rec findrec acc = function
     ((sp',obj) as hd)::t ->
      	       	   if sp = sp' then (acc,hd,t) else findrec (hd::acc) t
   | [] -> errorlabstrm "lib__split_lib" [< 'sTR"no such entry" >]
 in findrec [] !lib_stk
  
;;

let split_lib_P p = 
 let rec findrec acc = function
     ((sp',obj) as hd)::t ->
             if p hd then (acc,hd,t) else findrec (hd::acc) t
   | [] -> errorlabstrm "lib__split_lib_P" [< 'sTR"no such entry" >]
 in findrec [] !lib_stk
  
;;

let discard_context = 
 let rec discrec = function
    (sp,ClosedDir (_,_,_,ctxt))::t -> (Mhm.rmv lib_tab sp;
       	       	       	       	       discrec ctxt;
       	       	       	       	       discrec t)
  | (sp,_)::t -> (Mhm.rmv lib_tab sp;
       	       	  discrec t)
  | [] -> ()
 in discrec
  
;;

let restore_context = 
 let rec restrec = function
    ((sp,(ClosedDir (_,_,_,ctxt) as obj)))::t -> (Mhm.add lib_tab (sp,obj);
       	       	       	       	       	       	  restrec ctxt;
       	       	       	       	       	       	  restrec t)
  | (sp,obj)::t -> (Mhm.add lib_tab (sp,obj);
       	       	    restrec t)
  | [] -> ()
 in restrec
  
;;

(* adds a module right after the latest module entry in the library. *)
let add_module (sp,node) =
  if Mhm.in_dom lib_tab sp then begin
    let id = string_of_id (basename sp) in
    errorlabstrm "Lib.add_module"
      [< 'sTR "Cannot restore module"; 'sPC; 'sTR id; 'sPC; 
	 'sTR "(already something of that name)" >] end;
  let (pref,hd,tail) = 
    split_lib_P (function (_,ClosedDir(_,{module_p=true},_,_)) -> true
                   | _ -> false) in
  let newlib = rev_append pref ((sp,node)::hd::tail)
  in (lib_stk := newlib;
      restore_context [(sp,node)])
;;

let reset_to sp =
  let pl = dirpath sp in
  if (not((prefix_of (List.rev pl) (List.rev !path_prefix)))) then
    error "cannot reset to an object which is in a closed section";
  if (not((Mhm.in_dom lib_tab sp))) then
    error "cannot reset to a nonexistent object";
  let (discard,ent,save) = split_lib sp
  in lib_stk := save;
     discard_context (ent::discard);
     recalc_path_prefix()
;;

let reset_to_P p =
  let (discard,ent,save) = 
    try split_lib_P p
    with UserError("lib__split_lib_P",_) -> errorlabstrm "lib__reset_to_P"
                    [< 'sTR"cannot reset to a nonexistent object" >]
  in let sp,n = ent 
  in let pl = dirpath sp
  in if (not((prefix_of (List.rev pl) (List.rev !path_prefix)))) then
       error "cannot reset to an object which is in a closed section";
     (match n with
        ClosedDir(_,{module_p=true},_,_) -> errorlabstrm "lib__reset_to_P"
                        [< 'sTR"cannot reset to a module" >]
      | Import _ -> anomaly "reset to an 'Import' entry"
      | _ -> lib_stk := save;
             discard_context (ent::discard);
             recalc_path_prefix() )
;;

let reset_keeping sp =
  let pl = dirpath sp in
  if (not((prefix_of (List.rev pl) (List.rev !path_prefix)))) then
    error "cannot reset keeping an object which is in a closed section";
  if (not((Mhm.in_dom lib_tab sp))) then
    error "cannot reset keeping a nonexistent object";
  let (discard,itobj,save) = split_lib sp
  in lib_stk := itobj::save;
     discard_context discard;
     recalc_path_prefix()
;;

let reset_keeping_P p =
  let (discard,itobj,save) = split_lib_P p
  in lib_stk := itobj::save;
     discard_context discard;
     recalc_path_prefix()
;;

let map sp = Mhm.map lib_tab sp;;

let map_leaf sp =
  match Mhm.map lib_tab sp with
    LEAF x -> x
  | _      -> invalid_arg "lib__map_leaf"
;;

let leaf_object_tag sp =
  let lobj = map_leaf sp
  in Libobject.object_tag lobj
;;

let find_entry_P p = 
 let rec findrec = function
    (sp,_ as ent)::tl -> (if try p ent with _ -> false then sp else findrec tl)
  | []                -> failwith "lib__find_entry: no such entry"
 in findrec !lib_stk
  
;;

let find_dir = function
    None   -> find_entry_P (function (_,OpenDir _) -> true | _ -> false)
  | Some s -> find_entry_P (function (_,OpenDir(s',_)) -> s = s' | _ -> false)
;;

let open_dir s odc =
  let s_id = id_of_string s in
  let s_sp = make_path OBJ s_id in
  (if Mhm.in_dom lib_tab s_sp then
        errorlabstrm "lib__open_dir"
        [< 'sTR"Section " ; 'sTR s ; 'sTR" already exists" >]);
  let rv = add_entry (s_sp,OpenDir(s,odc))
  in path_prefix := s:: !path_prefix; rv
;;

let close_dir sp cdc =
  if (not((kind_of_path sp = OBJ))) then
    invalid_arg "lib__close_dir";
  if (not((Mhm.in_dom lib_tab sp))) then
    error "directory does not exist";
  match Mhm.map lib_tab sp with
    (OpenDir (s,odc)) -> Mhm.rmv lib_tab sp;
      	       	       	 let (ctxt,_,rest) = split_lib sp
      	       	       	 in lib_stk := rest;
       	       	       	 add_entry (sp,ClosedDir(s,odc,cdc,ctxt));
       	       	       	 path_prefix := List.tl !path_prefix
  | _ -> error "is not a directory"
;;

let contents_after = function
   Some sp -> let (ctxt,_,_) = split_lib sp
              in List.rev ctxt
 | None    -> !lib_stk
;;

let app f = 
 let rec apprec = function
     []   -> ()
   | h::t -> apprec t; f h
 in apprec !lib_stk
  
;;

let freeze () = !lib_stk;;

let unfreeze l =
    lib_stk := l;
    Mhm.empty lib_tab;
    restore_context l;
    recalc_path_prefix()
;;

let cwd () = !path_prefix;;

let node_name (sp,_) = basename sp;;

let anon_SA = "_";;

let is_anonymous_sp sp = atompart_of_id (basename sp) = anon_SA;;

let next_anonymous_id () = 
 let rec nextrec = function
    [] 		      -> make_ident anon_SA 0
  | (sp,OpenDir _)::_ -> make_ident anon_SA 0
  | (sp,_)::tl 	      -> if is_anonymous_sp sp then
              	       	   make_ident anon_SA (1+(index_of_id(basename sp)))
      	       	       	 else nextrec tl
 in nextrec (contents_after None)
;;

let last_entry () = 
  match !lib_stk with
    []        -> None
  | (sp,_)::_ -> Some sp
;;

(* $Id: lib.ml,v 1.10 1999/06/29 07:47:25 loiseleu Exp $ *)
