(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                summary.ml                                *)
(****************************************************************************)


open Std;;
open Pp;;
open Names;;

type 'a summary_declaration =
    {freeze_function : unit -> 'a;
     unfreeze_function : 'a -> unit;
     init_function : unit -> unit}
;;

let sUMMARIES = (Mlm.create() : (string,Dyn.t summary_declaration) Mlm.t);;

type summary_vector = (string,Dyn.t) Listmap.t;;

let declare_summary sumname sdecl =

  let (infun,outfun) = Dyn.create (sumname^"-SUMMARY") in

  let dyn_freeze () = infun(sdecl.freeze_function()) and
      dyn_unfreeze sum = sdecl.unfreeze_function(outfun sum) and
      dyn_init = sdecl.init_function in

  let ddecl = {freeze_function = dyn_freeze;
               unfreeze_function = dyn_unfreeze;
               init_function = dyn_init} in

  try Mlm.add sUMMARIES (sumname,ddecl)
  with Failure _ -> anomalylabstrm "summary__declare_summary"
                    [< 'sTR"Cannot declare a summary twice: " ; 'sTR sumname >]
;;

let freeze_summary () =
    List.map (fun (id,sdecl) -> (id,sdecl.freeze_function()))
        (Mlm.toList sUMMARIES)
;;

let unfreeze_summary sum_vector =
    List.iter (fun (id,sdecl) ->
                 if Listmap.in_dom sum_vector id then
                     sdecl.unfreeze_function (Listmap.map sum_vector id)
                 else sdecl.init_function())
            (Mlm.toList sUMMARIES)
;;

let init_summaries () =
    Mlm.app (fun (_,sdecl) -> sdecl.init_function()) sUMMARIES;;

let sUMMARY_STACK = ref ([] : (section_path * summary_vector) list);;

let init_caches () =
    init_summaries();
    sUMMARY_STACK := [];;

let discard_to_key_including sp = 
 let rec discrec = function
    []                -> anomaly "summary__discard_to_key_including"
  | ((sp',_)::l as l_0) -> if sp = sp' then l else discrec l
 in (sUMMARY_STACK := discrec !sUMMARY_STACK)
  
;;

(* Freeze the state of the summaries, storing it under the given key,
   in the summary-stack *)
let freeze_to_key sp =
    if Listmap.in_dom !sUMMARY_STACK sp then
        discard_to_key_including sp;
    sUMMARY_STACK := (sp,freeze_summary()):: !sUMMARY_STACK
;;

let unfreeze_from_key sp =
  let sum = try (Listmap.map !sUMMARY_STACK sp)
            with Not_found -> failwith "Summary.unfreeze_from_key"
  in try unfreeze_summary sum
     with e -> anomalylabstrm "Summary.unfreeze_from_key"
                         [< 'sTR"Exception in unfreezing summary: " ; 'cUT ;
                            Errors.explain_sys_exn e >]
;;

type frozen_t = (section_path * summary_vector) list;;

let get_frozen_summaries() = !sUMMARY_STACK;;
let set_frozen_summaries fs = sUMMARY_STACK := fs;;

let discard_to_key_excluding sp = 
 let rec discrec = function
    []                -> anomaly "summary__discard_to_key_excluding"
  | ((sp',_)::l as l_0) -> if sp = sp' then l_0 else discrec l
 in (sUMMARY_STACK := discrec !sUMMARY_STACK)
  
;;

let find_latest () =
  let cd = List.rev (Lib.cwd()) in
  let sp = try_find
        (fun (sp,_) ->
             if prefix_of (List.rev (dirpath sp)) cd then
                 try let _ = Lib.map sp in sp
                 with _ -> failwith "caught"
             else failwith "caught")
      	!sUMMARY_STACK
  in sp
;;

(* will unfreeze the latest summary in the stack, discarding any which
   come after it.  We return the section_path of that summary. *)
let unfreeze_latest sp =
    unfreeze_from_key sp;
    discard_to_key_excluding sp
;;

let list_frozen_summaries () = List.map fst !sUMMARY_STACK;;

(* $Id: summary.ml,v 1.9 1999/10/29 23:19:15 barras Exp $ *)
