(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               tacinterp.ml                               *)
(****************************************************************************)


open Std;;
open Names;;
open Proof_trees;;
open Pp;;

open Tacmach;;
open Macros;;
open CoqAst;;
open Ast;;
open Printer;;
open Term;;


let tactic_tab = (Mhm.create 17 : (string, tactic_arg list -> tactic) Mhm.t);;

let tacinterp_add (s,f) =
    try Mhm.add tactic_tab (s,f)
    with Failure _ ->
        errorlabstrm "tacinterp_add"
        [< 'sTR"Cannot add the tactic " ; 'sTR s ; 'sTR" twice" >]
;;

let overwriting_tacinterp_add (s,f) =
    if Mhm.in_dom tactic_tab s then
        (Mhm.rmv tactic_tab s;
         warning ("Overwriting definition of tactic interpreter command "^s));
    Mhm.add tactic_tab (s,f)
;;

let tacinterp_init () = Mhm.empty tactic_tab;;

let tacinterp_map s = Mhm.map tactic_tab s;;

let err_msg_tactic_not_found macro_loc macro =
  user_err_loc
    (macro_loc,"macro_expand",
     [< 'sTR"Tactic macro ";'sTR macro; 'sPC; 'sTR"not found" >])

let rec interp ast =
  match ast with
    Node(loc,opn,tl) ->
     (match (opn, tl) with
        ("TACTICLIST",_) -> interp_semi_list tclIDTAC tl
      | ("DO",[n;tac]) -> tclDO (num_of_ast n) (interp tac)
      | ("TRY",[tac]) -> tclTRY (interp tac)
      | ("INFO",[tac]) -> tclINFO (interp tac)
      | ("REPEAT",[tac]) -> tclREPEAT (interp tac)
      | ("ORELSE",[tac1;tac2]) -> tclORELSE (interp tac1) (interp tac2)
      |	("FIRST",l) -> tclFIRST (List.map interp l)
      |	("TCLSOLVE",l) -> tclSOLVE (List.map interp l)
      | ("CALL",macro::args) ->
	  interp (macro_expand loc (nvar_of_ast macro) (List.map cvt_arg args))
      | _ -> interp_atomic loc opn (List.map cvt_arg tl))

  | _ -> user_err_loc(Ast.loc ast,"Tacinterp.interp",
                      [< 'sTR"A non-ASTnode constructor was found" >])

and interp_atomic loc opn args = 
  try tacinterp_map opn args
  with Not_found ->
    try vernac_tactic(opn,args)
    with e -> Stdpp.raise_with_loc loc e

and interp_semi_list acc = function
    (Node(_,"TACLIST",seq))::l ->
      interp_semi_list (tclTHENS acc (List.map interp seq)) l
  | t::l -> interp_semi_list (tclTHEN acc (interp t)) l
  | [] -> acc
;;

let is_just_undef_macro ast =
  match ast with
      Node(_,"TACTICLIST",[Node(loc,"CALL",macro::_)]) ->
        let id = nvar_of_ast macro in
      	(try let _ = Macros.lookup id in None
         with Not_found -> Some id)
    | _ -> None;;

let vernac_interp =
let gentac =
    hide_tactic "Interpret"
    (fun vargs gl -> match vargs with 
	 [TACEXP com] -> interp com gl
       | _ -> assert false) in
  fun com -> gentac ([TACEXP com])
;;

let vernac_interp_atomic =
let gentac =
    hide_tactic "InterpretAtomic"
    (fun argl gl -> match argl with 
	 ((IDENTIFIER id)::args) -> 
	   interp_atomic dummy_loc (string_of_id id) args gl
       | _ -> assert false)
in fun opn args -> gentac ((IDENTIFIER opn)::args)
;;

(* $Id: tacinterp.ml,v 1.19 1999/07/09 18:02:05 mohring Exp $ *)
