(**************************************************************************)
(*                               coq2latex                                *)
(**************************************************************************)

open Lexer
open Filter

(* options *)

let redirect = ref false
let to_file  = ref ""

let latex = ref false
let dvi   = ref false
let ps    = ref false

let std_input = ref false

let nodoc     = ref false
let latex2e   = ref false
let latex_command = ref "latex"

(* pretty-printing on the channel chan_out *)

let chan_out = ref stdout

let print_s s = output_string !chan_out s
let print_c c = output_char !chan_out c

(* replacement of tabulations by spaces *)

let one_space = "~"

let tab = ref 8
let col = ref 0

let next_tab () =
  print_s one_space ; 
  col := succ !col ;
  while (!col mod !tab) != 0 do
    print_s one_space ; 
    col := succ !col
  done


let rec initial_spaces =
    let rec ispaces = parser
      [< 'Char ' '  ; s >] ->
        print_s one_space ;
        col := succ !col ;
        ispaces s
    | [< 'Char '\t' ; s >] ->
        next_tab () ;
        ispaces s
    | [< >]               -> () in
    function s -> col := 0 ; ispaces s


(* translations of tokens *)

let size_of_token = function
      Special s  -> String.length s
    | Kwd s      -> String.length s
    | Ident s    -> String.length s
    | String s   -> String.length (String.escaped s)
    | Char '\t'  -> 0
    | Char c     -> 1
    | CB | CE    -> 2

let cb () =
  print_s (if !latex2e then "\\texttt{\\textit{" else "{\\tt\\it ")

let ce () =
  print_s (if !latex2e then "}}" else "}")


let print_token_tex tok =
    let com = !clevel > 0 in
    if com then col := !col + (size_of_token tok) ;
    match tok with
      Special "->"  -> print_s (if com then "->" else "$\\rightarrow$")
    | Special "<-"  -> print_s (if com then "<-" else "$\\leftarrow$")
    | Special "=>"  -> print_s (if com then "=>" else "$\\Rightarrow$")
    | Special "<->" -> print_s (if com then "<->" else "$\\leftrightarrow$")
    | Special "/\\" -> print_s (if com then "/\\char'134" else "$\\land$")
    | Special "\\/" -> print_s (if com then "\\char'134/" else "$\\lor$")
    | Special "<="  -> print_s (if com then "<=" else "$\\le$")
    | Special ">="  -> print_s (if com then ">=" else "$\\ge$")
    | Special s     -> print_s s

    | Kwd s      -> print_s (if com then s else ("\\KW{" ^ s ^ "}"))

    | String s   ->
	let s' = Tex.escaped_string (String.escaped s) in
	  print_s ("\"" ^ s' ^ "\"")

    | Ident s    -> print_s (Tex.escaped_string s)

    | Char ' '   -> print_s (if com then one_space else " ")
    | Char '~'   -> print_s (if com then "\\char'176" else "$\\neg$")
    | Char '\t'  -> next_tab ()
    | Char c     -> print_s (Tex.escaped_char c)

    | CB         -> if !clevel = 1 then cb ();
                    print_s "(*" (**)

    | CE         -> print_s "*)" ;
                    if !clevel = 0 then ce ()

module P : PRINT = struct
  let initial = initial_spaces
  let print_token = print_token_tex
end
 
module Filter = FFilter(P)

let line_header = "\LINE{"
let line_trailer = "}"

let print_tex_line line =
  let st = lexer (Stream.of_string line) in
  print_s line_header ;
  if !clevel > 0 then cb ();
  Filter.filter st ;
  if !clevel > 0 then ce ();
  print_s line_trailer

let tex_header () =
  (if !latex2e then
     "\\documentclass[a4paper]{article}"
   else
     "\\documentstyle[a4,fullpage]{article}")
  ^
"
\\newcounter{linecount}
\\newcounter{linenext}

\\pagestyle{headings}
\\hsize=17.5cm

\\newcommand{\\File}[3]{%
\\setcounter{linecount}{0}
\\setcounter{linenext}{9}
\\setcounter{page}{1}
\\renewcommand{\\thepage}{{\\it{}#2 #3\\hfill Page \\arabic{page} of #1.}}
}

\\newcommand{\\KW}[1]{{\\sf #1}}      % Keyword

\\newcommand{\\linebox}{%
\\ifnum\\value{linecount}>\\value{linenext}\\addtocounter{linenext}{10}
\\hbox{\\rm\\scriptsize\\arabic{linecount}}\\fi}

\\newcommand{\\LINE}[1]{%
\\noindent\\hbox to \\hsize{\\addtocounter{linecount}{1}{#1}\\hfill{\\linebox}}}

\\begin{document}

"

let tex_trailer = "\

\\end{document}
"

(* processing of a given channel *)

let process_channel chan_in =
  try 
    while true do
      let line = input_line chan_in in
      print_tex_line line ;
      print_s "\n"
    done
  with End_of_file -> print_s "\n"

let headings texfilename mtime =
  let tm = Unix.localtime mtime in
  let texdate = (string_of_int tm.Unix.tm_mday) 
        ^ "/" ^ (string_of_int (succ tm.Unix.tm_mon))
        ^ "/" ^ (string_of_int tm.Unix.tm_year) in
  let textime = (string_of_int tm.Unix.tm_hour)
        ^ ":" ^ (string_of_int tm.Unix.tm_min) in
  "\n\\File{" ^ texfilename ^ "}{" ^ texdate ^ "}{" ^ textime ^ "}\n"

(* processing of files *)

let treat_one_file filename =
  let texfilename = Tex.escaped_string filename in
  let mtime = (Unix.stat filename).Unix.st_mtime in
  print_s (headings texfilename mtime) ;

  let chan_in = open_in filename in
  process_channel chan_in ;

  close_in chan_in ;
  if not !nodoc then print_s "\\vfill\\eject\n"

let treat_all_files files =
  List.iter treat_one_file files  

(* processing of standard input *)

let treat_stdin () =
  print_s (headings "stdin" (Unix.time())) ;
  process_channel stdin ;
  flush stdout

(* usage *)

let usage () =
  prerr_string "\
Usage: coq2latex <options> file...
Options:
  -               take the input on standard input
  -latex          produce the LaTeX document
  -dvi            produce the DVI document
  -ps             produce the PostScript document 
  -o file         redirect output to a file
  -c command      use a different command than latex (e.g. latex2e)
  -t n            set the tabbing value
  -2e             produce a LaTeX2e document
";
  flush stderr;
  exit 1

let tmp_prefixe = Filename.temp_file "coqpp" ""

let tmp suffixe = tmp_prefixe ^ suffixe

let rec parse = function
    "-"  :: ll        -> std_input := true ; parse ll

  | "-o" :: (f :: ll) -> to_file := if not (Filename.is_relative f) then f 
                                    else Filename.concat (Sys.getcwd()) f ; 
                         redirect := true ; parse ll
  | "-o" :: []        -> usage () 

  | "-latex" :: ll    -> latex := true ; parse ll
  | "-dvi"   :: ll    -> dvi   := true ; parse ll
  | "-ps"    :: ll    -> ps    := true ; parse ll

  | "-t" :: (n :: ll) -> tab := (int_of_string n) ; parse ll
  | "-t" :: []        -> usage ()

  | "-2e" :: ll       -> latex2e := true ; parse ll
  | "-nodoc" :: ll    -> nodoc := true ; parse ll
  | "-c" :: (com::ll) -> latex_command := com ; parse ll
  | "-c" :: []        -> usage ()

  | ("-h"|"-?") :: ll -> usage ()

  | f :: ll           -> f::(parse ll)

  | []                -> []

(* cat d'un fichier sur !chan_out *)

let cat f =
  let ch = open_in f in
  try
    while true do
      let c = input_char ch in print_c c
    done ;
  with End_of_file -> () ;
  close_in ch ;
  flush !chan_out

let rm_f f =
  if Sys.file_exists f then Sys.remove f

let clean_and_exit () =
  rm_f (tmp "");
  rm_f (tmp ".tex") ;
  rm_f (tmp ".log") ;
  rm_f (tmp ".aux") ;
  rm_f (tmp ".dvi") ;
  rm_f (tmp ".ps") ;
  exit 0

let fork_exec prog args =
  match Unix.fork () with
      0 (* child *)  -> Unix.execvp prog args ; exit 0
    | _ (* parent *) -> Unix.wait ()

let fork_silent_exec prog args =
  let fd_in = Unix.descr_of_in_channel (open_in "/dev/null") in
  let fd_out = Unix.descr_of_out_channel (open_out "/dev/null") in
  let fd_err = Unix.descr_of_out_channel (open_out "/dev/null") in
  let pid = Unix.create_process prog args fd_in fd_out fd_err in
    Unix.waitpid [] pid

let coqpp () =
  let lg_command = Array.length Sys.argv in
  if lg_command < 2 then 
    usage ()
  else
    let files = parse (List.tl (Array.to_list Sys.argv)) in
    
    if !latex then
      if !redirect then chan_out := open_out !to_file else ()
    else
      chan_out := open_out (tmp ".tex") ;

    if not !nodoc then print_s (tex_header()) ;
    if !std_input then
      treat_stdin ()
    else
      treat_all_files files ;
    if not !nodoc then print_s tex_trailer ;
    flush !chan_out ;

    if !latex then begin
      if !redirect then close_out !chan_out ;
      clean_and_exit ()
    end ;
    close_out !chan_out ;
    Sys.chdir (Filename.dirname (tmp ".tex")) ;

    let _ = 
      fork_silent_exec !latex_command [| !latex_command ; (tmp ".tex") |]  in

    if !dvi then begin
      chan_out := if !redirect then open_out !to_file else stdout ;
      cat (tmp ".dvi") ;
      if !redirect then close_out !chan_out ;
      clean_and_exit ()
    end ;

    if !ps then begin
      let _ = fork_exec "dvips" [| "dvips" ; "-q" ; "-t" ; "a4" ; "-o" ;
				   (tmp ".ps") ; (tmp ".dvi") |] in
      chan_out := if !redirect then open_out !to_file else stdout ;
      cat (tmp ".ps") ;
      if !redirect then close_out !chan_out ;
      clean_and_exit ()
    end ;

    let _ = 
      fork_exec "dvips" [| "dvips" ; "-q" ; "-t" ; "a4" ; (tmp ".dvi") |] in
    clean_and_exit ()
;;

Printexc.catch coqpp ();;
