/*
  Entry.cc

  Copyright (c) 1996 Roland Wunderling, Malte Zoeckler
  Copyright (c) 1998 Michael Meeks
  Copyright (c) 1999 Dragos Acostachioaie

  This file is part of DOC++.

  DOC++ is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public
  License as published by the Free Software Foundation; either
  version 2 of the license, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "doc.h"

#define max(a,b) ((a) > (b) ? (a) : (b))

// Keywords that might come up...
char cxxreserved[] = "bool char short int long void float double"
                     "register signed unsigned const static virtual"
                     "public private protected extern volatile export namespace"
		     "using enum struct union class throw friend operator"
		     "typedef inline";
char javareserved[]= "boolean char byte short int long float double";

static McHashTable<char*, char*> *cxxtable;
static McHashTable<char*, char*> *javatable;

// Simple round robin cache for `fastNotSmall == true'
static const int max_cache = 5;
static int cache_pos = 0;
// So we can remember blanks as well
static char *cache_name[max_cache];
static Entry *cache_entry[max_cache];

// Don't want general bits
#define worthDoing(m) !m->general

void makeSubLists(Entry *rt);
Entry *findJavaPackage(McString n, Entry *root, Entry *p = 0);
Entry *findJavaClass(McString n, Entry *root);
int isIt(const char *n, const Entry *m);
static McString* get1RefName(McString &str, int start, int end);
int equalFunctionType(McString const &a, McString const &b);
unsigned int countCommas(McString const &args);
void seperateArguments(char const *args, McDArray<McString *> &arg_list);
int equalFunctionArgs(McString const &a, McString const &b);
int equalSignature(Entry *a, Entry *b);
Entry *lookupFunctionBySignature(Entry *group, Entry *model);
int hasDocumentation(Entry const *entry);
void mergeDocumentation(Entry *original, Entry *ccentry);
int mergeCCFunctionInHeader(Entry *ccentry);
void mergeCCFunctions(Entry *root);
void getRefNames(McDArray<McString*> strings, const char *names);

Entry::Entry()
{
    static int n = 1;

    number = n++;
    subnumber = 0;
    
    if(java)
	language = LANG_JAVA;
    else 
	language = LANG_CXX;

    protection    = PUBL;
    section       = EMPTY_SEC;
    file	  = "No file";
    parent        = 0;
    next          = 0;
    sub           = 0;
    tex           = 0;
    ownPage       = 0;
    override      = 0;

    currentParent = 0;
    htype         = 0;
    hname         = 0;
    hargs         = 0;
    hmemo         = 0;
    hsee          = 0;
    hdoc          = 0;
    line          = 0;
    general	  = 0;
    fromUnknownPackage = 0;
}

void Entry::addBaseClass(Entry* base,PROTECTION state)
{
    Entry *tmp2=this;
    if( state == PUBL ) {
	pubBaseclasses.append( base ) ;
	base->pubChilds.insert( 0,1,&tmp2 ) ;
    }
    else if (state == PROT) {
	proBaseclasses.append( base ) ;
	base->proChilds.insert( 0,1,&tmp2 ) ;
    }
    else {
	priBaseclasses.append( base ) ;
	base->priChilds.insert( 0,1,&tmp2 ) ;
    }
    baseclasses.append( base ) ;
}

void Entry::addBaseClass(const char *classname,PROTECTION state)
{
    Entry*	tmp=0;    
    if (language==LANG_JAVA) {
	McString t(classname);
	makeFullName(t);
	tmp=findJavaClass(t,root);
    } else {
	Entry* p = (parent!=0) ? parent : this;
	tmp = getRefEntry( classname, p);
    }

  if(tmp && (tmp->section == CLASS_SEC || tmp->section == INTERFACE_SEC))
    addBaseClass(tmp, state);
   else 
    {
      McString	*str = new McString( classname ) ;
 	makeFullName(*str);

      if( state == PUBL )
	otherPubBaseclasses.append( str ) ;
      else if ( state == PROT )
	otherProBaseclasses.append( str ) ;
      else 
	otherPriBaseclasses.append( str ) ;
    }
}

void Entry::addSubEntry(Entry *current)
{
    int	i = 1;
    Entry **e = &sub;
    sublist.append(current);
    for(; *e; e = &((*e)->next), ++i);
    *e = current;
    current->parent = this;
    current->sub = 0;
    current->next = 0;
    current->subnumber = i;
    current->makeFullName();
}

void Entry::dump(FILE *foo, int recursive)
{
#ifdef DEBUG
    static char indent[1024] = "" ;
    static int indd = 0 ;
    int lp ;
  
    if (recursive)
    {
	if (indd>0)
	    indent[indd-1] = ' ' ;
	indent[indd] = (char)0 ;
	if (recursive==1)
	    fprintf (foo, "%sname '%s'\n", indent, (char *)name) ;
    }
    else
    {
	fprintf(foo, "%s--------------------------Start-%p-----------------------------\n",
	  indent, this);
	fprintf(foo, "%sLang %c section %d protect/pub %c\n", indent, language,
	  section, protection);
	fprintf(foo, "%sparent %p next %p sub %p\n", indent, parent, next, sub);
	fprintf(foo, "%stype `%s'\n", indent, (char *)type);
	fprintf(foo, "%sname `%s'\n", indent, (char *)name);
	fprintf(foo, "%sargs `%s'\n", indent, (char *)args);
	fprintf(foo, "%smemo `%s'\n", indent, (char *)memo);
	fprintf(foo, "%sdoc `%s'\n", indent, (char *)doc);
	fprintf(foo, "%sfile `%s'\n", indent, (char *)file);
	fprintf(foo, "%sfullName `%s'\n", indent, (char *)fullName);
	fprintf(foo, "%sauthor `%s' version `%s'\n", indent, (char *)author,
	  (char *)version);

	McDArray<McString*> mda ;
	McDArray<Entry*> mde ;

	{
	    mda = see ;
	    if (mda.size()) fprintf (foo, "see :\n") ;
	    for (lp=0;lp<mda.size();lp++)
		fprintf(foo, "%s    `%s'\n", indent, (char *)mda[lp]);
	}	
	{
	    mda = param ;
	    if (mda.size()) fprintf (foo, "param :\n") ;
	    for (lp=0;lp<mda.size();lp++)
		fprintf(foo, "%s    `%s'\n", indent, (char *)mda[lp]);
	}	
	{
	    mda = exception ;
	    if (mda.size()) fprintf (foo, "exception :\n") ;
	    for (lp=0;lp<mda.size();lp++)
		fprintf(foo, "%s    `%s'\n", indent, (char *)mda[lp]);
	}

	fprintf(foo, "%sHTML\n", indent);
	if(htype)
	  fprintf(foo, "%shtype `%s'", indent, htype);
	if(hname)
	  fprintf(foo, " hname `%s'", hname);
	if(hargs)
	  fprintf(foo, " hargs `%s'\n", hargs);

	if(hmemo)
	  fprintf(foo, "%shmemo `%s'", indent, hmemo);
	if(hsee)
	  fprintf(foo, " hsee `%s'", hsee);
	if(hdoc)
	  fprintf(foo, " hdoc `%s'\n", hdoc);

	fprintf(foo, "%sfileName `%s', line %d general %d ownPage %d\n", indent,
	  (char *)fileName, line, general, ownPage);
	fprintf(foo, "%stex %d number %d subnumber %d\n", indent, tex, number,
	  subnumber);
    }
    if (sublist.size())
    {
	if (recursive<2)
	    fprintf(foo, "%ssublist:\n", indent);
	for (lp=0;lp<sublist.size();lp++)
	{
	    if (recursive && sublist[lp])
	    {
		indd++ ;
		sublist[lp]->dump (foo, recursive);
		indd-- ;
	    }
	    else
		fprintf(foo, "%s    `%s'\n", indent, (char *)sublist[lp]->name);
	}
    }

    if (!recursive)
	fprintf(foo,"%s----------------------------End-%p-----------------------------\n",
	  indent, this);
    else if (indd>0)
	indent[indd]=0 ;
#endif
}

void Entry::dumpNumber(McString &s)
{
    static char buf[33];
    if(parent)
	{
	parent->dumpNumber(s);
	sprintf(buf, "%d", subnumber);
	if(parent->parent)
	    {
	    s += '.';
	    s += buf;
	    }
	else
	    s += buf;
	}
}

void Entry::dumpNumber(FILE *f)
{
  McString num;
  dumpNumber(num);
  fprintf(f, "%s", (const char*)num);
}

// This assumes they are sorted, and binary chop searches ?
Entry *Entry::findSub(const char *name, int first, int last)
{
    int i, n, result;
    if(last - first < 4)	// I assume this is to catch different adjacent duplicates
	{
	for(i = first + 1; i <= last - 1; i++)
	    if(isIt(name, sublist[i]))
		return sublist[i];
        return 0;
	}

    n = (first + last) / 2;
    result = strcmp(name, sublist[n]->name);

    if(result == 0 && worthDoing(sublist[n]))
	return(sublist[n]);
    if(result > 0)
	return findSub(name, n, last);
    return findSub(name, first, n);
}

// This routine is called `A HELL OF A LOT' (tm), and needs keeping small
Entry *Entry::findSub(const char *name)
{
    int i, max;
    if(sublist.size())
	{
	max = sublist.size() - 1;
	if(isIt(name, sublist[0]))
	    return sublist[0];
	if(isIt(name, sublist[max]))
	    return sublist[max];
	Entry *found = findSub(name, 0, max);
	if(found)
	    return found;
	}
    for(int i = sublist.size(); i-- > 0;)
	{
	if(sublist[i]->section == MANUAL_SEC)
	    {
	    Entry *found = sublist[i]->findSub(name);
	    if(found)
		return found;
	    }
	}
    return 0;
}

void Entry::getPackage(McString &n)
{
    Entry *tmp = this;
    n = "";
    while(tmp)
	{
	if(tmp->section == PACKAGE_SEC)
	    {
	    if(n.length() > 0)
		n.insert(0, ".");
	    n.insert(0, tmp->name);
	    }
	tmp = tmp->parent;
	}    
}

int Entry::isClass()
{
    return(section == CLASS_SEC || section == INTERFACE_SEC);
}

void Entry::makeFullName(McString &name)
{
#ifdef DEBUG
    if (verb)
      printf("Make full name from `%s'\n", (const char *)name);
#endif

    /* If name is allready a fully qualified name, just return.*/
    if (name.index('.')>=0)
	return;

    /* Search in the list of import statements.*/
    for (int i=0 ; i<import.size() ; i++) {
	int ri=import[i]->rindex('.');
	if (ri>=0) {
	    if (strcmp((char *)name,(char *)(*(import[i]))+ri+1)==0) {
		name=(char *)*(import[i]);
		return;
	    }
	}
    }
    /* OK, assume that its the same package !*/
    McString tmp;
    getPackage(tmp);
    if (tmp.length())
	tmp+='.';
    name.insert(0,tmp);
#ifdef DEBUG
    if (verb)
      printf("	-> `%s'\n", (const char *)name);
#endif
}

void Entry::makeFullName()
{
    Entry *tmp = parent;

    fullName = name;
    while(tmp && tmp->name.length() > 0)
	{
	if(language == LANG_CXX)
	    fullName.insert(0, "::");
	else
    	    fullName.insert(0, ".");
	fullName.insert(0, tmp->name);
	tmp = tmp->parent;
	}
}

void Entry::makeRefs()
{
  int	i ;
  if (language==LANG_CXX) {
      if (section==CLASS_SEC)
	  findBases() ;
  } else if (language==LANG_JAVA ) {
      if (section==CLASS_SEC || section==INTERFACE_SEC) {
	  for ( i=0 ; i<implements.size() ; i++) {
	      McString tmp=*(implements[i]);
	      makeFullName(tmp);
	      addBaseClass((char *) tmp,PUBL);
	  }
	  for ( i=0 ; i<extends.size() ; i++) {
	      McString tmp=*(extends[i]);
	      makeFullName(tmp);	      
	      addBaseClass((char *) tmp,PUBL);
	  }	  
      }
  } else {
      fprintf(stderr, "Entry::makeRefs(): Unknown language\n");
  }
  for( i = 0 ; i < sublist.size() ; ++i )
      sublist[i]->makeRefs();
}

void Entry::makeSubList()
{
    Entry*	tmp ;
    int		i = 0 ;

    sublist.clear() ;
    for( tmp = sub ; tmp ; tmp = tmp->next )
	++i ;

    if( i )
    {
	sublist.remax( i ) ;
	for( tmp = sub ; tmp ; tmp = tmp->next )
	{
	    sublist.append( (Entry*)0 ) ;
	    for( i = sublist.size()-1 ; i > 0 ; --i )
	    {
		if( sublist[i-1]->name <= (const char *) tmp->name )
		    break ;
		sublist[i] = sublist[i-1] ;
	    }
	    sublist[i] = tmp ;
	}
    }
}

Entry *Entry::newSubEntry()
{
    Entry *newentry = new Entry;
    addSubEntry(newentry);
    return newentry;
}

void Entry::removeSub(Entry *e)
{
    for (int i=0 ; i<sublist.size() ; i++)
	if (sublist[i]==e) {
	    sublist.remove(i);
	    break;
	}
    if (sub==e)
	sub=e->next;
    else {
	for (Entry *tmp=sub ; tmp->next; tmp=tmp->next)
	    if (tmp->next==e) {
		tmp->next=e->next;
		break;
	    }
    }
}

void Entry::setupLangHash()
{
  // C++
    char *st = cxxreserved ;
    char *tmp = st ;
    cxxtable = new McHashTable<char*,char*>("") ;
    while (*tmp)
	if (*tmp==' ') {
	    *tmp=0 ;
	    cxxtable->insert (st, "1") ;
	    st=++tmp ;
	}
	else tmp++ ;

  // Java
    st = javareserved ;
    tmp = st ;
    javatable = new McHashTable<char*,char*>("") ;
    while (*tmp)
	if (*tmp==' ') {
	    *tmp=0 ;
	    javatable->insert (st, "1") ;
	    st=++tmp ;
	}
	else tmp++ ;

    // Setup cache as well
    for (int lp=0;lp<max_cache;lp++) {
	cache_entry[lp] = 0 ;
	cache_name[lp] = 0 ;
    }
}

void makeSubLists( Entry* rt )
{
    rt->makeSubList() ;
    for( int i = rt->sublist.size() ; --i >= 0 ; )
	makeSubLists( rt->sublist[i] ) ;
}

Entry *findJavaPackage(McString n, Entry *root, Entry *p = 0)
{
    Entry *tmp, *newPackage, *result = 0;
    int i = n.index('.'), j;
    McString *pkgName;

    if(!root)
	return 0;

    if(root->section == EMPTY_SEC)
	{
	p = root;
	root = root->sub;
	}

    if((root->section != PACKAGE_SEC) && (root->section != EMPTY_SEC))
	return 0;

    if(i > -1)
	{
	for(tmp = root; tmp && !result; tmp = tmp->next)
	    if(tmp->section == PACKAGE_SEC)
		{
		i = n.index('.');

		pkgName = new McString(n, 0, i);

		if(tmp->name == *pkgName)
		    {
		    n.remove(0, i + 1);

		    if(n.length() == 0)
			result = tmp;
		    else
	    		result = findJavaPackage(n, tmp->sub, tmp);
		    }
		}
    	}
    else
	for(tmp = root; tmp && !result; tmp = tmp->next)
	    if(tmp->section == PACKAGE_SEC)
		if(tmp->name == n)
		    result = tmp;

    if(!result)
	{
	i = -1;

	do
	    {
	    j = n.index('.', i + 1);
	    newPackage = p->newSubEntry();

	    if(j != -1)
		newPackage->name = *(new McString(n, 0, j));
	    else
		newPackage->name = n;

	    newPackage->section = PACKAGE_SEC;
	    newPackage->fromUnknownPackage = 1;

	    p = newPackage;
	    n.remove(0, j + 1);
	    }
	while((j != -1) && (n.length() > 0));

	result = newPackage;
	}

    return result;
}

Entry *findJavaClass(McString n, Entry *root)
{
	Entry		*pkg,
			*tmp,
			*result=NULL;
	int 	 	 i=n.rindex('.');
	McString	*package;


	if (i!=-1)
		package = new McString(n,0,i);
	else
		package = &n;

	pkg = findJavaPackage(*package,root);

	if (pkg!=NULL){
		if (i!=-1)
			n.remove(0,i+1);

		for (tmp=pkg->sub; tmp && (result==NULL); tmp=tmp->next)
			if ((tmp->section==CLASS_SEC) || (tmp->section==INTERFACE_SEC)){
				if (tmp->name==n)
					result=tmp;
			}

		if (!result){
			result=pkg->newSubEntry();
			result->name=n;
			result->section=CLASS_SEC;
			result->fromUnknownPackage=1;
		}
	}

	return(result);
}

int isIt(const char* n, const Entry* m)
{
  if(worthDoing(m))
    return !strcmp(n, (const char *)m->name);
  else
    return 0;
}

static McString* get1RefName( McString& str, int start, int end )
{
  McString*	s = 0 ;

  while( (str[start] == ' ' || str[start] == '\t') && start < end )
    start++ ;
  while( (str[end] == ' ' || str[end] == '\t' || str[end] == ',')
	 && start < end )
    end-- ;

  if( start < end )
    {
      s = new McString(str, start, end ) ;
    }

  return s ;
}


int equalFunctionType (McString const &a, McString const &b)
{
  char const *type_a = (char const *) a;
  char const *type_b = (char const *) b;

  if (strncmp (type_a, "virtual", 7) == 0)
    type_a += 7;

  while (*type_a == ' ') type_a++;
  while (*type_b == ' ') type_b++;
  return strcmp (type_a, type_b) == 0;
}

unsigned int countCommas (McString const &args)
{
  unsigned int count = 0;
  int i;
  for(i = 0; i < args.size() && args[i] != 0; i++)
    if (args [i] == ',')
      count++;
  return count;
}

void seperateArguments(char const *args, McDArray<McString *> &arg_list)
{
  unsigned int start;
  unsigned int end;
  unsigned int i;
  for (i = 0;; i++)
    {
      if (args [i] == 0) return;
      if (args [i] == '(') break;
    }

  start = ++i;

  for (;; i++)
    {
      if (args [i] == 0) return;
      if (start != 0
	  && (args [i] == '=' || args [i] == ',' || args [i] == ')'))
	{
	  while (args [start] == ' ')
	    start++;
	  end = i - 1;
	  while (args [end] == ' ')
	    end--;

	  if (start < end)
	    {
	      McString *s = new McString (args, start, end - start + 1);
	      arg_list.append (s);
	    }
	  start = 0;
	}
      if (args [i] == ',')
	start = i + 1;
      if (args [i] == ')')
	return;
    }
}

int equalFunctionArgs (McString const &a, McString const &b)
{
  int i, equal = 1;

  if(countCommas(a) != countCommas(b))
    return 0;

  McDArray<McString *> a_list;
  seperateArguments (a, a_list);
  McDArray<McString *> b_list;
  seperateArguments (b,
    b_list);

  if (a_list.size () != b_list.size ())
    {
      // This is okay IF one list has zero entities and one has one.
      // That's because zero and one both get a comma count of zero.
      assert (countCommas (a) == 0 && countCommas (b) == 0);
      return 0;
    }

  for(i = 0; i < a_list.size(); i++)
    {
      if (strcmp (*(a_list [i]), *(b_list [i])) != 0)
	equal = 0;
      delete a_list [i];
      delete b_list [i];
    }

  return equal;
}

int equalSignature (Entry *a, Entry *b)
{
    if (strcmp (a->name, b->name) != 0) return 0; // names not the same

    if (!equalFunctionType (a->type, b->type)) return 0; // types not the same

    if (!equalFunctionArgs (a->args, b->args)) return 0; // args not the same

    return 1;
}

Entry *lookupFunctionBySignature (Entry *group, Entry *model)
{
    int i;

    for (i = 0; i < group->sublist.size (); i++)
        if (equalSignature (group->sublist [i], model))
            return group->sublist [i];
    return 0;
}

int hasDocumentation (Entry const *entry)
{
    return (strcmp (entry->memo, "") != 0
            || strcmp (entry->doc, "") != 0
            || entry->param.size () > 0
            || entry->exception.size () > 0
            || strcmp (entry->retrn, "") != 0);
}

void mergeDocumentation(Entry *original, Entry *ccentry)
{
    int i;

    if(hasDocumentation(original) && hasDocumentation(ccentry))
	fprintf(stderr, "Warning: `%s %s%s' already have documentation, "
	    "ignoring the new one\n", (char const *)original->type,
	    (char const *)original->fullName,
	    (char const *)original->args);
	return;
    if(hasDocumentation(original))
	return;

    original->memo = ccentry->memo;
    original->doc = ccentry->doc;
    for(i = 0; i < ccentry->param.size(); i++)
	original->param.append(new McString(*(ccentry->param[i])));
    for(i = 0; i < ccentry->exception.size(); i++)
	original->exception.append(new McString(*(ccentry->exception[i])));
    original->retrn = ccentry->retrn;
}

int mergeCCFunctionInHeader(Entry *ccentry)
{
    int scope;
    if(ccentry->section != FUNCTION_SEC)
	{
#ifdef DEBUG
	if(verb)
	    printf("mergeCCFunctionInHeader: `%s' is no function\n",
		(const char *)ccentry->name);
#endif
	return 0;
	}
    scope = ccentry->name.rindex(':');
    scope--;
    if(scope < 1 || ccentry->name[scope] != ':')
	{
#ifdef DEBUG
	if(verb)
	    printf("No reason to merge function `%s'\n",
		(const char *)ccentry->name);
#endif
	return 0;
	}

    McString class_name(ccentry->name, 0, scope);
    McString unqualified_name(ccentry->name, scope + 2,
	ccentry->name.size () - scope - 3);
    Entry *father = getRefEntry(class_name, ccentry->parent);

    if(!father)
	{
#ifdef DEBUG
	if(verb)
	    printf("No class `%s' for `%s' found\n", (const char *)class_name,
		(const char *)ccentry->name);
#endif
	return 0;
	}
    ccentry->parent->removeSub(ccentry);
    ccentry->name = unqualified_name;

    Entry *original = lookupFunctionBySignature(father, ccentry);
    if(original == 0)
	{
	fprintf(stderr, "Warning: function `%s %s %s' not found in class `%s'\n",
	    (char const *)ccentry->type, (char const *)ccentry->name,
	    (char const *)ccentry->args, (char const *)class_name);
	father->addSubEntry(ccentry);
	}
    else
	{
        mergeDocumentation(original, ccentry);
        delete ccentry;
	}
    return 1;
}

void mergeCCFunctions(Entry *root)
{
    for (int i=0 ; i<root->sublist.size() ; i++) {
	if (root->sublist[i]->section==FUNCTION_SEC) {
	    if (mergeCCFunctionInHeader(root->sublist[i])) {
		i--;
	    }
	} else if (root->sublist[i]->section==MANUAL_SEC ||
		   root->sublist[i]->section==PACKAGE_SEC)
	    mergeCCFunctions(root->sublist[i]);
    }
}

void getRefNames( McDArray<McString*> strings, const char* names )
{
  McString	str = names ;
    int		j ;
    int		i = 0 ;

    while( (j = str.index( ",", i) ) >= 0 )
    {
      McString*	s = get1RefName( str, i, j) ;
	if( s )
	    strings.append(s) ;
	i = j+1 ;
    }
    McString*	s = get1RefName( str, i, str.length() ) ;
    if( s )
	strings.append(s) ;
}

/* We are looking for an entry named name. We start on the level of entry.
   `name' can be a simple identifier or it can contain C++ scopes (::)

   This routine is called `A HELL OF A LOT' (tm), and needs keeping quick
*/
Entry *searchRefEntry(McString &name, Entry *entry)
{    
    int i, lp, e;
    Entry *rot = entry;

    // General entries have a) no page, b) no references
    // This saves a _huge_ amount of time!
    if(entry->general)
	return 0;

    if(entry->language == LANG_JAVA && name.index('.') >= 0)
	{
	Entry *result = findJavaClass(name, root);
	return result;
	}

    static McHashTable<char*, char*> *table;
    if(entry->language == LANG_CXX)
	table = cxxtable;
    else
	if(entry->language == LANG_JAVA)
    	    table = javatable;
	else
	    table = 0;
    if(table && table->lookup((char *)name))	// Its a keyword silly !
	return 0;

    // Don't bother caching keywords
    if(cache_pos >= max_cache)
	cache_pos = 0;
    if(fastNotSmall)				// Actualy use the cache :-)
	for(lp = 0; lp < max_cache; lp++)
	    if(cache_entry[lp] && isIt(name, cache_entry[lp]))
		return cache_entry[lp];		// Don't re-enter it (probably best)
    	    else
		if(cache_name[lp])
		    if(strcmp((char *)name, cache_name[lp]) == 0)
			return 0;
    if(cache_name[cache_pos])
	free(cache_name[cache_pos]);
    cache_name[cache_pos] = 0;

    while(rot->section == MANUAL_SEC && rot->parent)
	rot = rot->parent;        

    e = name.index("::");

    // Just a name
    if(e < 0)
	{
	Entry *tmp = rot->findSub(name);
	if(tmp)
	    return(cache_entry[cache_pos++] = tmp);

	// Otherwise search in parent classes
	for(i = 0; i < rot->pubBaseclasses.size(); i++)
	    {
	    entry = searchRefEntry(name, rot->pubBaseclasses[i]);
	    if(cache_pos >= max_cache)
		cache_pos = 0;
	    if(entry)
		return(cache_entry[cache_pos++] = entry);
	    }

	for(i = 0; i < rot->proBaseclasses.size(); i++)
	    {
	    entry = searchRefEntry(name, rot->proBaseclasses[i]);
	    if(cache_pos >= max_cache)
		cache_pos = 0;
	    if(entry)
		return(cache_entry[cache_pos++] = entry);
	    }

	// Last chance, search on a higher doc++-level
	if(rot->parent)
	    return(searchRefEntry(name, rot->parent));

	// Cache a name with no corresponding entry
	assert(cache_pos < max_cache);

	// Whilst entries are static, strings get realloced lots!
	cache_name[cache_pos] = strdup((char *)name);
	cache_entry[cache_pos++] = 0;
	return 0;
	}
    else
	{
	McString tmp((const char *)name);
	while((e = tmp.index("::")) >= 0)
	    {
	    if(e == 0)
		{
		rot = root;	       
		tmp.remove(0, 2);
		}
	    else
		{
		McString base(name, 0, e);
		tmp.remove(0, e + 2);
		rot = searchRefEntry(base, rot);
		if(!rot)
		    return 0;
		}
	    }
	return searchRefEntry(tmp, rot);
	}
}

Entry *getRefEntry(const char *name, Entry *entry)
{
    Entry *result;
    McString tmp(name);
    result = getRefEntry(tmp, entry);
    return result;
}

Entry *getRefEntry(McString &name, Entry *entry)
{
    extern McDArray<namespace_entry *> namespace_table;
    int i;
    McString tmp;
    Entry *result = searchRefEntry(name, entry);
    if(!result && entry->language == LANG_CXX)
	// try to search inside namespaces, if any
	for(i = 0; i < namespace_table.size(); i++)
	    {
	    tmp = namespace_table[i]->name;
	    if((result = searchRefEntry(name,
		searchRefEntry(tmp, root))))
		{
#ifdef DEBUG
		printf("Class `%s' found in namespace `%s'!\n",
		    (char *)name, (char *)namespace_table[i]->name);
#endif
		break;
		}
	    }
    return result;
}
