%option yylineno
%option noyywrap
%{
/*
 * Definitions section
 */

#include <stdio.h>
#include <string.h>
#include "y.tab.h"
#include "genparse.h"

/*
 * Help with debugging
 */

#define return_token(x) { if(DEBUG) fprintf(stderr,"LEX: Token %s: '%s'\n",#x,yytext); return x; }

%}

/*
 * More definitions
 */

char                    [A-Za-z]
first_char              ^[A-Za-z]
digit                   [0-9]
alnum                   [a-zA-Z_0-9]+
c_var                   [a-zA-Z_][a-zA-Z_0-9]*
file                    [a-zA-Z_][a-zA-Z_0-9/]*.h
whitespace              [ \t\n]
any  .

%%

{whitespace}            ; /* do nothing on whitespace */

"/*" 			{ /* Ignore C comments */
			  int c1 = 0;
			  int c2 = input();
            
                          for (;;)
                            {
                              if (c2 == EOF) break;
			      if (c1 == '*' && c2 == '/') break;
                              c1 = c2;
                              c2 = input();
                            }
			}

{first_char}		{
                          yylval.c = *yytext;
                          return_token(FIRST_CHAR);
			}

"{"			return_token(OPEN_BRACE);
"}"			return_token(CLOSE_BRACE);
"/"			return_token(SLASH);
"int"			return_token(INT);
"float" 		return_token(FLOAT);
"char"			return_token(CHAR);
"string"		return_token(STRING);
"flag"			return_token(FLAG);
"include"               return_token(INCLUDE);
"<"                     return_token(LT);
">"                     return_token(GT);
"#"                     return_token(POUND);
"mandatory"             return_token(MANDATORY);

{c_var}			{
                          yylval.str = strdup(yytext);
                          return_token(C_VAR);
			}

{file}			{
                          yylval.str = strdup(yytext);
                          return_token(FILENAME);
			}

{alnum}			{
                          yylval.str = strdup(yytext);
                          return_token(ALNUM);
			}

"["			return_token(BEGIN_RANGE);
"]"			return_token(END_RANGE);
".."			return_token(RANGE_SPEC1);
"..."			return_token(RANGE_SPEC2);
"()"			return_token(FUNC_SPEC);
"\""[^\"]*"\""          { 
                          yylval.str = strdup(yytext+1); 
                          yylval.str[strlen(yylval.str)-1]='\0';
                          return_token(QUOTED);
                        }

{any}			{ /* error */
			  fprintf(stderr,"parser error in line %d ('%s') \n",yylineno,yytext);
			  exit(1);
			}

%%
