/* $Header: /cvs/gnome/gIDE/src/gI_document.c,v 1.4 1999/12/05 07:17:51 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/utsname.h>
#include <netdb.h>
#include "structs.h"
#include "gI_document.h"
#include "gI_compile.h"
#include "gI_file.h"
#include "gI_window.h"
#include "gI_edit.h"
#include "gI_common.h"
#include "gI_help.h"
#include "gI_menus.h"

#ifdef HAVE_GTKTEXT_PATCH
#include <gtksctext.h>
#include <gtkeditor/gtkeditor.h>
#endif

/* externs */
extern gI_window *main_window;
extern gI_config *cfg;


/*
 ---------------------------------------------------------------------
     Function: gI_document_new()
     Desc: To Create A New Notebook-Page
 ---------------------------------------------------------------------
*/

gI_document *gI_document_new( gI_window *window )
{
	gI_document *document;
	GtkWidget *vscrollbar;
	GtkWidget *hbox;
	gchar label[25];
	static glong doc_count = 1;

	document = g_malloc0( sizeof( gI_document ) );

	if( window->notebook == NULL )
	{
		window->notebook = gtk_notebook_new();
		gtk_widget_show( window->notebook );
		gtk_notebook_set_scrollable( GTK_NOTEBOOK( window->notebook ), TRUE );
	}

	document->window = window;
	g_snprintf( label, sizeof(label), "Untitled%ld", doc_count++ );
	document->label = gtk_label_new( label );
	document->filename = NULL;
	document->compile_window = NULL;
	document->x = 0;
	document->y = 0;
	document->read_only = 0;
	document->working = 0;
	document->last_mod = 0;
	document->op_list = NULL;
	document->first_undo = 1;

	gtk_widget_show( document->label );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_widget_show( hbox );

#ifdef HAVE_GTKTEXT_PATCH
	document->text = gtk_editor_new( NULL, NULL );
#else
	document->text = gtk_text_new( NULL, NULL );
#endif
	gI_text_default_tab_width( document->text ) = cfg->tab_width;
	gI_text_tab_stops( document->text ) = NULL;
	gI_text_tab_stops( document->text ) = g_list_prepend( gI_text_tab_stops( document->text ), (void*)cfg->tab_width );
	gI_text_tab_stops( document->text ) = g_list_prepend( gI_text_tab_stops( document->text ), (void*)cfg->tab_width );
	gI_text_set_editable( document->text, TRUE );
	gI_text_set_word_wrap( document->text, cfg->wordwrap );
	gtk_box_pack_start( GTK_BOX( hbox ), document->text, TRUE, TRUE, 5 );
	gtk_widget_show( document->text );
	gI_text_set_point( document->text, 0 );

	document->changed = FALSE;
	document->changed_id = gtk_signal_connect( GTK_OBJECT( document->text ), "changed",
	                       GTK_SIGNAL_FUNC( gI_document_changed ), (gpointer) document );

	gtk_signal_connect_object(GTK_OBJECT(document->text),"event",
	                          GTK_SIGNAL_FUNC(gI_document_show_popup_menu),
	                          GTK_OBJECT(window->popup->menu));

#ifdef HAVE_GTKTEXT_PATCH
	if( !cfg->highlight )
	{
		gtk_editor_hilite_when_idle( GTK_EDITOR( document->text ), FALSE );
	}
	if( !cfg->edtab )
	{
		GTK_EDITOR( document->text )->one_line_ind = NULL;
	}
#endif

	gtk_signal_connect( GTK_OBJECT( document->text ), "key_press_event",
	                    GTK_SIGNAL_FUNC( gI_document_keypressed ), document );
	gtk_signal_connect( GTK_OBJECT( document->text ), "button_press_event",
	                    GTK_SIGNAL_FUNC( gI_document_buttonpressed ), document );
	gtk_signal_connect_after( GTK_OBJECT( document->text ), "button_press_event",
	                    GTK_SIGNAL_FUNC( gI_document_buttonpressed_after ), document );
	document->insert_id = gtk_signal_connect( GTK_OBJECT( document->text ), "insert_text",
	                      GTK_SIGNAL_FUNC( gI_document_insert_text ), document );
	document->delete_id = gtk_signal_connect( GTK_OBJECT( document->text ), "delete_text",
	                      GTK_SIGNAL_FUNC( gI_document_delete_text ), document );

	vscrollbar = gtk_vscrollbar_new( gI_text_vadj( document->text ) );
	gtk_box_pack_end( GTK_BOX( hbox ), vscrollbar, FALSE, TRUE, 5 );
	gtk_widget_show( vscrollbar );

	if( window->documents == NULL )
		window->documents = g_list_alloc();

	g_list_append( window->documents, document );

	gtk_notebook_append_page( GTK_NOTEBOOK( window->notebook ), hbox, document->label );

	gtk_notebook_set_page( GTK_NOTEBOOK( window->notebook ), g_list_length(( GTK_NOTEBOOK( window->notebook ) )->first_tab ) - 1 );

	if( !window->switch_page_id )
	{
		window->switch_page_id = gtk_signal_connect_after( GTK_OBJECT( window->notebook ), "switch_page",
		                         GTK_SIGNAL_FUNC( gI_window_switch_notebookpage ), window );
	}

	gtk_widget_grab_focus( document->text );

	/* Assing document with current style configuration. */
	if( cfg->style )
		gtk_widget_set_style( document->text, cfg->style );

	gtk_signal_emit_by_name( GTK_OBJECT( window->window ), "check_resize" );

	return( document );
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_set_unchanged()
     Desc: Sets the given document status to unchanged, corrects
           Notebook-Tab...
 ---------------------------------------------------------------------
*/

void gI_document_set_unchanged( gI_document *document )
{
	gchar str[STRLEN];
	GtkNotebookPage *page;
	glong index;
	gchar *label;

	if( !document )
		return;

	if( !document->filename )
	{
		gchar *ptr;

		gtk_label_get( GTK_LABEL( document->label ), &label );
		strcpy( str, label );
		ptr = strchr( str, '*' );
		if( ptr )
		{
			*ptr = '\0';
		}
	}
	else
	{
		g_snprintf( str, STRLEN, "%s", file_strip_name( document->filename ) );
	}

	gtk_label_set_text( GTK_LABEL( document->label ), str );

	index = g_list_index( main_window->documents, (gpointer) document );
	index -= 1;

	page = (GtkNotebookPage *) g_list_nth_data( GTK_NOTEBOOK( main_window->notebook )->children, index );
	gtk_label_set_text( GTK_LABEL( page->tab_label ), str );

	document->changed_id = gtk_signal_connect( GTK_OBJECT( document->text ), "changed",
	                       GTK_SIGNAL_FUNC( gI_document_changed ), document );
	document->changed = FALSE;

	gI_window_set_statusbar( document );
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_changed()
     Desc: This function is called, when the document is changed
 ---------------------------------------------------------------------
*/

void gI_document_changed( GtkWidget *widget, gI_document *document )
{
	gchar str[STRLEN];
	GtkNotebookPage *page;
	gchar *label = NULL;

	if( document->changed )
		return;

	document->changed = TRUE;

	if( document->filename != NULL )
	{
		g_snprintf( str, STRLEN, "%s*", file_strip_name( document->filename ) );
	}
	else
	{
		gtk_label_get( GTK_LABEL( document->label ), &label );
		g_snprintf( str, sizeof(str), "%s*", label );
	}

	if( label != NULL )
	{
		if( !strchr( label, '*' ) )
			gtk_label_set_text( GTK_LABEL( document->label ), str );
	}
	else
	{
		gtk_label_set_text( GTK_LABEL( document->label ), str );
	}

	page = GTK_NOTEBOOK( main_window->notebook )->cur_page;
	gtk_label_set_text( GTK_LABEL( page->tab_label ), str );

	if( document->changed_id )
	{
		gtk_signal_disconnect( GTK_OBJECT( document->text ), document->changed_id );
		document->changed_id = 0;
	}

	gI_window_set_statusbar( document );

	check_current_doc();
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_keypressed()
     Desc: This function is called, when a key is pressed
 ---------------------------------------------------------------------
*/

void gI_document_keypressed( GtkWidget *widget, GdkEventKey *event, gI_document *document )
{
	gI_window_clear_statusbar( document->window );

	gI_document_track_movement( document );

	gI_window_set_statusbar( document );
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_get_current()
     Desc: Returns the current Document
 ---------------------------------------------------------------------
*/

gI_document *gI_document_get_current( gI_window *window )
{
	gint current;
	gI_document *current_document;

	current_document = NULL;

	current = gtk_notebook_get_current_page( GTK_NOTEBOOK( window->notebook ) ) + 1;
	current_document = ( g_list_nth( window->documents, current ) )->data;

	return( current_document );
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_get_nth()
     Desc: Returns the current Document
 ---------------------------------------------------------------------
*/

gI_document *gI_document_get_nth( gI_window *window, guint index )
{
	gI_document *current_document;

	current_document = NULL;

	current_document = ( g_list_nth( window->documents, index ) )->data;

	return( current_document );
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_clear_text()
     Desc: Clears the given Document
 ---------------------------------------------------------------------
*/

void gI_document_clear_text( gI_document *document )
{
	gint i;
	gint flag;

	i = gI_text_get_length( document->text );
	if( i > 0 )
	{
		flag = document->changed;
		document->changed = TRUE;
		gtk_editable_delete_text( GTK_EDITABLE( document->text ),
		                          0,
		                          i );
		document->changed = flag;
	}
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_get_no()
     Desc: Returns the Number of open Documents 
 ---------------------------------------------------------------------
*/

gint gI_document_get_no( gI_window *window )
{
	return( g_list_length( window->documents ) - 1 );
}


/*
----------------------------------------------------------------------
Function gI_document_show_popup_menu
----------------------------------------------------------------------
*/
gint gI_document_show_popup_menu(GtkWidget *widget,GdkEvent *ev)
{
	if(ev->type==GDK_BUTTON_PRESS)
	{
		GdkEventButton *event=(GdkEventButton*)ev;
		if(event->button==3)
		{   gtk_menu_popup (GTK_MENU(widget), NULL, NULL, NULL, NULL,
			                    event->button, event->time);
			return TRUE;
		}
	}
	return FALSE;
}


/*
-----------------------------------------------------------------------
Function: gl_document_cmd_swaphc
-----------------------------------------------------------------------
*/
void gI_document_cmd_swaphc(GtkWidget *widget,gI_window *win)
{
	size_t len;
	gchar *newfname;
	gI_document *doc;
	newfname=0;
	doc=gI_document_get_current(win);
	if(!doc || !doc->filename)
		return;

	len=strlen(doc->filename);
	while(len){
		if(doc->filename[len]=='.')
			break;

		len--;
	};
	if(strcasecmp(&doc->filename[len],".h")==0) {
		len++;
		newfname=g_malloc(len+4);
		strcpy(newfname,doc->filename);  /* copy original filename */
		strcpy(&newfname[len],"cc");
		if(!file_check_if_exist(newfname,FALSE))  {
			strcpy(&newfname[len],"cpp");
			if(!file_check_if_exist(newfname,FALSE))  {
				newfname[len+1]=0;    /* test if *.c exist */
				if(!file_check_if_exist(newfname,TRUE)) {
					g_free(newfname);    /* must free up before return */
					return ;
				}
			}
		}
	}
	else if(strncasecmp(&doc->filename[len],".c",2)==0 ) {
		len++;
		newfname=g_strdup(doc->filename);
		newfname[len]='h';
		newfname[len+1]=0;
		if(!file_check_if_exist(newfname,TRUE)) {
			g_free(newfname);
			return ;
		}
	}

	if(newfname)
	{
		if( !check_file_open( newfname ) )
		{
			file_open_by_name(win,newfname);
		}
		else
		{
			goto_file( newfname );
		}
		g_free( newfname );
	}
}

/*
-----------------------------------------------------------------------
Function: gl_document_cmd_open_file_at_line
-----------------------------------------------------------------------
*/
void gI_document_cmd_open_file_at_line(GtkWidget *widget,gI_window *win)
{
	glong i;
	gI_document *current;
	gchar *line;
	gchar filename[MAXLEN];
	gchar *inc[64];
	gchar *fwp;
	gI_comp_set *cset = NULL;

	current = gI_document_get_current( win );
	if( !current )
		return;

	if( !GTK_EDITABLE( current->text )->has_selection )
	{
		gI_error_dialog( _("No selection in current document!") );
		return;
	}

	/* get selection */
	if( GTK_EDITABLE( current->text )->selection_end_pos
	        < GTK_EDITABLE( current->text )->selection_start_pos )
	{
		line = gtk_editable_get_chars( GTK_EDITABLE( current->text ),
		                               GTK_EDITABLE( current->text )->selection_end_pos,
		                               GTK_EDITABLE( current->text )->selection_start_pos );
	}
	else
	{
		line = gtk_editable_get_chars( GTK_EDITABLE( current->text ),
		                               GTK_EDITABLE( current->text )->selection_start_pos,
		                               GTK_EDITABLE( current->text )->selection_end_pos );
	}

	if( !line )
		return;

	if( !strstr( line, "#include" ) )
	{
		fwp = g_strconcat( getcwd( NULL, STRLEN ), "/", line, NULL );
		if( !check_file_open( fwp ) )
			file_open_by_name( win, fwp );
		else
			goto_file( fwp );
		g_free( line );
		g_free( fwp );
		return;
	}

	if( current->filename )
		cset = get_comp_set_from_filename( current->filename );

	if( strchr( line, '<' ) && strchr( line, '>' ) )
	{
		strncpy( filename, SK_GetBetween( line, '<', '>' ), MAXLEN );

		/* search includes path's, if available */
		if( cset || cfg->incpath )
		{
			if( cset )
			{
				SK_GetFields( cset->incpath, inc, ':' );
			}
			else
			{
				if( cfg->incpath )
				{
					SK_GetFields( cfg->incpath, inc, ':' );
				}
				else
				{
					gI_error_dialog( _("Can't find the included file in the specified paths") );

					g_free( line );
					return;
				}
			}

			for(i=1;i<=atoi(inc[0]);i++)
			{
				if( inc[i] != NULL )
				{
					fwp = g_strconcat( inc[i], "/", filename, NULL );
					if( file_exist( fwp ) )
					{
						if( !check_file_open( fwp ) )
							file_open_by_name( win, fwp );
						else
							goto_file( fwp );
						g_free( fwp );
						g_free( line );
						return;
					}
				}
			}
		}

		gI_error_dialog( _("Can't find the included file in the specified paths") );

		g_free( line );
		return;
	}

	if( strchr( line, '"' ) )
	{
		strncpy( filename, SK_GetBetween( line, '"', '"' ), MAXLEN );
		fwp = g_strconcat( getcwd( NULL, STRLEN ), "/", filename, NULL );
		if( !check_file_open( fwp ) )
			file_open_by_name( win, fwp );
		else
			goto_file( fwp );
		g_free( line );
		g_free( fwp );
		return;
	}

	g_free( line );
}


/*
 ---------------------------------------------------------------------
     Function: gI_document_track_movement()
     Desc: Tracks the movement in document
 ---------------------------------------------------------------------
*/
void gI_document_track_movement( gI_document *document )
{
	gint height = 14, width = 7;

	if( !document )
	{
		return;
	}

	/* code needed to track the cursor movement in every document */
	/* This routines is TO SLOW removed until someone implements faster or GtkText Widget is rewritten */

	/*document->x = get_column_from_point( gI_text_get_point( GTK_TEXT( document->text ) ) );*/
	/*	document->y = get_line_from_point( gI_text_get_point( GTK_TEXT( document->text ) ) ); */

	/*	document->x = (gint) GTK_TEXT( document->text )->current_line->data; */
	/* Height 14, Width 7 hard coded for now will need to be changed with gdk_char_height? and gdk_char_width? */

	/*width = gdk_text_width( document->text->style->font,
			   	  GTK_TEXT( document->text )->text.ch, strlen(GTK_TEXT( document->text )->text.ch) );
	height = gdk_text_height( document->text->style->font,
				    GTK_TEXT( document->text )->text.ch, strlen(GTK_TEXT( document->text )->text.ch) );*/

	document->x = gI_text_last_ver_value( document->text) + gI_text_cursor_pos_y( document->text ) / height;
	document->y = gI_text_cursor_pos_x( document->text ) / width;
}



glong gI_document_get_changed( gI_window *window )
{
	GList *docs;
	glong cnt = 0;
	gI_document *cdoc;

	docs = window->documents;

	while( docs != NULL )
	{
		cdoc = (gI_document *) docs->data;
		if( !cdoc )
		{
			docs = docs->next;
			continue;
		}

		if( cdoc->changed == TRUE )
		{
			cnt++;
		}

		docs = docs->next;
	}

	return( cnt );
}

/* this does not work, ........ */
glong get_column_from_point( glong point )
{
	glong i = point;
	glong col = 1;
	gI_document *current;

	if( point == 0 )
		return( col );

	current = gI_document_get_current( main_window );
	if( !current )
		return(-1);

	while( gtk_editable_get_chars( GTK_EDITABLE( current->text ), i, i+1 )[0] != '\n' )
	{
		col++;
		if( i == 0 )
		{
			col = 1;
			break;
		}
		i--;
	}

	return( col );
}


void gI_document_buttonpressed( GtkWidget *widget, GdkEventButton *event, gI_document *document )
{
	check_current_doc();

	/*gI_window_clear_statusbar( document->window );

	gI_document_track_movement( document );

	gI_window_set_statusbar( document );*/
}


void gI_document_buttonpressed_after( GtkWidget *widget, GdkEventButton *event, gI_document *document )
{
        /* jump to cursor mark */
        gI_text_set_point( document->text, gtk_editable_get_position( GTK_EDITABLE( document->text ) ) );

      	gI_window_clear_statusbar( document->window );
        gI_document_track_movement( document );
        gI_window_set_statusbar( document );
}


void special_remove_current_file( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	gchar errmsg[200];

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->filename )
		return;

	if( remove( current->filename ) < 0 )
	{
		g_snprintf( errmsg, sizeof(errmsg), _("File Error: %s"), strerror( errno ) );
		gI_error_dialog( errmsg);
		return;
	}

	check_current_doc();
}

/* this is for g_list_foreach callback function */
void update_document_CB( gI_document* doc, GtkStyle* style )
{
	if ( doc && doc->text )
		gtk_widget_set_style( doc->text, style );
}

void update_documents_style( gI_window* window )
{
	g_list_foreach( window->documents, (GFunc) update_document_CB, cfg->style );
}



void check_current_doc(void)
{
	gI_document *current;
	glong ret;
	gchar msg[STRLEN];

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->filename )
		return;

	ret = check_doc_changed( current );

	switch( ret )
	{
	case 0:
		{
			break;
		}

	case 1:
		{
			if( current->changed )
			{
				g_snprintf( msg, sizeof(msg), _("The file\n'%s'\nhas been changed from outside the editor!\nIf You reload, You'll loose the changes You did in gIDE!\nDo You really want to reload it?"), current->filename );
			}
			else
			{
				g_snprintf( msg, sizeof(msg), _("The file\n'%s'\nhas been changed from outside the editor!\nDo You want to reload it?"), current->filename );
			}

			if( gI_ask_dialog( msg) == 0) /* YES */
			{
				file_reload( NULL, (gpointer)1 );
				break;
			}
			else
			{
				gI_document_set_changed( current );
				break;
			}
		}

	case 2:
		{
			g_snprintf( msg, sizeof(msg), _("The file\n'%s'\nis not longer available.\nChoose YES to close it, or NO to keep it!"), current->filename );
			if( gI_ask_dialog( msg) == 0) /* YES */
			{
				file_close( NULL, NULL );
				break;
			}
			else
			{
				gI_document_set_changed( current );
				break;
			}
		}
	}
}


void special_insert_file( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	GList *files = NULL;
	FILE *f;
	gchar *filename;
	gchar buf[STRLEN];

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	files = gI_file_sel_new( _("Insert File..."), FALSE, FALSE );
	if( files && files->data )
	{
		filename = (gchar *) files->data; 

		f = fopen( filename, "r" );
	        if( !f )
	        {
	       		g_snprintf( buf, sizeof(buf), _("File Error: %s"), strerror( errno ) );
		        gI_error_dialog( buf);
		        return;
		}

	        while( fgets( buf, sizeof(buf), f ) )
       		{
	        	gI_document_insert( current, NULL, NULL, NULL, buf, strlen( buf ) );
     		}

	        fclose( f );
	}
}


void gI_document_set_changed( gI_document *document )
{
	gchar str[STRLEN];
	GtkNotebookPage *page;
	glong curpage;
	gchar *l;

	if( !document )
		return;

	/*** gI_document_set_changed() is called from check_current_doc() WHEN the
	     document is changed
		if( document->changed )
			return;	
	***/

	document->changed = TRUE;

	gtk_label_get( GTK_LABEL( document->label ), &l );
	strcpy( str, l );

	if( !strchr( l, '*' ) )
	{
		if( document->filename != NULL )
		{
			g_snprintf( str, STRLEN, "%s*", file_strip_name( document->filename ) );
		}
		else
		{
			g_snprintf( str, STRLEN, "%s*", l );
		}
	}

	gtk_label_set_text( GTK_LABEL( document->label ), str );

	curpage = gtk_notebook_get_current_page( GTK_NOTEBOOK( main_window->notebook ) );

	gtk_notebook_set_page( GTK_NOTEBOOK( main_window->notebook ),
	                       gI_document_get_pageno( document ) );

	page = GTK_NOTEBOOK( main_window->notebook )->cur_page;
	gtk_label_set_text( GTK_LABEL( page->tab_label ), str );

	gtk_notebook_set_page( GTK_NOTEBOOK( main_window->notebook ),
	                       curpage );

	if( document->changed_id )
	{
		gtk_signal_disconnect( GTK_OBJECT( document->text ), document->changed_id );
		document->changed_id = 0;
	}

	gI_window_set_statusbar( document );

	/* set last mod time */
	document->last_mod = get_last_mod( document->filename );
}


glong gI_document_get_pageno( gI_document *document )
{
	return( g_list_index( main_window->documents, (gpointer) document ) - 1 );
}


static void special_mail_doc_ok( GtkWidget *widget, GtkWidget *entry )
{
	gI_document *current;
	gchar *to = NULL, *smtp = NULL, *email = NULL;
	struct sockaddr_in socks;
	struct sockaddr_in sockd;
	gint skfd;
	struct utsname uts;
	/*	struct hostent *hostent = NULL;
		gchar *hwd;*/
	gchar *buf;
	/*	gchar *user;*/
	glong psize = 0;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	to = gtk_entry_get_text( GTK_ENTRY( entry ) );

	if( !cfg->smtp || isempty( cfg->smtp ) )
		smtp = g_strdup( "127.0.0.1" );
	else
		smtp = g_strdup( cfg->smtp );

	if( !cfg->email || isempty( cfg->email ) )
		email = getenv( "USER" );
	else
		email = cfg->email;

	skfd = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
	if( skfd < 0 )
	{
		perror( "socket" );
	}

	if( uname( &uts ) < 0 )
	{
		perror( "uname" );
	}

	/***
	printf("nodename: %s\n", uts.nodename );
	#ifdef __USE_GNU
	printf("domainname: %s\n", uts.domainname );
	#else
	printf("domainname: %s\n", uts.__domainname );
	#endif

	#ifdef __USE_GNU
		hwd = g_strconcat( uts.nodename, ".", uts.domainname, NULL );
	#else
		hwd = g_strconcat( uts.nodename, ".", uts.__domainname, NULL );
	#endif	

		hostent = gethostbyname( hwd );
	#ifdef __USE_GNU
		if( !hostent && !strcmp( uts.domainname, "(none)" ) )
	#else
		if( !hostent && !strcmp( uts.__domainname, "(none)" ) )
	#endif
		{
			hostent = gethostbyname( uts.nodename );			
		}

		g_free( hwd );

		if( !hostent )
		{
			error_dialog( "\n    Unable to determe local address!    \n", "Address Lookup Error" );
			return;
		}	
	***/

	socks.sin_family = AF_INET;
	socks.sin_port = htons( 0 );
	socks.sin_addr.s_addr = inet_addr( "0.0.0.0" );
	if( bind( skfd, (struct sockaddr *) &socks, sizeof( struct sockaddr_in ) ) < 0 )
		{
			perror( "bind" );
		}

	sockd.sin_family = AF_INET;
	sockd.sin_port = htons( 25 );
	sockd.sin_addr.s_addr = inet_addr( smtp );

	g_free( smtp );

	if( connect( skfd, (struct sockaddr *) &sockd, sizeof( struct sockaddr_in ) ) < 0 )
		{
			buf = g_strconcat( _("Connect Error: "), strerror(errno), NULL );
			gI_error_dialog( buf);
			g_free( buf );
			return;
		}

	/***printf( "connected\n" );***/

	/* process gtk stuff */
	while( gtk_events_pending() || gdk_events_pending() )
		gtk_main_iteration();

	buf = g_strconcat( "HELO", " ", uts.nodename, "\n", NULL );
	write( skfd, buf, strlen( buf ) );
	g_free( buf );

	/***
	#ifdef __USE_GNU
		if( !strcmp( uts.domainname, "(none)" ) )
	#else
		if( !strcmp( uts.__domainname, "(none)" ) )
	#endif
			buf = g_strconcat( "MAIL FROM: ", user, "@", uts.nodename, "\n", NULL ); 
		else
	#ifdef __USE_GNU
			buf = g_strconcat( "MAIL FROM: ", user, "@", uts.nodename, ".", uts.domainname, "\n", NULL ); 
	#else
			buf = g_strconcat( "MAIL FROM: ", user, "@", uts.nodename, ".", uts.__domainname, "\n", NULL ); 
	#endif
	***/
	buf = g_strconcat( "MAIL FROM: ", email, "\n", NULL );

	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	buf = g_strconcat( "RCPT TO: ", to, "\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	buf = g_strconcat( "DATA\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	buf = g_strconcat( "To: ", to, "\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	if( current->filename )
		buf = g_strconcat( "Subject: ", current->filename, "\n", NULL );
	else
		buf = g_strconcat( "Subject: File\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );


	for(psize=0;psize<(gI_text_get_length( current->text ));psize++)
	{
		buf = gtk_editable_get_chars( GTK_EDITABLE( current->text ),
		                              psize,
		                              psize+1 );

		write( skfd, buf, 1 );

		g_free( buf );

		gtk_main_iteration_do( FALSE );
	}

	buf = g_strconcat( ".\n", NULL );
	write( skfd, buf, 2 );
	g_free( buf );

	gI_ok_dialog( _("Mail sent.") );
}


void special_mail_doc( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	entry_dialog( _("Send To:  "), _("Mail Document"), special_mail_doc_ok );
}


void show_manpage( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( GTK_EDITABLE( current->text )->has_selection )
	{
		gchar *mp;

		mp = gtk_editable_get_chars( GTK_EDITABLE( current->text ),
		                             GTK_EDITABLE( current->text )->selection_start_pos,
		                             GTK_EDITABLE( current->text )->selection_end_pos );

		gman( mp, NULL );

		g_free( mp );
	}
	else
	{
		gchar *mp;
		gint point, noc;
		gchar *ptr;

		/* get the next gchars, max. 50 */
		point = gI_text_get_point( current->text );
		if( point+50 > gI_text_get_length( current->text ) )
		{
			noc = gI_text_get_length( current->text ) - point;
		}
		else
		{
			noc = 50;
		}

		mp = gtk_editable_get_chars( GTK_EDITABLE( current->text ),
		                             point, point+noc  );

		ptr = mp;

		while( *ptr != ' ' && *ptr != '(' && *ptr != ';' )
			ptr++;

		*ptr = '\0';

		if( !isempty( mp ) )
		{
			gman( mp, NULL );
		}

		g_free( mp );
	}
}


void gI_document_insert_text( GtkEditable *editable,
                              const gchar *new_text,
                              gint new_text_length,
                              gint *position,
                              gI_document *document )
{
	gI_UndoItem *undo;

	undo = gI_UndoItem_new( document,
	                        new_text,
	                        new_text_length,
	                        *position,
	                        ((*position)+new_text_length),
	                        GI_UNDO_INSERT );

	gI_UndoItem_add( undo );

	menus_set_sensitive( "/Edit/Undo", TRUE );
}


void gI_document_delete_text( GtkEditable *editable,
                              gint start_pos,
                              gint end_pos,
                              gI_document *document )
{
	gI_UndoItem *undo;
	gchar *data;

	data = gtk_editable_get_chars( GTK_EDITABLE( editable ),
	                               start_pos,
	                               end_pos );

	undo = gI_UndoItem_new( document,
	                        data,
	                        strlen(data),
	                        start_pos,
	                        end_pos,
	                        GI_UNDO_DELETE );

	gI_UndoItem_add( undo );

	g_free( data );

	menus_set_sensitive( "/Edit/Undo", TRUE );
}


void gI_document_insert( gI_document *document, GdkFont *font, GdkColor *fore,
                         GdkColor *back, const char *chars, gint length )
{
	g_return_if_fail( document != NULL );

	if( document->read_only )
	{
		return;
	}

	gI_text_insert( document->text, font, fore, back,
	                chars, length );
}


void gI_document_delete_selection( gI_document *document )
{
	g_return_if_fail( document != NULL );

	if( document->read_only )
	{
		return;
	}

	gtk_editable_delete_selection( GTK_EDITABLE( document->text ) );
}


void gI_document_delete( gI_document *document, gint from, gint to )
{
	g_return_if_fail( document != NULL );

	if( document->read_only )
	{
		return;
	}

	gtk_editable_delete_text( GTK_EDITABLE( document->text ),
	                          from, to );
}

