/* $Header: /cvs/gnome/gIDE/src/gI_edit.c,v 1.5 1999/12/05 19:10:48 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gnome.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "structs.h"
#include "gI_document.h"
#include "gI_window.h"
#include "gI_edit.h"
#include "gI_menus.h"
#include "gI_common.h"

/* externs */
extern gI_window *main_window;
extern glong ro_not_change;


/*
 ---------------------------------------------------------------------
     Function: edit_undo()
     Desc: Callback-Function /Edit/Undo
 ---------------------------------------------------------------------
*/

void edit_undo( GtkWidget *widget, gpointer data )
{
	gI_UndoItem *undo;
	gint from, to;
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
	{
		menus_set_sensitive( "/Edit/Undo", FALSE );
		menus_set_sensitive( "/Edit/Redo", FALSE );
		return;
	}

	if( !current->op_list )
	{
		/* no list? set both sensitive false */
		menus_set_sensitive( "/Edit/Undo", FALSE );
		menus_set_sensitive( "/Edit/Redo", FALSE );
		return;
	}

	if( current->read_only )
	{
		return;
	}

	if( current->first_undo )
	{
		current->first_undo = 0;

		current->op_list = g_list_last( current->op_list );
	}

	undo = (gI_UndoItem *) current->op_list->data;
	if( !undo )
	{
		/* no undo item? set sensitive false */
		menus_set_sensitive( "/Edit/Undo", FALSE );
		return;
	}

	gtk_signal_disconnect( GTK_OBJECT( undo->document->text ),
	                       undo->document->insert_id );
	gtk_signal_disconnect( GTK_OBJECT( undo->document->text ),
	                       undo->document->delete_id );

	from = MIN( undo->from, undo->to );
	to = MAX( undo->from, undo->to );

	switch( undo->type )
	{
	case GI_UNDO_BEGIN:
		{
			menus_set_sensitive( "/Edit/Undo", FALSE );
			break;
		}

	case GI_UNDO_INSERT:
		{
			gI_text_set_point( undo->document->text,
			                   undo->from );
			gtk_editable_select_region( GTK_EDITABLE( undo->document->text ),
			                            from,
			                            to );
			gI_document_delete_selection( undo->document );
			gtk_editable_set_position( GTK_EDITABLE( undo->document->text ),
			                           from );
			break;
		}

	case GI_UNDO_DELETE:
		{
			gI_text_set_point( undo->document->text,
			                   undo->from );
			gI_document_insert( undo->document,
			                    NULL,
			                    NULL,
			                    NULL,
			                    undo->data,
			                    (undo->to-undo->from) );
			gtk_editable_set_position( GTK_EDITABLE( undo->document->text ),
			                           to );
			break;
		}

	default:
		{
			g_warning( "Whats going on? Unknown UNDO type!\n" );
			break;
		}
	}

	/* set (un)changed */
	if( undo->changed )
		gI_document_set_changed( undo->document );
	else
		gI_document_set_unchanged( undo->document );

	/***
		gtk_signal_emit_stop_by_name( GTK_OBJECT( undo->document->text ),
									  "insert_text" );
		gtk_signal_emit_stop_by_name( GTK_OBJECT( undo->document->text ),
									  "delete_text" );
	***/

	undo->document->insert_id = gtk_signal_connect( GTK_OBJECT( undo->document->text ), "insert_text",
	                            GTK_SIGNAL_FUNC( gI_document_insert_text ), undo->document );
	undo->document->delete_id = gtk_signal_connect( GTK_OBJECT( undo->document->text ), "delete_text",
	                            GTK_SIGNAL_FUNC( gI_document_delete_text ), undo->document );

	/* undo called, now we can allow a redo */
	menus_set_sensitive( "/Edit/Redo", TRUE );

	/* no more operations to undo, set sensitive false */
	if( current->op_list->prev == NULL )
	{
		menus_set_sensitive( "/Edit/Undo", FALSE );
		menus_set_sensitive( "/Edit/Redo", TRUE );
	}
	else
	{
		current->op_list = g_list_previous( current->op_list );
	}
}


/*
 ---------------------------------------------------------------------
     Function: edit_redo()
     Desc: Callback-Function /Edit/Redo
 ---------------------------------------------------------------------
*/

void edit_redo( GtkWidget *widget, gpointer data )
{
	gI_UndoItem *undo;
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
	{
		menus_set_sensitive( "/Edit/Undo", FALSE );
		menus_set_sensitive( "/Edit/Redo", FALSE );
		return;
	}

	if( !current->op_list )
	{
		/* no list? set both sensitive false */
		menus_set_sensitive( "/Edit/Undo", FALSE );
		menus_set_sensitive( "/Edit/Redo", FALSE );
		return;
	}

	if( current->read_only )
	{
		return;
	}

	/* no more operations to redo, set sensitive false */
	if( current->op_list->next == NULL )
	{
		menus_set_sensitive( "/Edit/Redo", FALSE );
		return;
	}
	else
	{
		current->op_list = g_list_next( current->op_list );
	}

	if( current->op_list->next == NULL )
		menus_set_sensitive( "/Edit/Redo", FALSE );


	undo = (gI_UndoItem *) current->op_list->data;
	if( !undo )
	{
		/* no undo item? set sensitive false */
		menus_set_sensitive( "/Edit/Redo", FALSE );
		return;
	}

	gtk_signal_disconnect( GTK_OBJECT( undo->document->text ),
	                       undo->document->insert_id );
	gtk_signal_disconnect( GTK_OBJECT( undo->document->text ),
	                       undo->document->delete_id );

	switch( undo->type )
	{
	case GI_UNDO_DELETE:
		{
			gI_text_set_point( undo->document->text,
			                   undo->from );
			gtk_editable_select_region( GTK_EDITABLE( undo->document->text ),
			                            undo->from,
			                            undo->to );
			gI_document_delete_selection( undo->document );
			break;
		}

	case GI_UNDO_INSERT:
		{
			gI_text_set_point( undo->document->text,
			                   undo->from );
			gI_document_insert( undo->document,
			                    NULL,
			                    NULL,
			                    NULL,
			                    undo->data,
			                    (undo->to-undo->from) );
			break;
		}

	default:
		{
			g_warning( "Whats going on? Unknown UNDO type!\n" );
			break;
		}
	}

	/* set changed */
	gI_document_set_changed( undo->document );

	/***
		gtk_signal_emit_stop_by_name( GTK_OBJECT( undo->document->text ),
									  "insert_text" );
		gtk_signal_emit_stop_by_name( GTK_OBJECT( undo->document->text ),
									  "delete_text" );
	***/

	undo->document->insert_id = gtk_signal_connect( GTK_OBJECT( undo->document->text ), "insert_text",
	                            GTK_SIGNAL_FUNC( gI_document_insert_text ), undo->document );
	undo->document->delete_id = gtk_signal_connect( GTK_OBJECT( undo->document->text ), "delete_text",
	                            GTK_SIGNAL_FUNC( gI_document_delete_text ), undo->document );
	menus_set_sensitive( "/Edit/Undo", TRUE );

}


/*
 ---------------------------------------------------------------------
     Function: edit_cut()
     Desc: Callback-Function /Edit/Cut
 ---------------------------------------------------------------------
*/
void edit_cut( GtkWidget *widget, gpointer data )
{
	gI_document *current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_cut_clipboard( GTK_EDITABLE( current->text ) );
}


/*
 ---------------------------------------------------------------------
     Function: edit_copy()
     Desc: Callback-Function /Edit/Copy
 ---------------------------------------------------------------------
*/
void edit_copy( GtkWidget *widget, gpointer data )
{
	gI_document *current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_copy_clipboard( GTK_EDITABLE( current->text ) );
}


/*
 ---------------------------------------------------------------------
     Function: edit_paste()
     Desc: Callback-Function /Edit/Paste
 ---------------------------------------------------------------------
*/
void edit_paste( GtkWidget *widget, gpointer data )
{
	gI_document *current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_paste_clipboard( GTK_EDITABLE( current->text ) );
}


/*
 ---------------------------------------------------------------------
     Function: edit_select_all()
     Desc: Callback-Function /Edit/Select All
 ---------------------------------------------------------------------
*/
void edit_select_all( GtkWidget *widget, gpointer data )
{
	gI_document *current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_select_region( GTK_EDITABLE( current->text ), 0, gI_text_get_length( current->text ) );
}


/*
 ---------------------------------------------------------------------
     Function: edit_select_line()
     Desc: Callback-Function /Edit/Select Line
 ---------------------------------------------------------------------
*/

void edit_select_line( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	gint oldpoint;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	oldpoint = gI_text_get_point( current->text );

	gtk_editable_select_region( GTK_EDITABLE( current ->text ), oldpoint,
	                            get_end_of_line( current->text, oldpoint  ) );
}


void set_read_only( gboolean state )
{
	/* set menu items and toolbar items sensitive true/false here */

	/* menu items */
	menus_set_sensitive( "/Edit/Cut", state );
	menus_set_sensitive( "/Edit/Paste", state );
	menus_set_sensitive( "/Edit/Date-Time", state );
	menus_set_sensitive( "/Edit/Delete/To begin of file", state );
	menus_set_sensitive( "/Edit/Delete/To end of file", state );
	menus_set_sensitive( "/Edit/Delete/To begin of line", state );
	menus_set_sensitive( "/Edit/Delete/To end of line", state );
	menus_set_sensitive( "/Edit/Insert/GPL-Notice (C)", state );
	menus_set_sensitive( "/Edit/Insert/GPL-Notice (C++)", state );
	menus_set_sensitive( "/Edit/Replace/TABs with Spaces", state );

	/* toolbar items */
	/* FIXME: how to do this?*/
}


/*
 ---------------------------------------------------------------------
     Function: edit_read_only()
     Desc: Callback-Function /Edit/Read Only
 ---------------------------------------------------------------------
*/

void edit_read_only( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	if( ro_not_change )
		return;

	current = gI_document_get_current( main_window );
	if( !current )
	{
		return;
	}

	if( current->read_only )
	{
		current->read_only = 0;
		gI_text_set_editable( current->text, TRUE );
	}
	else
	{
		current->read_only = 1;
		gI_text_set_editable( current->text, FALSE );
	}

	set_read_only( !current->read_only );

	gI_window_set_statusbar( current );
}


glong get_end_of_line( GtkWidget *text, glong point )
{
	static glong eol;

	g_return_val_if_fail( text != NULL, 0 );

	for(eol=point;eol<gI_text_get_length( text );eol++)
	{
		if( gtk_editable_get_chars( GTK_EDITABLE( text ), eol, eol+1 )[0] == '\n' )
			break;
	}

	if( eol < 0 )
		eol = 0;
	if( eol > gI_text_get_length( text ) )
		eol = gI_text_get_length( text );

	return( eol );
}


glong get_begin_of_line( GtkWidget *text, glong point )
{
	static glong bol;

	g_return_val_if_fail( text != NULL, 0 );

	for(bol=point;bol>0;bol--)
	{
		if( gtk_editable_get_chars( GTK_EDITABLE( text ), bol, bol-1 )[0] == '\n' )
			break;
	}

	if( bol < 0 )
		bol = 0;
	if( bol > gI_text_get_length( text ) )
		bol = gI_text_get_length( text );

	return( bol );
}


void edit_date_time( GtkWidget *widget, gpointer data )
{
	time_t t;
	struct tm *st;
	gchar *timestr;
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	time(&t);

	st = localtime(&t);

	timestr = asctime( st );

	gI_document_insert( current, NULL, NULL, NULL,
	                    timestr, strlen(timestr) );
}


void edit_delete_to_bol( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_select_region( GTK_EDITABLE( current->text ),
	                            gI_text_get_point( current->text ),
	                            get_begin_of_line( current->text,
	                                               gI_text_get_point( current->text ) ) );

	gI_document_delete_selection( current );
}


void edit_delete_to_eol( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_select_region( GTK_EDITABLE( current->text ),
	                            gI_text_get_point( current->text ),
	                            get_end_of_line( current->text,
	                                             gI_text_get_point( current->text ) ) );

	gI_document_delete_selection( current );
}


void edit_delete_to_bof( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_select_region( GTK_EDITABLE( current->text ),
	                            gI_text_get_point( current->text ),
	                            0 );

	gI_document_delete_selection( current );
}


void edit_delete_to_eof( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	gtk_editable_select_region( GTK_EDITABLE( current->text ),
	                            gI_text_get_point( current->text ),
	                            gI_text_get_length( current->text ) );

	gI_document_delete_selection( current );
}


gI_UndoItem *gI_UndoItem_new( gI_document *document,
                              const gchar *data,
                              gint data_length,
                              gint from,
                              gint to,
                              gint type )
{
	gI_UndoItem *undo;

	g_return_val_if_fail( document != NULL, NULL );
	g_return_val_if_fail( data != NULL, NULL );

	undo = g_malloc0( sizeof( gI_UndoItem ) );

	undo->document = document;
	undo->data = g_malloc0( data_length+1 );
	strncpy( undo->data, data, data_length );
	undo->data[data_length] = '\0';
	undo->from = from;
	undo->to = to;
	undo->type = type;
	undo->changed = document->changed;

	return( undo );
}


void gI_UndoItem_destroy( gI_UndoItem *undo )
{
	g_return_if_fail( undo != NULL );

	if( undo->data )
		g_free( undo->data );

	g_free( undo );
}


void gI_UndoItem_add( gI_UndoItem *undo )
{
	g_return_if_fail( undo != NULL );
	g_return_if_fail( undo->document != NULL );
	g_return_if_fail( undo->data != NULL );

	menus_set_sensitive( "/Edit/Undo", TRUE );

	if( !undo->document->op_list )
	{
		/* Create undo buffer */
		gI_UndoItem *marker =
		    gI_UndoItem_new( undo->document, "", 0,0,0, GI_UNDO_BEGIN );
		undo->document->op_list =
		    g_list_append( undo->document->op_list, marker );
	}
	else if ( g_list_next(undo->document->op_list) != NULL )
	{
		GList* node_to_remove;

		while ( ( node_to_remove =
		              g_list_next( undo->document->op_list ) )!= NULL)
		{
			g_list_remove_link( undo->document->op_list,
			                    node_to_remove );
			gI_UndoItem_destroy( node_to_remove->data );
			g_list_free( node_to_remove );
		}
	}

	g_list_append( undo->document->op_list, undo );
	undo->document->op_list = g_list_last( undo->document->op_list );
}


void edit_gpl_c( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	gchar *GPL_Notice =
	    "/* Program Name\n"
	    " * Copyright (C) 1999 Author Name\n"
	    " *\n"
	    " * This program is free software; you can redistribute it and/or modify\n"
	    " * it under the terms of the GNU General Public License as published by\n"
	    " * the Free Software Foundation; either version 2 of the License, or\n"
	    " * (at your option) any later version.\n"
	    " *\n"
	    " * This program is distributed in the hope that it will be useful,\n"
	    " * but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	    " * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	    " * GNU General Public License for more details.\n"
	    " *\n"
	    " * You should have received a copy of the GNU General Public License\n"
	    " * along with this program; if not, write to the Free Software\n"
	    " * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
	    " */\n"
	    "\n";

	current = gI_document_get_current( main_window );
	if( !current )
	{
		return;
	}

	gI_document_insert( current, NULL, NULL, NULL,
	                    GPL_Notice, strlen(GPL_Notice) );
}


void edit_gpl_cpp( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	gchar *GPL_Notice =
	    "// Program Name\n"
	    "// Copyright (C) 1999 Author Name\n"
	    "//\n"
	    "// This program is free software; you can redistribute it and/or modify\n"
	    "// it under the terms of the GNU General Public License as published by\n"
	    "// the Free Software Foundation; either version 2 of the License, or\n"
	    "// (at your option) any later version.\n"
	    "//\n"
	    "// This program is distributed in the hope that it will be useful,\n"
	    "// but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	    "// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	    "// GNU General Public License for more details.\n"
	    "//\n"
	    "// You should have received a copy of the GNU General Public License\n"
	    "// along with this program; if not, write to the Free Software\n"
	    "// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
	    "\n";

	current = gI_document_get_current( main_window );
	if( !current )
	{
		return;
	}

	gI_document_insert( current, NULL, NULL, NULL,
	                    GPL_Notice, strlen(GPL_Notice) );
}



void edit_repl_tab_spaces( GtkWidget *widget, gpointer data )
{
	GtkWidget *dlg;
	gI_document *current;
	gint spaces;

	void _edit_repl_tab_spaces ( gchar * str, gpointer data)
	{
		gint i,j;
		gchar *c;

		if( !str || isempty( str )) {
			return;
		}

		spaces = atol( str );
		g_print( str );

		for(i=1; i<=(gI_text_get_length( current->text )); i++)
		{
			c = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
			if( *c == '\t' )
			{
				gtk_editable_delete_text( GTK_EDITABLE( current->text ), i-1, i );
				gtk_editable_set_position( GTK_EDITABLE( current->text ), i-1 );

				for(j=0; j<spaces; j++)
				{
					gI_document_insert( current, NULL, NULL, NULL, " ", 1 );
				}
				i += (spaces - 1);
			}
			g_free( c );
		}
	};


	current = gI_document_get_current( main_window );
	if( !current ) {
		return;
	}

	spaces = gI_text_default_tab_width( current->text );
	dlg = gnome_request_dialog( FALSE,
	                            _("Number of Spaces per TAB ?"),
	                            "4",   /* FIXME: strToInt( spaces), buy a C book ? */
	                            3,     /* max. of chars */
	                            _edit_repl_tab_spaces,
	                            NULL,  /* data */
	                            NULL); /* parent */

	gnome_dialog_run_and_close( GNOME_DIALOG( dlg ) );
}

