------------------------------------------------------------------------------
--                                                                          --
--                GNU ADA RUN-TIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                   S Y S T E M . O S _ I N T E R F A C E                  --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.3 $                             --
--                                                                          --
--          Copyright (C) 1992-1999, Free Software Foundation, Inc.         --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is the SGI Pthreads version of this package.

--  This package encapsulates all direct interfaces to OS services
--  that are needed by children of System.

--  PLEASE DO NOT add any with-clauses to this package
--  or remove the pragma Elaborate_Body.
--  It is designed to be a bottom-level (leaf) package.

with Interfaces.C;
package System.OS_Interface is

   pragma Preelaborate;

   pragma Linker_Options ("-lPtasking");
   pragma Linker_Options ("-lpthread");

   subtype int            is Interfaces.C.int;
   subtype short          is Interfaces.C.short;
   subtype long           is Interfaces.C.long;
   subtype unsigned       is Interfaces.C.unsigned;
   subtype unsigned_short is Interfaces.C.unsigned_short;
   subtype unsigned_long  is Interfaces.C.unsigned_long;
   subtype unsigned_char  is Interfaces.C.unsigned_char;
   subtype plain_char     is Interfaces.C.plain_char;
   subtype size_t         is Interfaces.C.size_t;

   -----------
   -- Errno --
   -----------

   function errno return int;
   pragma Import (C, errno, "__get_errno");

   EPERM        : constant := 1;  --   Not super-user
   ENOENT       : constant := 2;  --   No such file or directory
   ESRCH        : constant := 3;  --   No such process
   EINTR        : constant := 4;  --   interrupted system call
   EIO          : constant := 5;  --   I/O error
   ENXIO        : constant := 6;  --   No such device or address
   E2BIG        : constant := 7;  --   Arg list too long
   ENOEXEC      : constant := 8;  --   Exec format error
   EBADF        : constant := 9;  --   Bad file number
   ECHILD       : constant := 10; --   No children
   EAGAIN       : constant := 11; --   No more processes
   ENOMEM       : constant := 12; --   Not enough core
   EACCES       : constant := 13; --   Permission denied
   EFAULT       : constant := 14; --   Bad address
   ENOTBLK      : constant := 15; --   Block device required
   EBUSY        : constant := 16; --   Mount device busy
   EEXIST       : constant := 17; --   File exists
   EXDEV        : constant := 18; --   Cross-device link
   ENODEV       : constant := 19; --   No such device
   ENOTDIR      : constant := 20; --   Not a directory
   EISDIR       : constant := 21; --   Is a directory
   EINVAL       : constant := 22; --   Invalid argument
   ENFILE       : constant := 23; --   File table overflow
   EMFILE       : constant := 24; --   Too many open files
   ENOTTY       : constant := 25; --   Not a typewriter
   ETXTBSY      : constant := 26; --   Text file busy
   EFBIG        : constant := 27; --   File too large
   ENOSPC       : constant := 28; --   No space left on device
   ESPIPE       : constant := 29; --   Illegal seek
   EROFS        : constant := 30; --   Read only file system
   EMLINK       : constant := 31; --   Too many links
   EPIPE        : constant := 32; --   Broken pipe
   EDOM         : constant := 33; --   Math arg out of domain of func
   ERANGE       : constant := 34; --   Math result not representable
   ENOMSG       : constant := 35; --   No message of desired type
   EIDRM        : constant := 36; --   Identifier removed
   ECHRNG       : constant := 37; --   Channel number out of range
   EL2NSYNC     : constant := 38; --   Level 2 not synchronized
   EL3HLT       : constant := 39; --   Level 3 halted
   EL3RST       : constant := 40; --   Level 3 reset
   ELNRNG       : constant := 41; --   Link number out of range
   EUNATCH      : constant := 42; --   Protocol driver not attached
   ENOCSI       : constant := 43; --   No CSI structure available
   EL2HLT       : constant := 44; --   Level 2 halted
   EDEADLK      : constant := 45; --   Deadlock condition.
   ENOLCK       : constant := 46; --   No record locks available.

   --  Convergent Error Returns

   EBADE        : constant := 50; --   invalid exchange
   EBADR        : constant := 51; --   invalid request descriptor
   EXFULL       : constant := 52; --   exchange full
   ENOANO       : constant := 53; --   no anode
   EBADRQC      : constant := 54; --   invalid request code
   EBADSLT      : constant := 55; --   invalid slot
   EDEADLOCK    : constant := 56; --   file locking deadlock error

   EBFONT       : constant := 57; --   bad font file fmt

   --  stream problems

   ENOSTR       : constant := 60; --   Device not a stream
   ENODATA      : constant := 61; --   no data (for no delay io)
   ETIME        : constant := 62; --   timer expired
   ENOSR        : constant := 63; --   out of streams resources

   ENONET       : constant := 64; --   Machine is not on the network
   ENOPKG       : constant := 65; --   Package not installed
   EREMOTE      : constant := 66; --   The object is remote
   ENOLINK      : constant := 67; --   the link has been severed
   EADV         : constant := 68; --   advertise error
   ESRMNT       : constant := 69; --   srmount error

   ECOMM        : constant := 70; --   Communication error on send
   EPROTO       : constant := 71; --   Protocol error
   EMULTIHOP    : constant := 74; --   multihop attempted
   EBADMSG      : constant := 77; --   trying to read unreadable message
   ENAMETOOLONG : constant := 78; --   path name is too long
   EOVERFLOW    : constant := 79; --   value too large to be stored in data
   --                                  type
   ENOTUNIQ     : constant := 80; --   given log. name not unique
   EBADFD       : constant := 81; --   f.d. invalid for this operation
   EREMCHG      : constant := 82; --   Remote address changed

   --  shared library problems

   ELIBACC         : constant := 83; --   Can't access a needed shared lib.
   ELIBBAD         : constant := 84; --   Accessing a corrupted shared lib.
   ELIBSCN         : constant := 85; --   .lib section in a.out corrupted.
   ELIBMAX         : constant := 86; --   Attempting to link in too many libs.
   ELIBEXEC        : constant := 87; --   Attempting to exec a shared library.
   EILSEQ          : constant := 88; --   Illegal byte sequence.
   ENOSYS          : constant := 89; --   Unsupported file system operation
   ELOOP           : constant := 90; --   Symbolic link loop
   ERESTART        : constant := 91; --   Restartable system call
   ESTRPIPE        : constant := 92; --   if pipe/FIFO, don't sleep in stream
   --                                     head

   ENOTEMPTY       : constant := 93; --   directory not empty

   EUSERS          : constant := 94; --   Too many users (for UFS)

   --  BSD Networking Software
   --  argument errors

   ENOTSOCK        : constant := 95; --   Socket operation on non-socket
   EDESTADDRREQ    : constant := 96; --   Destination address required
   EMSGSIZE        : constant := 97; --   Message too long
   EPROTOTYPE      : constant := 98; --   Protocol wrong type for socket
   ENOPROTOOPT     : constant := 99; --   Protocol not available
   EPROTONOSUPPORT : constant := 120; --  Protocol not supported
   ESOCKTNOSUPPORT : constant := 121; --  Socket type not supported
   EOPNOTSUPP      : constant := 122; --  Operation not supported on socket
   EPFNOSUPPORT    : constant := 123; --  Protocol family not supported
   EAFNOSUPPORT    : constant := 124; --  Address family not supported by
   --                                     protocol family

   EADDRINUSE      : constant := 125; --   Address already in use
   EADDRNOTAVAIL   : constant := 126; --   Can't assign requested address

   --  operational errors

   ENETDOWN        : constant := 127; --   Network is down
   ENETUNREACH     : constant := 128; --   Network is unreachable
   ENETRESET       : constant := 129; --   Network dropped connection because
   --                                      of reset

   ECONNABORTED    : constant := 130; --   Software caused connection abort
   ECONNRESET      : constant := 131; --   Connection reset by peer
   ENOBUFS         : constant := 132; --   No buffer space available
   EISCONN         : constant := 133; --   Socket is already connected
   ENOTCONN        : constant := 134; --   Socket is not connected

   --  XENIX has 135 - 142

   ESHUTDOWN       : constant := 143; --   Can't send after socket shutdown
   ETOOMANYREFS    : constant := 144; --   Too many references: can't splice
   ETIMEDOUT       : constant := 145; --   Connection timed out
   ECONNREFUSED    : constant := 146; --   Connection refused
   EHOSTDOWN       : constant := 147; --   Host is down
   EHOSTUNREACH    : constant := 148; --   No route to host

   --  Operation would block

   EWOULDBLOCK     : constant := 11;

   EALREADY        : constant := 149; --   operation already in progress
   EINPROGRESS     : constant := 150; --   operation now in progress

   --  SUN Network File System

   ESTALE          : constant := 151; --   Stale NFS file handle

   --  Pyramid's AIO Compatibility - raw disk async I/O

   EIORESID   : constant := 500; --   block not fully transferred

   --  XENIX error numbers

   EUCLEAN    : constant := 135; --   Structure needs cleaning
   ENOTNAM    : constant := 137; --   Not a XENIX named type file
   ENAVAIL    : constant := 138; --   No XENIX semaphores available
   EISNAM     : constant := 139; --   Is a named type file
   EREMOTEIO  : constant := 140; --   Remote I/O error
   EINIT      : constant := 141; --   Reserved for future
   EREMDEV    : constant := 142; --   Error 142

   -------------
   -- Signals --
   -------------

   NSIG : constant := 65;
   type Signal is new int range 0 .. Interfaces.C."-" (NSIG, 1);
   for Signal'Size use int'Size;

   SIGHUP     : constant := 1; --  hangup
   SIGINT     : constant := 2; --  interrupt (rubout)
   SIGQUIT    : constant := 3; --  quit (ASCD FS)
   SIGILL     : constant := 4; --  illegal instruction (not reset)
   SIGTRAP    : constant := 5; --  trace trap (not reset)
   SIGIOT     : constant := 6; --  IOT instruction
   SIGABRT    : constant := 6; --  used by abort, replace SIGIOT in the
   --                              future
   SIGEMT     : constant := 7; --  EMT instruction
   SIGFPE     : constant := 8; --  floating point exception
   SIGKILL    : constant := 9; --  kill (cannot be caught or ignored)
   SIGBUS     : constant := 10; --  bus error
   SIGSEGV    : constant := 11; --  segmentation violation
   SIGSYS     : constant := 12; --  bad argument to system call
   SIGPIPE    : constant := 13; --  write on a pipe with no one to read it
   SIGALRM    : constant := 14; --  alarm clock
   SIGTERM    : constant := 15; --  software termination signal from kill
   SIGUSR1    : constant := 16; --  user defined signal 1
   SIGUSR2    : constant := 17; --  user defined signal 2
   SIGCLD     : constant := 18; --  alias for SIGCHLD
   SIGCHLD    : constant := 18; --  child status change
   SIGPWR     : constant := 19; --  power-fail restart
   SIGWINCH   : constant := 20; --  window size change
   SIGURG     : constant := 21; --  urgent condition on IO channel
   SIGPOLL    : constant := 22; --  pollable event occurred
   SIGIO      : constant := 22; --  I/O possible (Solaris SIGPOLL alias)
   SIGSTOP    : constant := 23; --  stop (cannot be caught or ignored)
   SIGTSTP    : constant := 24; --  user stop requested from tty
   SIGCONT    : constant := 25; --  stopped process has been continued
   SIGTTIN    : constant := 26; --  background tty read attempted
   SIGTTOU    : constant := 27; --  background tty write attempted
   SIGVTALRM  : constant := 28; --  virtual timer expired
   SIGPROF    : constant := 29; --  profiling timer expired
   SIGXCPU    : constant := 30; --  CPU time limit exceeded
   SIGXFSZ    : constant := 31; --  filesize limit exceeded
   SIGK32     : constant := 32; --  reserved for kernel (IRIX)
   SIGCKPT    : constant := 33; --  Checkpoint warning
   SIGRESTART : constant := 34; --  Restart warning
   SIGUME     : constant := 35; --  Uncorrectable memory error
   --  Signals defined for Posix 1003.1c.
   SIGPTINTR    : constant := 47;
   SIGPTRESCHED : constant := 48;
   --  Posix 1003.1b signals
   SIGRTMIN   : constant := 49; --  Posix 1003.1b signals
   SIGRTMAX   : constant := 64; --  Posix 1003.1b signals

   type sigset_t is private;
   type sigset_t_ptr is access all sigset_t;

   function sigaddset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigaddset, "sigaddset");

   function sigdelset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigdelset, "sigdelset");

   function sigfillset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigfillset, "sigfillset");

   function sigismember
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigismember, "sigismember");

   function sigemptyset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigemptyset, "sigemptyset");

   type siginfo_t is record
      si_signo     : int;
      si_code      : int;
      si_errno     : int;
      bit_field_substitute_1 : string (1 .. 116);
   end record;
   pragma Convention (C, siginfo_t);

   --  The types mcontext_t and gregset_t are part of the ucontext_t
   --  information, which is specific to Solaris2.4 for SPARC
   --  The ucontext_t info seems to be used by the handler
   --  for SIGSEGV to decide whether it is a Storage_Error (stack overflow) or
   --  a Constraint_Error (bad pointer). The original code that did this
   --  is suspect, so it is not clear whether we really need this part of
   --  the signal context information, or perhaps something else.
   --  More analysis is needed, after which these declarations may need to
   --  be changed.

   type reg_array is array (integer range 0 .. 31) of Interfaces.Unsigned_64;

   type struct_sigcontext is record
      sc_regmask   : unsigned;
      sc_status    : unsigned;
      sc_pc        : Interfaces.Unsigned_64;
      sc_regs      : reg_array;
      sc_fpregs    : reg_array;
      sc_ownedfp   : unsigned;
      sc_fpc_csr   : unsigned;
      sc_fpc_eir   : unsigned;
      sc_ssflags   : unsigned;
      sc_mdhi      : Interfaces.Unsigned_64;
      sc_mdlo      : Interfaces.Unsigned_64;
      sc_cause     : Interfaces.Unsigned_64;
      sc_badvaddr  : Interfaces.Unsigned_64;
      sc_triggersave  : Interfaces.Unsigned_64;
      sc_sigset    : sigset_t;
      sc_pad       : reg_array;
   end record;
   pragma Convention (C, struct_sigcontext);

   type array_type_2 is array (integer range 0 .. 1) of int;
   type struct_sigaction is record
      sa_flags     : int;
      sa_handler   : System.Address;
      sa_mask      : sigset_t;
      sa_resv      : array_type_2;
   end record;
   pragma Convention (C, struct_sigaction);

   type struct_sigaction_ptr is access all struct_sigaction;

   SIG_BLOCK   : constant := 1;
   SIG_UNBLOCK : constant := 2;
   SIG_SETMASK : constant := 3;

   --  SA_NOCLDSTOP : constant := 131072;
   --  not used
   --  SA_SIGINFO : constant := 8;
   --  not used

   SIG_DFL : constant := 0;
   SIG_IGN : constant := 1;
   --  SIG_ERR : constant := -1;
   --  not used

   function sigaction
     (sig  : Signal;
      act  : struct_sigaction_ptr;
      oact : struct_sigaction_ptr := null)
     return int;
   pragma Import (C, sigaction, "sigaction");

   ----------
   -- Time --
   ----------

   type timespec is private;
   type timespec_ptr is access all timespec;

   function nanosleep
     (rqtp : access timespec;
      rmtp : timespec_ptr := null) return int;
   pragma Import (C, nanosleep, "nanosleep");

   type clockid_t is private;

   CLOCK_REALTIME  : constant clockid_t;
   CLOCK_SGI_FAST  : constant clockid_t;
   CLOCK_SGI_CYCLE : constant clockid_t;

   SGI_CYCLECNTR_SIZE : constant := 165;

   function syssgi (request : Interfaces.C.int) return Interfaces.C.ptrdiff_t;
   pragma Import (C, syssgi, "syssgi");

   function clock_gettime
     (clock_id : clockid_t;
      tp       : access timespec)
     return int;
   pragma Import (C, clock_gettime, "clock_gettime");

   function clock_getres
     (clock_id : clockid_t;
      tp       : access timespec)
     return int;
   pragma Import (C, clock_getres, "clock_getres");

   function To_Duration (TS : timespec) return Duration;
   pragma Inline (To_Duration);

   function To_Timespec (D : Duration) return timespec;
   pragma Inline (To_Timespec);

   type struct_timezone is record
      tz_minuteswest  : int;
      tz_dsttime   : int;
   end record;
   pragma Convention (C, struct_timezone);
   type struct_timezone_ptr is access struct_timezone;

   type struct_timeval is private;
   --  This is needed on systems that do not have clock_gettime ()
   --  but do have gettimeofday ().

   function To_Duration (TV : struct_timeval) return Duration;
   pragma Inline (To_Duration);

   function To_Timeval (D : Duration) return struct_timeval;
   pragma Inline (To_Timeval);

   function gettimeofday
     (tv : access struct_timeval;
      tz : struct_timezone_ptr := null) return int;
   pragma Import (C, gettimeofday, "gettimeofday");

   procedure usleep (useconds : unsigned_long);
   pragma Import (C, usleep, "usleep");

   --  add a hook to locate the Epoch, for use with Calendar????

   -------------------------
   -- Priority Scheduling --
   -------------------------

   MIN_PRIO : constant := 0;
   MAX_PRIO : constant := 255;

   SCHED_FIFO  : constant := 1;
   SCHED_RR    : constant := 2;
   SCHED_TS    : constant := 3;
   SCHED_OTHER : constant := 3;
   SCHED_NP    : constant := 4;

   function sched_get_priority_min (Policy : int) return int;
   pragma Import (C, sched_get_priority_min, "sched_get_priority_min");

   function sched_get_priority_max (Policy : int) return int;
   pragma Import (C, sched_get_priority_max, "sched_get_priority_max");

   -------------
   -- Process --
   -------------

   type pid_t is private;

   function kill (pid : pid_t; sig : Signal) return int;
   pragma Import (C, kill, "kill");

   function getpid return pid_t;
   pragma Import (C, getpid, "getpid");

   -------------
   -- Threads --
   -------------

   type Thread_Body is access
     function (arg : System.Address) return System.Address;
   type pthread_t           is private;
   subtype Thread_Id        is pthread_t;

   type pthread_mutex_t     is limited private;
   type pthread_cond_t      is limited private;
   type pthread_attr_t      is limited private;
   type pthread_mutexattr_t is limited private;
   type pthread_condattr_t  is limited private;
   type pthread_key_t       is private;

   PTHREAD_CREATE_DETACHED : constant := 1;
   PTHREAD_CREATE_JOINABLE : constant := 0;

   ---------------------------
   --  POSIX.1c  Section 3  --
   ---------------------------

   function sigwait
     (set : access sigset_t;
      sig : access Signal)
     return int;
   pragma Import (C, sigwait, "sigwait");

   function pthread_kill
     (thread : pthread_t;
      sig    : Signal)
     return int;
   pragma Import (C, pthread_kill, "pthread_kill");

   function pthread_sigmask
     (how  : int;
      set  : sigset_t_ptr;
      oset : sigset_t_ptr)
     return int;
   pragma Import (C, pthread_sigmask, "pthread_sigmask");

   ----------------------------
   --  POSIX.1c  Section 11  --
   ----------------------------

   function pthread_mutexattr_init (attr : access pthread_mutexattr_t)
     return int;
   pragma Import (C, pthread_mutexattr_init, "pthread_mutexattr_init");

   function pthread_mutexattr_destroy (attr : access pthread_mutexattr_t)
     return int;
   pragma Import (C, pthread_mutexattr_destroy, "pthread_mutexattr_destroy");

   function pthread_mutex_init
     (mutex : access pthread_mutex_t;
      attr  : access pthread_mutexattr_t)
     return int;
   pragma Import (C, pthread_mutex_init, "pthread_mutex_init");

   function pthread_mutex_destroy (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_destroy, "pthread_mutex_destroy");

   function pthread_mutex_lock (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_lock, "pthread_mutex_lock");

   function pthread_mutex_trylock (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_trylock, "pthread_mutex_trylock");

   function pthread_mutex_unlock (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_unlock, "pthread_mutex_unlock");

   function pthread_condattr_init (attr : access pthread_condattr_t)
     return int;
   pragma Import (C, pthread_condattr_init, "pthread_condattr_init");

   function pthread_condattr_destroy (attr : access pthread_condattr_t)
     return int;
   pragma Import (C, pthread_condattr_destroy, "pthread_condattr_destroy");

   function pthread_cond_init
     (cond : access pthread_cond_t;
      attr : access pthread_condattr_t)
     return int;
   pragma Import (C, pthread_cond_init, "pthread_cond_init");

   function pthread_cond_destroy (cond : access pthread_cond_t) return int;
   pragma Import (C, pthread_cond_destroy, "pthread_cond_destroy");

   function pthread_cond_signal (cond : access pthread_cond_t) return int;
   pragma Import (C, pthread_cond_signal, "pthread_cond_signal");

   function pthread_cond_wait
     (cond  : access pthread_cond_t;
      mutex : access pthread_mutex_t)
     return int;
   pragma Import (C, pthread_cond_wait, "pthread_cond_wait");

   function pthread_cond_timedwait
     (cond    : access pthread_cond_t;
      mutex   : access pthread_mutex_t;
      abstime : access timespec)
     return int;
   pragma Import (C, pthread_cond_timedwait, "pthread_cond_timedwait");

   ----------------------------
   --  POSIX.1c  Section 13  --
   ----------------------------

   PTHREAD_PRIO_NONE    : constant := 0;
   PTHREAD_PRIO_PROTECT : constant := 2;
   PTHREAD_PRIO_INHERIT : constant := 1;

   function pthread_mutexattr_setprotocol
     (attr     : access pthread_mutexattr_t;
      protocol : int) return int;
   pragma Import
      (C, pthread_mutexattr_setprotocol, "pthread_mutexattr_setprotocol");

   function pthread_mutexattr_getprotocol
     (attr     : access pthread_mutexattr_t;
      protocol : access int) return int;
   pragma Import
     (C, pthread_mutexattr_getprotocol, "pthread_mutexattr_getprotocol");

   function pthread_mutexattr_setprioceiling
     (attr     : access pthread_mutexattr_t;
      prioceiling : int) return int;
   pragma Import
     (C, pthread_mutexattr_setprioceiling, "pthread_mutexattr_setprioceiling");

   function pthread_mutexattr_getprioceiling
     (attr        : access pthread_mutexattr_t;
      prioceiling : access int) return int;
   pragma Import (C, pthread_mutexattr_getprioceiling,
     "pthread_mutexattr_getprio_ceiling");

   type struct_sched_param is record
      sched_priority  : int;
   end record;
   pragma Convention (C, struct_sched_param);

   function pthread_getschedparam
     (thread : pthread_t;
      policy : int;
      param  : access struct_sched_param)
     return int;
   pragma Import (C, pthread_getschedparam, "pthread_getschedparam");

   function pthread_setschedparam
     (thread : pthread_t;
      policy : int;
      param  : access struct_sched_param)
     return int;
   pragma Import (C, pthread_setschedparam, "pthread_setschedparam");

   function pthread_attr_setscope
     (attr            : access pthread_attr_t;
      contentionscope : int)
     return int;
   pragma Import (C, pthread_attr_setscope, "pthread_attr_setscope");

   function pthread_attr_getscope
     (attr            : access pthread_attr_t;
      contentionscope : access int)
     return int;
   pragma Import (C, pthread_attr_getscope, "pthread_attr_getscope");

   function pthread_attr_setinheritsched
     (attr         : access pthread_attr_t;
      inheritsched : int)
     return int;
   pragma Import
     (C, pthread_attr_setinheritsched, "pthread_attr_setinheritsched");

   function pthread_attr_getinheritsched
     (attr         : access pthread_attr_t;
      inheritsched : access int)
     return int;
   pragma Import
     (C, pthread_attr_getinheritsched, "pthread_attr_getinheritsched");

   function pthread_attr_setschedpolicy
     (attr   : access pthread_attr_t;
      policy : int)
     return int;
   pragma Import
    (C, pthread_attr_setschedpolicy, "pthread_attr_setschedpolicy");

   function pthread_attr_getschedpolicy
     (attr   : access pthread_attr_t;
      policy : access int)
     return int;
   pragma Import
     (C, pthread_attr_getschedpolicy, "pthread_attr_getschedpolicy");

   function pthread_attr_setschedparam
     (attr        : access pthread_attr_t;
      sched_param : access struct_sched_param)
     return int;
   pragma Import (C, pthread_attr_setschedparam, "pthread_attr_setschedparam");

   function pthread_attr_getschedparam
     (attr        : access pthread_attr_t;
      sched_param : access struct_sched_param)
     return int;
   pragma Import (C, pthread_attr_getschedparam, "pthread_attr_getschedparam");

   function sched_yield return int;
   pragma Import (C, sched_yield, "sched_yield");

   -----------------------------
   --  P1003.1c - Section 16  --
   -----------------------------

   function pthread_attr_init
     (attributes : access pthread_attr_t)
     return int;
   pragma Import (C, pthread_attr_init, "pthread_attr_init");

   function pthread_attr_destroy
     (attributes : access pthread_attr_t)
     return int;
   pragma Import (C, pthread_attr_destroy, "pthread_attr_destroy");

   function pthread_attr_setstacksize
     (attr      : access pthread_attr_t;
      stacksize : size_t)
     return int;
   pragma Import (C, pthread_attr_setstacksize, "pthread_attr_setstacksize");

   function pthread_attr_getstacksize
     (attr      : access pthread_attr_t;
      stacksize : access size_t)
     return int;
   pragma Import (C, pthread_attr_getstacksize, "pthread_attr_getstacksize");

   function pthread_attr_setstackaddr
     (attr      : access pthread_attr_t;
      stackaddr : size_t)
     return int;
   pragma Import (C, pthread_attr_setstackaddr, "pthread_attr_setstackaddr");

   function pthread_attr_getstackaddr
     (attr      : access pthread_attr_t;
      stackaddr : access size_t)
     return int;
   pragma Import (C, pthread_attr_getstackaddr, "pthread_attr_getstackaddr");

   function pthread_attr_setdetachstate
     (attr        : access pthread_attr_t;
      detachstate : int)
     return int;
   pragma Import
     (C, pthread_attr_setdetachstate, "pthread_attr_setdetachstate");

   function pthread_attr_getdetachstate
     (attr        : access pthread_attr_t;
      detachstate : access int)
     return int;
   pragma Import
     (C, pthread_attr_getdetachstate, "pthread_attr_getdetachstate");

   function pthread_create
     (thread        : access pthread_t;
      attributes    : access pthread_attr_t;
      start_routine : Thread_Body;
      arg           : System.Address)
     return int;
   pragma Import (C, pthread_create, "pthread_create");

   function pthread_detach (thread : pthread_t) return int;
   pragma Import (C, pthread_detach, "pthread_detach");

   procedure pthread_exit (status : System.Address);
   pragma Import (C, pthread_exit, "pthread_exit");

   function pthread_self return pthread_t;
   pragma Import (C, pthread_self, "pthread_self");

   function pthread_equal (t1 : pthread_t; t2 : pthread_t)
      return int;
   --  be careful not to use "=" on thread_t!
   pragma Import (C, pthread_equal, "pthread_equal");

   ----------------------------
   --  POSIX.1c  Section 17  --
   ----------------------------

   function pthread_setspecific
     (key   : pthread_key_t;
      value : System.Address)
     return int;
   pragma Import (C, pthread_setspecific, "pthread_setspecific");

   function pthread_getspecific (key : pthread_key_t) return System.Address;
   pragma Import (C, pthread_getspecific, "pthread_getspecific");

   type destructor_pointer is access
      procedure (arg : System.Address);

   function pthread_key_create
     (key        : access pthread_key_t;
      destructor : destructor_pointer)
     return int;
   pragma Import (C, pthread_key_create, "pthread_key_create");

   procedure pthread_init;

private

   type array_type_1 is array (integer range 0 .. 3) of unsigned;
   type sigset_t is record
      X_X_sigbits : array_type_1;
   end record;
   pragma Convention (C, sigset_t);

   type pid_t is new long;

   type time_t is new long;

   type timespec is record
      tv_sec  : time_t;
      tv_nsec : long;
   end record;
   pragma Convention (C, timespec);

   type clockid_t is new int;
   CLOCK_REALTIME  : constant clockid_t := 1;
   CLOCK_SGI_CYCLE : constant clockid_t := 2;
   CLOCK_SGI_FAST  : constant clockid_t := 3;

   type struct_timeval is record
      tv_sec  : long;
      tv_usec : long;
   end record;
   pragma Convention (C, struct_timeval);

   type array_type_9 is array (integer range 0 .. 4) of long;
   type pthread_attr_t is record
      X_X_D : array_type_9;
   end record;
   pragma Convention (C, pthread_attr_t);

   type array_type_8 is array (integer range 0 .. 1) of long;
   type pthread_condattr_t is record
      X_X_D : array_type_8;
   end record;
   pragma Convention (C, pthread_condattr_t);

   type array_type_7 is array (integer range 0 .. 1) of long;
   type pthread_mutexattr_t is record
      X_X_D : array_type_7;
   end record;
   pragma Convention (C, pthread_mutexattr_t);

   type pthread_t is new unsigned;

   type array_type_10 is array (integer range 0 .. 7) of long;
   type pthread_mutex_t is record
      X_X_D : array_type_10;
   end record;
   pragma Convention (C, pthread_mutex_t);

   type array_type_11 is array (integer range 0 .. 7) of long;
   type pthread_cond_t is record
      X_X_D : array_type_11;
   end record;
   pragma Convention (C, pthread_cond_t);

   type pthread_key_t is new int;

end System.OS_Interface;
