------------------------------------------------------------------------------
--                                                                          --
--                GNU ADA RUN-TIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--           S Y S T E M . I N T E R R U P T _ M A N A G E M E N T          --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.16 $
--                                                                          --
--             Copyright (C) 1991-1999 Florida State University             --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is a HP-UX version of this package.

--  PLEASE DO NOT add any dependences on other packages.
--  This package is designed to work with or without tasking support.

--  Make a careful study of all signals available under the OS,
--  to see which need to be reserved, kept always unmasked,
--  or kept always unmasked.
--  Be on the lookout for special signals that
--  may be used by the thread library.

with Interfaces.C;
--  used for int

with System.OS_Interface;
--  used for various Constants, Signal and types

package body System.Interrupt_Management is

   use Interfaces.C;
   use System.OS_Interface;

   type Interrupt_List is array (Interrupt_ID range <>) of Interrupt_ID;
   Exception_Interrupts : constant Interrupt_List :=
     (SIGFPE, SIGILL, SIGSEGV, SIGBUS);

   Unreserve_All_Interrupts : Interfaces.C.int;
   pragma Import
     (C, Unreserve_All_Interrupts, "__gl_unreserve_all_interrupts");

   ----------------------
   -- Notify_Exception --
   ----------------------

   --  This function identifies the Ada exception to be raised using
   --  the information when the system received a synchronous signal.
   --  Since this function is machine and OS dependent, different code
   --  has to be provided for different target.

   --  Following code is intended for Solaris on Sparcstation.

   procedure Notify_Exception
     (signo   : Signal;
      info    : access siginfo_t;
      context : access ucontext_t);

   procedure Notify_Exception
     (signo   : Signal;
      info    : access siginfo_t;
      context : access ucontext_t)
   is
   begin

      --  As long as we are using a longjmp to return control to the
      --  exception handler on the runtime stack, we are safe. The original
      --  signal mask (the one we had before coming into this signal catching
      --  function) will be restored by the longjmp. Therefore, raising
      --  an exception in this handler should be a safe operation.

      --  Check that treatment of exception propagation here
      --  is consistent with treatment of the abort signal in
      --  System.Task_Primitives.Operations.

      case signo is
         when SIGFPE =>
            case info.si_code is
               when  FPE_INTDIV |
                     FPE_INTOVF |
                     FPE_FLTDIV |
                     FPE_FLTOVF |
                     FPE_FLTUND |
                     FPE_FLTRES |
                     FPE_FLTINV |
                     FPE_FLTSUB =>
               raise Constraint_Error;
               when others =>
                  pragma Assert (False);
                  null;
            end case;
         when SIGILL =>
            case info.si_code is
               when  ILL_ILLOPC |
                     ILL_ILLOPN |
                     ILL_ILLADR |
                     ILL_ILLTRP |
                     ILL_PRVOPC |
                     ILL_PRVREG |
                     ILL_COPROC |
                     ILL_BADSTK =>
               raise Constraint_Error;
               when others =>
                  pragma Assert (False);
                  null;
            end case;
         when SIGSEGV =>
         --  If the address that caused the error was in the first page, this
         --  was caused by accessing a null pointer.
            raise Storage_Error;
         when others =>
            pragma Assert (False);
            null;
      end case;
   end Notify_Exception;

   ---------------------------
   -- Initialize_Interrupts --
   ---------------------------

   --  Nothing needs to be done on this platform.

   procedure Initialize_Interrupts is
   begin
      null;
   end Initialize_Interrupts;

begin
   declare
      act     : aliased struct_sigaction;
      old_act : aliased struct_sigaction;
      mask    : aliased sigset_t;
      Result  : Interfaces.C.int;

   begin

      Abort_Task_Interrupt := SIGTERM;
      --  Note: on other targets, we usually use SIGABRT, but on HP/UX, it
      --  appears that SIGABRT can't be used in sigwait(), so we use SIGTERM.

      act.sa_handler := Notify_Exception'Address;

      act.sa_flags := 16#00000020#;
      --  Set sa_flags to SA_NODEFER so that during the handler execution
      --  we do not change the Signal_Mask to be masked for the Signal.
      --  This is a temporary fix to the problem that the Signal_Mask is
      --  not restored after the exception (longjmp) from the handler.
      --  The right fix should be made in sigsetjmp so that we save
      --  the Signal_Set and restore it after a longjmp.
      --  In that case, this field should be changed back to 0. ???

      Result := sigemptyset (mask'Access);
      pragma Assert (Result = 0);

      for I in Exception_Interrupts'Range loop
         Result := sigaddset (mask'Access, Signal (Exception_Interrupts (I)));
         pragma Assert (Result = 0);
      end loop;

      act.sa_mask := mask;

      for I in Exception_Interrupts'Range loop
         Keep_Unmasked (Exception_Interrupts (I)) := True;
         Result :=
           sigaction
             (Signal (Exception_Interrupts (I)), act'Unchecked_Access,
              old_act'Unchecked_Access);
         pragma Assert (Result = 0);
      end loop;

      Keep_Unmasked (Abort_Task_Interrupt) := True;

      Keep_Unmasked (SIGABRT) := True;
      Keep_Unmasked (SIGPIPE) := True;
      --  SIGABRT & SIGPIPE can't be masked when calling sigwait.

      Keep_Unmasked (SIGBUS)  := True;

      Keep_Unmasked (SIGTRAP) := True;
      --  Keep SIGTRAP unmasked to debug multithreaded programs.

      Keep_Unmasked (SIGTTIN) := True;
      Keep_Unmasked (SIGTTOU) := True;
      Keep_Unmasked (SIGTSTP) := True;
      --  To avoid interfering with IO and background processes

      Keep_Unmasked (SIGPROF) := True;
      --  To avoid disturbing the profiler

      Keep_Unmasked (SIGALRM) := True;
      Keep_Unmasked (SIGVTALRM) := True;
      --  To avoid disturbing sleeps and IOs

      Keep_Unmasked (SIGIO) := True;
      --  To avoid disturbing async IOs

      Keep_Unmasked (SIGCHLD) := True;
      --  To avoid disturbing fork(0 based operations

      --  By keeping SIGINT unmasked, allow the user to do a Ctrl-C, but in the
      --  same time, disable the ability of handling this signal
      --  via Ada.Interrupts.
      --  The pragma Unreserve_All_Interrupts let the user the ability to
      --  change this behavior.

      if Unreserve_All_Interrupts = 0 then
         Keep_Unmasked (SIGINT) := True;
      end if;

      --  Reserve this not to interfere with thread scheduling

      Reserve := Keep_Unmasked or Keep_Masked;
      Reserve (SIGALRM) := True;
      Reserve (SIGSTOP) := True;
      Reserve (SIGKILL) := True;

      Reserve := Reserve or Keep_Unmasked or Keep_Masked;
      Reserve (0) := True;
      --  We do not have Signal 0 in reality. We just use this value
      --  to identify not existing signals (see s-intnam.ads). Therefore,
      --  Signal 0 should not be used in all signal related operations hence
      --  mark it as reserved.
   end;
end System.Interrupt_Management;
