------------------------------------------------------------------------------
--                                                                          --
--                GNU ADA RUN-TIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                   S Y S T E M . O S _ I N T E R F A C E                  --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.17 $                            --
--                                                                          --
--          Copyright (C) 1992-1998, Free Software Foundation, Inc.         --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is a Sun OS (FSU THREADS) version of this package.

--  This package encapsulates all direct interfaces to OS services
--  that are needed by children of System.

--  PLEASE DO NOT add any with-clauses to this package
--  or remove the pragma Elaborate_Body.
--  It is designed to be a bottom-level (leaf) package.

with Interfaces.C;
package System.OS_Interface is
   pragma Preelaborate;

   pragma Linker_Options ("-lgthreads");
   pragma Linker_Options ("-lmalloc");

   subtype int            is Interfaces.C.int;
   subtype short          is Interfaces.C.short;
   subtype long           is Interfaces.C.long;
   subtype unsigned       is Interfaces.C.unsigned;
   subtype unsigned_short is Interfaces.C.unsigned_short;
   subtype unsigned_long  is Interfaces.C.unsigned_long;
   subtype unsigned_char  is Interfaces.C.unsigned_char;
   subtype plain_char     is Interfaces.C.plain_char;
   subtype size_t         is Interfaces.C.size_t;

   -----------
   -- Errno --
   -----------

   function errno return int;
   pragma Import (C, errno, "__get_errno");

   --  NAMEs not used are commented-out
   --  NAMEs not supported on this system have __NAME for value

   --  E2BIG    : constant := 7;
   --  EACCES   : constant := 13;
   EAGAIN   : constant := 11;
   --  EBADF    : constant := 9;
   --  EBUSY    : constant := 16;
   --  ECHILD   : constant := 10;
   --  EDEADLK  : constant := 78;
   --  EDOM     : constant := 33;
   --  EEXIST   : constant := 17;
   --  EFAULT   : constant := 14;
   --  EFBIG    : constant := 27;
   EINTR    : constant := 4;
   EINVAL   : constant := 22;
   --  EIO      : constant := 5;
   --  EISDIR   : constant := 21;
   --  EMFILE   : constant := 24;
   --  EMLINK   : constant := 31;
   --  ENAMETOOLONG : constant := 63;
   --  ENFILE   : constant := 23;
   --  ENODEV   : constant := 19;
   --  ENOENT   : constant := 2;
   --  ENOEXEC  : constant := 8;
   --  ENOLCK   : constant := 79;
   ENOMEM   : constant := 12;
   --  ENOSPC   : constant := 28;
   --  ENOSYS   : constant := 90;
   --  ENOTDIR  : constant := 20;
   --  ENOTEMPTY    : constant := 66;
   --  ENOTTY   : constant := 25;
   --  ENXIO    : constant := 6;
   --  EPERM    : constant := 1;
   --  EPIPE    : constant := 32;
   --  ERANGE   : constant := 34;
   --  EROFS    : constant := 30;
   --  ESPIPE   : constant := 29;
   --  ESRCH    : constant := 3;
   ETIMEDOUT    : constant := 60;
   --  EXDEV    : constant := 18;

   -------------
   -- Signals --
   -------------

   NSIG : constant := 32;
   type Signal is new int range 0 .. Interfaces.C."-" (NSIG, 1);
   for Signal'Size use int'Size;

   --  NAMEs not used are commented-out
   --  NAMEs not supported on this system have __NAME for value

   SIGHUP     : constant := 1; --  hangup
   SIGINT     : constant := 2; --  interrupt (rubout)
   SIGQUIT    : constant := 3; --  quit (ASCD FS)
   SIGILL     : constant := 4; --  illegal instruction (not reset)
   SIGTRAP    : constant := 5; --  trace trap (not reset)
   SIGIOT     : constant := 6; --  IOT instruction
   SIGABRT    : constant := 6; --  used by abort, replace SIGIOT in the  future
   SIGEMT     : constant := 7; --  EMT instruction
   SIGFPE     : constant := 8; --  floating point exception
   SIGKILL    : constant := 9; --  kill (cannot be caught or ignored)
   SIGBUS     : constant := 10; --  bus error
   SIGSEGV    : constant := 11; --  segmentation violation
   SIGSYS     : constant := 12; --  bad argument to system call
   SIGPIPE    : constant := 13; --  write on a pipe with no one to read it
   SIGALRM    : constant := 14; --  alarm clock
   SIGTERM    : constant := 15; --  software termination signal from kill
   SIGUSR1    : constant := 30; --  user defined signal 1
   SIGUSR2    : constant := 31; --  user defined signal 2
   SIGCLD     : constant := 20; --  alias for SIGCHLD
   SIGCHLD    : constant := 20; --  child status change
   SIGWINCH   : constant := 28; --  window size change
   SIGURG     : constant := 16; --  urgent condition on IO channel
   SIGPOLL    : constant := 23; --  pollable event occurred
   SIGIO      : constant := 23; --  I/O possible (Solaris SIGPOLL alias)
   SIGSTOP    : constant := 17; --  stop (cannot be caught or ignored)
   SIGTSTP    : constant := 18; --  user stop requested from tty
   SIGCONT    : constant := 19; --  stopped process has been continued
   SIGTTIN    : constant := 21; --  background tty read attempted
   SIGTTOU    : constant := 22; --  background tty write attempted
   SIGVTALRM  : constant := 26; --  virtual timer expired
   SIGPROF    : constant := 27; --  profiling timer expired
   SIGXCPU    : constant := 24; --  CPU time limit exceeded
   SIGXFSZ    : constant := 25; --  filesize limit exceeded

   type sigset_t is private;

   function sigaddset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigaddset, "sigaddset");

   function sigdelset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigdelset, "sigdelset");

   function sigfillset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigfillset, "sigfillset");

   function sigismember
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigismember, "sigismember");

   function sigemptyset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigemptyset, "sigemptyset");

   type union_type_1 is new int;
   type siginfo_t is record
      si_signo     : int;
      si_code      : int;
      si_value     : union_type_1;
   end record;
   pragma Convention (C, siginfo_t);

   --  The types mcontext_t and gregset_t are part of the ucontext_t
   --  information, which is specific to Solaris2.4 for SPARC
   --  The ucontext_t info seems to be used by the handler
   --  for SIGSEGV to decide whether it is a Storage_Error (stack overflow) or
   --  a Constraint_Error (bad pointer).  The original code that did this
   --  is suspect, so it is not clear whether we really need this part of
   --  the signal context information, or perhaps something else.
   --  More analysis is needed, after which these declarations may need to
   --  be changed.
   FPE_INTOVF_TRAP     : constant := 1;  --  integer overflow
   FPE_STARTSIG_TRAP   : constant := 2;  --  process using fp
   FPE_INTDIV_TRAP     : constant := 20;  --  integer divide by zero
   FPE_FLTINEX_TRAP    : constant := 196;  --  floating inexact result
   FPE_FLTDIV_TRAP     : constant := 200;  --  floating divide by zero
   FPE_FLTUND_TRAP     : constant := 204;  --  floating underflow
   FPE_FLTOPERR_TRAP   : constant := 208;  --  floating operand error
   FPE_FLTOVF_TRAP     : constant := 212;  --  floating overflow

   ILL_STACK           : constant := 0;  --  bad stack
   ILL_ILLINSTR_FAULT  : constant := 2;  --  illegal instruction fault
   ILL_PRIVINSTR_FAULT : constant := 3;  --  privileged instruction fault

   type array_type_1 is array (integer range 0 .. 30) of System.Address;
   type array_type_3 is array (integer range 0 .. 15) of int;
   type array_type_2 is array (integer range 0 .. 30) of array_type_3;
   type struct_sigcontext is record
      sc_onstack   : int;
      sc_mask      : int;
      sc_sp        : int;
      sc_pc        : int;
      sc_npc       : int;
      sc_psr       : int;
      sc_g1        : int;
      sc_o0        : int;
      sc_wbcnt     : int;
      sc_spbuf     : array_type_1;
      sc_wbuf      : array_type_2;
   end record;
   pragma Convention (C, struct_sigcontext);

   type struct_sigaction is record
      sa_handler   : System.Address;
      sa_mask      : sigset_t;
      sa_flags     : int;
   end record;
   pragma Convention (C, struct_sigaction);
   type struct_sigaction_ptr is access all struct_sigaction;

   SIG_BLOCK   : constant := 1;
   SIG_UNBLOCK : constant := 2;
   SIG_SETMASK : constant := 4;

   --  SA_NOCLDSTOP : constant := 8;
   --  not used
   --  SA_SIGINFO : constant := 0;
   --  not used

   SIG_DFL : constant := 0;
   SIG_IGN : constant := 1;
   --  SIG_ERR : constant := -1;
   --  not used

   function sigaction
     (sig  : Signal;
      act  : struct_sigaction_ptr;
      oact : struct_sigaction_ptr)
     return int;
   pragma Import (C, sigaction, "sigaction");

   ----------
   -- Time --
   ----------

   Time_Slice_Supported : constant boolean := False;
   --  Indicates wether time slicing is supported (i.e FSU threads have been
   --  compiled with DEF_RR)

   type timespec is private;

   function nanosleep (rqtp, rmtp : access timespec)  return int;
   --  FSU_THREADS has nonstandard nanosleep
   type clockid_t is private;

   CLOCK_REALTIME : constant clockid_t;

   function clock_gettime
     (clock_id : clockid_t;
      tp       : access timespec)
      return int;
   pragma Import (C, clock_gettime, "clock_gettime");

   function To_Duration (TS : timespec) return Duration;
   pragma Inline (To_Duration);

   function To_Timespec (D : Duration) return timespec;
   pragma Inline (To_Timespec);

   type struct_timezone is record
      tz_minuteswest  : int;
      tz_dsttime   : int;
   end record;
   pragma Convention (C, struct_timezone);
   type struct_timeval is private;
   --  This is needed on systems that do not have clock_gettime()
   --  but do have gettimeofday().

   function To_Duration (TV : struct_timeval) return Duration;
   pragma Inline (To_Duration);

   function To_Timeval (D : Duration) return struct_timeval;
   pragma Inline (To_Timeval);

   function gettimeofday
     (tv : access struct_timeval;
      tz : access struct_timezone) return int;
   pragma Import (C, gettimeofday, "gettimeofday");

   --  add a hook to locate the Epoch, for use with Calendar????

   -------------------------
   -- Priority Scheduling --
   -------------------------

   MIN_PRIO : constant := 0;
   MAX_PRIO : constant := 0;

   SCHED_FIFO  : constant := 0;
   SCHED_RR    : constant := 1;
   SCHED_OTHER : constant := 2;

   -------------
   -- Process --
   -------------

   type pid_t is private;

   function kill
     (pid : pid_t;
      sig : Signal)
   return int;
   pragma Import (C, kill, "kill");

   function getpid return pid_t;
   pragma Import (C, getpid, "getpid");

   -------------
   -- Threads --
   -------------

   type Thread_Body is access
     function (arg : System.Address) return System.Address;
   type pthread_t           is private;
   subtype Thread_Id        is pthread_t;

   type pthread_mutex_t     is limited private;
   type pthread_cond_t      is limited private;
   type pthread_attr_t      is limited private;
   type pthread_mutexattr_t is limited private;
   type pthread_condattr_t  is limited private;
   type pthread_key_t       is private;

   PTHREAD_CREATE_DETACHED : constant := 1;
   PTHREAD_CREATE_JOINABLE : constant := 0;

   -----------
   -- Stack --
   -----------

   Stack_Base_Available : constant boolean := False;
   --  Indicates wether the stack base is available on this target.
   --  This allows us to share s-osinte.adb between all the FSU run time.
   --  Note that this value can only be true if pthread_t has a complete
   --  definition that corresponds exactly to the C header files.

   function Get_Stack_Base (thread : pthread_t) return Address;
   pragma Inline (Get_Stack_Base);
   --  returns the stack base of the specified thread.
   --  Only call this function when Stack_Base_Available is True.

   function Get_Page_Size return size_t;
   function Get_Page_Size return Address;
   pragma Import (C, Get_Page_Size, "getpagesize");
   --  returns the size of a page, or 0 if this is not relevant on this
   --  target

   PROT_NONE  : constant := 0;
   PROT_READ  : constant := 1;
   PROT_WRITE : constant := 2;
   PROT_EXEC  : constant := 4;
   PROT_ALL   : constant := PROT_READ + PROT_WRITE + PROT_EXEC;

   PROT_ON    : constant := PROT_NONE;
   PROT_OFF   : constant := PROT_ALL;

   function mprotect
     (addr : Address; len : size_t; prot : int) return int;
   pragma Import (C, mprotect);

   -----------------------------------------
   --  Nonstandard Thread Initialization  --
   -----------------------------------------
   --  FSU_THREADS requires pthread_init, which is nonstandard
   --  and this should be invoked during the elaboration of s-taprop.adb
   procedure pthread_init;
   pragma Import (C, pthread_init, "pthread_init");

   ---------------------------
   --  POSIX.1c  Section 3  --
   ---------------------------

   function sigwait
     (set : access sigset_t;
      sig : access Signal)
     return int;
   --  FSU_THREADS has a nonstandard sigwait

   function pthread_kill
     (thread : pthread_t;
      sig    : Signal)
     return   int;
   pragma Import (C, pthread_kill, "pthread_kill");

   --  FSU threads does not have pthread_sigmask. Instead, it uses
   --  sigprocmask to do the signal handling when the thread library is
   --  sucked in.

   type sigset_t_ptr is access all sigset_t;

   function pthread_sigmask
     (how  : int;
      set  : sigset_t_ptr;
      oset : sigset_t_ptr)
      return int;
   pragma Import (C, pthread_sigmask, "sigprocmask");

   ----------------------------
   --  POSIX.1c  Section 11  --
   ----------------------------

   function pthread_mutexattr_init
     (attr : access pthread_mutexattr_t)
     return int;
   pragma Import (C, pthread_mutexattr_init, "pthread_mutexattr_init");

   function pthread_mutexattr_destroy
     (attr : access pthread_mutexattr_t)
     return int;
   pragma Import (C, pthread_mutexattr_destroy, "pthread_mutexattr_destroy");

   function pthread_mutex_init
     (mutex : access pthread_mutex_t;
      attr  : access pthread_mutexattr_t)
     return int;
   pragma Import (C, pthread_mutex_init, "pthread_mutex_init");

   function pthread_mutex_destroy
     (mutex : access pthread_mutex_t)
     return  int;
   pragma Import (C, pthread_mutex_destroy, "pthread_mutex_destroy");

   function pthread_mutex_lock
     (mutex : access pthread_mutex_t)
     return int;
   --  FSU_THREADS has nonstandard pthread_mutex_lock

   function pthread_mutex_unlock
     (mutex : access pthread_mutex_t)
     return int;
   --  FSU_THREADS has nonstandard pthread_mutex_lock

   function pthread_condattr_init
     (attr : access pthread_condattr_t)
     return int;
   pragma Import (C, pthread_condattr_init, "pthread_condattr_init");

   function pthread_condattr_destroy
     (attr : access pthread_condattr_t)
     return int;
   pragma Import (C, pthread_condattr_destroy, "pthread_condattr_destroy");

   function pthread_cond_init
     (cond : access pthread_cond_t;
      attr : access pthread_condattr_t)
     return int;
   pragma Import (C, pthread_cond_init, "pthread_cond_init");

   function pthread_cond_destroy
     (cond : access pthread_cond_t)
     return int;
   pragma Import (C, pthread_cond_destroy, "pthread_cond_destroy");

   function pthread_cond_signal
     (cond : access pthread_cond_t)
     return int;
   pragma Import (C, pthread_cond_signal, "pthread_cond_signal");

   function pthread_cond_wait
     (cond  : access pthread_cond_t;
      mutex : access pthread_mutex_t)
     return  int;
   --  FSU_THREADS has a nonstandard pthread_cond_wait

   function pthread_cond_timedwait
     (cond    : access pthread_cond_t;
      mutex   : access pthread_mutex_t;
      abstime : access timespec)
     return    int;
   --  FSU_THREADS has a nonstandard pthread_cond_timedwait

   ----------------------------
   --  POSIX.1c  Section 13  --
   ----------------------------

   PTHREAD_PRIO_NONE    : constant := 0;
   PTHREAD_PRIO_PROTECT : constant := 2;
   PTHREAD_PRIO_INHERIT : constant := 1;

   function pthread_mutexattr_setprotocol
     (attr     : access pthread_mutexattr_t;
      protocol : int)
     return int;
   pragma Import
      (C, pthread_mutexattr_setprotocol, "pthread_mutexattr_setprotocol");

   function pthread_mutexattr_getprotocol
     (attr     : access pthread_mutexattr_t;
      protocol : access int)
     return int;
   pragma Import
     (C, pthread_mutexattr_getprotocol, "pthread_mutexattr_getprotocol");

   function pthread_mutexattr_setprioceiling
     (attr     : access pthread_mutexattr_t;
      prioceiling : int)
     return int;
   pragma Import
     (C, pthread_mutexattr_setprioceiling,
      "pthread_mutexattr_setprio_ceiling");

   function pthread_mutexattr_getprioceiling
     (attr     : access pthread_mutexattr_t;
      prioceiling : access int)
     return int;
   pragma Import
     (C, pthread_mutexattr_getprioceiling,
      "pthread_mutexattr_getprioceiling");

   type struct_sched_param is record
      sched_priority : int;  --  scheduling priority
   end record;

   function pthread_getschedparam
     (thread : pthread_t;
      policy : access int;
      param  : access struct_sched_param)
     return int;
   --  FSU_THREADS does not have pthread_getschedparam

   function pthread_setschedparam
     (thread : pthread_t;
      policy : int;
      param  : access struct_sched_param)
     return int;
   --  FSU_THREADS does not have pthread_setschedparam

   function pthread_attr_setscope
     (attr            : access pthread_attr_t;
      contentionscope : int)
     return int;
   pragma Import (C, pthread_attr_setscope, "pthread_attr_setscope");

   function pthread_attr_getscope
     (attr            : access pthread_attr_t;
      contentionscope : access int)
     return int;
   pragma Import (C, pthread_attr_getscope, "pthread_attr_getscope");

   function pthread_attr_setinheritsched
     (attr            : access pthread_attr_t;
      inheritsched : int)
     return int;
   pragma Import
     (C, pthread_attr_setinheritsched, "pthread_attr_setinheritsched");

   function pthread_attr_getinheritsched
     (attr         : access pthread_attr_t;
      inheritsched : access int)
     return int;
   pragma Import
     (C, pthread_attr_getinheritsched, "pthread_attr_getinheritsched");

   function pthread_attr_setschedpolicy
     (attr   : access pthread_attr_t;
      policy : int)
     return int;
   pragma Import
    (C, pthread_attr_setschedpolicy,
     "pthread_attr_setsched");

   function pthread_attr_getschedpolicy
     (attr   : access pthread_attr_t;
      policy : access int)
     return int;
   pragma Import (C, pthread_attr_getschedpolicy,
     "pthread_attr_getsched");

   function pthread_attr_setschedparam
     (attr        : access pthread_attr_t;
      sched_param : int)
     return int;
   pragma Import (C, pthread_attr_setschedparam, "pthread_attr_setschedparam");

   function pthread_attr_getschedparam
     (attr        : access pthread_attr_t;
      sched_param : access int)
     return int;
   pragma Import (C, pthread_attr_getschedparam, "pthread_attr_getschedparam");

   function sched_yield return int;
   --  FSU_THREADS does not have sched_yield;

   -----------------------------
   --  P1003.1c - Section 16  --
   -----------------------------

   function pthread_attr_init
     (attributes : access pthread_attr_t)
     return int;
   pragma Import (C, pthread_attr_init, "pthread_attr_init");

   function pthread_attr_destroy
     (attributes : access pthread_attr_t)
     return int;
   pragma Import (C, pthread_attr_destroy, "pthread_attr_destroy");

   function pthread_attr_setdetachstate
     (attr        : access pthread_attr_t;
      detachstate : int)
     return int;
   --  FSU_THREADS has a nonstandard pthread_attr_setdetachstate

   function pthread_attr_getdetachstate
     (attr        : access pthread_attr_t;
      detachstate : access int)
     return int;
   pragma Import
     (C, pthread_attr_getdetachstate, "pthread_attr_getdetachstate");

   function pthread_attr_getstacksize
     (attr      : access pthread_attr_t;
      stacksize : access size_t)
     return int;
   pragma Import
     (C, pthread_attr_getstacksize, "pthread_attr_getstacksize");

   function pthread_attr_setstacksize
     (attr      : access pthread_attr_t;
      stacksize : size_t)
     return int;
   pragma Import
     (C, pthread_attr_setstacksize, "pthread_attr_setstacksize");

   function pthread_create
     (thread        : access pthread_t;
      attributes    : access pthread_attr_t;
      start_routine : Thread_Body;
      arg           : System.Address)
     return int;
   pragma Import (C, pthread_create, "pthread_create");

   function pthread_detach (thread : pthread_t) return int;
   --  FSU_THREADS has a nonstandard pthread_detach

   procedure pthread_exit (status : System.Address);
   pragma Import (C, pthread_exit, "pthread_exit");

   function pthread_self return pthread_t;
   pragma Import (C, pthread_self, "pthread_self");

   function pthread_equal (t1 : pthread_t; t2 : pthread_t)
      return int;
   --  be careful not to use "=" on thread_t!
   pragma Import (C, pthread_equal, "pthread_equal");

   ----------------------------
   --  POSIX.1c  Section 17  --
   ----------------------------

   function pthread_setspecific
     (key   : pthread_key_t;
      value : System.Address)
     return  int;
   pragma Import (C, pthread_setspecific, "pthread_setspecific");

   function pthread_getspecific (key : pthread_key_t) return System.Address;
   --  FSU_THREADS has a nonstandard pthread_getspecific

   type destructor_pointer is access
      procedure (arg : System.Address);

   function pthread_key_create
     (key        : access pthread_key_t;
      destructor : destructor_pointer)
     return int;
   pragma Import (C, pthread_key_create, "pthread_key_create");

private

   type sigset_t is new int;

   --  In Solaris 2.4 the component sa_handler turns out to
   --  be one a union type, and the selector is a macro:
   --  #define sa_handler __funcptr._handler
   --  #define sa_sigaction __funcptr._sigaction

   type pid_t is new int;

   type time_t is new long;

   type timespec is record
      tv_sec       : time_t;
      tv_nsec      : long;
   end record;
   pragma Convention (C, timespec);

   type clockid_t is new int;
   CLOCK_REALTIME : constant clockid_t := 0;

   type struct_timeval is record
      tv_sec       : long;
      tv_usec      : long;
   end record;
   pragma Convention (C, struct_timeval);

   type pthread_attr_t is record
      flags        : int;
      stacksize    : int;
      contentionscope  : int;
      inheritsched  : int;
      detachstate  : int;
      sched        : int;
      prio         : int;
      starttime    : timespec;
      deadline     : timespec;
      period       : timespec;
   end record;
   pragma Convention (C, pthread_attr_t);

   type pthread_condattr_t is record
      flags        : int;
   end record;
   pragma Convention (C, pthread_condattr_t);

   type pthread_mutexattr_t is record
      flags        : int;
      prio_ceiling  : int;
      protocol     : int;
   end record;
   pragma Convention (C, pthread_mutexattr_t);

   type sigjmp_buf is array (integer range 0 .. 9) of int;

   type array_type_6 is array (integer range 0 .. 2) of System.Address;
   type union_type_2 is new int;
   type record_type_4 is record
      si_signo     : int;
      si_code      : int;
      si_value     : union_type_2;
   end record;
   pragma Convention (C, record_type_4);
   type array_type_7 is array (integer range 0 .. 32) of record_type_4;
   type record_type_5 is record
      head         : System.Address;
      tail         : System.Address;
   end record;
   pragma Convention (C, record_type_5);
   type array_type_8 is array (integer range 0 .. 7) of System.Address;
   type record_type_6 is record
      aio_return   : int;
      aio_errno    : int;
   end record;
   pragma Convention (C, record_type_6);
   type array_type_9 is array (integer range 0 .. 7) of long;
   type record_type_7 is record
      fds_bits     : array_type_9;
   end record;
   pragma Convention (C, record_type_7);
   type array_type_10 is array (integer range 0 .. 7) of long;
   type record_type_8 is record
      fds_bits     : array_type_10;
   end record;
   pragma Convention (C, record_type_8);
   type array_type_11 is array (integer range 0 .. 7) of long;
   type record_type_9 is record
      fds_bits     : array_type_11;
   end record;
   pragma Convention (C, record_type_9);

   type record_type_3 is record
      context      : sigjmp_buf;
      pbody        : sigjmp_buf;
      errno        : int;
      ret          : int;
      stack_base   : System.Address;
      state        : int;
      next         : array_type_6;
      num_timers   : int;
      interval     : struct_timeval;
      sig_info     : array_type_7;
      sig          : int;
      code         : int;
      osp          : int;
      opc          : int;
      obp          : int;
      nscp         : System.Address;
      scp          : System.Address;
      joinq        : record_type_5;
      cond         : System.Address;
      queue        : System.Address;
      mask         : sigset_t;
      pending      : sigset_t;
      sigwaitset   : sigset_t;
      func         : System.Address;
      arg          : System.Address;
      result       : System.Address;
      key          : array_type_8;
      cleanup_top  : System.Address;
      attr         : pthread_attr_t;
      base_prio    : int;
      max_ceiling_prio  : int;
      new_prio     : int;
      resultp      : record_type_6;
      wait_on_select  : int;
      width        : int;
      how_many     : int;
      readfds      : record_type_7;
      writefds     : record_type_8;
      exceptfds    : record_type_9;
      tp           : struct_timeval;
   end record;
   pragma Convention (C, record_type_3);
   type pthread_t is access record_type_3;

   type record_type_1 is record
      head         : System.Address;
      tail         : System.Address;
   end record;
   pragma Convention (C, record_type_1);
   type pthread_mutex_t is record
      queue        : record_type_1;
      lock         : plain_char;
      owner        : System.Address;
      flags        : int;
      prio_ceiling  : int;
      protocol     : int;
      prev_max_ceiling_prio  : int;
   end record;
   pragma Convention (C, pthread_mutex_t);

   type record_type_2 is record
      head         : System.Address;
      tail         : System.Address;
   end record;
   pragma Convention (C, record_type_2);
   type pthread_cond_t is record
      queue        : record_type_2;
      flags        : int;
      waiters      : int;
      mutex        : System.Address;
   end record;
   pragma Convention (C, pthread_cond_t);

   type pthread_key_t is new int;

end System.OS_Interface;
