/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: lamboot.c,v 6.15 1999/11/11 18:02:53 jsquyres Exp $
 *
 *	Function:	- fully-connected LAM booting tool
 *			- boots Trollius on a network of UNIX hosts
 *			- hosts specified using host file syntax
 *			- uses hboot
 */

#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <kio.h>
#include <lamnet.h>
#include <net.h>
#include <sfh.h>
#include <terror.h>
#include <patchlevel.h>

/*
 * static variables
 */
static char		*bhost;		/* boot schema file */
static int		fl_debug;	/* debug mode */
static int		fl_verbose;	/* verbose mode */
static int		nboot;		/* # nodes booted */
static int		nrun;		/* # nodes running */

/*
 * external functions
 */
extern int		_lam_few();
extern int		bhostparse();
extern int		lambootagent();

/*
 * external variables
 */
extern struct kio_t	_kio;

/* 
 * private functions
 */
static void		bail();
static void		wipe();
static char		*findbhost();


int
main(argc, argv)  

int			argc;
char			*argv[];

{
	FILE		*fp;		/* boot schema file pointer */
	int		idup;		/* index of duplicate node */
	int		iorigin;	/* index of origin node */
	int		badhost;	/* bad host index */
	int		nlamnet;	/* lamnet size */
	struct lamnode	*lamnet;	/* network description array */
	int             j;

/*
 * Parse the command line.
 */
	validopts("dhvxHV");

	if (do_args(&argc, argv)) {
	  show_help("lamboot", "usage", NULL);
	  exit(EUSAGE);
	}

	if (opt_taken('V')) {
		printf("\nLAM %s", LAM_VERSION);
#if LAM_WANT_IMPI
		printf("/IMPI");
#endif
#if LAM_WANT_MPI2CPP
		printf("/MPI 2 C++");
#endif
#if LAM_WANT_ROMIO
		printf("/ROMIO");
#endif
		printf(" - University of Notre Dame\n\n");
		printf("\tArch:\t\t%s\n", LAM_ARCH);
		printf("\tRPI:\t\t%s\n", LAM_RPI);
		exit(0);
	}

	/* Ensure that we are not root */

	if (getuid() == 0 || geteuid() == 0) {
	  show_help(NULL, "deny-root", NULL);
	  exit(EACCES);
	}

	if ((errno = (argc <= 2) ? 0 : EUSAGE)) {
	  show_help("lamboot", "usage", NULL);
	  exit(errno);
	}
 
	if (opt_taken('h')) {
	  show_help("lamboot", "usage", NULL);
	  exit(0);
	}

	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;

	if (! opt_taken('H')) {
		printf("\nLAM %s", LAM_VERSION);
#if LAM_WANT_IMPI
		printf("/IMPI");
#endif
#if LAM_WANT_MPI2CPP
		printf("/MPI 2 C++");
#endif
#if LAM_WANT_ROMIO
		printf("/ROMIO");
#endif
		printf(" - University of Notre Dame\n\n");
	}
/*
 * Locate the system file.
 */
	bhost = findbhost(argc, argv);
/*
 * Open the system file.
 */
	DBUG("lamboot: opening hostfile %s\n", bhost);
	fp = fopen(bhost, "r");
	if (fp == 0) {
	  show_help("boot", "open-hostfile", "lamboot", bhost, NULL);
	  exit(errno);
	}
/*
 * Parse the system file.
 */
	if (bhostparse(fp, &lamnet, &nlamnet)) {
	  show_help("boot", "bhostparse", "lamboot", bhost, NULL);
	  exit(errno);
	}
	if (fl_debug) {
	  printf("lamboot: found the following hosts:\n");
	  for (j = 0; j < nlamnet; j++)
	    printf("lamboot:   n%d %s\n", j, lamnet[j].lnd_hname);
	}
/*
 * Close the system file.
 */
	if (fclose(fp)) {
	  show_help("boot", "close-hostfile", "lamboot", bhost, NULL);
	  exit(errno);
	}
/*
 * Locate the host nodes.
 */
	if (lamnet_findhosts(lamnet, nlamnet, &badhost)) {

		if (errno == EBADHOST) {
		  show_help("boot", "resolv-hostname", "lamboot",
			    lamnet[badhost].lnd_hname, bhost, NULL);
		  exit(LAM_EEXIT);
		} else {
		  char buffer[16];
		  sprintf(buffer, "%d", errno);
		  show_help("boot", "resolv-unknown", "lamboot", bhost,
			    buffer, NULL);
		}
		exit(errno);
	}
/*
 * Make rudimentary check for duplicate hosts.
 */
	if (lamnet_dups(lamnet, nlamnet, &idup)) {
	  show_help("boot", "duplicated-host", "lamboot", 
		    lamnet[idup].lnd_hname, NULL);
	  exit(EINVAL);
	}
/*
 * Find the origin node.
 */
	iorigin = lamnet_findorig(lamnet, nlamnet);

	if (iorigin < 0) {
	  show_help("boot", "no-localhost", "lamboot", bhost, NULL);
	  exit(EINVAL);
	}

	_kio.ki_origin = iorigin;
	lamnet[iorigin].lnd_type |= NT_ORIGIN | NT_ME;

	DBUG("lamboot: found %d host node(s)\n", nlamnet);
	DBUG("lamboot: origin node is %d (%s)\n", iorigin,
	     lamnet[iorigin].lnd_hname);
/*
 * Clean up on interrupt.
 */
	if (signal(SIGINT, bail) == SIG_ERR) lampanic("lamboot (signal)");
/*
 * Boot the system.
 */
	if (lambootagent(lamnet, nlamnet, &nboot, &nrun, fl_verbose)) {

	  /* If the localhost was not in the boot schema, we must
             print the error message here, because lambootagent() does
             not know the name of the file (which is printed in the
             help message) */
	  /* This shouldn't happen, though, since we check for the
             local host above.  But better safe than sorry... */
	  if (errno == EINVAL)
	    show_help("boot", "no-localhost", "lamboot", bhost, NULL);

	  /* lambootagent should print all other relevant error messages */
	  
	  bail();
	}

	DBUG("lamboot completed successfully\n");
	return(0);
}

/*
 *	findbhost
 *
 *	Function:	- locates boot schema file
 *	Accepts:	- argc of remaining command line
 *			- argv of remaining command line
 *	Returns:	- full pathname of boot schema file
 */
static char *
findbhost(cmdc, cmdv)

int			cmdc;
char			**cmdv;

{
	char		*fname;
	char		*full;		/* temporary for full pathname */
	char		**pathv;
	int		pathc;
/*
 * Set the directories for the boot schema file search.
 */
	pathc = 0;
	pathv = 0;
	argvadd(&pathc, &pathv, "");
	argvadd(&pathc, &pathv, "$TROLLIUSHOME/boot");
	argvadd(&pathc, &pathv, "$LAMHOME/boot");
	argvadd(&pathc, &pathv, DEFPBHOST);
/*
 * Set the boot schema file names.
 */
	if (cmdc == 2) {
		fname = cmdv[1];
	} else if ((fname = getenv("LAMBHOST"))) {
	} else if ((fname = getenv("TROLLIUSBHOST"))) {
	} else {
		fname = DEFFBHOST;
	}
/*
 * Search the directories.
 */
	full = sfh_path_find(fname, pathv, R_OK);

	if (full == 0) {
	  show_help("boot", "open-hostfile", "lamboot", fname, NULL);
	  exit(errno);
	}

	DBUG("lamboot: boot schema file: %s\n", full);
	return(full);
}

/*
 *	bail
 *
 *	Function:	- cleans up and bails out
 *	Returns:	- does not return, exits with error code
 */
static void
bail()

{
	int		err_save;	/* saved error code */

	err_save = errno;
	show_help("lamboot", "about-to-wipe", NULL);
	wipe();
	DBUG("lamboot did NOT complete successfully\n");
	exit(err_save);
}

/*
 *	wipe
 *
 *	Function:	- executes the wipe tool
 *			- kills all host Trollius sessions
 */
static void
wipe()

{
	int		cmdn;
	int		r;
	char		**cmdv;
	char		buf[16];

	if (nboot <= 0) {
		return;
	}

	cmdn = 0;
	cmdv = 0;
	argvadd(&cmdn, &cmdv, DEFTWIPE);

	if (fl_verbose) {
		argvadd(&cmdn, &cmdv, "-v");
	}

	argvadd(&cmdn, &cmdv, "-n");
	sprintf(buf, "%d", nboot);
	argvadd(&cmdn, &cmdv, buf);

	argvadd(&cmdn, &cmdv, bhost);

	VERBOSE("%s ...\n", DEFTWIPE);

	r = _lam_few(cmdv);

	if (r) {
		errno = r;
		show_help("boot", "wipe-fail", NULL);
		exit(errno);
	}
}
