%-----------------------------------------------------------------------------%
% Copyright (C) 1994-1998 The University of Melbourne.
% This file may only be copied under the terms of the GNU General
% Public License - see the file COPYING in the Mercury distribution.
%-----------------------------------------------------------------------------%

% peephole.m - local LLDS to LLDS optimizations based on pattern-matching.

% Authors: fjh and zs.

%-----------------------------------------------------------------------------%

:- module peephole.

:- interface.

:- import_module bool, list.
:- import_module llds, globals.

	% Peephole optimize a list of instructions.

:- pred peephole__optimize(gc_method, list(instruction), list(instruction),
		bool).
:- mode peephole__optimize(in, in, out, out) is det.

:- implementation.

:- import_module map, string, std_util.
:- import_module code_util, opt_util, opt_debug.

	% Patterns that can be switched off.

:- type pattern --->		incr_sp.

	% We zip down to the end of the instruction list, and start attempting
	% to optimize instruction sequences. As long as we can continue
	% optimizing the instruction sequence, we keep doing so;
	% when we find a sequence we can't optimize, we back up and try
	% to optimize the sequence starting with the previous instruction.

peephole__optimize(GC_Method, Instrs0, Instrs, Mod) :-
	peephole__invalid_opts(GC_Method, InvalidPatterns),
	peephole__optimize_2(InvalidPatterns, Instrs0, Instrs, Mod).

:- pred peephole__optimize_2(list(pattern), list(instruction),
		list(instruction), bool).
:- mode peephole__optimize_2(in, in, out, out) is det.
peephole__optimize_2(_, [], [], no).
peephole__optimize_2(InvalidPatterns, [Instr0 - Comment | Instrs0],
		Instrs, Mod) :-
	peephole__optimize_2(InvalidPatterns, Instrs0, Instrs1, Mod0),
	peephole__opt_instr(Instr0, Comment, InvalidPatterns, Instrs1,
		Instrs, Mod1),
	( Mod0 = no, Mod1 = no ->
		Mod = no
	;
		Mod = yes
	).

	% Try to optimize the beginning of the given instruction sequence.
	% If successful, try it again.

:- pred peephole__opt_instr(instr, string, list(pattern),
	list(instruction), list(instruction), bool).
:- mode peephole__opt_instr(in, in, in, in, out, out) is det.

peephole__opt_instr(Instr0, Comment0, InvalidPatterns, Instrs0, Instrs, Mod) :-
	(
		opt_util__skip_comments(Instrs0, Instrs1),
		peephole__match(Instr0, Comment0, InvalidPatterns, Instrs1,
			Instrs2)
	->
		( Instrs2 = [Instr2 - Comment2 | Instrs3] ->
			peephole__opt_instr(Instr2, Comment2, InvalidPatterns,
				Instrs3, Instrs, _)
		;
			Instrs = Instrs2
		),
		Mod = yes
	;
		Instrs = [Instr0 - Comment0 | Instrs0],
		Mod = no
	).

%-----------------------------------------------------------------------------%

	% Look for code patterns that can be optimized, and optimize them.

:- pred peephole__match(instr, string, list(pattern),
		list(instruction), list(instruction)).
:- mode peephole__match(in, in, in, in, out) is semidet.

	% A `computed_goto' with all branches pointing to the same
	% label can be replaced with an unconditional goto.

peephole__match(computed_goto(_, Labels), Comment, _, Instrs0, Instrs) :-
	list__all_same(Labels),
	Labels = [Target|_],
	Instrs = [goto(label(Target)) - Comment | Instrs0].

	% A conditional branch whose condition is constant
	% can be either eliminated or replaced by an unconditional goto.
	%
	% A conditional branch to an address followed by an unconditional
	% branch to the same address can be eliminated.
	%
	% A conditional branch to a label followed by that label
	% can be eliminated.

peephole__match(if_val(Rval, CodeAddr), Comment, _, Instrs0, Instrs) :-
	(
		opt_util__is_const_condition(Rval, Taken)
	->
		(
			Taken = yes,
			Instrs = [goto(CodeAddr) - Comment | Instrs0]
		;
			Taken = no,
			Instrs = Instrs0
		)
	;
		opt_util__skip_comments(Instrs0, Instrs1),
		Instrs1 = [Instr1 | _],
		Instr1 = goto(CodeAddr) - _
	->
		Instrs = Instrs0
	;
		CodeAddr = label(Label),
		opt_util__is_this_label_next(Label, Instrs0, _)
	->
		Instrs = Instrs0
	;
		fail
	).

	% If a `mkframe' is followed by an assignment to its redoip slot,
	% with the instructions in between containing only straight-line code,
	% we can delete the assignment and instead just set the redoip
	% directly in the `mkframe'.
	%
	%	mkframe(NFI, _)		=>	mkframe(NFI, Redoip)
	%	<straightline instrs>		<straightline instrs>
	%	assign(redoip(lval(_)), Redoip)
	%
	% If a `mkframe' is followed by a test that can fail, we try to
	% swap the two instructions to avoid doing the mkframe unnecessarily.
	%
	%	mkframe(NFI, dofail)	=>	if_val(test, redo)
	%	if_val(test, redo/fail)		mkframe(NFI, dofail)
	%
	%	mkframe(NFI, label)	=>	if_val(test, redo)
	%	if_val(test, fail)		mkframe(NFI, label)
	%
	%	mkframe(NFI, label)	=>	mkframe(NFI, label)
	%	if_val(test, redo)		if_val(test, label)
	%
	% These two patterns are mutually exclusive because if_val is not
	% straight-line code.

peephole__match(mkframe(NondetFrameInfo, Redoip1), Comment, _,
		Instrs0, Instrs) :-
	(
		% A mkframe sets curfr to point to the new frame
		% only for ordinary frames, not temp frames.
		( NondetFrameInfo = ordinary_frame(_, _, _) ->
			AllowedBases = [maxfr, curfr]
		;
			AllowedBases = [maxfr]
		),
		opt_util__next_assign_to_redoip(Instrs0, AllowedBases,
			[], Redoip2, Skipped, Rest),
		opt_util__touches_nondet_ctrl(Skipped, no)
	->
		list__append(Skipped, Rest, Instrs1),
		Instrs = [mkframe(NondetFrameInfo, Redoip2) - Comment
			| Instrs1]
	;
		opt_util__skip_comments_livevals(Instrs0, Instrs1),
		Instrs1 = [Instr1 | Instrs2],
		Instr1 = if_val(Test, Target) - Comment2,
		(
			Redoip1 = do_fail,
			( Target = do_redo ; Target = do_fail)
		->
			Instrs = [
				if_val(Test, do_redo)
					- Comment2,
				mkframe(NondetFrameInfo, do_fail)
					- Comment
				| Instrs2
			]
		;
			Redoip1 = label(_)
		->
			(
				Target = do_fail
			->
				Instrs = [
					if_val(Test, do_redo)
						- Comment2,
					mkframe(NondetFrameInfo, Redoip1)
						- Comment
					| Instrs2
				]
			;
				Target = do_redo
			->
				Instrs = [
					mkframe(NondetFrameInfo, Redoip1)
						- Comment,
					if_val(Test, Redoip1)
						- Comment2
					| Instrs2
				]
			;
				fail
			)
		;
			fail
		)
	).

	% If a `store_ticket' is followed by a `reset_ticket',
	% we can delete the `reset_ticket'.
	%
	%	store_ticket(Lval)	=>	store_ticket(Lval)
	%	reset_ticket(Lval, _R)

peephole__match(store_ticket(Lval), Comment, _, Instrs0, Instrs) :-
	opt_util__skip_comments(Instrs0, Instrs1),
	Instrs1 = [reset_ticket(lval(Lval), _Reason) - _Comment2 | Instrs2],
	Instrs = [store_ticket(Lval) - Comment | Instrs2].

	% If an assignment to a redoip slot is followed by another, with
	% the instructions in between containing only straight-line code,
	% we can delete one of the asignments:
	%
	%	assign(redoip(Fr), Redoip1) =>	assign(redoip(Fr), Redoip2)
	%	<straightline instrs>		<straightline instrs>
	%	assign(redoip(Fr), Redoip2)

	% If an assignment of do_fail to the redoip slot of the current frame
	% is followed by straight-line instructions except possibly for if_val
	% with do_fail or do_redo as target, until a goto to do_succeed(no),
	% and if the nondet stack linkages are not touched by the
	% straight-line instructions, then we can discard the nondet stack
	% frame early.

peephole__match(assign(redoip(lval(Base)), Redoip), Comment, _,
		Instrs0, Instrs) :-
	(
		opt_util__next_assign_to_redoip(Instrs0, [Base],
			[], Redoip2, Skipped, Rest),
		opt_util__touches_nondet_ctrl(Skipped, no)
	->
		list__append(Skipped, Rest, Instrs1),
		Instrs = [assign(redoip(lval(Base)),
			const(code_addr_const(Redoip2))) - Comment
			| Instrs1]
	;
		Base = curfr,
		Redoip = const(code_addr_const(do_fail)),
		opt_util__straight_alternative(Instrs0, Between, After),
		opt_util__touches_nondet_ctrl(Between, no)
	->
		list__condense([Between,
			[goto(do_succeed(yes)) - "early discard"], After],
				Instrs)
	;
		fail
	).

	% If a decr_sp follows an incr_sp of the same amount, with the code
	% in between not referencing the stack, except possibly for a
	% restoration of succip, then the two cancel out. Assignments to
	% stack slots are allowed and are thrown away.
	%
	%	incr_sp N
	%	<...>		=>	<...>
	%	decr_sp N
	%
	%	incr_sp N
	%	<...>		=>	<...>
	%	succip = detstackvar(N)
	%	decr_sp N

peephole__match(incr_sp(N, _), _, InvalidPatterns, Instrs0, Instrs) :-
	\+ list__member(incr_sp, InvalidPatterns),
	(
		opt_util__no_stackvars_til_decr_sp(Instrs0, N, Between, Remain)
	->
		list__append(Between, Remain, Instrs)
	;
		fail
	).

%-----------------------------------------------------------------------------%

	% Given a GC method, return the list of invalid peephole
	% optimizations.

:- pred peephole__invalid_opts(gc_method, list(pattern)).
:- mode peephole__invalid_opts(in, out) is det.

peephole__invalid_opts(GC_Method, InvalidPatterns) :-
	(
		GC_Method = accurate
	->
		InvalidPatterns = [incr_sp]
	;
		InvalidPatterns = []
	).

%-----------------------------------------------------------------------------%
