/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1999 David Baum.
 * All Rights Reserved.
 */


#include "parse_util.h"
#include "Expr.h"
#include "parser.h"
#include "AssignStmt.h"
#include "Condition.h"
#include "Clause.h"
#include "Symbol.h"
#include "GosubStmt.h"
#include "InlineStmt.h"
#include "Program.h"
#include "UnaryExpr.h"
#include "BinaryExpr.h"
#include "AtomExpr.h"
#include "ValueExpr.h"
#include "Error.h"

int GetConstantValue(const Expr *e)
{
	int v;
	bool ok;
	
	ok = e->Evaluate(v);
	delete e;
	
	if (!ok)
		Error(kErr_ConstantNeeded).RaiseLex();
	
	return v;
}


int GetLValue(const Expr *e)
{
	int lv = e->GetLValue();
	if (lv==kIllegalVar)
		Error(kErr_LValueNeeded).RaiseLex();
	delete e;
	return lv;
}


Expr *MakeUnaryExpr(int op, Expr *arg)
{
	if (UnaryExpr::NeedsConstant(op) &&
		!arg->PromiseConstant())
	{
		Error(kErr_ConstantNeeded).RaiseLex();
		// return a dummy expr so the parse tree is still valid
		return arg;
	}
	
	return new UnaryExpr(op, arg);
}


Expr *MakeBinaryExpr(Expr *lhs, int op, Expr *rhs)
{
	if (BinaryExpr::NeedsConstant(op) &&
		(!rhs->PromiseConstant() ||
		 !lhs->PromiseConstant()))
	{
		Error(kErr_ConstantNeeded).RaiseLex();
		// return a dummy expr so the parse tree is still valid
		delete rhs;
		return lhs;
	}
		
	return new BinaryExpr(lhs, op, rhs);
}


Expr *MakeValueExpr(Expr *e)
{
	if (!e->PromiseConstant())
	{
		Error(kErr_ConstantNeeded).RaiseLex();
		// return a dummy expr so the parse tree is still valid
		return e;
	}

	return new ValueExpr(e);
}
