/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "RCX_Link.h"
#include "RCX_Cmd.h"

#define kSerialPortEnv	"RCX_PORT"


#define kMaxTxData	(2*RCX_Link::kMaxCmdLength + 6)
#define kMaxRxData	(kMaxTxData + 2*RCX_Link::kMaxReplyLength + 5)

#define kFragmentChunk	20
#define kFirmwareChunk	200

#define	kDefaultRetryCount 5


static void DumpData(const UByte *ptr, int length);

static int FindRCXSync(const UByte *data, int length);
static int FindCMSync(const UByte *data, int length);
static RCX_Result VerifyReply(const UByte *data, int length, UByte cmd);

RCX_Link::Target RCX_Link::sDefaultTarget = RCX_Link::kTarget_RCX;


RCX_Link::RCX_Link()
{
	fTxData = new UByte[kMaxTxData];
	fRxData = new UByte[kMaxRxData];
	fVerbose = false;
	fSerial = PSerial::NewSerial();
	fTxLastCommand = 0;
	SetTarget(sDefaultTarget);
	fTarget = sDefaultTarget;
}


RCX_Link::~RCX_Link()
{
	Close();
	delete [] fTxData;
	delete [] fRxData;
	delete fSerial;
}


void RCX_Link::SetTarget(Target target)
{
	fTarget = target;
	fRxTimeout = kMinTimeout;
	fDynamicTimeout = true;
}


RCX_Result RCX_Link::Open(const char *portName)
{
	// see if an environment variable is set
	if (!portName) portName = getenv(kSerialPortEnv);
	
	// just use the default
	if (!portName) portName = PSerial::GetDefaultName();

	if (! fSerial->Open(portName)) return kRCX_OpenSerialError;
	
	fSerial->SetSpeed(2400, kPSerial_ParityOdd);
	
	if (fTarget == kTarget_CyberMaster)
	{
		fSerial->SetDTR(true);
		fSerial->SetRTS(false);
	}
	
	fSynced = false;
	fResult = kRCX_OK;
	
	return kRCX_OK;
}


void RCX_Link::Close()
{
	fSerial->Close();
}



void RCX_Link::BuildTxData(const UByte *data, int length)
{
	int i;
	UByte checksum = 0;
	UByte byte;
	
	UByte *ptr = fTxData;
	
	if (fTarget == kTarget_CyberMaster)
	{
		// CM sync pattern
		*ptr++ = 0xfe;
		*ptr++ = 0;
		*ptr++ = 0;
		*ptr++ = 0xff;
	}
	else
	{
		// RCX sync pattern
		*ptr++ = 0x55;
		*ptr++ = 0xff;
		*ptr++ = 0;
	}
	
	// interleaved data & inverse data
	for(i=0; i<length; i++)
	{
		byte = *data++;

		// correction for duplicate commands
		if (i==0)
		{
			if (byte==fTxLastCommand) byte ^=8;
			fTxLastCommand = byte;
		}
				
		*ptr++ = byte;
		*ptr++ = (UByte)~byte;
		checksum += byte;
	}
	
	// checksum
	*ptr++ = checksum;
	*ptr++ = (UByte)~checksum;
	
	fTxLength = ptr - fTxData;
}


RCX_Result RCX_Link::Sync()
{
	RCX_Cmd cmd;
	RCX_Result result;

	if (fSynced) return kRCX_OK;
	
	// always start with a ping
	result = Send(cmd.MakePing());
	// if error, try a second time
	if (result == kRCX_ReplyError)
		result = Send(&cmd);
	if (RCX_ERROR(result)) return result;

	// cybermaster requires an unlock also
	if (fTarget == kTarget_CyberMaster)
	{
		result = Send(cmd.MakeUnlockCM());
		if (RCX_ERROR(result)) return result;
	}

	fSynced = true;
	return kRCX_OK;
}


RCX_Result RCX_Link::SendFragment(RCX_FragmentType type, UByte number, const UByte *data, int length, int total)
{
	RCX_Cmd cmd;
	RCX_Result result;
	
	result = Sync();
	if (RCX_ERROR(result)) return result;
		
	result = Send(cmd.MakeBegin(type, number, (UShort)length));
	if (RCX_ERROR(result)) return result;

	// make sure we have enough room
	if (result != 1 ||
		GetReplyByte(0) != 0) return kRCX_MemFullError;	
	
	if (total == 0)
		total = length;
	
	if (total > 0)
		BeginProgress(total);

	result = Download(data, length, kFragmentChunk);
	return result;
}


RCX_Result RCX_Link::GetVersion(ULong &rom, ULong &ram)
{
	RCX_Cmd cmd;
	RCX_Result result;
	UByte reply[8];
	
	result = Sync();
	if (RCX_ERROR(result)) return result;
	
	result = Send(cmd.MakeUnlock());
	if (RCX_ERROR(result)) return result;
	
	if (result != 8) return kRCX_ReplyError;
	
	GetReply(reply,8);
	
	rom =	((ULong)reply[0] << 24) |
			((ULong)reply[1] << 16) |
			((ULong)reply[2] << 8) |
			((ULong)reply[3]);
	
	ram =	((ULong)reply[4] << 24) |
			((ULong)reply[5] << 16) |
			((ULong)reply[6] << 8) |
			((ULong)reply[7]);

	return kRCX_OK;
}


RCX_Result RCX_Link::GetValue(RCX_Value value)
{
	RCX_Cmd cmd;
	RCX_Result result;
	
	result = Sync();
	if (RCX_ERROR(result)) return result;
	
	result = Send(cmd.MakeRead(value));
	if (RCX_ERROR(result)) return result;
	if (result != 2) return kRCX_ReplyError;

	result = (int)GetReplyByte(0) + ((int)GetReplyByte(1) << 8);
	return result;	
}


RCX_Result RCX_Link::GetBatteryLevel()
{
	RCX_Cmd cmd;
	RCX_Result result;

	result = Sync();
	if (RCX_ERROR(result)) return result;
	
	result = Send(cmd.Set(kRCX_BatteryLevelOp));
	if (result != 2) return kRCX_ReplyError;
	
	result = (int)GetReplyByte(0) + ((int)GetReplyByte(1) << 8);

	return result;
}


RCX_Result RCX_Link::SendFirmware(const UByte *data, int length)
{
	RCX_Cmd cmd;
	RCX_Result result;
	int check;
	
	result = Sync();
	if (RCX_ERROR(result)) return result;
	
	result = Send(cmd.MakeUnlock());
	if (RCX_ERROR(result)) return result;

	result = Send(cmd.Set(kRCX_BootModeOp, 1, 3, 5, 7, 0xb));
	if (RCX_ERROR(result)) return result;
	
	check = 0;
	for(int i=0; i<length; i++)
		check += data[i];

	result = Send(cmd.Set(kRCX_BeginFirmwareOp, (UByte)(2*length), (UByte)((2*length)>>8),
		(UByte)check, (UByte)(check>>8), 0));
	if (RCX_ERROR(result)) return result;
	
	BeginProgress(length);
	result = Download(data, length, kFirmwareChunk);
	if (RCX_ERROR(result)) return result;
	
	result = Send(cmd.Set(kRCX_EndFirmwareOp, 0x4c, 0x45, 0x47, 0x4f, 0xae));
	return result;
}


RCX_Result RCX_Link::Download(const UByte *data, int length, int chunk)
{
	RCX_Cmd cmd;
	RCX_Result result;
	UShort seq;
	int remain = length;
	int n;
	
	seq = 1;
	while(remain > 0)
	{
		if (remain <= chunk)
		{
			seq = 0;
			n = remain;
		}
		else
			n = chunk;
	
		result = Send(cmd.MakeDownload(seq++, data, (UByte)n));
		if (result < 0) return result;
		
		remain -= n;
		data += n;
		if (!IncrementProgress(n)) return kRCX_AbortError;
	}

	return kRCX_OK;
}


RCX_Result RCX_Link::Send(const RCX_Cmd *cmd, bool retry)
{
	return Send(cmd->GetBody(), cmd->GetLength(), retry);
}


RCX_Result RCX_Link::Send(const UByte *data, int length, bool retry)
{
	if (length > kMaxCmdLength) return kRCX_RequestError;

	// format the command
	BuildTxData(data, length);

	// try sending
	int tries = retry ? kDefaultRetryCount : 1;
	int originalTimeout = fRxTimeout;
	int i;
	
	for(i=0; i<tries; i++)
	{
		fResult = _SendFromTxBuffer();
		
		if (!RCX_ERROR(fResult))
		{
			// success
			if (i==0)
			{
				// worked on first try, lets see if we can go faster next time
				if (fDynamicTimeout && fRxTimeout > kMinTimeout)
				{
					int newTimeout = fRxTimeout - (fRxTimeout / 10);
					if (newTimeout < kMinTimeout)
						newTimeout = kMinTimeout;
					fRxTimeout = newTimeout;
				}
			}
			return fResult;
		}
		
		// only the second kRCX_IREchoError is catastrophic
		// this is somewhat of a hack - I really should keep track
		// of the echo, but for now this makes sure that a serial
		// level failure on a single packet doesn't kill the entire
		// send
		if (fResult == kRCX_IREchoError && i > 0) break;

		if (fDynamicTimeout && i>0)
		{
			// increase retry time
			fRxTimeout *= 2;
			if (fRxTimeout > kMaxTimeout) fRxTimeout = kMaxTimeout;
		}
	}
	
	// retries exceeded, restore original timeout and lose the sync
	fRxTimeout = originalTimeout;
	fSynced = false;
	return fResult;
}


RCX_Result RCX_Link::_SendFromTxBuffer()
{
	int offset;
	RCX_Result result;
	
	// drain serial rx buffer
	fSerial->SetTimeout(0);
	while(fSerial->Read(fRxData, kMaxRxData) > 0)
		;

	// send command	
	fSerial->Write(fTxData, fTxLength);
	fSerial->FlushWrite();
	if (fVerbose)
	{
		printf("Tx: ");
		DumpData(fTxData, fTxLength);
	}

	// get the reply
	fSerial->SetTimeout(fRxTimeout);
	fRxLength = 0;
	while(fRxLength < kMaxRxData)
	{
		if (fSerial->Read(fRxData+fRxLength, 1) != 1) break;
		fRxLength++;
	}
	
	if (fVerbose)
	{
		printf("Rx: ");
		DumpData(fRxData, fRxLength);
	}
	
	if (fRxLength == 0)
		return fTarget==kTarget_CyberMaster ? kRCX_ReplyError : kRCX_IREchoError;
	
	// find reply
	offset = 0;
	while(1)
	{
		if (fTarget == kTarget_CyberMaster)
			result = FindCMSync(fRxData + offset, fRxLength - offset);
		else
			result = FindRCXSync(fRxData + offset, fRxLength - offset);
				
		if (!result) return kRCX_ReplyError;
	
		offset += result;
		
		result = VerifyReply(fRxData + offset, fRxLength - offset, fTxLastCommand);
		if (result >= 0) break;
	}

	fReplyStart = offset + 2;
	return result;
}


RCX_Result VerifyReply(const UByte *data, int length, UByte cmd)
{
	UByte checksum;
	const UByte *ptr;
	const UByte *end;
	const UByte *match;
	
	// always need a cmd and a checksum
	if (length < 4) return kRCX_ReplyError;
	
	// check the cmd
	if ((data[0] != (~cmd & 0xff)) ||
		(data[1] != cmd)) return kRCX_ReplyError;

	ptr = data + 2;
	end = data + length - 1;
	checksum = data[0];
	match = nil;
	
	while(ptr < end)
	{
		if (ptr[0] != (~ptr[1] & 0xff)) break;
		
		if (ptr[0] == checksum)
			match = ptr;
		
		checksum += ptr[0];
		ptr += 2;
	}
	
	if (!match) return kRCX_ReplyError;
	
	return ((match - data) / 2) - 1;
}


RCX_Result	RCX_Link::GetReply(UByte *data, int maxLength)
{
	if (fResult < 0) return fResult;
	
	RCX_Result length;
	const UByte *src = fRxData + fReplyStart;
	
	for(length=0; (length<maxLength) && (length<fResult); length++)
	{
		*data++ = *src++;
		src++;
	}
	
	return length;
}


void RCX_Link::BeginProgress(int total)
{
	fDownloadTotal = total;
	fDownloadSoFar = 0;
}


bool RCX_Link::IncrementProgress(int delta)
{
	fDownloadSoFar += delta;
	return DownloadProgress(fDownloadSoFar, fDownloadTotal);
}


bool RCX_Link::DownloadProgress(int /* soFar */, int /* total */)
{
	return true;
}


int FindCMSync(const UByte *data, int length)
{
	const UByte *end = data + length;
	const UByte *ptr;

	for(ptr=data; ptr<end; ptr++)
		if (*ptr == 0xff) return ptr-data+1;

	return 0;
}


int FindRCXSync(const UByte *data, int length)
{
	const UByte *end = data + length - 2;
	const UByte *ptr;
	
	for(ptr=data; ptr<end; ptr++)
	{
		if (ptr[0]==0x55 &&
			ptr[1]==0xff &&
			ptr[2]==0x00) return ptr-data+3;
	}
	
	return 0;
}


void DumpData(const UByte *ptr, int length)
{
	int i;
	
	for(i=0; i<length; i++)
	{
		printf("%02x ", *ptr++);
//		if ((i%16)==15) putchar('\n');
	}

	putchar('\n');
}
