/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/* FLASK */

%{
#include "policydb.h"

policydb_t *policydbp;
queue_t id_queue = 0;

extern char yytext[];
extern int yyerror(char *msg);

static char errormsg[255];

static int insert_separator(int);
static int insert_id(char *,int);
static int define_class(void);
static int define_sid(void);
static int define_common_perms(void);
static int define_av_perms(int);
static int define_role_types(void);
static role_datum_t *define_role_dom(role_datum_t*);
static role_datum_t *merge_roles_dom(role_datum_t*,role_datum_t*);
static int define_type(void);
static int define_transition(void);
static int define_sens(void);
static int define_dominance(void);
static int define_category(void);
static int define_level(void);
static int define_user(void);
static int define_sid_context(void);
static int define_common_base(void);
static int define_av_base(void);
static int define_default(int);
static int define_rule(int);
static int define_constraint(constraint_expr_t*);
static constraint_expr_t *define_constraint_expr(constraint_expr_t*,
						 constraint_expr_t*,
						 unsigned);
%}

%token COMMON
%token CLASS
%token CONSTRAIN
%token INHERITS
%token SID
%token ROLE
%token ROLES
%token TYPE
%token TYPES
%token ALIAS
%token TYPE_TRANSITION
%token SENSITIVITY
%token DOMINANCE
%token CATEGORY
%token LEVEL
%token RANGES
%token USER
%token DEFAULT
%token ALLOW
%token AUDITALLOW
%token AUDITDENY
%token NOTIFY
%token SOURCE
%token TARGET
%token SAMEUSER
%token NOT AND OR
%token IDENTIFIER

%left OR
%left AND
%right NOT
%%
policy			: classes initial_sids access_vectors mls te rbac constraints users initial_sid_contexts
			;
classes			: class_def
			| classes class_def
			;
class_def		: CLASS identifier
			{if (define_class()) return -1;}
			;
initial_sids 		: initial_sid_def
			| initial_sids initial_sid_def
			;
initial_sid_def		: SID identifier
                        {if (define_sid()) return -1;}
			;
access_vectors		: common_perms av_perms
			;
common_perms		: common_perms_def
			| common_perms common_perms_def
			;
common_perms_def	: COMMON identifier '{' identifier_list '}'
			{if (define_common_perms()) return -1;}
			;
av_perms		: av_perms_def
			| av_perms av_perms_def
			;
av_perms_def		: CLASS identifier '{' identifier_list '}'
			{if (define_av_perms(FALSE)) return -1;}
                        | CLASS identifier INHERITS identifier
			{if (define_av_perms(TRUE)) return -1;}
                        | CLASS identifier INHERITS identifier '{' identifier_list '}'
			{if (define_av_perms(TRUE)) return -1;}
			;
mls			: sensitivities dominance categories levels base_perms
			;
sensitivities	 	: sensitivity_def
			| sensitivities sensitivity_def
			;
sensitivity_def		: SENSITIVITY identifier alias_def ';'
			{if (define_sens()) return -1;}
			| SENSITIVITY identifier ';'
			{if (define_sens()) return -1;}
	                ;
alias_def		: ALIAS '{' identifier_list '}'
			| ALIAS identifier
			;
dominance		: DOMINANCE identifier
			{if (define_dominance()) return -1;}
                        | DOMINANCE '{' identifier_list '}'
			{if (define_dominance()) return -1;}
			;
categories 		: category_def
			| categories category_def
			;
category_def		: CATEGORY identifier alias_def ';'
			{if (define_category()) return -1;}
			| CATEGORY identifier ';'
			{if (define_category()) return -1;}
			;
levels	 		: level_def
			| levels level_def
			;
level_def		: LEVEL identifier ':' cat_set ';'
			{if (define_level()) return -1;}
			| LEVEL identifier ';'
			{if (define_level()) return -1;}
			;
base_perms		: common_base av_base
			;
common_base		: common_base_def
			| common_base common_base_def
			;
common_base_def	        : COMMON identifier '{' perm_base_list '}'
	                {if (define_common_base()) return -1;}
			;
av_base		        : av_base_def
			| av_base av_base_def
			;
av_base_def		: CLASS identifier '{' perm_base_list '}'
	                {if (define_av_base()) return -1;}
			| CLASS identifier
	                {if (define_av_base()) return -1;}
			;
perm_base_list		: perm_base
			| perm_base_list perm_base
			;
perm_base		: identifier ':' identifier
			{if (insert_separator(0)) return -1;}
                        | identifier ':' '{' identifier_list '}'
			{if (insert_separator(0)) return -1;}
			;
te			: types transitions defaults rules
			;
types			: type_def
			| types type_def
			;
type_def		: TYPE identifier alias_def ';'
                        {if (define_type()) return -1;}
	                | TYPE identifier ';'
                        {if (define_type()) return -1;}
    			;
transitions		: transition_def
			| transitions transition_def
			;
transition_def		: TYPE_TRANSITION identifier identifier ':' identifier identifier ';'
                        {if (define_transition()) return -1;}
    			;
defaults		: default_def |
			| defaults default_def
			;
default_def		: DEFAULT ALLOW identifier ';'
			{if (define_default(0)) return -1;}
			| DEFAULT AUDITALLOW identifier ';'
			{if (define_default(1)) return -1;}
			| DEFAULT AUDITDENY identifier ';'
			{if (define_default(2)) return -1;}
			| DEFAULT NOTIFY identifier ';'
			{if (define_default(3)) return -1;}
			;
rules			: rule_def
                        | rules rule_def
			;
rule_def		: allow_def
			| auditallow_def
			| auditdeny_def
			| notify_def
			;
allow_def		: ALLOW identifier identifier ':' names_sep names  ';'
			{if (define_rule(AVTAB_ALLOWED)) return -1; }
		        ;
auditallow_def		: AUDITALLOW identifier identifier ':' names_sep names ';'
			{if (define_rule(AVTAB_AUDITALLOW)) return -1; }
		        ;
auditdeny_def		: AUDITDENY identifier identifier ':' names_sep names ';'
			{if (define_rule(AVTAB_AUDITDENY)) return -1; }
		        ;
notify_def		: NOTIFY identifier identifier ':' names_sep names ';'
			{if (define_rule(AVTAB_NOTIFY)) return -1; }
		        ;
rbac			: role_types role_dominance
                        ;
role_types		: role_type_def
                        | role_types role_type_def
			;
role_type_def		: ROLE identifier TYPES names ';'
			{if (define_role_types()) return -1;}
                        ;
role_dominance		: DOMINANCE '{' roles '}'
			;
roles			: role_def
			{ $$ = $1; }
			| roles role_def
			{ $$ = (int) merge_roles_dom((role_datum_t*)$1, (role_datum_t*)$2); if ($$ == 0) return -1;}
			;
role_def		: ROLE identifier_push ';'
                        {$$ = (int) define_role_dom(NULL); if ($$ == 0) return -1;}
			| ROLE identifier_push '{' roles '}'
                        {$$ = (int) define_role_dom((role_datum_t*)$4); if ($$ == 0) return -1;}
			;
constraints		: constraint_def
			| constraints constraint_def
			;
constraint_def		: CONSTRAIN names_sep names_pushsep constraint_expr ';'
			{ if (define_constraint((constraint_expr_t*)$4)) return -1; }
			;
constraint_expr		: '(' constraint_expr ')'
			{ $$ = $2; }
			| constraint_expr AND constraint_expr
			{ $$ = (int) define_constraint_expr((constraint_expr_t*)$1,
					(constraint_expr_t*)$3,
					CONSTRAINT_EXPR_TYPE_AND);
			  if ($$ == 0) return -1; }
			| constraint_expr OR constraint_expr
			{ $$ = (int) define_constraint_expr((constraint_expr_t*)$1,
					(constraint_expr_t*)$3,
					CONSTRAINT_EXPR_TYPE_OR);
			  if ($$ == 0) return -1; }
			| NOT constraint_expr
			{ $$ = (int) define_constraint_expr((constraint_expr_t*)$2,
					NULL, CONSTRAINT_EXPR_TYPE_NOT);
			  if ($$ == 0) return -1; }
			| constraint_expr_prim
			{ $$ = $1; }
			;
constraint_expr_prim	: SAMEUSER
			{ $$ = (int) define_constraint_expr(NULL,NULL,
						CONSTRAINT_EXPR_TYPE_SAMEUSER);
			  if ($$ == 0) return -1; }
			| SOURCE ROLE names_push
			{ $$ = (int) define_constraint_expr(NULL,NULL,
						CONSTRAINT_EXPR_TYPE_ROLE_SOURCE);
			  if ($$ == 0) return -1; }
			| TARGET ROLE names_push
			{ $$ = (int) define_constraint_expr(NULL,NULL,
						CONSTRAINT_EXPR_TYPE_ROLE_TARGET);
			  if ($$ == 0) return -1; }
			| ROLE identifier_push
			{ $$ = (int) define_constraint_expr(NULL,NULL,
						CONSTRAINT_EXPR_TYPE_ROLE_RELATION);
			  if ($$ == 0) return -1; }
			| SOURCE TYPE names_push
			{ $$ = (int) define_constraint_expr(NULL,NULL,
						CONSTRAINT_EXPR_TYPE_TYPE_SOURCE);
			  if ($$ == 0) return -1; }
			| TARGET TYPE names_push
			{ $$ = (int) define_constraint_expr(NULL,NULL,
						CONSTRAINT_EXPR_TYPE_TYPE_TARGET);
			  if ($$ == 0) return -1; }
			;
users			: user_def
			| users user_def
			;
user_def		: USER identifier ROLES names_sep RANGES user_ranges ';'
	                {if (define_user()) return -1;}
			;
user_ranges		: mls_range_def
			| '{' user_range_def_list '}'
			;
user_range_def_list	: mls_range_def
			| user_range_def_list mls_range_def
			;
initial_sid_contexts	: initial_sid_context_def
			| initial_sid_contexts initial_sid_context_def
			;
initial_sid_context_def	: SID identifier security_context_def
			{if (define_sid_context()) return -1;}
			;
security_context_def	: identifier ':' identifier ':' identifier ':' mls_range_def
	                ;
mls_range_def		: mls_level_def '-' mls_level_def
			{if (insert_separator(0)) return -1;}
	                | mls_level_def
			{if (insert_separator(0)) return -1;}
	                ;
mls_level_def		: identifier ':' cat_set
			{if (insert_separator(0)) return -1;}
	                | identifier
			{if (insert_separator(0)) return -1;}
	                ;
cat_set			: identifier
			| cat_set ',' identifier
			;
names_sep 		: identifier
			{ if (insert_separator(0)) return -1; }
			| '{' identifier_list '}'
			{ if (insert_separator(0)) return -1; }
			;
names			: identifier
			| '{' identifier_list '}'
			;
names_pushsep		: identifier
			{ if (insert_separator(1)) return -1; }
			| '{' identifier_list '}'
			{ if (insert_separator(1)) return -1; }
			;
names_push		: identifier_push
			| '{' identifier_list_push '}'
			;
identifier_list_push	: identifier_push
			| identifier_list_push identifier_push
			;
identifier_push		: IDENTIFIER
			{ if (insert_id(yytext, 1)) return -1; }
			;
identifier_list		: identifier
			| identifier_list identifier
			;
identifier		: IDENTIFIER
			{ if (insert_id(yytext,0)) return -1; }
			;
%%
static int
insert_separator(int push)
{
	int             error;

	if (push)
		error = queue_push(id_queue, 0);
	else
		error = queue_insert(id_queue, 0);

	if (error) {
		yyerror("queue overflow");
		return -1;
	}
	return 0;
}

static int
insert_id(char *id, int push)
{
	char           *newid;
	int             error;

	newid = (char *) MALLOC(strlen(id) + 1);
	if (!newid) {
		yyerror("out of memory");
		return -1;
	}
	strcpy(newid, id);
	if (push)
		error = queue_push(id_queue, (queue_element_t) newid);
	else
		error = queue_insert(id_queue, (queue_element_t) newid);

	if (error) {
		yyerror("queue overflow");
		free(newid);
		return -1;
	}
	return 0;
}


static int
define_class(void)
{
	char           *id;
	class_datum_t  *datum;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no class name for class definition?");
		return -1;
	}
	datum = (class_datum_t *) MALLOC(sizeof(class_datum_t));
	if (!datum) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	datum->value = ++policydbp->classes->nprim;

	ret = hashtab_insert(policydbp->classes->table,
			     (hashtab_key_t) id, (hashtab_datum_t) datum);

	if (ret == HASHTAB_PRESENT) {
		--policydbp->classes->nprim;
		FREE(datum, sizeof(class_datum_t));
		FREE(id, strlen(id) + 1);
		yyerror("duplicate class definition");
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		FREE(id, strlen(id) + 1);
		FREE(datum, sizeof(class_datum_t));
		return -1;
	}
	return 0;
}

static int
define_sid(void)
{
	char           *id;
	initial_sid_datum_t *datum;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no sid name for SID definition?");
		return -1;
	}
	datum = (initial_sid_datum_t *) MALLOC(sizeof(initial_sid_datum_t));
	if (!datum) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	datum->sid = ++policydbp->sids->nprim;
	datum->isvalid = FALSE;
	root_context_init(&datum->context);

	ret = hashtab_insert(policydbp->sids->table,
			     (hashtab_key_t) id, (hashtab_datum_t) datum);

	if (ret == HASHTAB_PRESENT) {
		--policydbp->sids->nprim;
		FREE(datum, sizeof(initial_sid_datum_t));
		FREE(id, strlen(id) + 1);
		yyerror("duplicate SID definition");
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		FREE(id, strlen(id) + 1);
		FREE(datum, sizeof(initial_sid_datum_t));
		return -1;
	}
	return 0;
}

static int
define_common_perms(void)
{
	char           *id, *perm;
	symtab_t       *comdatum;
	perm_datum_t   *perdatum = 0;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no common name for common perm definition?");
		return -1;
	}
	comdatum = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!comdatum) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	ret = hashtab_insert(policydbp->commons->table,
			     (hashtab_key_t) id, (hashtab_datum_t) comdatum);

	if (ret == HASHTAB_PRESENT) {
		FREE(comdatum, sizeof(symtab_t));
		FREE(id, strlen(id) + 1);
		yyerror("duplicate common definition");
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		FREE(id, strlen(id) + 1);
		FREE(comdatum, sizeof(symtab_t));
		return -1;
	}
	symtab_init(comdatum);
	if (!comdatum->table) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		FREE(comdatum, sizeof(symtab_t));
		return -1;
	}
	while ((perm = queue_remove(id_queue))) {
		perdatum = (perm_datum_t *) MALLOC(sizeof(perm_datum_t));
		if (!perdatum) {
			yyerror("out of memory");
			FREE(perm, strlen(perm) + 1);
			return -1;
		}
		perdatum->value = ++comdatum->nprim;

		/*
		 * By default, we set all four base permissions on this
		 * permission. This means that if base_permissions is not
		 * explicitly defined for this permission, then this
		 * permission will only be granted in the equivalent case.
		 */
		perdatum->base_perms = MLS_BASE_READ | MLS_BASE_WRITE |
			MLS_BASE_READBY | MLS_BASE_WRITEBY;
		/* actual value set in define_av_base */

		if (perdatum->value >= (sizeof(access_vector_t) * 8)) {
			yyerror("too many permissions to fit in an access vector");
			FREE(perm, strlen(perm) + 1);
			FREE(perdatum, sizeof(perm_datum_t));
			return -1;
		}
		ret = hashtab_insert(comdatum->table,
				     (hashtab_key_t) perm,
				     (hashtab_datum_t) perdatum);

		if (ret == HASHTAB_PRESENT) {
			sprintf(errormsg, "duplicate permission %s in common %s",
				perm, id);
			yyerror(errormsg);
			FREE(perm, strlen(perm) + 1);
			FREE(perdatum, sizeof(perm_datum_t));
			return -1;
		}
		if (ret == HASHTAB_OVERFLOW) {
			yyerror("hash table overflow");
			FREE(perm, strlen(perm) + 1);
			FREE(perdatum, sizeof(perm_datum_t));
			return -1;
		}
	}

	return 0;
}

static int
define_av_perms(int common)
{
	char           *id;
	avdef_datum_t  avddatum, *avddatump;
	class_datum_t  *cladatum;
	symtab_t       *comdatum;
	perm_datum_t   *perdatum, *perdatum2 = 0;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no tclass name for av perm definition?");
		return -1;
	}
	cladatum = (class_datum_t *) hashtab_search(policydbp->classes->table,
						    (hashtab_key_t) id);
	if (!cladatum) {
		sprintf(errormsg, "class %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}

	FREE(id, strlen(id) + 1);

	avddatump = avdeftab_search(policydbp->avdefs, cladatum->value);
	if (avddatump) {
		yyerror("duplicate access vector definition");
		return -1;
	}
	avddatum.common = 0;
	if (symtab_init(&avddatum.private)) {
		yyerror("out of memory");
		return -1;
	}
	avddatum.constraints = NULL;

	avddatum.mlsperms.read = 0;	/* actual value set in define_av_base */
	avddatum.mlsperms.readby = 0;	/* actual value set in define_av_base */
	avddatum.mlsperms.write = 0;	/* actual value set in define_av_base */
	avddatum.mlsperms.writeby = 0;	/* actual value set in define_av_base */

	if (common) {
		id = (char *) queue_remove(id_queue);
		if (!id) {
			yyerror("no inherits name for access vector definition?");
			return -1;
		}
		comdatum = (symtab_t *) hashtab_search(policydbp->commons->table,
						       (hashtab_key_t) id);

		if (!comdatum) {
			sprintf(errormsg, "common %s is not defined", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			return -1;
		}
		avddatum.common = comdatum;

		/*
		 * private permissions will start with values after the last
		 * common perm
		 */
		avddatum.private.nprim += comdatum->nprim;

		FREE(id, strlen(id) + 1);
	}
	while ((id = queue_remove(id_queue))) {
		perdatum = (perm_datum_t *) MALLOC(sizeof(perm_datum_t));
		if (!perdatum) {
			yyerror("out of memory");
			FREE(id, strlen(id) + 1);
			return -1;
		}
		perdatum->value = ++avddatum.private.nprim;

		/*
		 * By default, we set all four base permissions on this
		 * permission. This means that if base_permissions is not
		 * explicitly defined for this permission, then this
		 * permission will only be granted in the equivalent case.
		 */
		perdatum->base_perms = MLS_BASE_READ | MLS_BASE_WRITE |
			MLS_BASE_READBY | MLS_BASE_WRITEBY;
		/* actual value set in define_av_base */

		if (perdatum->value >= (sizeof(access_vector_t) * 8)) {
			yyerror("too many permissions to fit in an access vector");
			FREE(perdatum, sizeof(perm_datum_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
		if (common) {
			/*
			 * private perms and common perms exist in the same
			 * name space
			 */
			perdatum2 = (perm_datum_t *) hashtab_search(avddatum.common->table,
							(hashtab_key_t) id);
			if (perdatum2) {
				sprintf(errormsg, "permission %s conflicts with an inherited permission", id);
				yyerror(errormsg);
				FREE(perdatum, sizeof(perm_datum_t));
				FREE(id, strlen(id) + 1);
				return -1;
			}
		}
		/*
		 * insert into the private permission table
		 */
		ret = hashtab_insert(avddatum.private.table,
				     (hashtab_key_t) id,
				     (hashtab_datum_t) perdatum);

		if (ret == HASHTAB_PRESENT) {
			sprintf(errormsg, "duplicate permission %s", id);
			yyerror(errormsg);
			FREE(perdatum, sizeof(perm_datum_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
		if (ret == HASHTAB_OVERFLOW) {
			yyerror("hash table overflow");
			FREE(perdatum, sizeof(perm_datum_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
	}

	ret = avdeftab_insert(policydbp->avdefs, cladatum->value, &avddatum);
	if (ret == HASHTAB_PRESENT) {
		yyerror("duplicate access vector definition");
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		return -1;
	}

	return 0;
}


static int
define_role_types(void)
{
	role_datum_t   *role;
	char           *role_id, *id;
	type_datum_t   *typdatum;
	int             ret;

	role_id = queue_remove(id_queue);

	role = (role_datum_t *) MALLOC(sizeof(role_datum_t));
	if (!role) {
		yyerror("out of memory");
		free(role_id);
		return -1;
	}
	ebitmap_init(&role->dominates);
	ebitmap_init(&role->types);
	role->value = ++policydbp->roles->nprim;

	while ((id = queue_remove(id_queue))) {
		typdatum = (type_datum_t *) hashtab_search(policydbp->types->table,
							(hashtab_key_t) id);
		if (!typdatum) {
			sprintf(errormsg, "unknown type %s used in role types definition", id);
			yyerror(errormsg);
			free(id);
			continue;
		}
		if (!ebitmap_set_bit(&role->types, typdatum->info->value - 1, TRUE)) {
			yyerror("out of memory");
			free(id);
			free(role_id);
			return -1;
		}
		/* no need to keep type name */
		free(id);
	}

	ret = hashtab_insert(policydbp->roles->table,
			   (hashtab_key_t) role_id, (hashtab_datum_t) role);

	if (ret == HASHTAB_PRESENT) {
		--policydbp->roles->nprim;
		free(role);
		sprintf(errormsg, "duplicate definition for role %s", role_id);
		yyerror(errormsg);
		free(role_id);
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		free(role);
		free(id);
		free(role_id);
		return -1;
	}
	return 0;
}


static role_datum_t *
merge_roles_dom(role_datum_t * r1, role_datum_t * r2)
{
	role_datum_t   *new;

	new = malloc(sizeof(role_datum_t));
	if (!new) {
		yyerror("out of memory");
		return NULL;
	}
	new->value = 0;		/* temporary role */
	if (!ebitmap_or(&new->dominates, &r1->dominates, &r2->dominates)) {
		FREE(new, sizeof(role_datum_t));
		yyerror("out of memory");
		new = NULL;
	}
	if (!r1->value) {
		/* free intermediate result */
		ebitmap_destroy(&r1->dominates);
		FREE(r1, sizeof(role_datum_t));
	}
	if (!r2->value) {
		/* free intermediate result */
		yyerror("right hand role is temporary?");
		ebitmap_destroy(&r2->dominates);
		FREE(r2, sizeof(role_datum_t));
	}
	return new;
}


static role_datum_t *
define_role_dom(role_datum_t * r)
{
	role_datum_t   *role;
	char           *role_id;


	role_id = queue_remove(id_queue);
	role = (role_datum_t *) hashtab_search(policydbp->roles->table,
					       role_id);
	if (!role) {
		sprintf(errormsg, "no types definition for role %s", role_id);
		yyerror(errormsg);
		free(role_id);
		return NULL;
	}
	if (r) {
		if (!ebitmap_cpy(&role->dominates, &r->dominates)) {
			yyerror("out of memory");
			free(role_id);
			return NULL;
		}
		if (!r->value) {
			/* free intermediate result */
			ebitmap_destroy(&r->dominates);
			free(r);
		}
	}
	if (!ebitmap_set_bit(&role->dominates, role->value - 1, TRUE)) {
		yyerror("out of memory");
		free(role_id);
		return NULL;
	}
	return role;
}


static int
define_type(void)
{
	char           *id;
	type_info_t    *info;
	type_datum_t   *datum, *aliasdatum;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no type name for type definition?");
		return -1;
	}
	info = (type_info_t *) MALLOC(sizeof(type_info_t));
	if (!info) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	info->value = ++policydbp->types->nprim;

	datum = (type_datum_t *) MALLOC(sizeof(type_datum_t));
	if (!datum) {
		yyerror("out of memory");
		FREE(info, sizeof(type_info_t));
		FREE(id, strlen(id) + 1);
		return -1;
	}
	datum->isalias = FALSE;
	datum->info = info;

	ret = hashtab_insert(policydbp->types->table,
			     (hashtab_key_t) id, (hashtab_datum_t) datum);

	if (ret == HASHTAB_PRESENT) {
		--policydbp->types->nprim;
		FREE(datum, sizeof(type_datum_t));
		FREE(info, sizeof(type_info_t));
		sprintf(errormsg, "duplicate definition for type %s", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		FREE(datum, sizeof(type_datum_t));
		FREE(info, sizeof(type_info_t));
		FREE(id, strlen(id) + 1);
		return -1;
	}
	while ((id = queue_remove(id_queue))) {
		aliasdatum = (type_datum_t *) MALLOC(sizeof(type_datum_t));
		if (!aliasdatum) {
			yyerror("out of memory");
			return -1;
		}
		aliasdatum->isalias = TRUE;
		aliasdatum->info = info;

		ret = hashtab_insert(policydbp->types->table,
			  (hashtab_key_t) id, (hashtab_datum_t) aliasdatum);

		if (ret == HASHTAB_PRESENT) {
			sprintf(errormsg, "duplicate definition for type %s", id);
			yyerror(errormsg);
			FREE(aliasdatum, sizeof(type_datum_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
		if (ret == HASHTAB_OVERFLOW) {
			yyerror("hash table overflow");
			FREE(aliasdatum, sizeof(type_datum_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
	}

	return 0;
}

static int
define_transition(void)
{
	char           *id;
	tr_key_t       *trkey;
	tr_datum_t     *trdatum;
	type_datum_t   *datum;
	class_datum_t  *cladatum;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no oldtype in transition definition?");
		return -1;
	}
	trkey = (tr_key_t *) MALLOC(sizeof(tr_key_t));
	if (!trkey) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	datum = (type_datum_t *) hashtab_search(policydbp->types->table,
						(hashtab_key_t) id);
	if (!datum) {
		sprintf(errormsg, "unknown type %s used in transition definition", id);
		yyerror(errormsg);
		FREE(trkey, sizeof(tr_key_t));
		FREE(id, strlen(id) + 1);
		return -1;
	}
	trkey->source_type = datum->info->value;

	/* no need to keep type name */
	FREE(id, strlen(id) + 1);

	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no type in transition definition?");
		FREE(trkey, sizeof(tr_key_t));
		return -1;
	}
	datum = (type_datum_t *) hashtab_search(policydbp->types->table,
						(hashtab_key_t) id);
	if (!datum) {
		sprintf(errormsg, "unknown type %s used in transition definition", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		FREE(trkey, sizeof(tr_key_t));
		return -1;
	}
	trkey->target_type = datum->info->value;

	/* no need to keep type name */
	FREE(id, strlen(id) + 1);

	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no class in transition definition?");
		FREE(trkey, sizeof(tr_key_t));
		return -1;
	}
	cladatum = (class_datum_t *) hashtab_search(policydbp->classes->table,
						    (hashtab_key_t) id);
	if (!cladatum) {
		sprintf(errormsg, "unknown class %s used in transition definition", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		FREE(trkey, sizeof(tr_key_t));
		return -1;
	}
	trkey->object_class = cladatum->value;

	/* no need to keep class name */
	FREE(id, strlen(id) + 1);

	trdatum = (tr_datum_t *) hashtab_search(policydbp->transitions->table,
						(hashtab_key_t) trkey);
	if (trdatum) {
		yyerror("duplicate transition definition");
		FREE(trkey, sizeof(tr_key_t));
		return -1;
	}
	trdatum = (tr_datum_t *) MALLOC(sizeof(tr_datum_t));
	if (!trdatum) {
		yyerror("out of memory");
		FREE(trkey, sizeof(tr_key_t));
		return -1;
	}
	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no newtype in transition definition?");
		FREE(trkey, sizeof(tr_key_t));
		FREE(trdatum, sizeof(tr_datum_t));
		return -1;
	}
	datum = (type_datum_t *) hashtab_search(policydbp->types->table,
						(hashtab_key_t) id);
	if (!datum) {
		sprintf(errormsg, "unknown type %s used in transition definition", id);
		yyerror(errormsg);
		FREE(trkey, sizeof(tr_key_t));
		FREE(trdatum, sizeof(tr_datum_t));
		FREE(id, strlen(id) + 1);
		return -1;
	}
	trdatum->new_type = datum->info->value;

	/* no need to keep type name */
	FREE(id, strlen(id) + 1);

	policydbp->transitions->nel++;
	ret = hashtab_insert(policydbp->transitions->table,
			  (hashtab_key_t) trkey, (hashtab_datum_t) trdatum);

	if (ret == HASHTAB_PRESENT) {
		yyerror("duplicate definition for transition rule");
		FREE(trkey, sizeof(tr_key_t));
		FREE(trdatum, sizeof(tr_datum_t));
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		FREE(trkey, sizeof(tr_key_t));
		FREE(trdatum, sizeof(tr_datum_t));
		return -1;
	}
	return 0;
}

static int
define_sens(void)
{
	char           *id;
	mls_level_t    *level;
	level_datum_t  *datum, *aliasdatum;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no sensitivity name for sensitivity definition?");
		return -1;
	}
	level = (mls_level_t *) MALLOC(sizeof(mls_level_t));
	if (!level) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	level->sens = 0;	/* actual value set in define_dominance */
	++policydbp->levels->nprim;
	ebitmap_init(&level->cat);	/* actual value set in define_level */

	datum = (level_datum_t *) MALLOC(sizeof(level_datum_t));
	if (!datum) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		FREE(level, sizeof(mls_level_t));
		return -1;
	}
	datum->isalias = FALSE;
	datum->level = level;

	ret = hashtab_insert(policydbp->levels->table,
			     (hashtab_key_t) id, (hashtab_datum_t) datum);

	if (ret == HASHTAB_PRESENT) {
		--policydbp->levels->nprim;
		FREE(level, sizeof(mls_level_t));
		FREE(datum, sizeof(level_datum_t));
		sprintf(errormsg, "duplicate definition for sensitivity %s", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		FREE(id, strlen(id) + 1);
		FREE(level, sizeof(mls_level_t));
		FREE(datum, sizeof(level_datum_t));
		return -1;
	}
	while ((id = queue_remove(id_queue))) {
		aliasdatum = (level_datum_t *) MALLOC(sizeof(level_datum_t));
		if (!aliasdatum) {
			yyerror("out of memory");
			return -1;
		}
		aliasdatum->isalias = TRUE;
		aliasdatum->level = level;

		ret = hashtab_insert(policydbp->levels->table,
			  (hashtab_key_t) id, (hashtab_datum_t) aliasdatum);

		if (ret == HASHTAB_PRESENT) {
			sprintf(errormsg, "duplicate definition for level %s", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			FREE(aliasdatum, sizeof(level_datum_t));
			return -1;
		}
		if (ret == HASHTAB_OVERFLOW) {
			yyerror("hash table overflow");
			FREE(id, strlen(id) + 1);
			FREE(aliasdatum, sizeof(level_datum_t));
			return -1;
		}
	}

	return 0;
}

static int
define_dominance(void)
{
	level_datum_t  *datum;
	int             order;
	char           *id;


	order = 0;
	while ((id = (char *) queue_remove(id_queue))) {
		datum = (level_datum_t *) hashtab_search(policydbp->levels->table,
							 (hashtab_key_t) id);
		if (!datum) {
			sprintf(errormsg, "unknown sensitivity %s used in dominance definition", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			continue;
		}
		if (datum->level->sens != 0) {
			sprintf(errormsg, "sensitivity %s occurs multiply in dominance definition", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			return -1;
		}
		datum->level->sens = ++order;

		/* no need to keep sensitivity name */
		FREE(id, strlen(id) + 1);
	}

	if (order != policydbp->levels->nprim) {
		yyerror("all sensitivities must be specified in dominance definition");
		return -1;
	}
	return 0;
}

static int
define_category(void)
{
	char           *id;
	cat_datum_t    *datum, *aliasdatum;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no category name for category definition?");
		return -1;
	}
	datum = (cat_datum_t *) MALLOC(sizeof(cat_datum_t));
	if (!datum) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	datum->isalias = FALSE;
	datum->value = ++policydbp->cats->nprim;

	ret = hashtab_insert(policydbp->cats->table,
			     (hashtab_key_t) id, (hashtab_datum_t) datum);

	if (ret == HASHTAB_PRESENT) {
		--policydbp->cats->nprim;
		FREE(datum, sizeof(cat_datum_t));
		sprintf(errormsg, "duplicate definition for category %s", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		FREE(id, strlen(id) + 1);
		FREE(datum, sizeof(cat_datum_t));
		return -1;
	}
	while ((id = queue_remove(id_queue))) {
		aliasdatum = (cat_datum_t *) MALLOC(sizeof(cat_datum_t));
		if (!aliasdatum) {
			yyerror("out of memory");
			return -1;
		}
		aliasdatum->isalias = TRUE;
		aliasdatum->value = datum->value;

		ret = hashtab_insert(policydbp->cats->table,
			  (hashtab_key_t) id, (hashtab_datum_t) aliasdatum);

		if (ret == HASHTAB_PRESENT) {
			sprintf(errormsg, "duplicate definition for category %s", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			FREE(aliasdatum, sizeof(cat_datum_t));
			return -1;
		}
		if (ret == HASHTAB_OVERFLOW) {
			yyerror("hash table overflow");
			FREE(id, strlen(id) + 1);
			FREE(aliasdatum, sizeof(cat_datum_t));
			return -1;
		}
	}

	return 0;
}

static int
define_level(void)
{
	int             n;
	char           *id, *levid;
	level_datum_t  *levdatum;
	cat_datum_t    *catdatum;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no level name for level definition?");
		return -1;
	}
	levdatum = (level_datum_t *) hashtab_search(policydbp->levels->table,
						    (hashtab_key_t) id);
	if (!levdatum) {
		sprintf(errormsg, "unknown sensitivity %s used in level definition", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	if (ebitmap_length(&levdatum->level->cat)) {
		sprintf(errormsg, "sensitivity %s used in multiple level definitions", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	levid = id;
	n = 1;
	while ((id = queue_remove(id_queue))) {
		catdatum = (cat_datum_t *) hashtab_search(policydbp->cats->table,
							(hashtab_key_t) id);
		if (!catdatum) {
			sprintf(errormsg, "unknown category %s used in level definition", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			continue;
		}
		if (!ebitmap_set_bit(&levdatum->level->cat, catdatum->value - 1, TRUE)) {
			yyerror("out of memory");
			FREE(id, strlen(id) + 1);
			FREE(levid, strlen(levid) + 1);
			return -1;
		}
		/* no need to keep category name */
		FREE(id, strlen(id) + 1);

		n = n * 2;
	}

	FREE(levid, strlen(levid) + 1);

	policydbp->nlevels += n;

	return 0;
}

static int
define_user(void)
{
	char           *id, *levid;
	mls_range_t    *range;
	user_datum_t   *usrdatum;
	role_datum_t   *role;
	level_datum_t  *levdatum;
	cat_datum_t    *catdatum;
	int             ret, relation;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no user name for user definition?");
		return -1;
	}
	usrdatum = (user_datum_t *) MALLOC(sizeof(user_datum_t));
	if (!usrdatum) {
		yyerror("out of memory");
		FREE(id, strlen(id) + 1);
		return -1;
	}
	usrdatum->value = ++policydbp->users->nprim;
	ebitmap_init(&usrdatum->roles);
	usrdatum->ranges = queue_create();
	if (!usrdatum->ranges) {
		yyerror("out of memory");
		FREE(usrdatum, sizeof(user_datum_t));
		FREE(id, strlen(id) + 1);
		return -1;
	}
	ret = hashtab_insert(policydbp->users->table,
			     (hashtab_key_t) id, (hashtab_datum_t) usrdatum);

	if (ret == HASHTAB_PRESENT) {
		--policydbp->users->nprim;
		queue_destroy(usrdatum->ranges);
		FREE(usrdatum, sizeof(user_datum_t));
		FREE(id, strlen(id) + 1);
		yyerror("duplicate user definition");
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		yyerror("hash table overflow");
		queue_destroy(usrdatum->ranges);
		FREE(usrdatum, sizeof(user_datum_t));
		FREE(id, strlen(id) + 1);
		return -1;
	}
	while ((id = queue_remove(id_queue))) {
		role = (role_datum_t *) hashtab_search(policydbp->roles->table,
						       (hashtab_key_t) id);
		if (!role) {
			sprintf(errormsg, "unknown role %s used in user role definition", id
				);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			continue;
		}
		if (!ebitmap_set_bit(&usrdatum->roles, role->value - 1, TRUE)) {
			yyerror("out of memory");
			FREE(id, strlen(id) + 1);
			return -1;
		}
		/* no need to keep role name */
		FREE(id, strlen(id) + 1);
	}

	while ((id = queue_remove(id_queue))) {
		range = (mls_range_t *) MALLOC(sizeof(mls_range_t));
		if (!range) {
			yyerror("out of memory");
			FREE(id, strlen(id) + 1);
			return -1;
		}
		levdatum = (level_datum_t *) hashtab_search(policydbp->levels->table,
							(hashtab_key_t) id);
		if (!levdatum) {
			sprintf(errormsg, "unknown sensitivity %s used in user range definition", id);
			yyerror(errormsg);
			FREE(range, sizeof(mls_range_t));
			FREE(id, strlen(id) + 1);
			continue;
		}
		range->low.sens = levdatum->level->sens;
		ebitmap_init(&range->low.cat);

		levid = id;

		while ((id = queue_remove(id_queue))) {
			catdatum = (cat_datum_t *) hashtab_search(policydbp->cats->table,
							(hashtab_key_t) id);
			if (!catdatum) {
				sprintf(errormsg, "unknown category %s used in user range definition", id);
				yyerror(errormsg);
				FREE(id, strlen(id) + 1);
				continue;
			}
			if (!(ebitmap_get_bit(&levdatum->level->cat, catdatum->value - 1))) {
				sprintf(errormsg, "category %s cannot be associated with level %s", id, levid);
				yyerror(errormsg);
				FREE(id, strlen(id) + 1);
				continue;
			}
			if (!ebitmap_set_bit(&range->low.cat, catdatum->value - 1, TRUE)) {
				yyerror("out of memory");
				FREE(id, strlen(id) + 1);
				FREE(levid, strlen(levid) + 1);
				ebitmap_destroy(&range->low.cat);
				FREE(range, sizeof(mls_range_t));
				return -1;
			}
			/*
			 * no need to keep category name
			 */
			FREE(id, strlen(id) + 1);
		}

		/*
		 * no need to keep low sensitivity name
		 */
		FREE(levid, strlen(levid) + 1);

		id = queue_remove(id_queue);
		if (!id) {
			range->high.sens = range->low.sens;
			if (!ebitmap_cpy(&range->high.cat, &range->low.cat)) {
				yyerror("out of memory");
				FREE(id, strlen(id) + 1);
				ebitmap_destroy(&range->low.cat);
				FREE(range, sizeof(mls_range_t));
				return -1;
			}
		} else {
			levdatum = (level_datum_t *) hashtab_search(policydbp->levels->table,
							(hashtab_key_t) id);
			if (!levdatum) {
				sprintf(errormsg, "unknown sensitivity %s used in user range definition", id);
				yyerror(errormsg);
				FREE(id, strlen(id) + 1);
				ebitmap_destroy(&range->low.cat);
				FREE(range, sizeof(mls_range_t));
				return -1;
			}
			range->high.sens = levdatum->level->sens;
			ebitmap_init(&range->high.cat);

			levid = id;

			while ((id = queue_remove(id_queue))) {
				catdatum = (cat_datum_t *) hashtab_search(policydbp->cats->table,
							(hashtab_key_t) id);
				if (!catdatum) {
					sprintf(errormsg, "unknown category %s used in user range definition", id);
					yyerror(errormsg);
					FREE(id, strlen(id) + 1);
					continue;
				}
				if (!(ebitmap_get_bit(&levdatum->level->cat, catdatum->value - 1))) {
					sprintf(errormsg, "category %s cannot be associated with level %s", id, levid);
					yyerror(errormsg);
					FREE(id, strlen(id) + 1);
					continue;
				}
				if (!ebitmap_set_bit(&range->high.cat, catdatum->value - 1, TRUE)) {
					yyerror("out of memory");
					FREE(levid, strlen(levid) + 1);
					FREE(id, strlen(id) + 1);
					ebitmap_destroy(&range->low.cat);
					ebitmap_destroy(&range->high.cat);
					FREE(range, sizeof(mls_range_t));
					return -1;
				}
				/*
				 * no need to keep category name
				 */
				FREE(id, strlen(id) + 1);
			}

			/*
			 * no need to keep high sensitivity name
			 */
			FREE(levid, strlen(levid) + 1);

			id = queue_remove(id_queue);
			if (id) {
				yyerror("parse error");
				ebitmap_destroy(&range->low.cat);
				ebitmap_destroy(&range->high.cat);
				FREE(range, sizeof(mls_range_t));
				return -1;
			}
		}

		relation = mls_level_relation(range->high, range->low);
		if (!(relation & (MLS_RELATION_DOM | MLS_RELATION_EQ))) {
			/* high does not dominate low */
			yyerror("high does not dominate low");
			ebitmap_destroy(&range->low.cat);
			ebitmap_destroy(&range->high.cat);
			FREE(range, sizeof(mls_range_t));
			return -1;
		}
		if (queue_insert(usrdatum->ranges, (queue_element_t) range)) {
			yyerror("queue overflow");
			ebitmap_destroy(&range->low.cat);
			ebitmap_destroy(&range->high.cat);
			FREE(range, sizeof(mls_range_t));
			return -1;
		}
	}

	return 0;
}


static int
define_sid_context(void)
{
	char           *id, *levid;
	initial_sid_datum_t *siddatum;
	ss_root_context_t context;
	role_datum_t   *role;
	type_datum_t   *typdatum;
	level_datum_t  *levdatum;
	cat_datum_t    *catdatum;
	user_datum_t   *usrdatum;

	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no sid name for SID context definition?");
		return -1;
	}
	siddatum = (initial_sid_datum_t *) hashtab_search(policydbp->sids->table,
							(hashtab_key_t) id);
	if (!siddatum) {
		sprintf(errormsg, "SID %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	if (siddatum->isvalid) {
		sprintf(errormsg, "The context for SID %s is multiply defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	/* no need to keep the sid name */
	FREE(id, strlen(id) + 1);

	root_context_init(&context);

	/* parse the security context */

	/* extract the user */
	id = queue_remove(id_queue);
	if (!id) {
		yyerror("no effective user?");
		goto bad;
	}
	usrdatum = (user_datum_t *) hashtab_search(policydbp->users->table,
						   (hashtab_key_t) id);
	if (!usrdatum) {
		sprintf(errormsg, "user %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		goto bad;
	}
	context.user = usrdatum->value;

	/* no need to keep the user name */
	FREE(id, strlen(id) + 1);

	/* extract the role */
	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no role name for sid context definition?");
		return -1;
	}
	role = (role_datum_t *) hashtab_search(policydbp->roles->table,
					       (hashtab_key_t) id);
	if (!role) {
		sprintf(errormsg, "role %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	context.role = role->value;

	/* no need to keep the role name */
	FREE(id, strlen(id) + 1);


	/* extract the type */
	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no type name for sid context definition?");
		return -1;
	}
	typdatum = (type_datum_t *) hashtab_search(policydbp->types->table,
						   (hashtab_key_t) id);
	if (!typdatum) {
		sprintf(errormsg, "type %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	context.type = typdatum->info->value;

	/* no need to keep the type name */
	FREE(id, strlen(id) + 1);

	/* extract the low sensitivity */
	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no sensitivity name for sid context definition?");
		return -1;
	}
	levdatum = (level_datum_t *) hashtab_search(policydbp->levels->table,
						    (hashtab_key_t) id);
	if (!levdatum) {
		sprintf(errormsg, "Sensitivity %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	context.range.low.sens = levdatum->level->sens;

	/* extract low category set */
	levid = id;
	while ((id = queue_remove(id_queue))) {
		catdatum = (cat_datum_t *) hashtab_search(policydbp->cats->table,
							(hashtab_key_t) id);
		if (!catdatum) {
			sprintf(errormsg, "unknown category %s used in initial sid context", id);
			yyerror(errormsg);
			FREE(levid, strlen(levid) + 1);
			FREE(id, strlen(id) + 1);
			goto bad;
		}
		if (!ebitmap_set_bit(&context.range.low.cat,
				     catdatum->value - 1, TRUE)) {
			yyerror("out of memory");
			FREE(levid, strlen(levid) + 1);
			FREE(id, strlen(id) + 1);
			goto bad;
		}
		/* no need to keep category name */
		FREE(id, strlen(id) + 1);
	}

	/* no need to keep the sensitivity name */
	FREE(levid, strlen(levid) + 1);

	/* extract high sensitivity */
	id = (char *) queue_remove(id_queue);
	if (id) {
		levdatum = (level_datum_t *) hashtab_search(policydbp->levels->table,
							(hashtab_key_t) id);
		if (!levdatum) {
			sprintf(errormsg, "Sensitivity %s is not defined", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			goto bad;
		}
		context.range.high.sens = levdatum->level->sens;

		/* extract high category set */
		levid = id;
		while ((id = queue_remove(id_queue))) {
			catdatum = (cat_datum_t *) hashtab_search(policydbp->cats->table,
							(hashtab_key_t) id);
			if (!catdatum) {
				sprintf(errormsg, "unknown category %s used in initial sid context", id);
				yyerror(errormsg);
				FREE(levid, strlen(levid) + 1);
				FREE(id, strlen(id) + 1);
				goto bad;
			}
			if (!ebitmap_set_bit(&context.range.high.cat,
					     catdatum->value - 1, TRUE)) {
				yyerror("out of memory");
				FREE(levid, strlen(levid) + 1);
				FREE(id, strlen(id) + 1);
				goto bad;
			}
			/*
			 * no need to keep category name
			 */
			FREE(id, strlen(id) + 1);
		}

		/*
		 * no need to keep the sensitivity name
		 */
		FREE(levid, strlen(levid) + 1);

		id = (char *) queue_remove(id_queue);
		if (id) {
			yyerror("parse error");
			goto bad;
		}
	} else {
		context.range.high.sens = context.range.low.sens;
		if (!ebitmap_cpy(&context.range.high.cat, &context.range.low.cat)) {

			yyerror("out of memory");
			goto bad;
		}
	}

	/* intentional aliasing */
	siddatum->context = context;
	siddatum->isvalid = TRUE;

	return 0;

bad:
	root_context_destroy(&context);

	return -1;
}


static int
define_common_base(void)
{
	char           *id, *perm, *base;
	symtab_t       *comdatum;
	perm_datum_t   *perdatum;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no common name for common base definition?");
		return -1;
	}
	comdatum = (symtab_t *) hashtab_search(policydbp->commons->table,
					       (hashtab_key_t) id);
	if (!comdatum) {
		sprintf(errormsg, "common %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	while ((perm = queue_remove(id_queue))) {
		perdatum = (perm_datum_t *) hashtab_search(comdatum->table,
						      (hashtab_key_t) perm);
		if (!perdatum) {
			sprintf(errormsg, "permission %s is not defined for common %s", perm, id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			FREE(perm, strlen(perm) + 1);
			return -1;
		}
		/*
		 * An explicit definition of base_permissions for this
		 * permission.  Reset the value to zero.
		 */
		perdatum->base_perms = 0;

		while ((base = queue_remove(id_queue))) {
			if (!strcmp(base, "read"))
				perdatum->base_perms |= MLS_BASE_READ;
			else if (!strcmp(base, "write"))
				perdatum->base_perms |= MLS_BASE_WRITE;
			else if (!strcmp(base, "readby"))
				perdatum->base_perms |= MLS_BASE_READBY;
			else if (!strcmp(base, "writeby"))
				perdatum->base_perms |= MLS_BASE_WRITEBY;
			else if (strcmp(base, "none")) {
				sprintf(errormsg, "base permission %s is not defined", base);
				yyerror(errormsg);
				FREE(base, strlen(base) + 1);
				return -1;
			}
			FREE(base, strlen(base) + 1);
		}

		FREE(perm, strlen(perm) + 1);
	}

	FREE(id, strlen(id) + 1);

	return 0;
}


static int
common_base_set(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	perm_datum_t   *perdatum;
	avdef_datum_t  *avddatum;

	perdatum = (perm_datum_t *) datum;
	avddatum = (avdef_datum_t *) p;

	if (perdatum->base_perms & MLS_BASE_READ)
		avddatum->mlsperms.read |= (1 << (perdatum->value - 1));

	if (perdatum->base_perms & MLS_BASE_WRITE)
		avddatum->mlsperms.write |= (1 << (perdatum->value - 1));

	if (perdatum->base_perms & MLS_BASE_READBY)
		avddatum->mlsperms.readby |= (1 << (perdatum->value - 1));

	if (perdatum->base_perms & MLS_BASE_WRITEBY)
		avddatum->mlsperms.writeby |= (1 << (perdatum->value - 1));

	return 0;
}


static int
define_av_base(void)
{
	char           *id, *base;
	avdef_datum_t  *avddatum;
	class_datum_t  *cladatum;
	perm_datum_t   *perdatum;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no tclass name for av base definition?");
		return -1;
	}
	cladatum = (class_datum_t *) hashtab_search(policydbp->classes->table,
						    (hashtab_key_t) id);
	if (!cladatum) {
		sprintf(errormsg, "class %s is not defined", id);
		yyerror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}

	FREE(id, strlen(id) + 1);

	avddatum = avdeftab_search(policydbp->avdefs, cladatum->value);
	if (!avddatum) {
		yyerror("no access vector definition for this class pair");
		return -1;
	}
	/*
	 * Determine which common permissions should be included in each MLS
	 * vector for this access vector definition.
	 */
	if (avddatum->common)
		hashtab_map(avddatum->common->table, common_base_set, avddatum);

	while ((id = queue_remove(id_queue))) {
		perdatum = (perm_datum_t *) hashtab_search(avddatum->private.table,
							(hashtab_key_t) id);
		if (!perdatum) {
			sprintf(errormsg, "permission %s is not defined", id);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			return -1;
		}
		/*
		 * An explicit definition of base_permissions for this
		 * permission.  Reset the value to zero.
		 */
		perdatum->base_perms = 0;

		while ((base = queue_remove(id_queue))) {
			if (!strcmp(base, "read")) {
				perdatum->base_perms |= MLS_BASE_READ;
				avddatum->mlsperms.read |= (1 << (perdatum->value - 1));
			} else if (!strcmp(base, "write")) {
				perdatum->base_perms |= MLS_BASE_WRITE;
				avddatum->mlsperms.write |= (1 << (perdatum->value - 1));
			} else if (!strcmp(base, "readby")) {
				perdatum->base_perms |= MLS_BASE_READBY;
				avddatum->mlsperms.readby |= (1 << (perdatum->value - 1));
			} else if (!strcmp(base, "writeby")) {
				perdatum->base_perms |= MLS_BASE_WRITEBY;
				avddatum->mlsperms.writeby |= (1 << (perdatum->value - 1));
			} else if (strcmp(base, "none")) {
				sprintf(errormsg, "base permission %s is not defined", base);
				yyerror(errormsg);
				FREE(base, strlen(base) + 1);
				continue;
			}
			FREE(base, strlen(base) + 1);
		}

		FREE(id, strlen(id) + 1);
	}

	return 0;
}


static int
define_default(int which)
{
	int             value;
	char           *id;

	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no default name?");
		return -1;
	}
	if (!strcmp(id, "none"))
		value = TE_DEFAULT_NONE;
	else if (!strcmp(id, "all"))
		value = TE_DEFAULT_ALL;
	else if (!strcmp(id, "self"))
		value = TE_DEFAULT_SELF;
	else {
		free(id);
		sprintf(errormsg, "unrecognized default %s", id);
		yyerror(errormsg);
		return -1;
	}

	free(id);

	switch (which) {
	case 0:
		policydbp->defallow = value;
		break;
	case 1:
		policydbp->defauditallow = value;
		break;
	case 2:
		policydbp->defauditdeny = value;
		break;
	case 3:
		policydbp->defnotify = value;
		break;
	default:
		break;
	}

	return 0;
}


static int
define_rule(int which)
{
	char           *id, *tclassid, *stype, *ttype;
	avtab_key_t     avkey;
	avtab_datum_t   avdatum, *avdatump;
	avdef_datum_t  *avddatum;
	class_datum_t  *cladatum;
	type_datum_t   *typdatum;
	perm_datum_t   *perdatum;
	ebitmap_t       classmap;
	int             ret, i;


	stype = (char *) queue_remove(id_queue);
	if (!stype) {
		yyerror("no source type name for permission definition?");
		return -1;
	}
	typdatum = (type_datum_t *) hashtab_search(policydbp->types->table,
						   (hashtab_key_t) stype);
	if (!typdatum) {
		sprintf(errormsg, "unknown type %s used in permission definition", stype);
		yyerror(errormsg);
		FREE(stype, strlen(stype) + 1);
		return -1;
	}
	avkey.source_type = typdatum->info->value;

	ttype = (char *) queue_remove(id_queue);
	if (!ttype) {
		yyerror("no target type name for permission definition?");
		FREE(stype, strlen(stype) + 1);
		return -1;
	}
	typdatum = (type_datum_t *) hashtab_search(policydbp->types->table,
						   (hashtab_key_t) ttype);
	if (!typdatum) {
		sprintf(errormsg, "unknown type %s used in permission definition", ttype);
		yyerror(errormsg);
		FREE(ttype, strlen(ttype) + 1);
		FREE(stype, strlen(stype) + 1);
		return -1;
	}
	avkey.target_type = typdatum->info->value;

	ebitmap_init(&classmap);
	while ((tclassid = queue_remove(id_queue))) {
		cladatum = (class_datum_t *) hashtab_search(policydbp->classes->table,
						  (hashtab_key_t) tclassid);
		if (!cladatum) {
			sprintf(errormsg, "unknown class %s used in permission definition", tclassid);
			yyerror(errormsg);
			ebitmap_destroy(&classmap);
			FREE(tclassid, strlen(tclassid) + 1);
			FREE(ttype, strlen(ttype) + 1);
			FREE(stype, strlen(stype) + 1);
			return -1;
		}
		if (!ebitmap_set_bit(&classmap, cladatum->value - 1, TRUE)) {
			yyerror("out of memory");
			ebitmap_destroy(&classmap);
			FREE(tclassid, strlen(tclassid) + 1);
			FREE(ttype, strlen(ttype) + 1);
			FREE(stype, strlen(stype) + 1);
			return -1;
		}
		avkey.target_class = cladatum->value;

		avddatum = avdeftab_search(policydbp->avdefs, cladatum->value);
		if (!avddatum) {
			sprintf(errormsg, "no access vector definition for %s\n",
				tclassid);
			yyerror(errormsg);
			ebitmap_destroy(&classmap);
			FREE(tclassid, strlen(tclassid) + 1);
			FREE(ttype, strlen(ttype) + 1);
			FREE(stype, strlen(stype) + 1);
			return -1;
		}
		avdatump = avtab_search(policydbp->rules, &avkey);
		if (avdatump) {
			if (avdatump->specified & which) {
				sprintf(errormsg, "duplicate rule for (%s,%s:%s)",
					stype, ttype, tclassid);
				ebitmap_destroy(&classmap);
				yyerror(errormsg);
				FREE(tclassid, strlen(tclassid) + 1);
				FREE(ttype, strlen(ttype) + 1);
				FREE(stype, strlen(stype) + 1);
				return -1;
			}
			avdatump->specified |= which;
		} else {
			memset(&avdatum, 0, sizeof avdatum);
			avdatum.specified |= which;
			avdatum.avdefdatum = avddatum;
			ret = avtab_insert(policydbp->rules, &avkey, &avdatum);
			if (ret) {
				ebitmap_destroy(&classmap);
				yyerror("hash table overflow");
				FREE(tclassid, strlen(tclassid) + 1);
				FREE(ttype, strlen(ttype) + 1);
				FREE(stype, strlen(stype) + 1);
				return -1;
			}
		}

		free(tclassid);
	}

	FREE(ttype, strlen(ttype) + 1);
	FREE(stype, strlen(stype) + 1);

	while ((id = queue_remove(id_queue))) {
		for (i = 0; i < ebitmap_length(&classmap); i++) {
			if (ebitmap_get_bit(&classmap, i)) {
				avkey.target_class = i + 1;

				avddatum = avdeftab_search(policydbp->avdefs,
							   avkey.target_class);
				if (!avddatum) {
					yyerror("no access vector definition for class");
					ebitmap_destroy(&classmap);
					return -1;
				}
				avdatump = avtab_search(policydbp->rules, &avkey);

				perdatum = (perm_datum_t *) hashtab_search(avddatum->private.table,
							(hashtab_key_t) id);
				if (!perdatum) {
					if (avddatum->common) {
						perdatum = (perm_datum_t *) hashtab_search(avddatum->common->table,
							(hashtab_key_t) id);
					}
					if (!perdatum) {
						sprintf(errormsg, "permission %s is not defined for class", id);
						yyerror(errormsg);
						FREE(id, strlen(id) + 1);
						continue;
					}
				}
				switch (which) {
				case AVTAB_ALLOWED:
					avdatump->allowed |= (1 << (perdatum->value - 1));
					break;
#ifdef CONFIG_FLASK_AUDIT
				case AVTAB_AUDITALLOW:
					avdatump->auditallow |= (1 << (perdatum->value - 1));
					break;
				case AVTAB_AUDITDENY:
					avdatump->auditdeny |= (1 << (perdatum->value - 1));
					break;
#endif
#ifdef CONFIG_FLASK_NOTIFY
				case AVTAB_NOTIFY:
					avdatump->notify |= (1 << (perdatum->value - 1));
					break;
#endif
				}
			}
		}

		FREE(id, strlen(id) + 1);
	}

	ebitmap_destroy(&classmap);

	return 0;
}


static int
define_constraint(constraint_expr_t * expr)
{
	struct constraint_node *node;
	char           *id;
	class_datum_t  *cladatum;
	perm_datum_t   *perdatum;
	avdef_datum_t  *avddatum;
	ebitmap_t classmap;
	int i;


	ebitmap_init(&classmap);
	while ((id = queue_remove(id_queue))) {
		cladatum = (class_datum_t *) hashtab_search(policydbp->classes->table,
							(hashtab_key_t) id);
		if (!cladatum) {
			sprintf(errormsg, "class %s is not defined", id);
			ebitmap_destroy(&classmap);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			constraint_expr_destroy(expr);
			return -1;
		}
		if (!ebitmap_set_bit(&classmap, cladatum->value - 1, TRUE)) {
			yyerror("out of memory");
			ebitmap_destroy(&classmap);
			FREE(id, strlen(id) + 1);
			constraint_expr_destroy(expr);
			return -1;
		}

		avddatum = avdeftab_search(policydbp->avdefs, cladatum->value);
		if (!avddatum) {
			sprintf(errormsg, "class %s does not have an access vector interpretation", id);
			ebitmap_destroy(&classmap);
			yyerror(errormsg);
			FREE(id, strlen(id) + 1);
			constraint_expr_destroy(expr);
			return -1;
		}
		node = malloc(sizeof(struct constraint_node));
		if (!node) {
			yyerror("out of memory");
			constraint_expr_destroy(expr);
			return -1;
		}
		node->expr = expr;
		expr->count++;
		node->permissions = 0;

		if (!avddatum->constraints)
			policydbp->avdefs->ncons++;

		node->next = avddatum->constraints;
		avddatum->constraints = node;

		FREE(id, strlen(id) + 1);
	}

	expr->count--;

	while ((id = queue_remove(id_queue))) {
		for (i = 0; i < ebitmap_length(&classmap); i++) {
			if (ebitmap_get_bit(&classmap, i)) {
				avddatum = avdeftab_search(policydbp->avdefs,
							   i+1);

				node = avddatum->constraints;

				perdatum = (perm_datum_t *) hashtab_search(avddatum->private.table,
							(hashtab_key_t) id);
				if (!perdatum) {
					if (avddatum->common) {
						perdatum = (perm_datum_t *) hashtab_search(avddatum->common->table,
							(hashtab_key_t) id);
					}
					if (!perdatum) {
						sprintf(errormsg, "permission %s is not defined", id);
						yyerror(errormsg);
						FREE(id, strlen(id) + 1);
						ebitmap_destroy(&classmap);
						return -1;
					}
				}
				node->permissions |= (1 << (perdatum->value - 1));
			}
		}
		FREE(id, strlen(id) + 1);
	}

	ebitmap_destroy(&classmap);

	return 0;
}


static constraint_expr_t *
define_constraint_expr(constraint_expr_t * left,
		       constraint_expr_t * right,
		       unsigned expr_type)
{
	struct constraint_expr *expr;
	type_datum_t   *typdatum;
	role_datum_t   *role;
	char           *id;

	expr = malloc(sizeof(struct constraint_expr));
	if (!expr) {
		yyerror("out of memory");
		return NULL;
	}
	expr->left = left;
	expr->right = right;
	expr->expr_type = expr_type;
	expr->relation = 0;
	expr->count = 1;
	ebitmap_init(&expr->bitmap);

	switch (expr_type) {
	case CONSTRAINT_EXPR_TYPE_ROLE_RELATION:
		id = (char *) queue_remove(id_queue);
		if (!id) {
			yyerror("no id for relation?");
			free(expr);
			return NULL;
		}
		if (!strcmp(id, "dom"))
			expr->relation = CONSTRAINT_EXPR_VALUE_RELATION_DOM;
		else if (!strcmp(id, "domby"))
			expr->relation = CONSTRAINT_EXPR_VALUE_RELATION_DOMBY;
		else if (!strcmp(id, "eq"))
			expr->relation = CONSTRAINT_EXPR_VALUE_RELATION_EQ;
		else if (!strcmp(id, "incomp"))
			expr->relation = CONSTRAINT_EXPR_VALUE_RELATION_INCOMP;
		else {
			sprintf(errormsg, "unknown relation %s", id);
			yyerror(errormsg);
			free(expr);
			return NULL;
		}
		free(id);
		break;
	case CONSTRAINT_EXPR_TYPE_TYPE_SOURCE:
	case CONSTRAINT_EXPR_TYPE_TYPE_TARGET:
		while ((id = (char *) queue_remove(id_queue))) {
			typdatum = (type_datum_t *) hashtab_search(policydbp->types->table,
							(hashtab_key_t) id);
			if (!typdatum) {
				sprintf(errormsg, "unknown type %s", id);
				yyerror(errormsg);
				ebitmap_destroy(&expr->bitmap);
				free(expr);
				return NULL;
			}
			if (!ebitmap_set_bit(&expr->bitmap, typdatum->info->value - 1, TRUE)) {
				yyerror("out of memory");
				ebitmap_destroy(&expr->bitmap);
				free(expr);
				return NULL;
			}
			free(id);
		}
		break;
	case CONSTRAINT_EXPR_TYPE_ROLE_SOURCE:
	case CONSTRAINT_EXPR_TYPE_ROLE_TARGET:
		while ((id = (char *) queue_remove(id_queue))) {
			role = (role_datum_t *) hashtab_search(policydbp->roles->table,
							(hashtab_key_t) id);
			if (!role) {
				sprintf(errormsg, "unknown role %s", id);
				yyerror(errormsg);
				ebitmap_destroy(&expr->bitmap);
				free(expr);
				return NULL;
			}
			if (!ebitmap_set_bit(&expr->bitmap, role->value - 1, TRUE)) {
				yyerror("out of memory");
				ebitmap_destroy(&expr->bitmap);
				free(expr);
				return NULL;
			}
			free(id);
		}
		break;
	}

	return expr;
}

/* FLASK */
