/*
 * GSPOLF.C - polygon fill rendering routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

typedef struct s_PG_poly_fill_data PG_poly_fill_data;

struct s_PG_poly_fill_data
   {PG_device *dev;
    int nd;
    REAL **a;
    REAL *x;
    REAL *y;
    REAL *aext;
    byte *cnnct;
    pcons *alist;};

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_FILL_POLY - setup a window for a fill-poly rendering
 *                            - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_fill_poly(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int nde, nre, hvf, change;
    REAL nvxmn, nvxmx, nvymn, nvymx;
    REAL *dpex, *ddex, *pdx, *rpex, *rdex, *prx, *vwprt;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    pd->legend_fl = FALSE;

    alst = pd->alist;
    pri  = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = PLOT_FILL_POLY;};};

/* setup the viewport */
    vwprt = pd->viewport;
    hvf   = _PG_palette_orientation;
    if (vwprt != NULL)
       {nvxmn = vwprt[0];
        nvxmx = vwprt[1];
        nvymn = vwprt[2];
        nvymx = vwprt[3];}
    else
       {if (hvf == VERTICAL)
	   {nvxmn = 0.175;
	    nvxmx = 0.725 - _PG_palette_width;
	    nvymn = 0.175;
	    nvymx = 0.825;}

	else if (hvf == HORIZONTAL)
	   {nvxmn = 0.175;
	    nvxmx = 0.85;
	    nvymn = 0.23 + _PG_palette_width;
	    nvymx = 0.825;};}

    if (change)
       {PG_set_viewport(dev, nvxmn, nvxmx, nvymn, nvymx);

/* find the extrema for this frame */
	PG_find_extrema(data, 0.0, &dpex, &rpex, &nde, &ddex, &nre, &rdex);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

/* setup the range limits */
	prx = ((dev->autorange == TRUE) || (rpex == NULL)) ? rdex : rpex;
	PG_register_range_extrema(dev, nre, prx);

	SFREE(ddex);
	SFREE(rdex);};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FILL_HAND - draw and fill the polygons for the given data set */

static void PG_fill_hand(dev, g, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *g;
   PFByte fnc_zc, fnc_nc;
   {int i, npts, centering, clip, prec, nd, same;
    char bf[MAXLINE], *mtype, *s;
    REAL charspace, chupx, chupy;
    REAL chpthx, chpthy;
    REAL *rx, *ry, **afs, **afd, **r;
    REAL dextr[4], *fextr, *rextr, *aext;
    byte *cnnct;
    PM_mapping *h;
    PM_set *domain, *range;
    PG_dev_attributes *attr;
    pcons *alst;

    fextr = dev->range_extrema;

    for (h = g->f; h != NULL; h = h->next)
        {domain = h->domain;
	 npts   = domain->n_elements;
	 r      = (REAL **) domain->elements;

	 rx = PM_array_real(domain->element_type, r[0], npts, NULL);
	 ry = PM_array_real(domain->element_type, r[1], npts, NULL);

	 PM_array_real(domain->element_type, domain->extrema, 4, dextr);

	 range = h->range;
	 strcpy(bf, range->element_type);
	 mtype = SC_strtok(bf, " *", s);
	 npts  = range->n_elements;
	 nd    = range->dimension_elem;

	 same = (strcmp(mtype, SC_REAL_S) == 0);
	 if (same)
	    afd  = (REAL **) range->elements;
	 else
	    {afs = (REAL **) range->elements;
	     afd = FMAKE_N(REAL *, nd, "PG_FILL_HAND:afd");
	     for (i = 0; i < nd; i++)
	         CONVERT(SC_REAL_S, &afd[i], mtype, afs[i], npts, FALSE);};

/* find the range limits if any */
	 rextr = PM_get_limits(range);
	 aext  = ((rextr != NULL) && !dev->autorange) ? rextr : fextr;

/* find the additional mapping information */
	 centering = N_CENT;
	 alst = PM_mapping_info(h,
				"CENTERING", &centering,
				NULL);

/* save user's values for various attributes */
	 attr = PG_get_attributes(dev);

/* set attribute values */
	 clip      = TRUE;
	 prec      = TEXT_CHARACTER_PRECISION;
	 charspace = 0.0;
	 chupx     = 0.0;
	 chupy     = 1.0;
	 chpthx    = 1.0;
	 chpthy    = 0.0;
	 PG_set_clipping(dev, clip);
	 PG_set_char_precision(dev, prec);
	 PG_set_char_path(dev, chpthx, chpthy);
	 PG_set_char_up(dev, chupx, chupy);
	 PG_set_char_space(dev, charspace);
  
	 cnnct = PM_connectivity(h);

	 switch (centering)
	    {case Z_CENT :
	          (*fnc_zc)(dev, nd, afd, rx, ry, aext, cnnct, alst);
		  break;

	     case N_CENT :
                  (*fnc_nc)(dev, nd, afd, rx, ry, aext, cnnct, alst);
		  break;

             case F_CENT :
             case U_CENT :
             default     :
                  break;};

	 PG_draw_domain_boundary(dev, h);

	 PG_update_vs(dev);

	 PG_set_clipping(dev, FALSE);

/* reset user's values for various attributes */
	 PG_set_attributes(dev, attr);
	 SFREE(attr);

	 SFREE(rx);
	 SFREE(ry);
	 if (!same)
	    {for (i = 0; i < nd; i++)
	         SFREE(afd[i]);
	     SFREE(afd);};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FILL_CHUNK_ZC_LR - do a chunk of the fill poly work */

static void *_PG_fill_chunk_zc_lr(arg)
   void *arg;
   {int in, iz, j, k, l, nn, nmap, nz, color, kz, lz;
    int xok, yok, xso, yso;
    int lmn, lmx, nd;
    int *maxes, kmax, lmax, eflag;
    REAL xmn, xmx, ymn, ymx;
    REAL rx[5], ry[5], *vl;
    REAL *x1, *x2, *x3, *x4;
    REAL *y1, *y2, *y3, *y4;
    REAL **a, *x, *y, *aext;
    char *emap;
    byte *cnnct, *rv;
    pcons *alist;
    PG_device *dev;
    PG_poly_fill_data *par;

    par = (PG_poly_fill_data *) arg;

    dev   = par->dev;
    nd    = par->nd;
    a     = par->a;
    x     = par->x;
    y     = par->y;
    aext  = par->aext;
    cnnct = par->cnnct;
    alist = par->alist;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    nn    = kmax*lmax;
    nmap  = (kmax - 1) * (lmax - 1);
    nz    = SC_arrlen(a[0])/sizeof(REAL);

    LR_MAPPING_INFO(alist, nmap);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

    PM_LOGICAL_ZONE(x, x1, x2, x3, x4, kmax);
    PM_LOGICAL_ZONE(y, y1, y2, y3, y4, kmax);

    PG_get_window(dev, &xmn, &xmx, &ymn, &ymx);

    kz = kmax - 1;
    lz = lmax - 1;

    SC_chunk_split(&lmn, &lmx, &rv);

    vl = FMAKE_N(REAL, nd, "_PG_FILL_CHUNK_ZC_LR:vl");

    for (l = lmn; l < lmx; l++)
        for (k = 0; k < kz; k++)
            {in = l*kmax + k;
	     iz = l*kz + k;
	     if (emap[iz] == 0)
	        continue;

	     PG_SET_WITH_CLIP_INIT(xok, xso);
	     PG_SET_WITH_CLIP_INIT(yok, yso);

	     PG_SET_WITH_CLIP(rx[0], x1[in], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[0], y1[in], yok, yso, ymn, ymx);

	     PG_SET_WITH_CLIP(rx[1], x2[in], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[1], y2[in], yok, yso, ymn, ymx);

	     PG_SET_WITH_CLIP(rx[2], x3[in], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[2], y3[in], yok, yso, ymn, ymx);

	     PG_SET_WITH_CLIP(rx[3], x4[in], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[3], y4[in], yok, yso, ymn, ymx);

	     if (xok && yok)
	        {rx[4] = x1[in];
		 ry[4] = y1[in];

		 for (j = 0; j < nd; j++)
		     vl[j] = a[j][iz];

		 color = PG_select_color(dev, nd, vl, aext);
         
		 PG_set_color_fill(dev, color, TRUE);

		 PG_shade_poly(dev, rx, ry, 5);};};

    SFREE(vl);
    if (eflag)
        SFREE(emap);

    return(rv);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FILL_POLY_ZC_LR - fill polygons assuming a zone centered
 *                    - Logical-Rectangular mesh
 *                    - the mesh topology is always ignored and is there
 *                    - to be call compatible with the AC version
 */

void PG_fill_poly_zc_lr(dev, nd, a, x, y, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   REAL **a, *x, *y, *aext;
   byte *cnnct;
   pcons *alist;
   {int in, iz, k, l, nn, nmap, nz, kz, lz;
    int xok, yok, xso, yso;
    REAL xmn, xmx, ymn, ymx;
    REAL rx[5], ry[5];
    REAL *x1, *x2, *x3, *x4;
    REAL *y1, *y2, *y3, *y4;
    int *maxes, kmax, lmax, eflag;
    char *emap;
    PG_poly_fill_data par;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    nn    = kmax*lmax;
    nmap  = (kmax - 1) * (lmax - 1);
    nz    = SC_arrlen(a[0])/sizeof(REAL);

    kz = kmax - 1;
    lz = lmax - 1;

    LR_MAPPING_INFO(alist, nmap);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

    PM_LOGICAL_ZONE(x, x1, x2, x3, x4, kmax);
    PM_LOGICAL_ZONE(y, y1, y2, y3, y4, kmax);

    PG_get_window(dev, &xmn, &xmx, &ymn, &ymx);

    par.dev   = dev;
    par.nd    = nd;
    par.a     = a;
    par.x     = x;
    par.y     = y;
    par.aext  = aext;
    par.cnnct = cnnct;
    par.alist = alist;

    SC_chunk_loop(_PG_fill_chunk_zc_lr, 0, lz, TRUE, &par);

    if (dev->draw_fill_bound)
       {PG_set_color_fill(dev, dev->WHITE, TRUE);

	for (l = 0; l < lz; l++)
	    for (k = 0; k < kz; k++)
	        {in = l*kmax + k;
		 iz = l*kz + k;
		 if (emap[iz] == 0)
		    continue;

		 PG_SET_WITH_CLIP_INIT(xok, xso);
		 PG_SET_WITH_CLIP_INIT(yok, yso);

		 PG_SET_WITH_CLIP(rx[0], x1[in], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[0], y1[in], yok, yso, ymn, ymx);

		 PG_SET_WITH_CLIP(rx[1], x2[in], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[1], y2[in], yok, yso, ymn, ymx);

		 PG_SET_WITH_CLIP(rx[2], x3[in], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[2], y3[in], yok, yso, ymn, ymx);

		 PG_SET_WITH_CLIP(rx[3], x4[in], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[3], y4[in], yok, yso, ymn, ymx);

		 if (xok && yok)
		    {rx[4] = x1[in];
		     ry[4] = y1[in];

		     PG_draw_polyline(dev, rx, ry, 5, TRUE);};};};

    if (eflag)
        SFREE(emap);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FILL_CHUNK_NC_LR - do a chunk of the fill poly work */

static void *_PG_fill_chunk_nc_lr(arg)
   void *arg;
   {int i, i1, j, k, l, nn, nmap, color, km, lm;
    int xok, yok, xso, yso;
    int lmn, lmx, nd;
    int *maxes, kmax, lmax, eflag;
    REAL xmn, xmx, ymn, ymx;
    REAL rx[5], ry[5], *vl, *at;
    REAL *x1, *x2, *x3, *x4;
    REAL *y1, *y2, *y3, *y4;
    REAL **a, *x, *y, *aext;
    char *emap;
    byte *cnnct, *rv;
    pcons *alist;
    PG_device *dev;
    PG_poly_fill_data *par;

    par = (PG_poly_fill_data *) arg;

    dev   = par->dev;
    nd    = par->nd;
    a     = par->a;
    x     = par->x;
    y     = par->y;
    aext  = par->aext;
    cnnct = par->cnnct;
    alist = par->alist;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    nn    = kmax*lmax;

    km   = kmax - 1;
    lm   = lmax - 1;
    nmap = km*lm;

    LR_MAPPING_INFO(alist, nmap);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

    PM_LOGICAL_ZONE(x, x1, x2, x3, x4, kmax);
    PM_LOGICAL_ZONE(y, y1, y2, y3, y4, kmax);

    PG_get_window(dev, &xmn, &xmx, &ymn, &ymx);

    SC_chunk_split(&lmn, &lmx, &rv);

    vl = FMAKE_N(REAL, nd, "_PG_FILL_CHUNK_NC_LR:vl");

    for (l = lmn; l < lmx; l++)
        for (k = 0; k < km; k++)
            {i  = l*kmax + k;
             i1 = l*km +k;
	     if (emap[i1] == 0)
	        continue;

	     PG_SET_WITH_CLIP_INIT(xok, xso);
	     PG_SET_WITH_CLIP_INIT(yok, yso);

	     PG_SET_WITH_CLIP(rx[0], x1[i], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[0], y1[i], yok, yso, ymn, ymx);

	     PG_SET_WITH_CLIP(rx[1], x2[i], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[1], y2[i], yok, yso, ymn, ymx);

	     PG_SET_WITH_CLIP(rx[2], x3[i], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[2], y3[i], yok, yso, ymn, ymx);

	     PG_SET_WITH_CLIP(rx[3], x4[i], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ry[3], y4[i], yok, yso, ymn, ymx);

	     if (xok && yok)
	        {rx[4] = x1[i];
		 ry[4] = y1[i];

		 for (j = 0; j < nd; j++)
		     {at    = a[j];
		      vl[j] = 0.25*(at[i+1] + at[i+kmax+1] +
				    at[i+kmax] + at[i]);};

		 color = PG_select_color(dev, nd, vl, aext);
         
		 PG_set_color_fill(dev, color, TRUE);

		 PG_shade_poly(dev, rx, ry, 5);};};

    SFREE(vl);
    if (eflag)
        SFREE(emap);

    return(rv);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FILL_POLY_NC_LR - draw filled polygons for the variable A on the
 *                    - Logical-Rectangular mesh specified by coordinates
 *                    - X and Y which is KMAX by LMAX
 */

void PG_fill_poly_nc_lr(dev, nd, a, x, y, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   REAL **a, *x, *y, *aext;
   byte *cnnct;
   pcons *alist;
   {int i, k, l, n, nmap, km, lm;
    int xok, yok, xso, yso;
    REAL xmn, xmx, ymn, ymx;
    REAL rx[5], ry[5];
    REAL *x1, *x2, *x3, *x4;
    REAL *y1, *y2, *y3, *y4;
    int *maxes, kmax, lmax, eflag;
    char *emap;
    PG_poly_fill_data par;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    n     = kmax*lmax;

    km   = kmax - 1;
    lm   = lmax - 1;
    nmap = km*lm;

    LR_MAPPING_INFO(alist, nmap);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

    PM_LOGICAL_ZONE(x, x1, x2, x3, x4, kmax);
    PM_LOGICAL_ZONE(y, y1, y2, y3, y4, kmax);

    PG_get_window(dev, &xmn, &xmx, &ymn, &ymx);

    par.dev   = dev;
    par.nd    = nd;
    par.a     = a;
    par.x     = x;
    par.y     = y;
    par.aext  = aext;
    par.cnnct = cnnct;
    par.alist = alist;

    SC_chunk_loop(_PG_fill_chunk_nc_lr, 0, lm, TRUE, &par);

    if (dev->draw_fill_bound)
       {PG_set_color_fill(dev, dev->WHITE, TRUE);

	for (l = 0; l < lm; l++)
	    for (k = 0; k < km; k++)
	        {i = l*kmax + k;
		 if (emap[i] == 0)
		    continue;

		 PG_SET_WITH_CLIP_INIT(xok, xso);
		 PG_SET_WITH_CLIP_INIT(yok, yso);

		 PG_SET_WITH_CLIP(rx[0], x1[i], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[0], y1[i], yok, yso, ymn, ymx);

		 PG_SET_WITH_CLIP(rx[1], x2[i], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[1], y2[i], yok, yso, ymn, ymx);

		 PG_SET_WITH_CLIP(rx[2], x3[i], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[2], y3[i], yok, yso, ymn, ymx);

		 PG_SET_WITH_CLIP(rx[3], x4[i], xok, xso, xmn, xmx);
		 PG_SET_WITH_CLIP(ry[3], y4[i], yok, yso, ymn, ymx);

		 if (xok && yok)
		    {rx[4] = x1[i];
		     ry[4] = y1[i];

		     PG_draw_polyline(dev, rx, ry, 5, TRUE);};};};

    if (eflag)
        SFREE(emap);

    return;}
  
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FILL_POLY_ZC_AC - fill polygons assuming a zone centered
 *                    - Arbitrarily-Connected mesh of with mesh topology MT
 */

void PG_fill_poly_zc_ac(dev, nd, a, x, y, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   REAL **a, *x, *y, *aext;
   byte *cnnct;
   pcons *alist;
   {int j, is, os, iz, oz, is1, is2, in1, in2;
    int xok, yok, xso, yso;
    int *nc, nz, *np, nzp, nsp, npt, color;
    long **cells, *zones, *sides;
    REAL xmn, xmx, ymn, ymx;
    REAL *xpt, *ypt, *vl;
    PM_mesh_topology *mt;

    mt = (PM_mesh_topology *) cnnct;

    cells = mt->boundaries;
    zones = cells[2];
    sides = cells[1];

    nc = mt->n_cells;
    nz = nc[2];

    np  = mt->n_bound_params;
    nzp = np[2];
    nsp = np[1];

    vl  = FMAKE_N(REAL, nd, "PG_FILL_POLY_ZC_AC:vl");

    PG_get_window(dev, &xmn, &xmx, &ymn, &ymx);

/* can compute this by looping over the sides */
    xpt = FMAKE_N(REAL, 20, "PG_FILL_POLY_ZC_AC:xpt");
    ypt = FMAKE_N(REAL, 20, "PG_FILL_POLY_ZC_AC:ypt");

    for (iz = 0; iz < nz; iz++)
        {oz  = iz*nzp;
	 is1 = zones[oz];
	 is2 = zones[oz+1];

	 npt = 0;
	 PG_SET_WITH_CLIP_INIT(xok, xso);
	 PG_SET_WITH_CLIP_INIT(yok, yso);
	 for (is = is1; is <= is2; is++)
	     {os  = is*nsp;
	      in1 = sides[os];
	      in2 = sides[os+1];

	      PG_SET_WITH_CLIP(xpt[npt], x[in1], xok, xso, xmn, xmx);
	      PG_SET_WITH_CLIP(ypt[npt], y[in1], yok, yso, ymn, ymx);

              npt++;};

	 PG_SET_WITH_CLIP(xpt[npt], x[in2], xok, xso, xmn, xmx);
	 PG_SET_WITH_CLIP(ypt[npt], y[in2], yok, yso, ymn, ymx);

	 npt++;

	 if (xok && yok)
	    {for (j = 0; j < nd; j++)
	         vl[j] = a[j][iz];

	     color = PG_select_color(dev, nd, vl, aext);
         
	     PG_set_color_fill(dev, color, TRUE);

	     PG_shade_poly(dev, xpt, ypt, npt);};};

    if (dev->draw_fill_bound)
       {PG_set_color_fill(dev, dev->WHITE, TRUE);

	for (iz = 0; iz < nz; iz++)
	    {oz  = iz*nzp;
	     is1 = zones[oz];
	     is2 = zones[oz+1];

	     npt = 0;
	     PG_SET_WITH_CLIP_INIT(xok, xso);
	     PG_SET_WITH_CLIP_INIT(yok, yso);
	     for (is = is1; is <= is2; is++)
	         {os  = is*nsp;
		  in1 = sides[os];
		  in2 = sides[os+1];

		  PG_SET_WITH_CLIP(xpt[npt], x[in1], xok, xso, xmn, xmx);
		  PG_SET_WITH_CLIP(ypt[npt], y[in1], yok, yso, ymn, ymx);

		  npt++;};

	     PG_SET_WITH_CLIP(xpt[npt], x[in2], xok, xso, xmn, xmx);
	     PG_SET_WITH_CLIP(ypt[npt], y[in2], yok, yso, ymn, ymx);

	     npt++;

	     if (xok && yok)
	        PG_draw_polyline(dev, xpt, ypt, npt, TRUE);};};

    SFREE(vl);
    SFREE(xpt);
    SFREE(ypt);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FILL_POLY_NC_AC - draw filled polygons for the variable A on the
 *                    - Arbitrarily-Connected mesh specified by coordinates
 *                    - X and Y with topology MT
 */

void PG_fill_poly_nc_ac(dev, nd, a, x, y, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   REAL **a, *x, *y, *aext;
   byte *cnnct;
   pcons *alist;
   {REAL **ap;
    int j;

    ap = FMAKE_N(REAL *, nd, "PG_FILL_POLY_NC_AC:ap");
    for (j = 0; j < nd; j++)
        ap[j] = PM_node_zone_ac_2d(a[j], cnnct, alist);

    PG_fill_poly_zc_ac(dev, nd, ap, x, y, aext, cnnct, alist);

    for (j = 0; j < nd; j++)
        SFREE(ap[j]);
    SFREE(ap);

    return;}
  
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_POLY_FILL_CORE - polygon fill plot skeleton routine */

static void _PG_poly_fill_core(dev, data, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *data;
   PFByte fnc_zc, fnc_nc;
   {PG_graph *g;
    PG_picture_desc *pd;

    data->rendering = PLOT_FILL_POLY;

    pd = PG_setup_picture(dev, data, TRUE, TRUE, TRUE);

/* plot all of the current functions */
    for (g = data; g != NULL; g = g->next)
        PG_fill_hand(dev, g, fnc_zc, fnc_nc);

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_POLY_FILL_PLOT - main polygon fill plot control routine */

#ifdef PCC

void PG_poly_fill_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_poly_fill_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {

    if (strcmp(data->f->category, PM_LR_S) == 0)
       _PG_poly_fill_core(dev, data, PG_fill_poly_zc_lr, PG_fill_poly_nc_lr);

    else if (strcmp(data->f->category, PM_AC_S) == 0)
       _PG_poly_fill_core(dev, data, PG_fill_poly_zc_ac, PG_fill_poly_nc_ac);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
