/*
 * GSPR_RST.C - PGS raster device primitive routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"
#include "gsrast.h"

typedef struct s_EI EI;

struct s_EI
   {REAL x;
    REAL dx;
    REAL dy;
    EI *next;};

#define SMALLNO   1.0e-9
#define NODIFF    1.0e-8

#define SET_PIXEL(fr, x, y, wd, sz, clr, lwd)                          \
   {int i, j, m, xi, yi;                                               \
    for (i = -lwd/2; i <= (lwd+1)/2; i++)                              \
        {xi = x + i;                                                   \
         for (j = -lwd/2; j <= (lwd+1)/2; j++)                         \
	     {yi = y + j;                                              \
              m = xi + yi*wd;                                          \
              if ((0 <= m) && (m < sz))                                \
                 {fr->Y[m]  = clr.red;                                 \
                  fr->Cr[m] = clr.green;                               \
                  fr->Cb[m] = clr.blue;};};};}

static int
 SC_DECLARE(_PG_rst_clip_line,
         (PG_device *dev, REAL *x1, REAL *y1, REAL *x2, REAL *y2));

/*--------------------------------------------------------------------------*/

/*                         STATE QUERY ROUTINES                             */

/*--------------------------------------------------------------------------*/

/* _PG_RST_GET_TEXT_EXT_NDC - return the text extent in NDC
 *                          - of the given string
 */

void _PG_rst_get_text_ext_NDC(dev, s, px, py)
   PG_device *dev;
   char *s;
   REAL *px, *py;
   {REAL tx, ty;

    tx = strlen(s)*(dev->char_width_s + dev->char_space_s) -
         dev->char_space_s;
    ty = dev->char_height_s;

    *px = tx;
    *py = ty;

    return;}

/*--------------------------------------------------------------------------*/

/*                          STATE CHANGE ROUTINES                           */

/*--------------------------------------------------------------------------*/
 
/* _PG_RST_SET_LOGICAL_OP - set the logical operation */
 
void _PG_rst_set_logical_op(dev, lop)
   PG_device *dev;
   int lop;
   {return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_SET_LINE_STYLE - set the line style */
 
void _PG_rst_set_line_style(dev, style)
   PG_device *dev;
   int style;
   {dev->line_style = style;
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_SET_LINE_WIDTH - set the line width */
 
void _PG_rst_set_line_width(dev, width)
   PG_device *dev;
   REAL width;
   {dev->line_width = width;
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_SET_LINE_COLOR - set the line color */
 
void _PG_rst_set_line_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {
    if (mapped)
       dev->line_color = _PG_trans_color(dev, color);
    else
       dev->line_color = color;
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_SET_TEXT_COLOR - set the color of the text */
 
void _PG_rst_set_text_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {

    if (mapped)
       dev->text_color = _PG_trans_color(dev, color);
    else
       dev->text_color = color;
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SET_FILL_COLOR - sets current fill color */

void _PG_rst_set_fill_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {

    if (mapped)
       dev->fill_color = _PG_trans_color(dev, color);
    else
       dev->fill_color = color;
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SET_FONT - set the character font */

int _PG_rst_set_font(dev, face, style, size)
   PG_device *dev;
   char *face, *style;
   int size;
   {int nfont, nstyle, dx, dy, nc;
    REAL scale;
    char *font_name;

    if (!PG_setup_font(dev, face, style, size, &font_name, &nfont, &nstyle))
       return(FALSE);

    PG_query_screen(dev, &dx, &dy, &nc);

    scale              = 0.6;

    dev->char_width_s  = scale * (REAL)size/(REAL)dx;
    dev->char_height_s = scale * (REAL)size/(REAL)dy;

    dev->char_space_s  = 0.001 * dev->char_width_s;

    dev->marker_scale  = (REAL)size/(3.0 * (REAL)dx);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SET_CHAR_SIZE_NDC - set the character size in NCD */

void _PG_rst_set_char_size_NDC(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {dev->char_height_s = (REAL)y;
    dev->char_width_s  = (REAL)x;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SET_CHAR_PATH - set the direction along which text will be written
 *                       - defaults to (1, 0)
 */

void _PG_rst_set_char_path(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {dev->char_path_x = x;
    dev->char_path_y = y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SET_CHAR_PRECISION - set the character precision
 *                            - fast and fixed size or
 *                            - slow and flexible
 */

void _PG_rst_set_char_precision(dev, p)
   PG_device *dev;
   int p;
   {dev->char_precision = p;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SET_CHAR_SPACE - set the space between characters */

void _PG_rst_set_char_space(dev, s)
   PG_device *dev;
   REAL s;
   {dev->char_space = (REAL)s;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SET_CHAR_UP - set the direction which is up for individual
 *                     - characters
 *                     - defaults to (0, 1)
 */

void _PG_rst_set_char_up(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {dev->char_up_x = (REAL)x;
    dev->char_up_y = (REAL)y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_SET_CHAR_LINE - set the number characters per line */
 
void _PG_rst_set_char_line(dev, n)
   PG_device *dev;
   int n;
   {return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_SET_CLIPPING - set clipping
 *                      - flag = FALSE  -->  clipping off
 *                      - flag = TRUE   -->  clipping on
 */

void _PG_rst_set_clipping(dev, flag)
   PG_device *dev;
   int flag;
   {REAL xmin, xmax, ymin, ymax;

    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);
 
    dev->clipping = flag;

    return;}
 
/*--------------------------------------------------------------------------*/

/*                          MOVE AND DRAW ROUTINES                          */

/*--------------------------------------------------------------------------*/
 
/* _PG_RST_MOVE_GR_ABS - move the current graphics cursor position to the
 *                     - given absolute coordinates in WC
 */
 
void _PG_rst_move_gr_abs(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {

/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->gcurx = x;
    dev->gcury = y;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_MOVE_TX_ABS - move the current text cursor position to the
 *                     - given coordinates in WC
 */
 
void _PG_rst_move_tx_abs(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {

/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->tcurx = x;
    dev->tcury = y;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_MOVE_TX_REL - move the current text cursor position to the
 *                     - given relative coordinates in WC
 */
 
void _PG_rst_move_tx_rel(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {

/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->tcurx += x;
    dev->tcury += y;
     
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_CLIP_LINE - clip the line P1 = (x1, y1) to P2 = (x2, y2)
 *                   - to the box (xmn, ymn) to (xmx, ymx)
 *                   - return TRUE (visible) or FALSE (invisible)
 *                   - if visible, return new clipped points
 *                   - if not visible, no change to P1 and P2.
 */

static int _PG_rst_clip_line(dev, rx1, ry1, rx2, ry2)
   PG_device *dev;
   REAL *rx1, *ry1, *rx2, *ry2;
   {int visible, Icount, i, Sum1, Sum2, Inter, P1orP2;
    int ixmn, iymn, ixmx, iymx, P1code[4], P2code[4];
    REAL m, x1, x2, y1, y2, px1, px2, py1, py2, px, py, x, y;
    REAL xmn, xmx, ymn, ymx;

    if (!dev->clipping) return(TRUE);

    x1 = *rx1;
    x2 = *rx2;
    y1 = *ry1;
    y2 = *ry2;

    PG_get_viewport_PC(dev, &ixmn, &ixmx, &iymn, &iymx);

#if 0
    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
    WtoS(dev, xmn, ymn);
    WtoS(dev, xmx, ymx);
    
    if (dev->quadrant == QUAD_ONE)
       {StoP(dev, xmn, ymn, ixmn, iymn);
        StoP(dev, xmx, ymx, ixmx, iymx);}

    else if (dev->quadrant == QUAD_FOUR)
       {StoP(dev, xmn, ymx, ixmn, iymn);
        StoP(dev, xmx, ymn, ixmx, iymx);};
#endif

    xmn = (REAL)ixmn;
    ymn = (REAL)iymn;
    xmx = (REAL)ixmx;
    ymx = (REAL)iymx;

/* first end point P1 */
    if (x1 < xmn)
       P1code[3] = 1;
    else
       P1code[3] = 0;

    if (x1 > xmx)
       P1code[2] = 1;
    else
       P1code[2] = 0;

    if (y1 < ymn)
       P1code[1] = 1;
    else
       P1code[1] = 0;

    if (y1 > ymx)
       P1code[0] = 1;
    else
       P1code[0] = 0;

/* second end point P2 */
    if (x2 < xmn)
       P2code[3] = 1;
    else
       P2code[3] = 0;

    if (x2 > xmx)
       P2code[2] = 1;
    else
       P2code[2] = 0;

    if (y2 < ymn)
       P2code[1] = 1;
    else
       P2code[1] = 0;

    if (y2 > ymx)
       P2code[0] = 1;
    else
       P2code[0] = 0;

/* initialize the visibility flag, 
 * the drawing points (px1, py1) and (px2, py2),
 * and the counter
 */
    Icount = 0;
    visible = TRUE;
    P1orP2 = 0;
    m = DBL_MAX;   /* initialize to an infinite slope */

    px1 = x1;
    py1 = y1;
    px2 = x2;
    py2 = y2;

/* check for totally visible line */
    Sum1 = Sum2 = 0;
    for (i = 0; i < 4; i++)
        {Sum1 += P1code[i];
	 Sum2 += P2code[i];}
    if (Sum1 == 0 && Sum2 == 0) return(visible);

/* line is not totally visible
 * check for trivial invisible case
 */
    Inter = 0;   /* calculate intersection of P1 & P2 */
    for (i = 0; i < 4; i++)
        {Inter += (P1code[i] + P2code[i])/2;
	 if (Inter != 0)
	    {visible = FALSE;
	     return(visible);};}

/* line may be partially visible
 * check for first point inside window
 */
    if (Sum1 == 0)
       {Icount = 2;
	P1orP2 = 2;
	px1 = x1;
	py1 = y1;
	px  = x2;
	py  = y2;
	goto a2;}

/* check for second point inside window */
    if (Sum2 == 0)
       {Icount = 2;
	P1orP2 = 1;
	px2 = x2;
	py2 = y2;
	px  = x1;
	py  = y1;
	goto a2;}

/* neither end point inside window */
a1:
    if (P1orP2 == 1)
       {px1 = px;
	py1 = py;}
    if (P1orP2 == 2)
       {px2 = px;
	py2 = py;}

    Icount++;
    if (Icount > 2) goto a7;
    px = (Icount == 1) ? px1 : px2;
    py = (Icount == 1) ? py1 : py2;
    P1orP2 = (Icount == 1) ? 1 : 2;

/* check the left-side intercept */
a2:
    if ((x2 - x1) == 0) goto a4;   /* check for vertical line */
    m = (y2 - y1)/(x2 - x1);
    if (xmn < px) goto a3;
    y = m*(xmn - px) + py;
    if (y > ymx) goto a3;
    if (y < ymn) goto a3;

/* a proper intercept has been found */
    py = y;
    px = xmn;
    goto a1;

/* check the right-side intercept */
a3:
    if (xmx > px) goto a4;
    y = m*(xmx - px) + py;
    if (y > ymx) goto a4;
    if (y < ymn) goto a4;

/* a proper intercept has been found */
    py = y;
    px = xmx;
    goto a1;

/* check the top edige intercept */
a4:
    if (m == 0) goto a1;   /* check for horizontal line */
    if (ymx > py) goto a5;
    x = (1/m)*(ymx - py) + px;
    if (x < xmn) goto a5;
    if (x > xmx) goto a5;

/* a proper intercept has been found */
    px = x;
    py = ymx;
    goto a1;

/* check the bottom intercept */
a5:
    x = (1/m)*(ymn - py) + px;
    if (x < xmn) goto a6;
    if (x > xmx) goto a6;

/* a proper intercept has been found */
    px = x;
    py = ymn;
    goto a1;

/* the line is really invisible */
a6:
    visible = FALSE;

/* get clipping points P1 and P2 */
a7:
    if (!visible) return(visible);

#if 0
/* if P1 and P2 are on xmx or ymx lines */
  if (((ABS(px1 - xmx) < NODIFF) && (ABS(px2 - xmx) < NODIFF)) ||
	((ABS(py1 - ymx) < NODIFF) || (ABS(py2 - ymx) < NODIFF)))
     visible = FALSE;
  else
     {*ix1 = px1;
	*iy1 = py1;
	*ix2 = px2;
	*iy2 = py2;}
#endif

    *rx1 = px1;
    *ry1 = py1;
    *rx2 = px2;
    *ry2 = py2;

    return(visible);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_DRAW_LINE - draw a line on the current frame buffer */

static void _PG_rst_draw_line(dev, ix1, iy1, ix2, iy2)
   PG_device *dev;
   int ix1, iy1, ix2, iy2;
   {int i, dx, dy, nc, x, y, wd, hg, sz, lwd, visible, rgb;
    int limit, il, icnt, lim0, lim1, nn, new_limit, set_pixel, dl[4];
    unsigned short *pv;
    REAL scale;
    REAL xinc, yinc, xreal, yreal, rx1, ry1, rx2, ry2;
    PG_palette *pal;
    RGB_color_map *true_cm, clr;
    PG_RAST_device *mdv;
    frame *fr;

    rx1 = (REAL) ix1;
    ry1 = (REAL) iy1;
    rx2 = (REAL) ix2;
    ry2 = (REAL) iy2;
    visible = _PG_rst_clip_line(dev, &rx1, &ry1, &rx2, &ry2);
    if (!visible)
       return;

    GET_RAST_DEVICE(dev, mdv);

    rgb = mdv->rgb_mode;
    fr  = mdv->inner_frame;
    wd  = fr->width;
    hg  = fr->height;
    sz  = wd*hg;

    lwd     = 3.0*max(dev->line_width, 0.0);
    pal     = dev->current_palette;
    nc      = pal->n_pal_colors;
    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

    if (rgb)
       {scale = 255.0/((REAL) MAXPIX);

	clr        = true_cm[dev->line_color];
        clr.red   *= scale;
	clr.green *= scale;
	clr.blue  *= scale;}

    else
       {scale = ((REAL) nc)/((REAL) MAXPIX);

	clr.red   = (pv != NULL) ? pv[dev->line_color] : dev->line_color;
	clr.green = 0.0;
	clr.blue  = 0.0;}

    dx = ix2 - ix1;
    dy = iy2 - iy1;

    if (abs(dx) > abs(dy))
       {xinc = (dx > 0) ? 1.0 : -1.0 ;
        yinc = ((REAL) abs(dy)) / ((REAL) abs(dx));
        if (dy < 0)
	   yinc = -yinc;}

    else if (abs(dy) > abs(dx))
       {yinc = (dy > 0) ? 1.0 : -1.0 ;
        xinc = ((REAL) abs(dx)) / ((REAL) abs(dy));
        if (dx < 0)
	   xinc = -xinc;}
    else
       {xinc = (dx > 0) ? 1.0 : -1.0;
        yinc = (dy > 0) ? 1.0 : -1.0;}

    nn = 0;
    switch (dev->line_style)
       {case LINE_DASHED:
            nn = 2;
            dl[0] = 7;
            dl[1] = 3+lwd;
	    break;

        case LINE_DOTTED:
            nn = 2;
            dl[0] = 3;
            dl[1] = 3+lwd;
	    break;

        case LINE_DOTDASHED:
            nn = 4;
            dl[0] = 7;
            dl[1] = 3+lwd;
            dl[2] = 3;
            dl[3] = 3+lwd;
	    break;

        case LINE_SOLID:
        default:
	    break;};

    il    = 1;
    icnt  = 0;
    lim0  = 0;
    lim1  = dl[0];
/*=============================================================
    limit = max(abs(dx), abs(dy));
    x     = ix1;
    y     = iy1;
    xreal = x;
    yreal = y;
    SET_PIXEL(fr, x, y, wd, sz, clr, lwd);

01/16/98 note: the above coding replaced by the following...
==============================================================*/

/* ??? should we use "+ 0.49" or not ??? */
    limit = max(ABS(rx1-rx2), ABS(ry1-ry2)) + 0.49;
    xreal = rx1;
    yreal = ry1;
    x = (int) (xreal + 0.49);
    y = (int) (yreal + 0.49);
    SET_PIXEL(fr, x, y, wd, sz, clr, lwd);

    for (i = 1; i < limit; i++)
        {xreal += xinc;
         yreal += yinc;
         x = (int) (xreal + 0.49);
         y = (int) (yreal + 0.49);

	 il++;
	 new_limit = FALSE;
	 set_pixel = FALSE;
	 switch (nn)
	    {case 0:                /* SOLID or default */
	         set_pixel = TRUE;
	         break;
	     default:               /* DASHED, DOTTED or DOTDASHED */
	        if (icnt % 2 == 0)
		   {if (il <= lim1)
		       set_pixel = TRUE;
		    else
		       new_limit = TRUE;}
	        else
		   {if (il > lim1)
		       {new_limit = TRUE;
		        set_pixel = TRUE;};};
		if (new_limit)
		   {lim0 = lim1;
		    lim1 = lim1 + dl[++icnt % nn];}
	        break;}

	if (set_pixel)
	   SET_PIXEL(fr, x, y, wd, sz, clr, lwd);};

    return;}

#if 0
void _PG_rst_draw_line(dev, ix1, iy1, ix2, iy2)
   PG_device *dev;
   int ix1, iy1, ix2, iy2;
   {int dx, dy, d, ie, ine, x, y, wd, hg, sz, lwd;
    REAL scale;
    PG_palette *pal;
    RGB_color_map *true_cm, clr;
    PG_RAST_device *mdv;
    frame *fr;

    lwd     = (int) (3.0*(dev->line_width));
    pal     = dev->current_palette;
    true_cm = pal->true_colormap;
    clr     = true_cm[dev->line_color];

    scale = 255.0/((REAL) MAXPIX);
    clr.red   *= scale;
    clr.green *= scale;
    clr.blue  *= scale;

    GET_RAST_DEVICE(dev, mdv);

    fr  = mdv->inner_frame;
    wd  = fr->width;
    hg  = fr->height;
    sz  = wd*hg;

    dx = ix2 - ix1;
    dy = iy2 - iy1;

    d   = 2*dy - dx;
    ie  = 2*dy;
    ine = 2*(dy - dx);

    x = ix1;
    y = iy1;
    
    SET_PIXEL(fr, x, y, wd, sz, clr, lwd);
    while (x < ix2)
       {if (d <= 0)
           {d += ine;
	    x++;}
	else
	   {d += ine;
	    x++;
	    y++;};

	SET_PIXEL(fr, x, y, wd, sz, clr, lwd);};

    return;}
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_DRAW_TO_ABS - draw a line from current position to
 *                     - absolute position (x, y)
 *                     - in WC
 */

void _PG_rst_draw_to_abs(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {REAL x1, y1, x2, y2;
    int ix1, iy1, ix2, iy2;

/* if log axis options are on, take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);

    x1 = dev->gcurx;
    y1 = dev->gcury;
    x2 = dev->gcurx = x;
    y2 = dev->gcury = y;

    WtoS(dev, x1, y1);
    WtoS(dev, x2, y2);
    StoP(dev, x1, y1, ix1, iy1);
    StoP(dev, x2, y2, ix2, iy2);
 
    _PG_rst_draw_line(dev, ix1, iy1, ix2, iy2);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_DRAW_TO_REL - draw a line from current position to
 *                     - relative position (x, y)
 *                     - in WC
 */
 
void _PG_rst_draw_to_rel(dev, x, y)
   PG_device *dev;
   REAL x, y;
   {REAL x1, y1, x2, y2;
    int ix1, iy1, ix2, iy2;

/* if log axis options are on, take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    x1 = dev->gcurx;
    y1 = dev->gcury;
    x2 = dev->gcurx + x;
    y2 = dev->gcury + y;

    dev->gcurx = x2;
    dev->gcury = y2;

    WtoS(dev, x1, y1);
    WtoS(dev, x2, y2);
    StoP(dev, x1, y1, ix1, iy1);
    StoP(dev, x2, y2, ix2, iy2);
 
    _PG_rst_draw_line(dev, ix1, iy1, ix2, iy2);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_DRAW_CURVE - draw a PG_curve */
 
void _PG_rst_draw_curve(dev, crv, clip)
   PG_device *dev;
   PG_curve *crv;
   int clip;
   {int n, j, xo, yo;
    int ix1, iy1, ix2, iy2;
    int *px, *py;
    
    n  = crv->n;
    px = crv->x;
    py = crv->y;
    xo = crv->x_origin;
    yo = crv->y_origin;
    
    ix1 = *px++ + xo;
    iy1 = *py++ + yo;
    for (j = 1; j < n; j++)
        {ix2 = *px++ + xo;
	 iy2 = *py++ + yo;

	 _PG_rst_draw_line(dev, ix1, iy1, ix2, iy2);

	 ix1 = ix2;
	 iy1 = iy2;};

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_DRAW_DISJOINT_POLYLINE_2 - draws disjoint two dimensional
 *                                  - line segments specified in WC
 */

void _PG_rst_draw_disjoint_polyline_2(dev, x, y, n, flag, coord)
   PG_device *dev;
   REAL *x, *y;
   long n;
   int flag, coord;
   {int i, ix1, ix2, iy1, iy2;
    REAL *px, *py;
    REAL rx1, ry1, rx2, ry2;

/* if auto ranging or domaining is on the data will control the WC system */
    if (flag && (dev->autorange || dev->autodomain))
       PG_set_limits(dev, x, y, 2*n, CARTESIAN);

    px = x;
    py = y;

    PG_move_gr_abs(dev, x[0], y[0]);

    for (i = 0; i < n; i++)
        {rx1 = *px++;
         ry1 = *py++;
         rx2 = *px++;
         ry2 = *py++;

         if (coord)
            {if (dev->ifxlog)
                {rx1 = log10(ABS(rx1) + SMALL);
                 rx2 = log10(ABS(rx2) + SMALL);};
             if (dev->ifylog)
                {ry1 = log10(ABS(ry1) + SMALL);
                 ry2 = log10(ABS(ry2) + SMALL);};

             WtoS(dev, rx1, ry1);
             WtoS(dev, rx2, ry2);}

         StoP(dev, rx1, ry1, ix1, iy1);
         StoP(dev, rx2, ry2, ix2, iy2);

         if ((ix1 == ix2) && (iy1 == iy2))
            continue;

	 _PG_rst_draw_line(dev, ix1, iy1, ix2, iy2);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_LEFT_EDGE_SCAN - algorithm from Foley and vanDam */

#if 0

void _PG_left_edge_scan(xmn, ymn, xmx, ymx, v)
   int xmn, ymn, xmx, ymx, v;
   {int x, y, n, d, i;

    x = xmn;
    n = xmx - xmn;
    d = ymx - ymn;
    i = d;
    for (y = ymn; y <= ymx; y++)
        {SET_PIXEL(fr, x, y, wd, sz, clr);
	 i += n;
	 if (i > d)
	    {x++;
	     i -= d;};};

    return;}

#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* MAKE_EDGE_TABLE - make and fill an edge table */

static EI **make_edge_table(dev, x, y, n, iymax, iymin)
   PG_device *dev;
   REAL *x, *y;
   int n, iymax, iymin;
   {int i, j, narr, *flags;
    int ix1, ix2, iy1, iy2;
    REAL x1, x2, y1, y2, dx, dy, scanln;
    EI **et, *cptr, *pptr, *ei;

    narr  = iymax - iymin;
    et    = FMAKE_N(EI*, narr, "MAKE_EDGE_TABLE:et");
    flags = FMAKE_N(int, n-1, "MAKE_EDGE_TABLE:flags");
    memset(flags, 0, sizeof(int)*(n-1));

    for (i = iymax-1; i >= iymin; i--)
        {scanln = (REAL)i + 0.5;
	 cptr   = pptr = NULL;
         for (j = 0; j < n-1; j++)
	     if (flags[j] == 0)
		{x1 = x[j];
                 y1 = y[j];
                 x2 = x[j+1];
                 y2 = y[j+1];
		 WtoS(dev, x1, y1);
		 WtoS(dev, x2, y2);
		 StoP(dev, x1, y1, ix1, iy1);
		 StoP(dev, x2, y2, ix2, iy2);

                 if (((scanln < iy1) || (scanln < iy2)) && (iy1 != iy2))
		    {flags[j] = 1;
		     ei = FMAKE(EI, "MAKE_EDGE_TABLE:ei");
		     if (cptr == NULL)
		        cptr = ei;
		     if (ix1 == ix2)
		        {ei->x = ix1;
			 ei->dx = 0.0;}
		     else
		        {dx = ix1 - ix2;
		         dy = iy1 - iy2;
			 ei->x = ix1 + (scanln - iy1)*(dx/dy);
		         ei->dx = -(dx/dy);};

		     ei->dy   = (iy1 > iy2) ? (iy1 - iy2) : (iy2 - iy1);
		     ei->next = NULL;
		     if (pptr == NULL)
		        pptr = ei;
		     else
		        pptr->next = ei;
		     pptr = ei;};};

        et[i-iymin] = cptr;};

    SFREE(flags);

    return(et);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

static void free_edge_table(et, iymin, iymax)
   EI **et;
   int iymin, iymax;
   {int i;
    EI *ei, *t;

    for (i = iymin; i < iymax; i++)
        for (ei = et[i-iymin]; ei != NULL; ei = ei->next)
	    {t = ei;
	     SFREE(t);};
    SFREE(et);
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

static int find_active(ei, aet, size)
   EI *ei, **aet;
   int size;
   {int k, l;
    EI *ptr;

/* try for the next scanline */
    for (k = 0; k < size; k++)
        {aet[k]->x += aet[k]->dx;
	 aet[k]->dy--;
	 if (aet[k]->dy == 0)
	    {for (l = k; l < size-1; l++)
		 aet[l] = aet[l+1];
	     size--;
	     k--;};};

    for (ptr = ei; ptr != NULL; ptr = ptr->next)
	aet[size++] = ptr;

    return(size);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* BSORT_X - bubble sort is best for this problem */

static void bsort_x(aet, limit)
   EI **aet;
   int limit;
   {int i, sorted;
    EI *ei;

    sorted = FALSE;
    while (!sorted)
       {sorted = TRUE;
	for (i = 0; i < limit-1; i++)
	    if (aet[i]->x > aet[i+1]->x)
	       {sorted = FALSE;
		ei = aet[i];
		aet[i] = aet[i+1];
		aet[i+1] = ei;};};
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

static void fill_scan_line(aet, size, scanline, dev)
   EI **aet;
   int size, scanline;
   PG_device *dev;
   {int i, ix0, ix1;
    REAL rx0, rx1;

    for (i = 0; i < size; i += 2)
        {rx0 = aet[i]->x + 0.5*aet[i]->dx + SMALLNO;
	 ix0 = rx0;
	 if (ABS(rx0 - (REAL) ix0) > NODIFF)
	    ix0 = ceil(rx0);
	 else if (aet[i]->x > (REAL) ix0)
	    ix0 = floor(rx0);

	 rx1 = aet[i+1]->x + 0.5*aet[i+1]->dx + SMALLNO;
	 ix1 = rx1;
	 if (ABS(rx1 - (REAL)ix1) > NODIFF)
	    ix1 = ceil(rx1);
	 else if (aet[i+1]->x > (REAL) ix1)
	    ix1 = ceil(rx1);

	 _PG_rst_draw_line(dev, ix0, scanline, ix1, scanline);}

   return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_SHADE_POLY - polygon shading routine
 *                    - X and Y are in WC
 */

void _PG_rst_shade_poly(dev, x, y, n)
   PG_device *dev;
   REAL *x, *y;
   int n;
   {int i, ixdum, iymin, iymax, iytmp, aet_size, color, tmp_clr;
    EI **aet, **et;
    REAL ymin, ymax, xdum, lwd;

    xdum = 0.0;

    PM_maxmin(y, &ymin, &ymax, n);
    WtoS(dev, xdum, ymin);
    StoP(dev, xdum, ymin, ixdum, iymin);
    WtoS(dev, xdum, ymax);
    StoP(dev, xdum, ymax, ixdum, iymax);
    if (dev->quadrant == QUAD_FOUR)
       {iytmp = iymin;
	iymin = iymax;
	iymax = iytmp;};

    et = make_edge_table(dev, x, y, n, iymax, iymin);
    if (et == NULL)
       return;

    aet_size = 0;
    aet = FMAKE_N(EI *, n-1, "_PG_RST_SHADE_POL:aet");

    PG_get_line_color(dev, &color);
    tmp_clr = dev->fill_color;
    PG_set_color_line(dev, tmp_clr, FALSE);

    PG_get_line_width(dev, &lwd);
    PG_set_line_width(dev, _PG_line_width);

    for (i = iymax-1; i >= iymin; i--)
        {aet_size = find_active(et[i-iymin], aet, aet_size);
	 bsort_x(aet, aet_size);
	 fill_scan_line(aet, aet_size, i, dev);};

    PG_set_line_width(dev, lwd);
    PG_set_line_color(dev, color);

    SFREE(aet);
    free_edge_table(et, iymin, iymax);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_RST_FILL_CURVE - fill a closed PG_curve */

void _PG_rst_fill_curve(dev, crv)
   PG_device *dev;
   PG_curve *crv;
   {int i, n, xo, yo, ixc, iyc;
    int *x, *y;
    REAL wxc, wyc;
    REAL *wx, *wy;

    n  = crv->n;
    x  = crv->x;
    y  = crv->y;
    xo = crv->x_origin;
    yo = crv->y_origin;
    
/* the fill poly routine requires WC - sigh - so convert */
    wx = FMAKE_N(REAL, n, "_PG_RST_FILL_CURVE:wx");
    wy = FMAKE_N(REAL, n, "_PG_RST_FILL_CURVE:wy");

    for (i = 0; i < n; i++)
        {ixc = xo + x[i];
	 iyc = yo + y[i];

	 PtoS(dev, ixc, iyc, wxc, wyc);
	 StoW(dev, wxc, wyc);

	 wx[i] = wxc;
	 wy[i] = wyc;};

    _PG_rst_shade_poly(dev, wx, wy, n);

    SFREE(wx);
    SFREE(wy);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_PUT_IMAGE - put the image on the screen
 *                   - the image buffer may be overwritten by the pseudo color
 *                   - mapping if it is needed!!
 */

void _PG_rst_put_image(dev, bf, ix, iy, nx, ny)
   PG_device *dev;
   unsigned char *bf;
   int ix, iy, nx, ny;
   {int bpp, l, k, m, dx, dy, sz, rgb, pc;
    int nc, color, red_fl, green_fl, blue_fl, ok;
    unsigned short *pv;
    unsigned char *pbf, *r, *g, *b;
    REAL scale;
    FILE *fp;
    PG_palette *pal;
    RGB_color_map *true_cm;
    PG_RAST_device *mdv;
    frame *fr;

    pal     = dev->current_palette;
    nc      = pal->n_pal_colors;
    true_cm = pal->true_colormap;
    pv      = pal->pixel_value;

    pbf = bf;
    bpp = log((REAL) (dev->absolute_n_color))/log(2.0) + 0.5;
    fp  = dev->file;
    if (bpp == 1)
       nx = _PG_byte_bit_map(bf, nx, ny, TRUE);

    GET_RAST_DEVICE(dev, mdv);

    rgb = mdv->rgb_mode;
    fr  = mdv->inner_frame;
    GET_RGB(fr, r, g, b);

    dx = fr->width;
    dy = fr->height;
    sz = dx*dy;

    if (dev->quadrant == QUAD_FOUR)
       PG_invert_image_data(bf, nx, ny, 1);

    ok = TRUE;
    if (rgb)
       {scale = 255.0/((REAL) MAXPIX);

	for (l = 0; l < ny; l++)
	    {pbf = bf + l*nx;
	     for (k = 0; k < nx; k++)
	         {color    = *pbf++;
		  red_fl   = scale*true_cm[color].red;
		  green_fl = scale*true_cm[color].green;
		  blue_fl  = scale*true_cm[color].blue;
		  if ((red_fl < 0) || (255 < red_fl))
		     {if (ok)
			 {PRINT(stdout,
				"Bad Red Value for index %d: %d\n",
				color, red_fl);
			  ok = FALSE;};};

		  if ((green_fl < 0) || (255 < green_fl))
		     {if (ok)
			 {PRINT(stdout,
				"Bad Green Value for index %d: %d\n",
				color, green_fl);
			  ok = FALSE;};};

		  if ((blue_fl < 0) || (255 < blue_fl))
		     {if (ok)
			 {PRINT(stdout,
				"Bad Blue Value for index %d: %d\n",
				color, blue_fl);
			  ok = FALSE;};};

		  m = (k + ix) + (l + iy)*dx;
		  if ((m < 0) || (m >= sz))
		     {if (ok)
			 {PRINT(stdout,
				"Bad index %d < 0 or %d >= %d\n",
				m, m, sz);
			  ok = FALSE;};};

		  r[m] = red_fl;
		  g[m] = green_fl;
		  b[m] = blue_fl;};};}
    else
       {for (l = 0; l < ny; l++)
	    {pbf = bf + l*nx;
	     for (k = 0; k < nx; k++)
	         {m = (k + ix) + (l + iy)*dx;
		  if ((m < 0) || (m >= sz))
		     {if (ok)
			 {PRINT(stdout,
				"Bad index %d < 0 or %d >= %d\n",
				m, m, sz);
			  ok = FALSE;};};

		  pc   = *pbf++;
		  r[m] = (pv != NULL) ? pv[pc] : pc;};};};

    if (SC_mem_trace() < 0)
       {PRINT(stdout, "MEMORY CORRUPTED - _PG_RST_PUT_IMAGE\n");
	exit(2);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RST_GET_IMAGE - get the image on the screen into
 *                   - the image buffer
 */

void _PG_rst_get_image(dev, bf, ix, iy, nx, ny)
   PG_device *dev;
   unsigned char *bf;
   int ix, iy, nx, ny;
   {int k, l, mo, mi, mx, my;
    unsigned char *r, *g, *b;
    char *name;
    frame *fr;
    PG_RAST_device *mdv;

    GET_RAST_DEVICE(dev, mdv);

    fr   = mdv->inner_frame;
    mx   = mdv->width;
    my   = mdv->height;
    name = mdv->out_fname;

    GET_RGB(fr, r, g, b);

    for (l = 0; l < ny; l++)
        {for (k = 0; k < nx; k++)
	     {mo = k + l*nx;
	      mi = (k + ix) + (l + iy)*mx;

	      bf[mo] = r[mi];};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
