/*
 * GSVECT.C - vector plot routines for Portable Graphics System
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

static int
 _PG_ifix     = FALSE,
 _PG_ifixhead = FALSE,
 _PG_maxsize  = 0;

static REAL
 _PG_scale    = 1.0,
 _PG_hdangle  = 22.5,
 _PG_headsize = 0.05,
 _PG_fixsize,
 _PG_sizemax;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_VECTOR - setup a window for a vector rendering
 *                         - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_vector(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int nde, nre, change;
    REAL nvxmn, nvxmx, nvymn, nvymx;
    REAL *dpex, *ddex, *pdx, *rpex, *rdex, *prx, *vwprt;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    pd->palette_fl = FALSE;
    pd->legend_fl  = FALSE;

    alst = pd->alist;
    pri  = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = PLOT_VECTOR;};};

    vwprt = pd->viewport;
    if (vwprt != NULL)
       {nvxmn = vwprt[0];
        nvxmx = vwprt[1];
        nvymn = vwprt[2];
        nvymx = vwprt[3];}
    else
       {nvxmn = 0.175;
        nvxmx = 0.90;
        nvymn = 0.175;
        nvymx = 0.90;};

    if (change)
       {PG_set_viewport(dev, nvxmn, nvxmx, nvymn, nvymx);

/* find the extrema for this frame */
	PG_find_extrema(data, 0.05, &dpex, &rpex, &nde, &ddex, &nre, &rdex);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

/* setup the range limits */
	prx = ((dev->autorange == TRUE) || (rpex == NULL)) ? rdex : rpex;
	PG_register_range_extrema(dev, nre, prx);

	SFREE(ddex);
	SFREE(rdex);

	PG_set_palette(dev, "standard");

	PG_set_color_line(dev, dev->WHITE, TRUE);
	PG_set_color_text(dev, dev->WHITE, TRUE);

/*        PG_set_vec_attr(dev, VEC_SCALE, scale, 0); */

	PG_set_clipping(dev, TRUE);};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_VCT_HAND - plot arrows with heads (vectors)
 *             - The first argument is the device to draw to and
 *             - the second argument is the graph to draw.
 */

static void PG_vct_hand(dev, g, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *g;
   PFByte fnc_zc, fnc_nc;
   {int npts, centering;
    int color, style;
    REAL width;
    REAL **r, **u;
    REAL *rx, *ry, *ux, *uy;
    PM_mapping *h;
    PM_set *domain, *range;
    PG_dev_attributes *attr;
    byte *cnnct;
    pcons *alst;

    h = g->f;

    domain = h->domain;
    npts   = domain->n_elements;
    r      = (REAL **) domain->elements;

    rx = PM_array_real(domain->element_type, r[0], npts, NULL);
    ry = PM_array_real(domain->element_type, r[1], npts, NULL);

    range = h->range;
    u     = (REAL **) range->elements;

    ux = PM_array_real(range->element_type, u[0], npts, NULL);
    uy = PM_array_real(range->element_type, u[1], npts, NULL);

/* save user's values for various attributes */
    attr = PG_get_attributes(dev);

/* set the default plotting attributes for vectors */
    if ((g->info != NULL) && (strcmp(g->info_type, SC_PCONS_P_S) == 0))
       {int *pc, *ps;
        REAL *pw;

        SC_assoc_info((pcons *) g->info,
                      "LINE-COLOR", &pc,
                      "LINE-WIDTH", &pw,
                      "LINE-STYLE", &ps,
                      NULL);

	color = (pc == NULL) ? dev->BLUE : *pc;
	width = (pw == NULL) ? 0.0 : *pw;
	style = (ps == NULL) ? SOLID : *ps;}

    else
       {color = dev->BLUE;
        width = 0.0;
        style = SOLID;};

    PG_set_color_line(dev, color, TRUE);
    PG_set_line_style(dev, style);
    PG_set_line_width(dev, width);

/* find the additional mapping information */
    centering = N_CENT;
    alst = PM_mapping_info(h,
			   "CENTERING", &centering,
			   NULL);

    cnnct = PM_connectivity(h);

/* this is done consistently with PG_draw_contour */
    switch (centering)
       {case Z_CENT :
             (*fnc_zc)(dev, ux, uy, rx, ry, cnnct, alst);
	     break;

        case N_CENT :
             (*fnc_nc)(dev, ux, uy, rx, ry, cnnct, alst);
	     break;

        case F_CENT :
        case U_CENT :
        default     :
	     break;};

    PG_draw_domain_boundary(dev, h);

/* reset user's values for various attributes */
    PG_set_attributes(dev, attr);
    SFREE(attr);

    SFREE(rx);
    SFREE(ry);
    SFREE(ux);
    SFREE(uy);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_VCT_ZC_LR - draw zone centered logical rectangular vector plot */

void PG_draw_vct_zc_lr(dev, ux, uy, rx, ry, cnnct, alst)
   PG_device *dev;
   REAL *ux, *uy, *rx, *ry;
   byte *cnnct;
   pcons *alst;
   {int i, j, nn, nmap, nz, kmax, lmax, km, lm, *maxes;
    int eflag;
    REAL *vx, *vx1, *vx2, *vx3, *vx4;
    REAL *vy, *vy1, *vy2, *vy3, *vy4;
    REAL valx, valy;
    char *emap;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    nn    = kmax*lmax;
    nmap  = (kmax - 1) * (lmax - 1);

    LR_MAPPING_INFO(alst, nmap);

    PM_CHECK_EMAP(alst, nmap, eflag, emap);

    vx = FMAKE_N(REAL, nn, "PG_DRAW_VCT_ZC_LR:vx");
    vy = FMAKE_N(REAL, nn, "PG_DRAW_VCT_ZC_LR:vy");

    PM_LOGICAL_ZONE(vx, vx1, vx2, vx3, vx4, kmax);
    PM_LOGICAL_ZONE(vy, vy1, vy2, vy3, vy4, kmax);

    km = kmax - 1;
    lm = lmax - 1;
    nz = km*lm;
    for (j = 0; j < nz; j++)
        {valx = 0.25*ux[j];
	 valy = 0.25*uy[j];
	 i    = j + j/km;
	 vx1[i] += valx;
	 vx2[i] += valx;
	 vx3[i] += valx;
	 vx4[i] += valx;
	 vy1[i] += valy;
	 vy2[i] += valy;
	 vy3[i] += valy;
	 vy4[i] += valy;};

    _PG_draw_vct(dev, rx, ry, vx, vy, nn);

    SFREE(vx);
    SFREE(vy);

    if (eflag)
        SFREE(emap);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_VCT_NC_LR - draw node centered logical rectangular vector plot */

void PG_draw_vct_nc_lr(dev, ux, uy, rx, ry, cnnct, alst)
   PG_device *dev;
   REAL *ux, *uy, *rx, *ry;
   byte *cnnct;
   pcons *alst;
   {int nn, kmax, lmax, *maxes;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    nn    = kmax*lmax;

    _PG_draw_vct(dev, rx, ry, ux, uy, nn);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_VCT_ZC_AC - draw zone centered arbitrarily connected vector plot */

void PG_draw_vct_zc_ac(dev, ux, uy, rx, ry, cnnct, alst)
   PG_device *dev;
   REAL *ux, *uy, *rx, *ry;
   byte *cnnct;
   pcons *alst;
   {REAL *ax, *ay;

    ax = PM_zone_node_ac_2d(ux, cnnct, alst);
    ay = PM_zone_node_ac_2d(uy, cnnct, alst);

    PG_draw_vct_nc_ac(dev, ax, ay, rx, ry, cnnct, alst);

    SFREE(ax);
    SFREE(ay);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_VCT_NC_AC - draw node centered arbitrarily connected vector plot */

void PG_draw_vct_nc_ac(dev, ux, uy, rx, ry, cnnct, alst)
   PG_device *dev;
   REAL *ux, *uy, *rx, *ry;
   byte *cnnct;
   pcons *alst;
   {int nn, *nc;
    PM_mesh_topology *mt;

    mt = (PM_mesh_topology *) cnnct;
    nc = mt->n_cells;
    nn = nc[0];

    _PG_draw_vct(dev, rx, ry, ux, uy, nn);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_VCT - plots arrows (vectors)
 *              - X    - array of x coordinates
 *              - Y    - array of y coordinates
 *              - U    - array of velocity in x direction
 *              - V    - array of velocity in y direction
 *              - NPTS - number of elements in the arrays (arrows)
 */

void _PG_draw_vct(dev, x, y, u, v, npts)
   PG_device *dev;
   REAL *x, *y, *u, *v;
   int npts;
   {int i;
    REAL xp, yp, x0, y0, x1, y1, x2, y2, x3, y3, dx, dy, sx, sy;
    REAL cb, sb, ca, sa, r, f, lscale, size;
    REAL vecmax, angle;

    if (POSTSCRIPT_DEVICE(dev))
       {FILE *fp;

        fp = dev->file;
        io_printf(fp, "\n%% Begin Vectors\n\n");};

/* calculate the positions for the 2 wings on the normalized vector */
    angle = _PG_hdangle*(atan(1.0)/45.0);
    ca    = cos(angle);
    sa    = sin(angle);

/* if a maximum size was specified, compute the max vector length */
    if (_PG_maxsize != 0)
       {vecmax = 0.0;
        for (i = 0; i < npts; i++)
            {xp     = u[i];
             yp     = v[i];
             size   = sqrt(xp*xp + yp*yp);
             vecmax = max(vecmax, size);};
        lscale = _PG_sizemax/(vecmax + SMALL);}
    else
       lscale = _PG_scale;

/* compute the vectors */
    for (i = 0; i < npts; ++i)
        {x0 = x[i];
         y0 = y[i];

         WtoS(dev, x0, y0);

         xp = dev->bxw_s*u[i]*lscale;
         yp = dev->byw_s*v[i]*lscale;

/* determine the cosine and sine of the angle the vector is to be
 * plotted at 
 */
         size = sqrt(xp*xp + yp*yp) + SMALL;
         cb   = xp/size;
         sb   = yp/size;

/* set the x and y scale factor */
         if (_PG_ifix)
            r = _PG_fixsize;
         else
            r = size;

         if (_PG_ifixhead)
            f = _PG_headsize;
         else
            f = _PG_headsize*r;

         dx = r*cb;
         dy = r*sb;

/* compute new coordinates */
         x1  = x0 + dx;
         y1  = y0 + dy;

         sx = f*ca;
         sy = f*sa;

         x2 = x1 - (sx*cb + sy*sb);
         y2 = y1 - (sx*sb - sy*cb);
         x3 = x1 - (sx*cb - sy*sb);
         y3 = y1 - (sx*sb + sy*cb);

         StoW(dev, x0, y0);
         StoW(dev, x1, y1);
         StoW(dev, x2, y2);
         StoW(dev, x3, y3);

/* draw the shaft */
         PG_move_gr_abs(dev, x0, y0);
         PG_draw_to_abs(dev, x1, y1);

/* draw the head */
         PG_move_gr_abs(dev, x2, y2);
         PG_draw_to_abs(dev, x1, y1);
         if (POSTSCRIPT_DEVICE(dev))
            PG_move_gr_abs(dev, x1, y1);
         PG_draw_to_abs(dev, x3, y3);};

    if (POSTSCRIPT_DEVICE(dev))
       {FILE *fp;

        fp = dev->file;
        io_printf(fp, "\n%% End Vectors\n\n");};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_VEC_DISTINCT - plots arrows (vectors) 
 *                       - guarantee all endpoints to be distinct
 *                       - X    - array of x coordinates
 *                       - Y    - array of y coordinates
 *                       - U    - array of velocity in x direction
 *                       - V    - array of velocity in y direction
 *                       - NPTS - number of elements in the arrays (arrows)
 *                       - NOTE: this is only not static to shut up compilers
 */

void _PG_draw_vec_distinct(dev, x, y, u, v, npts)
   PG_device *dev;
   REAL *x, *y, *u, *v;
   int npts;
   {int i;
    REAL xp, yp, x0, y0, x1, y1, x2, y2, x3, y3, dx, dy, sx, sy;
    REAL cg, sg, cb, sb, ca, sa, r, f, lscale, aspect, size;
    REAL vecmax, angle, den;
    REAL xmax, xmin, ymax, ymin, dpx, dpy;

    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);
    aspect = (ymax - ymin)/(xmax - xmin + SMALL);

    PtoS(dev, 1, 1, dpx, dpy);
    StoW(dev, dpx, dpy);

/* if a maximum size was specified, compute the max vector length */
    if (_PG_maxsize != 0)
       {vecmax = 0.0;
        for (i = 0; i < npts; i++)
            {xp     = u[i];
             yp     = v[i];
             size   = sqrt(xp*xp + yp*yp);
             vecmax = max(vecmax, size);};};

/* calculate the positions for the 2 wings on the normalized vector */
    angle = _PG_hdangle*(atan(1.0)/45.0);

    f   = _PG_headsize;
    ca  = cos(angle);
    sa  = sin(angle);
    den = sqrt(1.0 + f*(f - 2.0*ca)) + SMALL;
    sg  = f*sa/den;
    cg  = (1.0 - f*ca)/den;

    if (_PG_maxsize != 0)
       lscale = _PG_sizemax/(vecmax + SMALL);
    else
       lscale = _PG_scale;

/* compute the vectors */
    for (i = 0; i < npts; ++i)
        {x0 = x[i];
         y0 = y[i];

/* determine the cosine and sine of the angle the vector is to be
 * plotted at 
 */
         xp   = u[i];
         yp   = v[i];
         size = sqrt(xp*xp + yp*yp) + SMALL;
         cb   = xp/size;
         sb   = yp/size;

/* set the x and y scale factor */
         if (_PG_ifix != 0)
            r = _PG_fixsize;
         else
            r = size*lscale;

         dx = r*cb;
         dy = r*sb;
         sx = den*dx;
         sy = den*dy;

/* compute new coordinates */
         x1 = x0 + dx;
         y1 = y0 + dy*aspect;

         x2 = x0 + sx*cg - sy*sg;
         y2 = y0 + (sy*cg + sx*sg)*aspect;
         x3 = x0 + sx*cg + sy*sg;
         y3 = y0 + (sy*cg - sx*sg)*aspect;

         if ((x0 == x1) && (y0 == y1))
            {x1 += dpx;
             y1 += dpy;};
         if ((x2 == x1) && (y2 == y1))
            {x2 -= dpx;
             y2 += dpy;};
         if ((x3 == x1) && (y3 == y1))
            {x3 -= dpx;
             y3 -= dpy;};

/* draw the shaft */
         PG_move_gr_abs(dev, x0, y0);
         PG_draw_to_abs(dev, x1, y1);

/* draw the head */
         PG_move_gr_abs(dev, x2, y2);
         PG_draw_to_abs(dev, x1, y1);
         PG_draw_to_abs(dev, x3, y3);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_VEC_ATTR - set the properties of the vectors for the
 *                 - next vector plot
 *                 - The parameters are paired, optional, and
 *                 - can be in any order.  For each pair, the first value
 *                 - describes the option, the second, the value.  The
 *                 - options are ints.  The values can be ints, REALs,
 *                 - or chars.  What  type the values are is determined
 *                 - by the option. The list must be ended  with a zero.
 *                 - 
 *                 -   VEC_SCALE     - a scale factor on vector lengths
 *                 -   VEC_ANGLE     - the angle between the arrow wings
 *                 -   VEC_FIXHEAD   - a fixed head size
 *                 -   VEC_HEADSIZE  - length of the wings
 *                 -   VEC_FIXSIZE   - a fixed vector length
 *                 -   VEC_MAXSIZE   - a maximum vector length
 *                 -   VEC_LINESTYLE - line style of vectors
 *                 -   VEC_LINETHICK - line width of vectors
 *                 -   VEC_COLOR     - line color of vectors
 */

#ifdef PCC

void PG_set_vec_attr(dev, va_alist)
   PG_device *dev;
   va_dcl

#endif

#ifdef ANSI

void PG_set_vec_attr(PG_device *dev, ...)

#endif

   {int type, itemp;
    REAL diffvp, diffwd, temp;
    REAL xmin, xmax, ymin, ymax;

/* get the current viewport & window -- this is used to go from NDC to
 * WC for specifying _PG_fixsize & _PG_maxsize 
 */
    PG_get_viewport(dev, &xmin, &xmax, &ymin, &ymax);
    diffvp = xmax - xmin;
    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);
    diffwd = xmax - xmin;

/* get the attributes */
    SC_VA_START(dev);
    while ((type = SC_VA_ARG(int)) != 0)
       {switch (type)
           {case VEC_SCALE :
	         _PG_scale = (REAL) SC_VA_ARG(double);
		 break;

            case VEC_ANGLE :
	         _PG_hdangle = (REAL) SC_VA_ARG(double);
		 break;

            case VEC_FIXHEAD :
	         _PG_ifixhead = SC_VA_ARG(int);
		 break;

            case VEC_HEADSIZE :
	         _PG_headsize = (REAL) SC_VA_ARG(double);
		 break;

            case VEC_FIXSIZE :
	         temp = SC_VA_ARG(double);
		 if (temp > 0.0)
		    {_PG_ifix = TRUE;
		     _PG_fixsize = temp*diffwd/diffvp;};
		 break;

            case VEC_MAXSIZE :
	         temp = SC_VA_ARG(double);
		 if (temp > 0.0)
		    {_PG_maxsize = TRUE;
		     _PG_sizemax = temp*diffwd/diffvp;};
		 break;

            case VEC_LINESTYLE :
	         itemp = SC_VA_ARG(int);
		 PG_set_line_style(dev, itemp);
		 break;

            case VEC_LINETHICK :
	         temp = SC_VA_ARG(double);
		 PG_set_line_width(dev, temp);
		 break;

            case VEC_COLOR :
	         itemp = SC_VA_ARG(int);
		 PG_set_color_line(dev, itemp, TRUE);
		 break;

            default :
	         break;};};

    SC_VA_END;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSVAT - set the properties of the vectors for the
 *        - next vector plot
 *        - (see the documentation for PD_SET_VEC_ATTR)
 */

#ifdef PCC

void F77_ID(pgsvat_, pgsvat, PGSVAT)(devid, va_alist)
   FIXNUM *devid;
   va_dcl

#endif

#ifdef ANSI

void F77_ID(pgsvat_, pgsvat, PGSVAT)(FIXNUM *devid, ...)

#endif

   {int type, itemp;
    REAL diffvp, diffwd, temp;
    REAL xmin, xmax, ymin, ymax;
    PG_device *dev;

    dev = *(PG_device **) devid;

/* get the current viewport & window -- this is used to go from NDC to
 * WC for specifying _PG_fixsize & _PG_maxsize 
 */
    PG_get_viewport(dev, &xmin, &xmax, &ymin, &ymax);
    diffvp = xmax - xmin;
    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);
    diffwd = xmax - xmin;

/* get the attributes */
    SC_VA_START(devid);
    while (TRUE)
       {type = *SC_VA_ARG(FIXNUM *);
        if (type == 0)
           break;

        switch (type)
           {case VEC_SCALE :
	         _PG_scale = (REAL) *SC_VA_ARG(double *);
		 break;

            case VEC_ANGLE :
	         _PG_hdangle = (REAL) *SC_VA_ARG(double *);
		 break;

            case VEC_FIXHEAD :
	         _PG_ifixhead = *SC_VA_ARG(FIXNUM *);
		 break;

            case VEC_HEADSIZE :
	         _PG_headsize = (REAL) *SC_VA_ARG(double *);
		 break;

            case VEC_FIXSIZE :
	         temp = *SC_VA_ARG(double *);
		 if (temp > 0.0)
		    {_PG_ifix = TRUE;
		     _PG_fixsize = temp*diffwd/diffvp;};
		 break;

            case VEC_MAXSIZE :
	         temp = *SC_VA_ARG(double *);
		 if (temp > 0.0)
		    {_PG_maxsize = TRUE;
		     _PG_sizemax = temp*diffwd/diffvp;};
		 break;

            case VEC_LINESTYLE :
	         itemp = *SC_VA_ARG(FIXNUM *);
		 PG_set_line_style(dev, itemp);
		 break;

            case VEC_LINETHICK :
	         temp = *SC_VA_ARG(double *);
		 PG_set_line_width(dev, temp);
		 break;

            case VEC_COLOR :
	         itemp = *SC_VA_ARG(FIXNUM *);
		 PG_set_color_line(dev, itemp, TRUE);
		 break;

            default :
	         break;};};

    SC_VA_END;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_VECTOR_CORE - vector plot skeleton routine */

static void _PG_vector_core(dev, data, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *data;
   PFByte fnc_zc, fnc_nc;
   {PG_graph *g;
    PG_picture_desc *pd;

    data->rendering = PLOT_VECTOR;

    pd = PG_setup_picture(dev, data, TRUE, TRUE, TRUE);

/* plot all of the current functions */
    for (g = data; g != NULL; g = g->next)
        PG_vct_hand(dev, g, fnc_zc, fnc_nc);

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_VECTOR_PLOT - main vector plot control routine */

#ifdef PCC

void PG_vector_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_vector_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {

    if (strcmp(data->f->category, PM_LR_S) == 0)
       _PG_vector_core(dev, data, PG_draw_vct_zc_lr, PG_draw_vct_nc_lr);

    else if (strcmp(data->f->category, PM_AC_S) == 0)
       _PG_vector_core(dev, data, PG_draw_vct_zc_ac, PG_draw_vct_nc_ac);

    return;}

/*--------------------------------------------------------------------------*/

/*                            FORTRAN API ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* PGPLVC - low level vector plot routine */

FIXNUM F77_ID(pgplvc_, pgplvc, PGPLVC)(devid, px, py, pu, pv, pn, pal)
   FIXNUM *devid;
   REAL *px, *py, *pu, *pv;
   FIXNUM *pn, *pal;
   {PG_device *dev;
    pcons *alst;

    dev  = SC_GET_POINTER(PG_device, *devid);
    alst = SC_GET_POINTER(pcons, *pal);

    _PG_draw_vct(dev, px, py, pu, pv, (int) *pn);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
