;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   <box-graphic>
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  persistent object model
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-class <box-graphic> (<leaf-object>)
  (width type: <real>)
  (height type: <real>))

(define-method status-line-when-sel ((self <box-graphic>))
  (format #f "Box ~d" (id self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   drawing
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-method pick-list* ((self <box-graphic>) pt ctm)
  (pick-on-path self
		pt
		ctm
		(list (make-point 0 0)
		      (make-point (width self) 0)
		      (make-point (width self) (height self))
		      (make-point 0 (height self))
		      (make-point 0 0))))


(define-method paint-artwork* ((self <box-graphic>) dev)
  (moveto dev (make-point 0 0))
  (lineto dev (make-point (width self) 0))
  (lineto dev (make-point (width self) (height self)))
  (lineto dev (make-point 0 (height self)))
  (lineto dev (make-point 0 0))
  (closepath dev)
  (stroke dev))

(define-method paint-object* ((self <box-graphic>) dev)
  (with-gstate-saved
   dev
   (lambda ()
     (if (get-property self 'stroke-color #f)
	 (setcolor dev (device-color dev (get-property self 'stroke-color))))
     (moveto dev (make-point 0 0))
     (lineto dev (make-point (width self) 0))
     (lineto dev (make-point (width self) (height self)))
     (lineto dev (make-point 0 (height self)))
     (lineto dev (make-point 0 0))
     (closepath dev)
     (stroke dev))))

(define-method start-active-drag ((self <box-graphic>) 
				  (in-view <open-view>)
				  (initial-pt <point>)) ;; device-sheet coords
  (let ((initial-posn (origin self))
	(ctm (invert-transform (compute-view-ctm-for self in-view))))
    (lambda ((new-pt <point>) flags)
      (set! new-pt (window->sheet in-view new-pt))
      (if (shift-state? flags)
	  (set! new-pt (shift-constraint-filter new-pt initial-pt)))
      (set-origin! self
		   (point+ initial-posn
			   (transform (point- new-pt initial-pt) ctm)))
      (mark-as-dirty (in-document in-view))
      ; expensive, but effective...
      (clear-all-areas (in-document in-view)))))

;;; handles are as follows:
;;;
;;;            3                2
;;;            *----------------*
;;;            |                |
;;;            |                |
;;;            |                |
;;;            *----------------*
;;;            0                1
;;;

(define-method accum-handles ((self <box-graphic>) accum)
  (accum self (make-point 0 0) 0)
  (accum self (make-point (width self) 0) 1)
  (accum self (make-point (width self) (height self)) 2)
  (accum self (make-point 0 (height self)) 3))


(define (box-adjuster (self <box-graphic>) handle-id)
  (let ((initial-origin (origin self))
	(initial-width (width self))
	(initial-height (height self)))
    (case handle-id
      ((0)
       (lambda ((adj <size>))
	 (set-origin! self (point+ initial-origin adj))
	 (set-width! self (+ initial-width (- (dx adj))))
	 (set-height! self (+ initial-height (- (dy adj))))))
      ((1)
       (lambda ((adj <size>))
	 (set-origin! self (point+ initial-origin
				   (make-size 0 (dy adj))))
	 (set-width! self (+ initial-width (dx adj)))
	 (set-height! self (+ initial-height (- (dy adj))))))
      ((2)
       (lambda ((adj <size>))
	 (set-width! self (+ initial-width (dx adj)))
	 (set-height! self (+ initial-height (dy adj)))))
      ((3)
       (lambda ((adj <size>))
	 (set-origin! self (point+ initial-origin
				   (make-size (dx adj) 0)))
	 (set-width! self (+ initial-width (- (dx adj))))
	 (set-height! self (+ initial-height (dy adj))))))))


(define-method start-active-drag-handle ((self <box-graphic>) 
					 (in-view <open-view>)
					 handle-id
					 (initial-pt <point>))
  (let* ((ctm (view-ctm (underlying-object in-view)))
	 (ictm (invert-transform ctm))
	 (ctm2 (invert-transform (translate ictm
					    (view-origin
					     (underlying-object in-view)))))
	 (adjuster (box-adjuster self handle-id))
	 (win (content-window in-view))
	 (gc (transient-gc win))
	 (last #f))
    ;
    (define (update-self new-pt)
      (let ((new-pt (window->sheet in-view new-pt)))
	;(dm "update box: ~s ..> ~s" initial-pt new-pt)
	(adjuster (transform (point- new-pt initial-pt) ictm))))
    ;
    (vector
     (lambda ((new-pt <point>) flags)
       (update-self new-pt)
       (let ((dp0 (transform (origin self) ctm2))
	     (dp1 (transform (point+ (origin self)
				     (make-size (width self) 0))
			     ctm2))
	     (dp2 (transform (point+ (origin self)
				     (make-size (width self) (height self)))
			     ctm2))
	     (dp3 (transform (point+ (origin self)
				     (make-size 0 (height self)))
			     ctm2)))
	 (if last
	     (draw-lines win gc last))
	 (set! last (map inexact->exact
			  (list (x dp0) (y dp0)
				(x dp1) (y dp1)
				(x dp2) (y dp2)
				(x dp3) (y dp3)
				(x dp0) (y dp0))))
	 (draw-lines win gc last)
	 (flush-client)))
     (lambda ((new-pt <point>) flags)
       (if last
	   (draw-lines win gc last))
       (mark-as-dirty (in-document in-view))
       (update-self new-pt)
       (clear-all-areas (in-document in-view))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   box-drawing tool
;;;
;;;   (nb, this is really a graphic command functionality)
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (drawbox-button-press (in-view <open-view>)
			      (dp0 <point>) ;; window device coords
			      modifier-state)
  (let* ((fwd-ctm (view-ctm (underlying-object in-view)))
	 ;; `ctm' converts window-device coords to user coords
	 (ctm (translate (invert-transform fwd-ctm)
			 (view-origin (underlying-object in-view))))
	 (fwd-ctm (invert-transform ctm)) ;; and back again
	 (win (content-window in-view))
	 (dp0 (tweak-point in-view dp0)) ;; snap to geometry (window d.c.)
	 (up0 (transform dp0 ctm))
	 (gc (transient-gc win))
	 (last #f))
    (set-active-drag-proc!
     in-view
     (vector
      (lambda ((dp2 <point>) flags)
	; we have to convert to user coordinates to draw the box
	; because our view might be rotated.  For just straight lines,
	; this is not a problem, because they are invariant under
	; affine xforms.
	(let* ((up2 (transform dp2 ctm))
	       (dp1 (transform (make-point (x up2) (y up0)) fwd-ctm))
	       (dp3 (transform (make-point (x up0) (y up2)) fwd-ctm)))
	  (if last
	      (draw-lines win gc last))
	  (set! last (map inexact->exact
			  (list (x dp0) (y dp0)
				(x dp1) (y dp1)
				(x dp2) (y dp2)
				(x dp3) (y dp3)
				(x dp0) (y dp0))))
	  (draw-lines win gc last)
	  (flush-client)))
      (lambda ((dp2 <point>) flags)
	(let* ((up2 (transform dp2 ctm))
	       (par (page-contents
		     (view-page
		      (underlying-object in-view))))
	       (box (make <box-graphic>
			  in-document: (in-document par)
			  parent-object: par
			  origin: (make-point (min (x up0) (x up2))
					      (min (y up0) (y up2)))
			  graphic-bounding-box: (make-rect 0 0 0 0)
			  width: (abs (- (x up2) (x up0)))
			  height: (abs (- (y up2) (y up0))))))
	  (if last
	      (draw-lines win gc last))
	  (clear-all-areas (in-document in-view))
	  (do-select in-view box 0)
	  (update-handles in-view)))))))

(add-major-mode!
 (make <major-mode>
       name: 'draw-box
       button-press-proc: drawbox-button-press))

(define-interactive (draw-box-mode view)
  (interactive (owner))
  (set-major-mode! view (get-major-mode 'draw-box)))

(graphic-set-key #\M-2 draw-box-mode)

;;;

(define-method externalize ((self <box-graphic>))
  `(box origin-x: ,(x (origin self))
	origin-y: ,(y (origin self))
	width: ,(width self) 
	height: ,(height self)))

(define (paste-box-from-extern extern group offset)
  (apply (lambda (#key (origin-x default: 0)
		       (origin-y default: 0)
		       width
		       height
		       (stroke-color default: #f)
		       (fill-color default: #f)
		       (stroke-width default: #f))
	   (let* ((sw (or stroke-width 1))
		  (b (make <box-graphic>
			  in-document: (in-document group)
			  parent-object: group
			  origin: (point+ (make-point origin-x origin-y) 
					  offset)
			  graphic-bounding-box: (inset-rect
						  (make-rect 0 0 width height)
						  (- sw)
						  (- sw))
						 ;(dx offset)
						 ;(dy offset))
			  width: width
			  height: height)))
	     (if stroke-color (set-property! b 'stroke-color stroke-color))
	     (if stroke-width (set-property! b 'stroke-width stroke-width))
	     (if fill-color (set-property! b 'fill-color fill-color))
	     b))
	 (cdr extern)))
