      SUBROUTINE PBDSYR2K( ICONTXT, MATBLK, UPLO, TRANS, N, K, NB,
     $                     ALPHA, A, LDA, B, LDB, BETA, C, LDC, IAPOS,
     $                     IBPOS, ICROW, ICCOL, ABCOMM, AWORK, BWORK,
     $                     CWORK, MULLEN, WORK )
*
*  -- PB-BLAS routine (version 2.1) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory.
*     April 28, 1996
*
*     Jaeyoung Choi, Oak Ridge National Laboratory
*     Jack Dongarra, University of Tennessee and Oak Ridge National Lab.
*     David Walker,  Oak Ridge National Laboratory
*
*     .. Scalar Arguments ..
      CHARACTER*1        ABCOMM, AWORK, BWORK, CWORK, MATBLK, TRANS,
     $                   UPLO
      INTEGER            IAPOS, IBPOS, ICCOL, ICONTXT, ICROW, K, LDA,
     $                   LDB, LDC, MULLEN, N, NB
      DOUBLE PRECISION   ALPHA, BETA
*     ..
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), B( LDB, * ), C( LDC, * ),
     $                   WORK( * )
*     ..
*
*  Purpose
*  =======
*
*  PBDSYR2K is a parallel blocked version of the Level-3 BLAS DSYR2K.
*  PBDSYR2K performs one of the symmetric rank k operations
*
*     C := alpha*A*B' + alpha*B*A' + beta*C,
*
*  or
*
*     C := alpha*A'*B + alpha*B'*A + beta*C,
*
*  where  alpha and beta  are scalars,  C is an n by n  symmetric matrix
*  and  A and B  are an  n by k  matrix in the first case  and  a k by n
*  matrix in the second case. k is limited to the block size NB, so that
*  A and B are block columns (if TRANS = 'N') or block rows (if TRANS =
*  'T'/'C').
*
*  The first elements  of the matrices A, B, and C should be located  at
*  the beginnings of their first blocks. (not the middle of the blocks.)
*  A and B  can  be broadcast  if necessary  and  then  transposed.  The
*  communication scheme can be selected.
*
*  Parameters
*  ==========
*
*  MATBLK  (input) CHARACTER*1
*          MATBLK specifies whether C is a (full) block matrix or
*          a single block as follows:
*
*             MATBLK = 'M',  C is a (full) block matrix,
*             MATBLK = 'B',  C is a single block.
*
*  UPLO    (input) CHARACTER*1
*          UPLO specifies  whether the upper or lower triangular part
*          of the array C is to be referenced as follows:
*
*             UPLO = 'U',   Only the  upper triangular part of  C
*                           is to be referenced.
*             UPLO = 'L',   Only the  lower triangular part of  C
*                           is to be referenced.
*
*  TRANS   (input) CHARACTER*1
*          TRANS specifies the operation to be performed as follows:
*
*             TRANS = 'N',      C := alpha*A*B' + alpha*B*A' + beta*C.
*             TRANS = 'T'/'C',  C := alpha*A'*B + alpha*B'*A + beta*C.
*
*  N       (input) INTEGER
*          NB specifies the row and column block size of the matrix C.
*          It also specifies the row block size of the matrices A and B
*          if MATBLK = 'M' and TRANS = 'N', or MATBLK = 'B' and TRANS =
*          'T'/'C'; and the column block size of the matrices A and B if
*          MATBLK = 'M' and TRANS = 'T'/'C', or MATBLK = 'B' and TRANS =
*          'N'.  NB >= 1.
*
*  K       (input) INTEGER
*          If TRANS = 'N',  K specifies  the number of columns of the
*          matrix A, and if TRANS = 'T'/'C', K specifies the number of
*          rows of the matrix  A.  K >= 0.
*
*  NB      (input) INTEGER
*          NB specifies the row and column block size of the matrix C.
*          It also specifies the row block size of the matrices A and B
*          if MATBLK = 'M' and TRANS = 'N', or MATBLK = 'B' and TRANS =
*          'T'/'C'; and the column block size of the matrices A and B
*          if MATBLK = 'M' and TRANS = 'T'/'C', or MATBLK = 'B' and
*          TRANS = 'N'.  NB >= 1.
*
*  ALPHA   (input) DOUBLE PRECISION
*          ALPHA specifies the scalar alpha.
*
*  A       (input) DOUBLE PRECISION array of local DIMENSION (LDA, ka),
*          where  ka is Kq when TRANS = 'N', or Nq otherwise.
*          If TRANS = 'N', the leading Np-by-Kq part of the array A
*          must contain the (local) matrix A, otherwise the leading Kp-
*          by-Nq part of the array A must contain the (local) matrix A.
*
*  LDA     (input) INTEGER
*          LDA specifies the first dimension of (local) A as declared
*          in the calling (sub) program.
*          LDA >= MAX(1,Np) if MATBLK = 'M' & TRANS = 'N',
*                           or MATBLK = 'B' & TRANS = 'T'/'C',
*          LDA >= MAX(1,Kp) if MATBLK = 'M' & TRANS = 'T'/'C',
*                           or MATBLK = 'B' & TRANS = 'N'.
*
*  B       (input) DOUBLE PRECISION array of local DIMENSION (LDB, ka),
*          where ka is Kq when TRANS = 'N', or Nq otherwise.
*          If TRANS = 'N', the leading Np-by-Kq part of the array B
*          must contain the (local) matrix B, otherwise the leading Kp-
*          by-Nq part of the array B must contain the (local) matrix B.
*
*  LDB     (input) INTEGER
*          LDB specifies the leading dimension of (local) B as declared
*          in the calling (sub) program.
*          LDB >= MAX(1,Np) if MATBLK = 'M' & TRANS = 'N',
*                           or MATBLK = 'B' & TRANS = 'T'/'C',
*          LDB >= MAX(1,Kp) if MATBLK = 'M' & TRANS = 'T'/'C',
*                           or MATBLK = 'B' & TRANS = 'N'.
*
*  BETA    (input) DOUBLE PRECISION
*          BETA specifies the scalar beta.
*
*  C       (input/output) DOUBLE PRECISION array of local DIMENSION
*          ( LDC, Nq ).
*           On entry with UPLO='U', the leading N-by-N upper triangular
*           part of the (global) array C  must contain the upper trian-
*           gular part of the symmetric matrix and the strictly lower
*           triangular part  of C is not referenced.  On exit, the upper
*           triangular part of the array  C is overwritten by the upper
*           triangular part of the updated  matrix.  Before entry with
*           UPLO = 'L', the leading  N-by-N lower triangular part of the
*           (global) array C  must  contain the lower triangular part of
*           the symmetric matrix and the strictly upper triangular part
*           of C is not referenced.  On exit the lower triangular part
*           of the array C is overwritten by the lower triangular part
*           of the updated matrix.
*
*  LDC     (input) INTEGER
*          LDC specifies the leading dimension of (local) C as declared
*          in the calling (sub) program.  LDC >= MAX(1,Np).
*
*  IAPOS   (input) INTEGER
*          If TRANS = 'N', IAPOS specifies a column of process
*          template, which holds the first block of A.  And if TRANS =
*          'T'/'C', IAPOS specifies a row of the template, which holds
*          the first block of A.
*          If MATBLK = 'M' and all columns or rows of processes have
*          a copy of A, then set IAPOS = -1.
*          And if MATBLK = 'B', IAPOS should be the same as IBPOS.
*
*  IBPOS   (input) INTEGER
*          If TRANS = 'N', IAPOS specifies a column of process
*          template, which holds the first block of B.  And if TRANS =
*          'T'/'C', IBPOS specifies a row of the template, which holds
*          the first block of A.
*          If MATBLK = 'M' and all columns or rows of processes have
*          a copy of B, then set IBPOS = -1.
*          And if MATBLK = 'B', IBPOS should be the same as IAPOS.
*
*  ICROW   (input) INTEGER
*          It specifies a row of process template which has the
*          first block of C.  It also represents the first (row)
*          process of the column block C if TRANS = 'N'.
*          When MATBLK = 'B', and all rows of processes have their
*          own copies of C, set ICROW =  -1.
*
*  ICCOL   (input) INTEGER
*          It specifies a column of process template which has the
*          first block of C.  It also represents the  first (column)
*          process  of the row block C if TRANS = 'T'/'C'.
*          When MATBLK = 'B', and all columns of processes have
*          their own copies of C, set ICCOL = -1.
*
*  ABCOMM  (input) CHARACTER*1
*          When MATBLK = 'M', ABCOMM specifies the communication scheme
*          of column or row block of A and B if communication is
*          necessary. It follows topology definition of BLACS.
*          When MATBLK = 'B', the argument is ignored.
*
*  AWORK   (input) CHARACTER*1
*          When MATBLK = 'M', AWORK determines whether A is a
*          workspace or not.
*
*             AWORK = 'Y':  A is workspace in other processes.
*                           A is sent to A position in other processes.
*                           It is assumed that processes have
*                           sufficient space to store (local) A.
*             AWORK = 'N':  Data of A in other processes will be
*                           untouched (unchanged).
*
*          When MATBLK = 'B', it is ignored.
*
*  BWORK   (input) CHARACTER*1
*          When MATBLK = 'M', BWORK determines whether B is a
*          workspace or not.
*
*             BWORK = 'Y':  B is workspace in other processes.
*                           B is sent to B position in other processes.
*                           It is assumed that processes have
*                           sufficient space to store (local) B.
*             BWORK = 'N':  Data of B in other processes will be
*                           untouched (unchanged).
*
*          When MATBLK = 'B', it is ignored.
*
*  CWORK   (input) CHARACTER*1
*          When MATBLK = 'M', CWORK determines whether the other
*          triangular part of C is accessed and modified or not.
*
*            CWORK = 'N': if UPLO = 'U', only upper triangular portion
*                         portion of the matrix C is accessed and the
*                         lower triangular portion is untouched.
*                         Likewise if UPLO = 'L', only lower triangular
*                         portion of the matrix C is accessed and the
*                         upper triangular portion is untouched.
*            CWORK = 'Y': if UPLO = 'U', only lower triangular portion
*                         of the matrix C may be accessed and modified
*                         for fast computation.  And if UPLO = 'L', the
*                         upper triangular portion of the matrix C may
*                         be accessed and modified for fast computation.
*
*          And when MATBLK = 'B', CWORK determines whether C is a
*          workspace or not.
*
*            CWORK = 'Y':  C is workspace in other processes.
*                          C is sent to C position in other processes.
*                          It is assumed that processes have
*                          sufficient space to store C.
*            CWORK = 'N':  Data of C in other processes will be
*                          untouched (unchanged).
*
*  MULLEN  (input) INTEGER
*          When MATBLK = 'M', MULLEN specifies multiplication length of
*          the optimum column number of a row block A if TRANS='T'/'C',
*          or A' if TRANS = 'N' for multiplying A with A'.  The value
*          depends on machine characteristics.
*          When MATBLK = 'B', it is ignored.
*
*  WORK    (workspace) DOUBLE PRECISION array of dimension Size(WORK)
*          It will store copy of A, A' B, and/or B'.
*
*  Parameters Details
*  ==================
*
*  Lx      It is  a local portion  of L  owned  by  a process,  (L is
*          replaced by M, or N,  and x  is replaced  by  either  p
*          (=NPROW) or q (=NPCOL)).  The value is determined by  L, LB,
*          x, and MI,  where  LB is  a block size  and MI is a  row  or
*          column position in a process template.  Lx is equal to  or
*          less than  Lx0 = CEIL( L, LB*x ) * LB.
*
*  Memory Requirement of WORK
*  ==========================
*
*  Npb    = CEIL( N, NB*NPROW )
*  Nqb    = CEIL( N, NB*NPCOL )
*  Np0    = NUMROC( N, NB, 0, 0, NPROW ) ~= Npb * NB
*  Nq0    = NUMROC( N, NB, 0, 0, NPCOL ) ~= Nqb * NB
*  LCMQ   = LCM / NPCOL
*  LCMP   = LCM / NPROW
*  ISZCMP = CEIL(MULLEN, LCMQ*NB)
*  SZCMP  = ISZCMP * ISZCMP * LCMQ*NB * LCMP*NB
*
*  (1) MATBLK = 'M'
*    (a) TRANS = 'N'
*      Size(WORK) = K * Nq0
*                 + K * Np0        ( if IAPOS <> -1 and AWORK <> 'Y' )
*                 + K * Np0        ( if IBPOS <> -1 and BWORK <> 'Y' )
*                 + MAX[ SZCMP                     ( if CWORK <> 'Y' ),
*                        K*CEIL(Nqb,LCMQ)*NB*MIN(LCMQ,CEIL(N,NB) ]
*    (b) TRANS = 'T'/'C'
*      Size(WORK) = K * Np0
*                 + K * Nq0        ( if IAPOS <> -1 and AWORK <> 'Y' )
*                 + K * Nq0        ( if IBPOS <> -1 and BWORK <> 'Y' )
*                 + MAX[ SZCMP                     ( if CWORK <> 'Y' ),
*                        K*CEIL(Npb,LCMP)*NB*MIN(LCMP,CEIL(N,NB) ]
*
*  (2) MATBLK = 'B'
*    (a) TRANS = 'N'     (on a row of processes ICROW only )
*      Size(WORK) = N * (N+1) / 2     ( if CWORK =  'Y')
*      Size(WORK) = N * N             ( if CWORK <> 'Y')
*    (b) TRANS = 'T'/'C' (on a column of processes ICCOL only )
*      Size(WORK) = N * (N+1) / 2     ( if CWORK =  'Y')
*      Size(WORK) = N * N             ( if CWORK <> 'Y')
*
*  Notes
*  -----
*  More precise space can be computed as
*
*  CEIL(Nqb,LCMQ)*NB => NUMROC( NUMROC(N,NB,0,0,NPCOL), NB, 0, 0, LCMQ )
*                    = NUMROC( Nq0, NB, 0, 0, LCMQ )
*  CEIL(Npb,LCMP)*NB => NUMROC( NUMROC(N,NB,0,0,NPROW), NB, 0, 0, LCMP )
*                    = NUMROC( Np0, NB, 0, 0, LCMP )
*
*  =====================================================================
*
*     ..
*     .. Parameters ..
      DOUBLE PRECISION   ONE, ZERO
      PARAMETER          ( ONE = 1.0D+0, ZERO = 0.0D+0 )
*     ..
*     .. Local Scalars ..
      CHARACTER*1        COMMAB, FORM
      LOGICAL            ADATA, ASPACE, BDATA, BSPACE, CMAT, CSPACE,
     $                   NOTRAN, UPPER
      INTEGER            INFO, IPB, IPBZ, IPT, IPW, IQBZ, ISZCMP, JJ,
     $                   JNPBZ, JPBZ, JQBZ, KI, KIZ, KJ, KJZ, LCM, LCMP,
     $                   LCMQ, LKK, LMW, LNW, LPBZ, LQBZ, MRCOL, MRROW,
     $                   MYCOL, MYROW, MZCOL, MZROW, NP, NPCOL, NPROW,
     $                   NQ
*     ..
*     .. External Functions ..
      LOGICAL            LSAME
      INTEGER            ICEIL, ILCM, NUMROC
      EXTERNAL           ICEIL, ILCM, LSAME, NUMROC
*     ..
*     .. External Subroutines ..
      EXTERNAL           BLACS_GRIDINFO, DGEBR2D, DGEBS2D, DGEMM,
     $                   DSYR2K, PBDMATADD, PBDT1CPY, PBDT2CPY,
     $                   PBDT3CPY, PBDTRADD, PBDTRAN, PXERBLA
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN
*     ..
*     .. Executable Statements ..
*
*     Quick return if possible.
*
      IF( N.EQ.0 .OR.
     $    ( ( ALPHA.EQ.ZERO .OR. K.EQ.0 ) .AND. BETA.EQ.ONE ) )
     $   RETURN
*
      CALL BLACS_GRIDINFO( ICONTXT, NPROW, NPCOL, MYROW, MYCOL )
*
      CMAT   = LSAME( MATBLK, 'M' )
      UPPER  = LSAME( UPLO,   'U' )
      NOTRAN = LSAME( TRANS,  'N' )
*
*     Test the input parameters.
*
      INFO = 0
      IF(      ( .NOT.CMAT                 ).AND.
     $         ( .NOT.LSAME( MATBLK, 'B' ) )         ) THEN
         INFO = 2
      ELSE IF( ( .NOT.UPPER                ).AND.
     $         ( .NOT.LSAME( UPLO,   'L' ) )         ) THEN
         INFO = 3
      ELSE IF( ( .NOT.NOTRAN               ).AND.
     $         ( .NOT.LSAME( TRANS, 'T' )  ).AND.
     $         ( .NOT.LSAME( TRANS, 'C' )  )         ) THEN
         INFO = 4
      ELSE IF( N  .LT.0                              ) THEN
         INFO = 5
      ELSE IF( K  .LT.0                              ) THEN
         INFO = 6
      ELSE IF( NB .LT.1                              ) THEN
         INFO = 7
      END IF
*
   10 CONTINUE
      IF( INFO.NE.0 ) THEN
         CALL PXERBLA( ICONTXT, 'PBDSYR2K ', INFO )
         RETURN
      END IF
*
*     Start the operations.
*
* === If C is a matrix ===
*
      IF( CMAT ) THEN
        NP = NUMROC( N, NB, MYROW, ICROW, NPROW )
        NQ = NUMROC( N, NB, MYCOL, ICCOL, NPCOL )
*
        IF( LDC.LT.MAX(1,NP)                    ) THEN
          INFO = 15
        ELSE IF( ICROW.LT.0 .OR. ICROW.GE.NPROW ) THEN
          INFO = 18
        ELSE IF( ICCOL.LT.0 .OR. ICCOL.GE.NPCOL ) THEN
          INFO = 19
        END IF
*
        ADATA  = .FALSE.
        IF( IAPOS.EQ.-1 ) ADATA = .TRUE.
        BDATA  = .FALSE.
        IF( IBPOS.EQ.-1 ) BDATA = .TRUE.
        ASPACE = LSAME( AWORK, 'Y' )
        BSPACE = LSAME( BWORK, 'Y' )
        CSPACE = LSAME( CWORK, 'Y' )
        COMMAB = ABCOMM
        IF( LSAME( COMMAB, ' ' ) ) COMMAB = '1'
*
*       LCM : the least common multiple of NPROW and NPCOL
*
        LCM  = ILCM( NPROW, NPCOL )
        LCMP = LCM  / NPROW
        LCMQ = LCM  / NPCOL
        LPBZ = LCMP * NB
        LQBZ = LCMQ * NB
*
        MRROW = MOD( NPROW+MYROW-ICROW, NPROW )
        MRCOL = MOD( NPCOL+MYCOL-ICCOL, NPCOL )
        LKK = MAX( 1, K )
*
*       PART 1: Distribute a column (or row) block A & B and transpose B
*       ================================================================
*
        IF( NOTRAN ) THEN
*
*         Form  C := alpha*A*B' + alpha*B*A' + beta*C.
*       __________     _                  _                  __________
*      |\_        |   | |                | |                |\_        |
*      |  \_      |   | |                | |                |  \_      |
*      |    \     |   | |  __________    | |  __________    |    \     |
*      |     C_   |=a*|A|*|___(B')___|+a*|B|*|___(A')___|+b*|     C_   |
*      |       \_ |   | |                | |                |       \_ |
*      |_________\|   |_|                |_|                |_________\|
*
          IF( LDA.LT.MAX(1,NP) .AND. ( ASPACE .OR.
     $        IAPOS.EQ.MYCOL .OR. IAPOS.EQ.-1 )    ) THEN
            INFO = 10
          ELSE IF( LDB.LT.MAX(1,NP) .AND. ( BSPACE .OR.
     $        IBPOS.EQ.MYCOL .OR. IBPOS.EQ.-1 )    ) THEN
            INFO = 12
          ELSE IF( IAPOS.LT.-1 .OR. IAPOS.GE.NPCOL ) THEN
            INFO = 16
          ELSE IF( IBPOS.LT.-1 .OR. IBPOS.GE.NPCOL ) THEN
            INFO = 17
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Broadcast A and B if necessary
*
          IPT = 1
          IPB = 1
          IF( .NOT.ADATA ) THEN
            IF( ASPACE ) THEN
              IF( MYCOL.EQ.IAPOS ) THEN
                CALL DGEBS2D( ICONTXT, 'Row', COMMAB, NP, K, A, LDA )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Row', COMMAB, NP, K, A, LDA,
     $                        MYROW, IAPOS )
              END IF
              ADATA = .TRUE.
            ELSE
              IF( MYCOL.EQ.IAPOS ) THEN
                CALL PBDMATADD( ICONTXT, 'V', NP, K, ONE, A, LDA, ZERO,
     $                          WORK, NP )
                CALL DGEBS2D( ICONTXT, 'Row', COMMAB, NP, K, WORK, NP )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Row', COMMAB, NP, K, WORK, NP,
     $                        MYROW, IAPOS )
              END IF
              IPT = NP * K + 1
              IPB = IPT
            END IF
          END IF
*
          IF( .NOT.BDATA ) THEN
            IF( BSPACE ) THEN
              IF( MYCOL.EQ.IBPOS ) THEN
                CALL DGEBS2D( ICONTXT, 'Row', COMMAB, NP, K, B, LDB )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Row', COMMAB, NP, K, B, LDB,
     $                        MYROW, IBPOS )
              END IF
              BDATA = .TRUE.
            ELSE
              IF( MYCOL.EQ.IBPOS ) THEN
                CALL PBDMATADD( ICONTXT, 'V', NP, K, ONE, B, LDB, ZERO,
     $                          WORK(IPB), NP )
                CALL DGEBS2D( ICONTXT, 'Row', COMMAB, NP, K,
     $                        WORK(IPB), NP )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Row', COMMAB, NP, K,
     $                        WORK(IPB), NP, MYROW, IBPOS )
              END IF
              IPT = NP * K + IPT
            END IF
          END IF
*
*         Transpose column blocks of B to WORK(IPT).
*
          IPW = K * NQ + IPT
          IF( BDATA ) THEN
            CALL PBDTRAN( ICONTXT, 'Col', 'T', N, K, NB, B, LDB, ZERO,
     $                    WORK(IPT), K, ICROW, -1, -1, ICCOL,
     $                    WORK(IPW) )
          ELSE
            CALL PBDTRAN( ICONTXT, 'Col', 'T', N, K, NB, WORK(IPB), NP,
     $                    ZERO, WORK(IPT),K, ICROW, -1, -1, ICCOL,
     $                    WORK(IPW) )
          END IF
*
        ELSE
*
*         Form  C := alpha*A'*B + alpha*B'*A + beta*C.
*     __________     _                   _                   __________
*    |\_        |   | |                 | |                 |\_        |
*    |  \_      |   | |                 | |                 |  \_      |
*    |    \     |   | |  ___________    | |  ___________    |    \     |
*    |     C_   |=a* A'*|_____B_____|+a* B'*|_____A_____|+b*|     C_   |
*    |       \_ |   | |                 | |                 |       \_ |
*    |_________\|   |_|                 |_|                 |_________\|
*
          IF( LDA.LT.MAX(1,K) .AND. ( ASPACE .OR.
     $        IAPOS.EQ.MYROW .OR. IAPOS.EQ.-1 )         ) THEN
            INFO = 10
          ELSE IF( LDB.LT.MAX(1,K) .AND. ( BSPACE .OR.
     $        IBPOS.EQ.MYROW .OR. IBPOS.EQ.-1 )         ) THEN
            INFO = 12
          ELSE IF( IAPOS.LT.-1 .OR. IAPOS.GE.NPROW      ) THEN
            INFO = 16
          ELSE IF( IBPOS.LT.-1 .OR. IBPOS.GE.NPROW      ) THEN
            INFO = 17
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Broadcast A and B if necessary
*
          IPT = 1
          IPB = 1
          IF( .NOT.ADATA ) THEN
            IF( ASPACE ) THEN
              IF( MYROW.EQ.IAPOS ) THEN
                CALL DGEBS2D( ICONTXT, 'Col', COMMAB, K, NQ, A, LDA )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Col', COMMAB, K, NQ, A, LDA,
     $                        IAPOS, MYCOL )
              END IF
              ADATA = .TRUE.
            ELSE
              IF( MYROW.EQ.IAPOS ) THEN
                CALL PBDMATADD( ICONTXT, 'G', K, NQ, ONE, A, LDA, ZERO,
     $                          WORK, K )
                CALL DGEBS2D( ICONTXT, 'Col', COMMAB, K, NQ, WORK, K )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Col', COMMAB, K, NQ, WORK, K,
     $                        IAPOS, MYCOL )
              END IF
              IPT = K * NQ + 1
              IPB = IPT
            END IF
          END IF
*
          IF( .NOT.BDATA ) THEN
            IF( BSPACE ) THEN
              IF( MYROW.EQ.IBPOS ) THEN
                CALL DGEBS2D( ICONTXT, 'Col', COMMAB, K, NQ, B, LDB )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Col', COMMAB, K, NQ, B, LDB,
     $                        IBPOS, MYCOL )
              END IF
              BDATA = .TRUE.
            ELSE
              IF( MYROW.EQ.IBPOS ) THEN
                CALL PBDMATADD( ICONTXT, 'G', K, NQ, ONE, B, LDB, ZERO,
     $                          WORK(IPB), K )
                CALL DGEBS2D( ICONTXT, 'Col', COMMAB, K, NQ,
     $                        WORK(IPB), K )
              ELSE
                CALL DGEBR2D( ICONTXT, 'Col', COMMAB, K, NQ,
     $                        WORK(IPB), K, IBPOS, MYCOL )
              END IF
              IPT = K * NQ + IPT
            END IF
          END IF
*
*         Transpose column blocks of B to WORK(IPT).
*
          IPW = NP * K + IPT
          IF( BDATA ) THEN
            CALL PBDTRAN( ICONTXT, 'Row', 'T', K, N, NB, B, LDB, ZERO,
     $                    WORK(IPT), NP, -1, ICCOL, ICROW, -1,
     $                    WORK(IPW) )
          ELSE
            CALL PBDTRAN( ICONTXT, 'Row', 'T', K, N, NB, WORK(IPB), K,
     $                    ZERO, WORK(IPT), NP, -1, ICCOL, ICROW, -1,
     $                    WORK(IPW) )
          END IF
        END IF
*
*       PART 2: Update C with A and B'
*       ==============================
*
        IF( NP.EQ.0 .OR. NQ.EQ.0 ) GO TO 80
*
*       If C is a symmetric upper triangular matrix,
*
        IF( UPPER ) THEN
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ = ISZCMP * LPBZ
          IQBZ = ISZCMP * LQBZ
          JPBZ = 0
          JQBZ = 0
*
          DO 40 JJ = 1, ICEIL(NQ, IQBZ)
            LMW = MIN( IPBZ, NP-JPBZ )
            LNW = MIN( IQBZ, NQ-JQBZ )
            JNPBZ = JPBZ + LMW
*
*           Modify (change) data in the lower triangular part
*
            IF( CSPACE ) THEN
*
*             Update C := alpha*A*B' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        A, LDA, WORK(JQBZ*K+IPT), LKK, BETA,
     $                        C(1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        WORK, NP, WORK(JQBZ*K+IPT), LKK, BETA,
     $                        C(1,JQBZ+1), LDC )
                END IF
*
*             Update C := alpha*B'*A + beta*C, if TRANS = 'T'/'C'
*
              ELSE
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, A(1,JQBZ+1), LDA, BETA,
     $                        C(1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, WORK(JQBZ*K+1), LKK, BETA,
     $                        C(1,JQBZ+1), LDC )
                END IF
              END IF
*
*           Update data in the upper triangular matrix
*           and save data in the lower triangular matrix
*
            ELSE
*
*             Update C := alpha*A*B' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA, A, LDA,
     $                        WORK(JQBZ*K+IPT), LKK, BETA,
     $                        C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        A(JPBZ+1,1), LDA, WORK(JQBZ*K+IPT), LKK,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA,
     $                        WORK, NP, WORK(JQBZ*K+IPT), LKK, BETA,
     $                        C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+1), NP, WORK(JQBZ*K+IPT), LKK,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
*
*             Update C := alpha*B'*A + beta*C, if TRANS = 'T'/'C'
*
              ELSE
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, A(1,JQBZ+1), LDA, BETA,
     $                        C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, A(1,JQBZ+1), LDA,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, WORK(JQBZ*K+1), LKK,
     $                        BETA, C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, WORK(JQBZ*K+1), LKK,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
              END IF
*
*             Compute diagonal blocks.
*
              MZROW = MRROW
              MZCOL = MRCOL
              KI = 0
*
              DO 30 KJ = 0, LCMQ-1
   20           CONTINUE
                IF( MZROW.LT.MZCOL ) THEN
                  MZROW = MZROW + NPROW
                  KI = KI + 1
                  GO TO 20
                END IF
                KIZ = KI * NB
                KJZ = KJ * NB
                IF( KJZ.GE.LNW ) GO TO 40
                FORM = 'G'
                IF( MZROW.EQ.MZCOL ) FORM = 'T'
                MZCOL = MZCOL + NPCOL
*
                CALL PBDTRADD( ICONTXT, 'Upper', FORM, KIZ, NB, ONE,
     $                         WORK(KJZ*LMW+IPW), LMW, BETA,
     $                         C(JPBZ+1,JQBZ+KJZ+1), LDC,
     $                         LPBZ, LQBZ, LMW, LNW-KJZ )
   30         CONTINUE
            END IF
*
            JPBZ = JNPBZ
            JQBZ = JQBZ + LNW
   40     CONTINUE
*
*       If C is a symmetric lower triangular matrix,
*
        ELSE
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ = ISZCMP * LPBZ
          IQBZ = ISZCMP * LQBZ
          JPBZ = 0
          JQBZ = 0
*
          DO 70 JJ = 1, ICEIL(NQ, IQBZ)
            LMW = MIN( IPBZ, MAX(NP-JPBZ, 0) )
            LNW = MIN( IQBZ, MAX(NQ-JQBZ, 0) )
            JNPBZ = JPBZ + LMW
*
*           Modify (change) data in the upper triangular part
*
            IF( CSPACE ) THEN
*
*             Update C := alpha*A*B' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        A(JPBZ+1,1), LDA, WORK(JQBZ*K+IPT), LKK,
     $                        BETA, C(JPBZ+1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        WORK(JPBZ+1), NP, WORK(JQBZ*K+IPT), LKK,
     $                        BETA, C(JPBZ+1,JQBZ+1), LDC )
                END IF
*
*             Update C := alpha*B'*A + beta*C, if TRANS = 'T'/'C'
*
              ELSE
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, A(1,JQBZ+1), LDA,
     $                        BETA, C(JPBZ+1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, WORK(JQBZ*K+1), LKK,
     $                        BETA, C(JPBZ+1,JQBZ+1), LDC )
                END IF
              END IF
*
*           Update data in the lower triangular matrix
*           and save data in the upper triangular matrix
*
            ELSE
*
*             Update C := alpha*A*B' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        A(JNPBZ+1,1), LDA, WORK(JQBZ*K+IPT), LKK,
     $                        BETA, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        A(JPBZ+1,1), LDA, WORK(JQBZ*K+IPT), LKK,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        WORK(JNPBZ+1), NP, WORK(JQBZ*K+IPT), LKK,
     $                        BETA, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+1), NP, WORK(JQBZ*K+IPT), LKK,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
*
*             Update C := alpha*B'*A + beta*C, if TRANS = 'T'/'C'
*
              ELSE
                IF( ADATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        WORK(JNPBZ+IPT), NP, A(1,JQBZ+1), LDA,
     $                        BETA, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, A(1,JQBZ+1), LDA,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        WORK(JNPBZ+IPT), NP, WORK(JQBZ*K+1), LKK,
     $                        BETA, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, WORK(JQBZ*K+1), LKK,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
              END IF
*
*             Compute diagonal blocks.
*
              MZROW = MRROW
              MZCOL = MRCOL
              KI = 0
*
              DO 60 KJ = 0, LCMQ-1
   50           CONTINUE
                IF( MZROW.LT.MZCOL ) THEN
                  MZROW = MZROW + NPROW
                  KI = KI + 1
                  GO TO 50
                END IF
                KIZ = KI * NB
                KJZ = KJ * NB
                IF( KJZ.GE.LNW ) GO TO 70
                FORM = 'G'
                IF( MZROW.EQ.MZCOL ) FORM = 'T'
                MZCOL = MZCOL + NPCOL
*
                CALL PBDTRADD( ICONTXT, 'Lower', FORM, KIZ, NB, ONE,
     $                          WORK(KJZ*LMW+IPW), LMW, BETA,
     $                          C(JPBZ+1,JQBZ+KJZ+1), LDC,
     $                          LPBZ, LQBZ, LMW, LNW-KJZ )
   60         CONTINUE
            END IF
*
            JPBZ = JNPBZ
            JQBZ = JQBZ + LNW
   70     CONTINUE
        END IF
*
   80   CONTINUE
*
*       PART 3: Transpose A' (A is already distributed)
*       ===============================================
*
        IF( NOTRAN ) THEN
          IF( ADATA ) THEN
            CALL PBDTRAN( ICONTXT, 'Col', 'T', N, K, NB, A, LDA, ZERO,
     $                    WORK(IPT), K, ICROW, -1, -1, ICCOL,
     $                    WORK(IPW) )
          ELSE
            CALL PBDTRAN( ICONTXT, 'Col', 'T', N, K, NB, WORK, NP, ZERO,
     $                    WORK(IPT), K, ICROW, -1, -1, ICCOL,
     $                    WORK(IPW) )
          END IF
*
        ELSE
          IF( ADATA ) THEN
            CALL PBDTRAN( ICONTXT, 'Row', 'T', K, N, NB, A, LDA, ZERO,
     $                    WORK(IPT), NP, -1, ICCOL, ICROW, -1,
     $                    WORK(IPW) )
          ELSE
            CALL PBDTRAN( ICONTXT, 'Row', 'T', K, N, NB, WORK, K, ZERO,
     $                    WORK(IPT), NP, -1, ICCOL, ICROW, -1,
     $                    WORK(IPW) )
          END IF
        END IF
*
*       PART 4: Update C with B and A'
*       =====================================
*
        IF( NP.EQ.0 .OR. NQ.EQ.0 ) RETURN
*
*       If C is a symmetric upper triangular matrix,
*
        IF( UPPER ) THEN
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ = ISZCMP * LPBZ
          IQBZ = ISZCMP * LQBZ
          JPBZ = 0
          JQBZ = 0
*
          DO 110 JJ = 1, ICEIL(NQ, IQBZ)
            LMW = MIN( IPBZ, NP-JPBZ )
            LNW = MIN( IQBZ, NQ-JQBZ )
            JNPBZ = JPBZ + LMW
*
*           Modify (change) data in the lower triangular part
*
            IF( CSPACE ) THEN
*
*             Update C := alpha*B*A' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        B, LDB, WORK(JQBZ*K+IPT), LKK,
     $                        ONE, C(1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        WORK(IPB), NP, WORK(JQBZ*K+IPT),
     $                        LKK, ONE, C(1,JQBZ+1), LDC )
                END IF
*
*             Update C := alpha*A'*B + beta*C, if TRANS = 'T'/'C'
*
              ELSE
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, B(1,JQBZ+1), LDB,
     $                        ONE, C(1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', JNPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, WORK(JQBZ*K+IPB),
     $                        LKK, ONE, C(1,JQBZ+1), LDC )
                END IF
              END IF
*
*           Update data in the upper triangular matrix
*           and save data in the lower triangular matrix
*
            ELSE
*
*             Update C := alpha*B*A' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA,
     $                        B, LDB, WORK(JQBZ*K+IPT), LKK,
     $                        ONE, C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        B(JPBZ+1,1), LDB, WORK(JQBZ*K+IPT),
     $                        LKK, ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA,
     $                        WORK(IPB), NP, WORK(JQBZ*K+IPT),
     $                        LKK, ONE, C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(JQBZ*K+IPT),
     $                        LKK, ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
*
*             Update C := alpha*A'*B + beta*C, if TRANS = 'T'/'C'
*
              ELSE
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, B(1,JQBZ+1), LDB,
     $                        ONE, C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, B(1,JQBZ+1), LDB,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', JPBZ, LNW, K, ALPHA,
     $                        WORK(IPT), NP, WORK(JQBZ*K+IPB),
     $                        LKK, ONE, C(1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, WORK(JQBZ*K+IPB),
     $                        LKK, ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
              END IF
*
*             Compute diagonal blocks.
*
              MZROW = MRROW
              MZCOL = MRCOL
              KI = 0
*
              DO 100 KJ = 0, LCMQ-1
   90           CONTINUE
                IF( MZROW.LT.MZCOL ) THEN
                  MZROW = MZROW + NPROW
                  KI = KI + 1
                  GO TO 90
                END IF
                KIZ = KI * NB
                KJZ = KJ * NB
                IF( KJZ.GE.LNW ) GO TO 110
                FORM = 'G'
                IF( MZROW.EQ.MZCOL ) FORM = 'T'
                MZCOL = MZCOL + NPCOL
*
                CALL PBDTRADD( ICONTXT, 'Upper', FORM, KIZ, NB, ONE,
     $                         WORK(KJZ*LMW+IPW), LMW, ONE,
     $                         C(JPBZ+1,JQBZ+KJZ+1), LDC,
     $                         LPBZ, LQBZ, LMW, LNW-KJZ )
  100         CONTINUE
            END IF
*
            JPBZ = JNPBZ
            JQBZ = JQBZ + LNW
  110     CONTINUE
*
*       If C is a symmetric lower triangular matrix,
*
        ELSE
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ = ISZCMP * LPBZ
          IQBZ = ISZCMP * LQBZ
          JPBZ = 0
          JQBZ = 0
*
          DO 140 JJ = 1, ICEIL(NQ, IQBZ)
            LMW = MIN( IPBZ, MAX(NP-JPBZ, 0) )
            LNW = MIN( IQBZ, MAX(NQ-JQBZ, 0) )
            JNPBZ = JPBZ + LMW
*
*           Modify (change) data in the upper triangular part
*
            IF( CSPACE ) THEN
*
*             Update C := alpha*B*A' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        B(JPBZ+1,1), LDB, WORK(JQBZ*K+IPT),
     $                        LKK, ONE, C(JPBZ+1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(JQBZ*K+IPT),
     $                        LKK, ONE, C(JPBZ+1,JQBZ+1), LDC )
                END IF
*
*             Update C := alpha*A'*B+alpha*B'*A+beta*C, if TRANS='T'/'C'
*
              ELSE
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, B(1,JQBZ+1), LDB,
     $                        ONE, C(JPBZ+1,JQBZ+1), LDC )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JPBZ, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, WORK(JQBZ*K+IPB),
     $                        LKK, ONE, C(JPBZ+1,JQBZ+1), LDC )
                END IF
              END IF
*
*           Update data in the lower triangular matrix
*           and save data in the upper triangular matrix
*
            ELSE
*
*             Update C := alpha*B*A' + beta*C, if TRANS = 'N'
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        B(JNPBZ+1,1), LDB, WORK(JQBZ*K+IPT),
     $                        LKK, ONE, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        B(JPBZ+1,1), LDB, WORK(JQBZ*K+IPT),
     $                        LKK, ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        WORK(JNPBZ+IPB), NP, WORK(JQBZ*K+IPT),
     $                        LKK, ONE, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(JQBZ*K+IPT),
     $                        LKK, ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
*
*             Update C := alpha*A'*B + beta*C, if TRANS = 'T'/'C'
*
              ELSE
                IF( BDATA ) THEN
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        WORK(JNPBZ+IPT), NP, B(1,JQBZ+1), LDB,
     $                        ONE, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, B(1,JQBZ+1), LDB,
     $                        ZERO, WORK(IPW), MAX(1,LMW) )
                ELSE
                  CALL DGEMM( 'No', 'No', NP-JNPBZ, LNW, K, ALPHA,
     $                        WORK(JNPBZ+IPT), NP, WORK(JQBZ*K+IPB),
     $                        LKK, ONE, C(JNPBZ+1,JQBZ+1), LDC )
                  CALL DGEMM( 'No', 'No', LMW, LNW, K, ALPHA,
     $                        WORK(JPBZ+IPT), NP, WORK(JQBZ*K+IPB),
     $                        LKK, ZERO, WORK(IPW), MAX(1,LMW) )
                END IF
              END IF
*
*             Compute diagonal blocks.
*
              MZROW = MRROW
              MZCOL = MRCOL
              KI = 0
*
              DO 130 KJ = 0, LCMQ-1
  120           CONTINUE
                IF( MZROW.LT.MZCOL ) THEN
                  MZROW = MZROW + NPROW
                  KI = KI + 1
                  GO TO 120
                END IF
                KIZ = KI * NB
                KJZ = KJ * NB
                IF( KJZ.GE.LNW ) GO TO 140
                FORM = 'G'
                IF( MZROW.EQ.MZCOL ) FORM = 'T'
                MZCOL = MZCOL + NPCOL
*
                CALL PBDTRADD( ICONTXT, 'Lower', FORM, KIZ, NB, ONE,
     $                         WORK(KJZ*LMW+IPW), LMW, ONE,
     $                         C(JPBZ+1,JQBZ+KJZ+1), LDC,
     $                         LPBZ, LQBZ, LMW, LNW-KJZ )
  130         CONTINUE
            END IF
*
            JPBZ = JNPBZ
            JQBZ = JQBZ + LNW
  140     CONTINUE
        END IF
*
* === If C is just a block ===
*
      ELSE
        IF( NOTRAN .AND. MYROW.EQ.ICROW ) THEN
*
*         Form  C := alpha*A*(B') + alpha*B*(A') +beta * C.
*                                  _                        _
*                                 | |                      | |
*                                 | |                      | |
*         _       _____________   | |      _____________   | |      _
*        |_| = a*|______A______|*(B') + a*|______B______|*(A') + b*|_|
*         C                       | |                      | |      C
*                                 | |                      | |
*                                 |_|                      |_|
*
          NQ = NUMROC( K, NB, MYCOL, IAPOS, NPCOL )
          CSPACE = LSAME( CWORK, 'Y' )
*
          IF( LDA.LT.MAX(1,N)                          ) THEN
            INFO = 10
          ELSE IF( LDB.LT.MAX(1,N)                     ) THEN
            INFO = 12
          ELSE IF( LDC.LT.MAX(1,N) .AND. ( CSPACE .OR.
     $             ICCOL.EQ.MYCOL .OR. ICCOL.EQ.-1 )   ) THEN
            INFO = 15
          ELSE IF( IAPOS.LT.0  .OR. IAPOS.GE.NPCOL     ) THEN
            INFO = 16
          ELSE IF( IBPOS.NE.IAPOS                      ) THEN
            INFO = 17
          ELSE IF( ICROW.LT.0  .OR. ICROW.GE.NPROW     ) THEN
            INFO = 18
          ELSE IF( ICCOL.LT.-1 .OR. ICCOL.GE.NPCOL     ) THEN
            INFO = 19
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Compute C
*
          IF( MYCOL.EQ.ICCOL ) THEN
            CALL DSYR2K( UPLO, TRANS, N, NQ, ALPHA, A, LDA, B, LDB,
     $                   BETA, C, LDC )
            CALL PBDT1CPY( UPLO, JJ, N, C, LDC, WORK )
            CALL DGSUM2D( ICONTXT, 'Row', '1-tree', 1, JJ, WORK, 1,
     $                    ICROW, ICCOL )
            CALL PBDT2CPY( UPLO, JJ, N, C, LDC, WORK )
*
          ELSE
            IF( CSPACE ) THEN
              CALL DSYR2K( UPLO, TRANS, N, NQ, ALPHA, A, LDA, B, LDB,
     $                     ZERO, C, LDC )
              CALL PBDT1CPY( UPLO, JJ, N, C, LDC, WORK )
              CALL DGSUM2D( ICONTXT, 'Row', '1-tree', 1, JJ, WORK, 1,
     $                      ICROW, ICCOL )
            ELSE
              CALL DSYR2K( UPLO, TRANS, N, NQ, ALPHA, A, LDA, B, LDB,
     $                     ZERO, WORK, N )
              CALL PBDT3CPY( UPLO, JJ, N, WORK )
              CALL DGSUM2D( ICONTXT, 'Row', '1-tree', 1, JJ, WORK, 1,
     $                      ICROW, ICCOL )
            END IF
          END IF
*
        ELSE IF( .NOT.NOTRAN .AND. MYCOL.EQ.ICCOL ) THEN
*
*         Form  B := alpha*B / op( A ).
*                                   _                         _
*                                  | |                       | |
*                                  | |                       | |
*         _       _____________    | |      _____________    | |      _
*        |_| = a*|_____(B')____| * |A| + a*|_____(A')____| * |B| + b*|_|
*         C                        | |                       | |      C
*                                  | |                       | |
*                                  |_|                       |_|
*
          NP = NUMROC( K, NB, MYROW, IAPOS, NPROW )
          CSPACE = LSAME( CWORK, 'Y' )
*
          IF( LDA.LT.MAX(1,NP)                         ) THEN
            INFO = 10
          ELSE IF( LDB.LT.MAX(1,NP)                    ) THEN
            INFO = 12
          ELSE IF( LDC.LT.MAX(1,N) .AND. ( CSPACE .OR.
     $             ICROW.EQ.MYROW .OR. ICROW.EQ.-1 )   ) THEN
            INFO = 15
          ELSE IF( IAPOS.LT.0  .OR. IAPOS.GE.NPROW     ) THEN
            INFO = 16
          ELSE IF( IBPOS.NE.IAPOS                      ) THEN
            INFO = 17
          ELSE IF( ICROW.LT.-1 .OR. ICROW.GE.NPROW     ) THEN
            INFO = 18
          ELSE IF( ICCOL.LT.0  .OR. ICCOL.GE.NPCOL     ) THEN
            INFO = 19
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Compute C
*
          IF( MYROW.EQ.ICROW ) THEN
            CALL DSYR2K( UPLO, TRANS, N, NP, ALPHA, A, LDA, B, LDB,
     $                   BETA, C, LDC )
            CALL PBDT1CPY( UPLO, JJ, N, C, LDC, WORK )
            CALL DGSUM2D( ICONTXT, 'Col', '1-tree', 1, JJ, WORK, 1,
     $                    ICROW, ICCOL )
            CALL PBDT2CPY( UPLO, JJ, N, C, LDC, WORK )
*
          ELSE
            IF( CSPACE ) THEN
              CALL DSYR2K( UPLO, TRANS, N, NP, ALPHA, A, LDA, B, LDB,
     $                     ZERO, C, LDC )
              CALL PBDT1CPY( UPLO, JJ, N, C, LDC, WORK )
              CALL DGSUM2D( ICONTXT, 'Col', '1-tree', 1, JJ, WORK, 1,
     $                      ICROW, ICCOL )
            ELSE
              CALL DSYR2K( UPLO, TRANS, N, NP, ALPHA, A, LDA, B, LDB,
     $                     ZERO, WORK, N )
              CALL PBDT3CPY( UPLO, JJ, N, WORK )
              CALL DGSUM2D( ICONTXT, 'Col', '1-tree', 1, JJ, WORK, 1,
     $                      ICROW, ICCOL )
            END IF
          END IF
        END IF
      END IF
*
      RETURN
*
*     End of PBDSYR2K
*
      END
