/* GI parser -*- objc -*-
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: parse.y,v 1.37 1998/07/21 14:54:52 tiggr Exp $  */

%{
#import "gi.h"
#import <time.h>

/* Forward declarations.  */
extern int gi_lex ();
void start_line (id s, int no_space);

extern char *yytext;

#define YYDEBUG 1
#define YYERROR_VERBOSE

/* Iff !0, we're currently not interested in comments.  */
int no_comments;

/* Iff !0, we're in the global scope, the only scope in which we output
   something.  */
static int global_scope = 1;

/* Iff !0, we're to eat newlines.  */
static int eat_newlines;

/* Iff !0, we're to suppress the next space.  */
static int eat_space;

/* Iff !0, we're to start a new line.  */
static int start_new_line;

/* The current column.  */
static int column;

/* The current indentation.  */
static int current_indent;

/* Iff !0, we're reading the body of a method, and the lexer is supposed
   to skip everything but braces.  */
int num_body_brace;

int current_line;
id <TLString> current_filename;

/* The current unit.  */
LTTUnit *current_unit;

/* The current output file.  It is NIL when parsing a file which is not to
   be output.  */
id <TLOutputStream> of;

void
yyerror (char *s)
{
  error (@"parse error at `%s' (%s)", yytext, s);
} /* yyerror */

void
emit1 (id <TLString> s, int no_space)
{
  if (eat_space)
    {
      no_space = 1;
      eat_space = 0;
    }

  if (start_new_line)
    start_line (@"", 0);

  if (column > current_indent && !no_space)
    {
      [of writeByte: ' '];
      column++;
    }

  if (of)
    [(id) s print: of quoted: NO];

  column += [s length];
} /* emit1 */

void
emit (id <TLString> s)
{
  emit1 (s, 0);
} /* emit */

void
emit_posing (id <TLString> s)
{
  emit (formac (nil, @"posing %@", s));
}

void
emit_no_space (id <TLString> s)
{
  emit1 (s, 1);
} /* emit_no_space */

void
start_line (id s, int no_space)
{
  if (column != current_indent || !eat_newlines)
    {
      if (column)
	[of writeByte: '\n'];
      if (current_indent && s && !no_space)
	formac (of, @"%s", indent (current_indent));
    }

  column = current_indent;
  start_new_line = 0;

  if (s)
    emit1 (s, no_space);
} /* start_line */

void
skip_line (void)
{
  if (column > current_indent)
    [of writeByte: '\n'];
  [of writeByte: '\n'];
  column = 0;
  start_line (@"", 0);
}

void
emit_another_list (id t, id sep, int first, void (*emitter) (id))
{
  id s;

  while (t)
    {
      DECONS (t, s, t);

      if (first)
	{
	  eat_space = 1;
	  first = 0;
	}
      else
	emit_no_space (@",");

      emitter (s);
    }
}

void
emit_list (id t, id sep, void (*emitter) (id))
{
  emit_another_list (t, sep, 1, emitter);
}

void
emit_possible_tuple (id t)
{
  if ([t consp])
    {
      emit (@"(");

      emit_list (t, @",", emit_possible_tuple);

      emit_no_space (@")");
    }
  else
    emit (t);
} /* emit_possible_tuple */

%}

/* Dangling else.  */
%expect 1

%pure_parser

%union {
  int i;
  id v;
}

/* Keywords.  */
%token CLASS END EXTENSION EXTERN IMPLEMENTATION INSTANCE INTERFACE POSING
%token RECEIVER CONST
%token IF ELSE DO FOR WHILE BREAK CONTINUE PRE POST OLD

/* Qualifiers.  */
%token LOCAL PUBLIC PRIVATE PROTECTED MUTABLE STATIC DEFERRED REDEFINE REDECLARE

/* Other stuff.  */
%token DYNAMIC
%token <v> DOC COMMENT
%token DOTDOT EQ GE LE NE AND OR IMPLIES SHL SHR
%token TYPEDEF VOID NIL
%token PLUSPLUS MINMIN
%token MULIS DIVIS MODIS PLUSIS MINIS SHLIS SHRIS
%token BANDIS BORIS BEORIS SCANDIS SCORIS
%token <v> STRING_CST IDENTIFIER NUMBER TYPE BASIC_TYPE SELECTOR

%type <v> atom array_reference class class_name
%type <v> .colon_super_list. entity_type expr expr_skip_comments extension
%type <v> .extension. extension_name identifier_list instance
%type <v> method_invocation method_invocation_part method_invocation_parts
%type <v> method_name_part non_ident_method_name_part selector_decl_part
%type <v> object_type rest_of_method_invocation selector_decl_parts
%type <v> super_indication super_list tuple tuple_field tuple_field_list
%type <v> tuple_type tuple_type_field_list type_cast type_cast_type
%type <v> tuple_type_string

%type <i> .foreign.

%left ','
%right '='
%right '?' ':'
%left IMPLIES
%left OR
%left AND
%left '<' LE EQ NE GE '>'
%left '^'
%left '|'
%left '&'
%left SHL SHR
%left '+' '-'
%left '*' '/' '%'
%left '~' '!' UNARY_MINUS OLD
%right '['

%start file

%%
file:
	  /* empty */
	| file file_element semicolon
	    {
	      start_line (AT"", 1);
	    }
	| file c
	    /* Since `C' can be empty, this loops at parse errors at the
               end of the input file.  */
	;

c:
	  /* empty */
	| c DOC
	    {
	      if (global_scope && flag_documentation)
		{
		  skip_line ();
		  start_line (formac (nil, AT"%@", $2), 1);
		  start_new_line = eat_newlines = 1;
		}
	    }
	| c COMMENT
	    {
	      if (global_scope && flag_comments)
		{
		  skip_line ();
		  start_line (formac (nil, AT"%@", $2), 1);
		  start_new_line = eat_newlines = 1;
		}
	    }
	| error
	;

file_element:
	  class_interface
	| instance_interface
	| class_implementation
	| instance_implementation
	;

class_interface:
	  interface .top_foreign. .qualifiers. class .extension.
	  .colon_super_list. .object_variables. decl_list end
	;

class_implementation:
	  implementation .top_foreign. .qualifiers. class .extension.
	  .colon_super_list.
	  .object_variables.
	    {
	      if (uf)
		{
		  if ($5)
		    formac (uf, AT"\n  extension %@ (%@)", $4, $5);
		  else
		    formac (uf, AT"\n  class %@", $4);

		  if ($6)
		    {
		      int first = YES;
		      id s;

		      while ($6)
			{
			  DECONS ($6, s, $6);
			  formac (uf, (first ? AT" posing %@"
				       : AT", %@"), s);
			}
		    }

		  formac (uf, AT";");
		}
	    }
	  def_list end
	;

instance_interface:
	  interface .top_foreign. .qualifiers. instance .extension.
	  .colon_super_list.
	  .object_variables.
	  decl_list end
	;

instance_implementation:
	  implementation .top_foreign. .qualifiers. instance .extension.
	  .colon_super_list.
	  .object_variables.
	  def_list end
	;

end:
	  c END
	    {
	      skip_line ();
	      start_line (AT"end", 1);
	    }
	;

class:
	  CLASS class_name
	    {
	      if (global_scope)
		emit (formac (nil, AT"class %@", $2));
	      $$ = $2;
	    }
	;

instance:
	  INSTANCE class_name
	    {
	      if (global_scope)
		emit (formac (nil, AT"instance %@", $2));
	      $$ = $2;
	    }
	;

interface:
	  INTERFACE
	    {
	      start_line (AT"interface", 1);
	    }
	;

implementation:
	  IMPLEMENTATION
	    {
	      start_line (AT"interface", 1);
	    }
	;

.object_variables.:
	  /* empty */
	| brace_open object_variables brace_close
	;

brace_open:
	  '{'
	    {
	      if (global_scope)
		{
		  start_line (AT"{", 1);
		  start_new_line = 1;
		  current_indent += 2;
		}
	    }
	;

brace_close:
	  '}'
	    {
	      if (global_scope)
		{
		  current_indent -= 2;
		  start_new_line = 1;
		  start_line (AT"}", 1);
		}
	    }
	;

object_variables:
	  /* empty */
	| object_variables object_variable semicolon
	;

object_variable:
	  c .qualifiers. unq_icvar
	    {
	    }
	| c .qualifiers. CONST IDENTIFIER '=' expr_skip_comments
	    {
	      if (global_scope)
		emit (formac (nil, AT"const %@ = %@", $4, $6));
	    }
	;

unq_icvar:
	  entity_type identifier_list
	    {
	      if (global_scope)
		{
		  id <TLEnumerator> e = [$2 enumerator];
		  int first = 1;
		  id obj;

		  emit ($1);

		  while ((obj = [e nextObject]))
		    if (first)
		      {
			emit (obj);
			first = 0;
		      }
		    else
		      emit_no_space (formac (nil, AT", %@", obj));
		}
	    }
	;

void_or_entity_type:
	  VOID
	    {
	      if (global_scope)
		emit (AT"void");
	    }
	| DYNAMIC
	    {
	      if (global_scope)
		emit (AT"dynamic");
	    }
	| entity_type
	    {
	      if (global_scope)
		emit_possible_tuple ($1);
	    }
	;

/* Types of entities.  */
entity_type:
	  BASIC_TYPE
	| tuple_type
	| object_type
	;

/* The entity with this type is a reference to an object which is an
   instance of the class named CLASS_NAME, unless modified by CLASS or
   INSTANCE, of course.  */
object_type:
	  class_name
	| RECEIVER
	    {
	      $$ = AT"id";
	    }
	| CLASS '(' object_type ')'
	    {
	      $$ = formac (nil, AT"class (%@)", $3);
	    }
	| INSTANCE '(' object_type ')'
	    {
	      $$ = formac (nil, AT"instance (%@)", $3);
	    }
	;

tuple_type:
	  '(' tuple_type_field_list ')'
	    {
	      $$ = $2;
	    }
	;

tuple_type_field_list:
	  entity_type
	    {
	      $$ = CONS ($1, nil);
	    }
	| tuple_type_field_list ',' entity_type
	    {
	      $$ = [$1 nconc: CONS ($3, nil)];
	    }
	;

.colon_super_list.:
	  /* empty */ { $$ = nil; }
	| ':'
	    {
	      emit_no_space (AT": ");
	    }
	  super_list
	    {
	      if ([$3 cdr])
		emit_list ([$3 cdr], AT",", emit_posing);
	      if ([$3 car])
		emit_another_list ([$3 car], AT",", ![$3 cdr], emit);
	      $$ = [$3 cdr];
	    }
	;

/* Return a CONS, of which the car lists the non-posing supers, and the
   CDR lists the posing supers.  */
super_list:
	  super_indication
	    {
	      $$ = CONS (CONS ($1, nil), nil);
	    }
	| POSING super_indication
	    {
	      $$ = CONS (nil, CONS ($2, nil));
	    }
	| super_list ',' super_indication
	    {
	      if ([$1 car])
		[[$1 car] nconc: CONS ($3, nil)];
	      else
		[$1 setCar: CONS ($3, nil)];

	      $$ = $1;
	    }
	| super_list ',' POSING super_indication
	    {
	      if ([$1 cdr])
		[[$1 cdr] nconc: CONS ($4, nil)];
	      else
		[$1 setCdr: CONS ($4, nil)];

	      $$ = $1;
	    }
	;

super_indication:
	  class_name
	| INSTANCE '(' class_name ')'
	    {
	      $$ = formac (nil, AT"instance (%@)", $3);
	    }
	| CLASS '(' class_name ')'
	    {
	      $$ = formac (nil, AT"class (%@)", $3);
	    }
	;

.extension.:
	  /* empty */ { $$ = NULL; }
	| extension
	;

extension:
	  EXTENSION extension_name
	    {
	      emit (formac (nil, AT"extension %@", $2));
	      $$ = $2;
	    }
	;

decl_list:
	  /* empty */
	| decl_list annotated_method_decl
	| decl_list typedef
	;

annotated_method_decl:
	  shared_method_part
	;

method_def:
	  shared_method_part method_body
	;

shared_method_part:
	  c
 	    {
	      start_new_line = 1;
	    }
	  .qualifiers. .foreign. method_decl .pre. .post.
	;

.top_foreign.:
	  .foreign.
	    {
	      if ($1)
		emit (AT"extern");
	    }
	;

.foreign.:
	  /* empty */ { $$ = 0; }
	| EXTERN { $$ = 1; }
	;

.qualifiers.:
	  /* empty */
	| .qualifiers. STATIC
	    {
	      if (global_scope)
		emit (AT"static");
	    }
	| .qualifiers. LOCAL
	    {
	      if (global_scope)
		emit (AT"local");
	    }
	| .qualifiers. DEFERRED
	    {
	      if (global_scope)
		emit (AT"deferred");
	    }
	| .qualifiers. REDEFINE
	    {
	      if (global_scope)
		emit (AT"redefine");
	    }
	| .qualifiers. REDECLARE
	    {
	      if (global_scope)
		emit (AT"redeclare");
	    }
	| .qualifiers. PUBLIC
	    {
	      if (global_scope)
		emit (AT"public");
	    }
	| .qualifiers. PROTECTED
	    {
	      if (global_scope)
		emit (AT"protected");
	    }
	| .qualifiers. PRIVATE
	    {
	      if (global_scope)
		emit (AT"private");
	    }
	| .qualifiers. MUTABLE
	    {
	      if (global_scope)
		emit (AT"mutable");
	    }
	| .qualifiers. POSING
	    {
	      if (global_scope)
		emit (AT"posing");
	    }
	;

method_decl:
	  void_or_entity_type method_decl_parts
	;

.pre.:
	    /* empty */
	| PRE c expr_skip_comments
	    {
	      if (global_scope)
		start_line (formac (nil, AT" pre %@", $3), 0);
	    }
	;

.post.:
	  /* empty */
	| POST c expr_skip_comments
	    {
	      if (global_scope)
		start_line (formac (nil, AT" post %@", $3), 0);
	    }
	;

method_decl_parts:
	  method_decl_part
	| method_decl_part method_decl_parts
	;

method_decl_part:
	  '('
	    {
	      if (global_scope)
		{
		  emit (AT"(");
		  eat_space = 1;
		}
	    }
	| ')'
	    {
	      if (global_scope)
		emit_no_space (AT")");
	    }
	| ','
	    {
	      if (global_scope)
		emit_no_space (AT",");
	    }
	| ':'
	    {
	      if (global_scope)
		emit_no_space (AT":");
	    }
	| '=' expr_skip_comments
	    {
	      if (global_scope)
		emit (formac (nil, AT"= %@", $2));
	    }
	| class_name
	    {
	      if (global_scope)
		emit ($1);
	    }
	| BASIC_TYPE
	    {
	      if (global_scope)
		emit ($1);
	    }
	| SELECTOR
	    {
	      if (global_scope)
		emit ($1);
	    }
	| non_ident_method_name_part
	    {
	      if (global_scope)
		emit (formac (nil, AT"%@", $1));
	    }
	;

non_ident_method_name_part:
	  CLASS { $$ = AT"class"; }
	| DYNAMIC { $$ = AT"dynamic"; }
	| EXTENSION { $$ = AT"extension"; }
	| INSTANCE { $$ = AT"instance"; }
	| END { $$ = AT"end"; }
	| EXTERN { $$ = AT"extern"; }
	| IMPLEMENTATION { $$ = AT"implementation"; }
	| INTERFACE { $$ = AT"interface"; }
	| POSING { $$ = AT"posing"; }
	| RECEIVER { $$ = AT"id"; }
	| CONST { $$ = AT"const"; }
	| IF { $$ = AT"if"; }
	| ELSE { $$ = AT"else"; }
	| DO { $$ = AT"do"; }
	| FOR { $$ = AT"for"; }
	| WHILE { $$ = AT"while"; }
	| BREAK { $$ = AT"break"; }
	| CONTINUE { $$ = AT"continue"; }
	| PUBLIC { $$ = AT"public"; }
	| PRIVATE { $$ = AT"private"; }
	| PROTECTED { $$ = AT"protected"; }
	| MUTABLE { $$ = AT"mutable"; }
	| STATIC { $$ = AT"static"; }
	| DEFERRED { $$ = AT"deferred"; }
	| REDEFINE { $$ = AT"redefine"; }
	| REDECLARE { $$ = AT"redeclare"; }
	| DYNAMIC { $$ = AT"dynamic"; }
	| TYPEDEF { $$ = AT"typedef"; }
	| VOID { $$ = AT"void"; }
	| NIL { $$ = AT"nil"; }
	;

method_name_part:
	  IDENTIFIER
	| non_ident_method_name_part
	;

def_list:
	  /* empty */
	| def_list method_def
	| typedef
	;

method_body:
	  semicolon
	|
	    {
	      emit_no_space (AT";");
	      start_new_line = 1;
	      global_scope = 0;
	    }
	  method_body_expr
	    {
	      global_scope = 1;
	    }
	;

method_body_expr:
	   compound
	;

typedef:
	  TYPEDEF entity_type IDENTIFIER
	;

compound:
	  '{'
	    {
	      num_body_brace++;
	    }
	  compound_contents '}'
	    {
	      num_body_brace--;
	    }
	;

compound_contents:
	  /* empty */
	| compound compound_contents
	;

array_reference:
	  expr '[' expr ']'
	    {
	      $$ = formac (nil, AT"%@[%@]", $1, $3);
	    }
	;

expr_skip_comments:
	  {
	    no_comments++;
	  }
	  expr
	  {
	    $$ = $2;
	    no_comments--;
	  }
	;

expr:
	  atom
	| array_reference
	| '-' expr %prec UNARY_MINUS
	    {
	      $$ = formac (nil, AT"-%@", $2);
	    }
	| OLD expr
	    {
	      $$ = formac (nil, AT"old %@", $2);
	    }
	| '~' expr
	    {
	      $$ = formac (nil, AT"~%@", $2);
	    }
	| '!' expr
	    {
	      $$ = formac (nil, AT"!%@", $2);
	    }
	| expr '*' expr
	    {
	      $$ = formac (nil, AT"%@ * %@", $1, $3);
	    }
	| expr '/' expr
	    {
	      $$ = formac (nil, AT"%@ / %@", $1, $3);
	    }
	| expr '%' expr
	    {
	      $$ = formac (nil, AT"%@ % %@", $1, $3);
	    }
	| expr '+' expr
	    {
	      $$ = formac (nil, AT"%@ + %@", $1, $3);
	    }
	| expr '-' expr
	    {
	      $$ = formac (nil, AT"%@ - %@", $1, $3);
	    }
	| expr SHL expr
	    {
	      $$ = formac (nil, AT"%@ << %@", $1, $3);
	    }
	| expr SHR expr
	    {
	      $$ = formac (nil, AT"%@ >> %@", $1, $3);
	    }
	| expr '&' expr
	    {
	      $$ = formac (nil, AT"%@ & %@", $1, $3);
	    }
	| expr '|' expr
	    {
	      $$ = formac (nil, AT"%@ | %@", $1, $3);
	    }
	| expr '^' expr
	    {
	      $$ = formac (nil, AT"%@ ^ %@", $1, $3);
	    }
	| expr '<' expr
	    {
	      $$ = formac (nil, AT"%@ < %@", $1, $3);
	    }
	| expr LE expr
	    {
	      $$ = formac (nil, AT"%@ <= %@", $1, $3);
	    }
	| expr EQ expr
	    {
	      $$ = formac (nil, AT"%@ == %@", $1, $3);
	    }
	| expr NE expr
	    {
	      $$ = formac (nil, AT"%@ != %@", $1, $3);
	    }
	| expr GE expr
	    {
	      $$ = formac (nil, AT"%@ >= %@", $1, $3);
	    }
	| expr '>' expr
	    {
	      $$ = formac (nil, AT"%@ > %@", $1, $3);
	    }
	| expr AND expr
	    {
	      $$ = formac (nil, AT"%@ && %@", $1, $3);
	    }
	| expr OR expr
	    {
	      $$ = formac (nil, AT"%@ || %@", $1, $3);
	    }
	| expr IMPLIES expr
	    {
	      $$ = formac (nil, AT"%@ -> %@", $1, $3);
	    }
	;
atom:
	  NUMBER
	| STRING_CST
	| IDENTIFIER
	| VOID
	    {
	      $$ = AT"void";
	    }
	| NIL
	    {
	      $$ = AT"nil";
	    }
	| tuple
	| method_invocation
	| type_cast
	;

type_cast:
	  type_cast_type '(' expr ')'
	    {
	      $$ = formac (nil, AT"%@ (%@)", $1, $3);
	    }
	| SELECTOR '(' selector_decl_parts ')'
	    {
	      $$ = formac (nil, AT"%@ (%@)", $1, $3);
	    }
	;

selector_decl_parts:
	  /* empty */ { $$ = formac (nil, AT""); }
	| selector_decl_parts selector_decl_part
	    {
	      $$ = formac ($1, AT"%@ ", $2);
	    }
	| selector_decl_parts selector_decl_part ':'
	    {
	      $$ = formac ($1, AT"%@: ", $2);
	    }
	;

selector_decl_part:
	  class_name
	| BASIC_TYPE
	| IDENTIFIER
	| tuple_type_string
	;

tuple_type_string:
	  tuple_type
		{
		  int first = YES;
		  $$ = formac (nil, AT"(");
		  while ($1)
		    {
		      id type;
		      DECONS ($1, type, $1);
		      if (first)
			first = NO;
		      else
			formac ($$, AT", ");
		      formac ($$, AT"%@", type);
		    }
		  formac ($$, AT")");
		}
	;

type_cast_type:
	  object_type
	| BASIC_TYPE
	;

method_invocation:
	  '[' expr rest_of_method_invocation ']'
	    {
	      $$ = formac (nil, AT"[%@ %@]", $2, $3);
	    }
	| '[' object_type rest_of_method_invocation ']'
	    {
	      $$ = formac (nil, AT"[%@ %@]", $2, $3);
	    }
	| IDENTIFIER tuple
	    {
	      $$ = formac (nil, AT"%@ %@", $1, $2);
	    }
	;

rest_of_method_invocation:
	  IDENTIFIER
	| method_invocation_parts
	;

method_invocation_parts:
	  method_invocation_part
	| method_invocation_parts method_invocation_part
	;

method_invocation_part:
	  method_name_part expr
	    {
	      $$ = formac (nil, AT"%@ %@", $1, $2);
	    }
	;

tuple:
	  '(' tuple_field_list ')'
	    {
	      if (!$2)
		$$ = formac (nil, AT"(");
	      else
		{
		  $$ = nil;
		  while ($2)
		    {
		      id s;

		      DECONS ($2, s, $2);
		      $$ = formac ($$, $$ ? AT", %@" : AT"(%@", s);
		    }
		}
	      formac ($$, AT")");
	    }
	;

tuple_field_list:
	  tuple_field
	    {
	      $$ = CONS ($1, nil);
	    }
	| tuple_field_list ',' tuple_field
	    {
	      $$ = [$1 nconc: CONS ($3, nil)];
	    }
	;

tuple_field:
	  /* empty */ { $$ = AT""; }
	| expr
	;

identifier_list:
	  IDENTIFIER
	    {
	      $$ = CONS ($1, nil);
	    }
	| identifier_list ',' IDENTIFIER
	    {
	      $$ = [$1 nconc: CONS ($3, nil)];
	    }
	;

semicolon:
	  ';'
	    {
	      if (global_scope)
		{
		  emit_no_space (AT";");
		  start_new_line = 1;
		}
	    }
	| error
	    {
	      error (AT"`;' expected");
	    }
	;

class_name:
	  TYPE
	| SELECTOR
	| IDENTIFIER '.' TYPE
	    {
	      $$ = formac (nil, AT"%@.%@", $1, $3);
	    }
	| TYPE '.' TYPE
	    {
	      $$ = formac (nil, AT"%@.%@", $1, $3);
	    }
	| IDENTIFIER
	| IDENTIFIER '.' IDENTIFIER
	    {
	      $$ = formac (nil, AT"%@.%@", $1, $3);
	    }
	;

extension_name:
	  IDENTIFIER
	| TYPE
	;

%%
