/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: LTTUnit.m,v 1.30 1998/01/22 17:27:13 tiggr Exp $  */

#import "ltt.h"

/* All units currently known.  */
TLDictionary *ltt_units;

/* All instances known in all known units.  The values stored are
   TLVectors of LTTInstances.  */
static TLDictionary *ltt_instances;

/* Used when searching in the hierarchy.  */
static int search_mark;

@implementation LTTUnit

+(LTTUnit *) findUnitNamed: (TLString *) n
{
  LTTUnit *u = [self unitNamed: n];

  if (!u)
    {
      [self searchUnit: n alongPath: load_path];
      u = [self unitNamed: n];
    }

  return u;
}

+initialize
{
  if (!ltt_units)
    {
      ltt_units = [TLDictionary dictionary];
      [ltt_units gcLock];
      ltt_instances = [TLDictionary dictionary];
      [ltt_instances gcLock];
    }

  return self;
}

+(TLVector *) instancesNamed: (id <TLString>) n
{
  return [ltt_instances objectForKey: n];
}

+(LTTUnit *) searchUnit: (TLString *) n alongPath: (TLVector *) path
{
  TLString *filename;
  LTTUnit *u;

  if (load_unit (n, path, nil, (id) self, &filename))
    return nil;

  u = [ltt_units objectForKey: n];
  if (!u)
    error (@"`%@' fails to define the unit `%@'", filename, n);

  return u;
}

+(id <LTTUnitAcceptor>) startUnitNamed: (id <TLString>) n
{
  TLString *s = [current_filename string];

  /* The directory must be deduced from the current filename.  */
  id <TLRange> r = [s rangeOfString: @"/" options: TLSEARCH_BACKWARD];
  TLString *d = (r ? [s stringWithRange:
			  [TLRange rangeWithStart: 0 end: [r _start]]] : nil);

  return [self unitWithName: n directory: d];
}

+(LTTUnit *) unitNamed: (TLString *) n
{
  return [ltt_units objectForKey: n];
}

+(LTTUnit *) unitWithName: (TLString *) n directory: (TLString *) d
{
  LTTUnit *u = [[self gcAlloc] initWithName: n directory: d];

  [ltt_units setObject: u forKey: n];

  return u;
}

-(void) addExtension: (LTTExtension *) e
{
  [extensions addElement: e];
}

-(void) addInstance: (LTTInstance *) m
{
  TLString *s = [[m lttName] internal];
  TLVector *v;

  /* Add the instance to ourself.  */
  [instances setObject: m forKey: s];

  /* Add the instance to the dict of all instances.  */
  v = [ltt_instances objectForKey: s];
  if (!v)
    {
      v = [TLVector vector];
      [ltt_instances setObject: v forKey: s];
    }
  [v addElement: m];
}

-(void) addSuperReferenceFrom: (LTTMeta *) from
			   to: (LTTMeta *) to
{
  TLSet *s = nil;

  if (!super_refs)
    super_refs = [TLDictionary new];
  else
    s = [super_refs objectForKey: from];

  if (!s)
    {
      s = [TLSet new];
      [super_refs setObject: s forKey: from];
    }

  [s addElement: to];
}

-(BOOL) dependsOn: (LTTUnit *) unit
	     mark: (int) k
{
  if (mark == k)
    return NO;
  mark = k;

  if (self == unit)
    return YES;

  {
    int i, n = [units_used length];

    for (i = 0; i < n; i++)
      {
	LTTUnit *u = [units_used _elementAtIndex: i];

	if (u == unit || [u dependsOn: unit mark: k])
	  return YES;
      }

    return NO;
  }
}

-(BOOL) dependsOn: (LTTUnit *) u
{
  return [self dependsOn: u mark: ++search_mark];
}

-(TLString *) directory
{
  return dir;
}

-(id <TLEnumerator>) extensions
{
  return [extensions enumerator];
}

-(id <TLEnumerator>) files
{
  return [files valueEnumerator];
}

-(void) gcReference
{
  MARK (dir);
  MARK (files);
  MARK (instances);
  MARK (extensions);
  MARK (units_used);
  MARK (super_refs);

  [super gcReference];
}

-(id) initWithName: (TLString *) n
	 directory: (TLString *) d
{
  if (![super initWithName: [CO_LTTName nameWithInternal: n]])
    return nil;

  dir = d;
  files = [TLDictionary dictionary];
  instances = [TLDictionary dictionary];
  extensions = [TLVector vector];
  units_used = [TLVector vector];

  return self;
}

-(LTTInstance *) findInstanceNamed: (id <TLString>) s
			      mark: (int) k
{
  LTTInstance *ins, *pins;
  int i, n;

  if (mark == k)
    return nil;
  mark = k;

  ins = [self instanceNamed: s];
  if (ins)
    return ins;

  for (pins = 0, i = 0, n = [units_used length]; i < n; i++)
    {
      LTTUnit *u = [units_used _elementAtIndex: i];

      ins = [u findInstanceNamed: s mark: k];
      if (ins)
	if (pins)
	  {
	    error (@"ambiguous reference to %@", s);
	    return nil;
	  }
	else
	  pins = ins;
    }

  return pins;
}

-(LTTInstance *) findInstanceNamed: (id <TLString>) s
{
  return [self findInstanceNamed: s mark: ++search_mark];
}

-(LTTInstance *) instanceNamed: (TLString *) s
{
  return [instances objectForKey: s];
}

-(id <TLEnumerator>) instances
{
  return [instances valueEnumerator];
}

-(LTTInstance *) metaWithName: (TLString *) s
{
  LTTInstance *m = [self instanceNamed: s];

  if (m)
    error (@"attempt to add duplicate class `%@' to unit `%@'",
	   s, [name internal]);
  else
    {
      m = [CO_LTTMeta instanceAndClassWithName: s unit: self];
      [self addInstance: m];
    }

  return m;
}

-(LTTFile *) fileNamed: (TLString *) s
{
  return [files objectForKey: ltt_filename_without_extension ([s basename])];
}

-(TLCons *) orderedUnits: (TLCons *) l
		    mark: (int) k
{
  int i, n;

  if (mark == k)
    return l;
  mark = k;

  if (units_used)
    for (i = 0, n = [units_used length]; i < n; i++)
      l = [[units_used _elementAtIndex: i] orderedUnits: l mark: k];

  return l ? [l nconc: CONS (self, nil)] : CONS (self, nil);
}

-(TLCons *) orderedUnits: (TLCons *) l
{
  return [self orderedUnits: l mark: ++search_mark];
}

-(id <TLString>) outputDefinitionName
{
  return (id) formac (nil, @"%@%@", TO_UNIT_DEF_PREFIX, [name external]);
}

-(TLCons *) posers
{
  return posers;
}

-(void) resolvePosing
{
  while (posing)
    {
      TLString *meta_name, *unit_name = nil;
      LTTExtension *x;
      TLCons *entry;
      LTTMeta *m;

      DECONS (posing, entry, posing);
      DECONS (entry, x, entry);

      while (entry)
	{
	  DECONS (entry, meta_name, entry);

	  if ([meta_name consp])
	    DECONS ((id) meta_name, unit_name, meta_name);

	  m = ltt_instance_in_unit (unit_name, self,
				    [isa instancesNamed: meta_name]);
	  if (m)
	    {
	      if ([[x meta] poser])
		warning (@"%@.%@ posing %@.%@ actually poses %@.%@",
			 [[m unit] lttName], [m lttName],
			 [[[x meta] unit] lttName], [[x meta] lttName],
			 [[[[x meta] posedSelf] unit] lttName],
			 [[[x meta] posedSelf] lttName]);

	      [m setPoser: [x meta]];
	      if (!posers)
		posers = CONS (CONS ([x meta], m), nil);
	      else
		[posers nconc: CONS (CONS ([x meta], m), nil)];
	      m = [m itsClass];
	      [m setPoser: [[x meta] itsClass]];
	      [posers nconc: CONS (CONS ([[x meta] itsClass], m), nil)];
	    }
	}
    }
}

-(TLDictionary *) superReferences
{
  return super_refs;
}

-(TLVector *) unitsUsed
{
  return units_used;
}

/******************** LTTUnitAcceptor protocol ********************/

-(id <LTTUnitAcceptor>) class: (id <TLString>) c
		       inFile: (id <TLString>) f
		       posing: (TLCons *) p
{
  return [self extension: nil inFile: f ofClass: c
	       inUnit: [name internal] posing: p];
}

-(id <LTTUnitAcceptor>) endUnit
{
  return isa;
}

-(id <LTTUnitAcceptor>) extension: (id <TLString>) ext_name
			   inFile: (id <TLString>) filename
			  ofClass: (id <TLString>) classname
			   inUnit: (id <TLString>) unit_name
			   posing: (TLCons *) p
{
  LTTFile *f;
  LTTMeta *m;
  LTTUnit *u;
  TLString *basename_se;

  if (!unit_name)
    u = self;
  else
    {
      u = [CO_LTTUnit unitNamed: unit_name];
      if (!u)
	{
	  error (@"unknown unit `%@'", unit_name);
	  return self;
	}
    }

  if (filename)
    basename_se
      = ltt_filename_without_extension ([(TLString *) filename basename]);
  else
    {
      filename = ext_name ? ext_name : classname;
      basename_se = filename;
      filename = formac (nil, @"%@" TOM_IMPLEMENTATION_SUFFIX, filename);
    }

  f = [self fileNamed: basename_se];
  if (!f)
    {
      TLString *jfile;

      if (dir)
	jfile = formac (nil, @"%@/%@." TOM_INTERFACE_SUFFIX, dir, basename_se);
      else
	jfile = formac (nil, @"%@." TOM_INTERFACE_SUFFIX, basename_se);

      f = [CO_LTTFile fileWithName: [LTTName nameWithInternal: filename
					     external: jfile] unit: self];
      [files setObject: f forKey: basename_se];
    }

  m = ltt_instance_in_unit (unit_name, self,
			    [isa instancesNamed: classname]);
  if (m && !ext_name && [m extensionNamed: nil]
       && [m unit] == self)
    {
      error (@"attempt to redefine class `%@' in `%@'", classname, filename);
      cerror (@"class `%@' already defined in `%@'", classname,
	      [[[[m extensionNamed: nil] container] lttName] external]);
    }
  else if (m && ext_name && [m extensionNamed: ext_name])
    {
      error (@"attempt to redefine %@(%@) in `%@'",
	     classname, ext_name, filename);
      cerror (@"%@(%@) already defined in `%@'", classname, ext_name,
	      [[[[m extensionNamed: nil] container] lttName] external]);
    }
  else
    {
      LTTExtension *x;

      if (!m)
	m = [self metaWithName: classname];

      x = [CO_LTTExtension extensionWithName: ext_name file: f meta: m];
      [CO_LTTExtension extensionWithName: ext_name file: f meta: [m itsClass]];

      if (p)
	posing = CONS (CONS (x, p), posing);
    }

  return self;
}

-(id <LTTUnitAcceptor>) startUnitNamed: (id <TLString>) name
{
  ABORT ();
}

-(id <LTTUnitAcceptor>) usesUnit: (id <TLString>) n
{
  LTTUnit *u = [isa findUnitNamed: n];

  if (u)
    {
      [units_used addElement: u];
      if ([u dependsOn: self])
	error (@"circular dependency between units `%@' and `%@'",
	       [name internal], n);
    }

  return self;
}

@end
