/* Implementation of TLRunLoop class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLDate.m,v 1.4 1998/02/23 14:17:28 tiggr Exp $  */

#define TLDATE_DECLARE_PRIVATE_METHODS
#import "tl/support.h"
#import "tl/TLObject.h"
#import <limits.h>
#import "tl/TLPatchedRoots.h"
#import "tl/TLDate.h"
#import "tl/TLString.h"
#import "tl/TLMutableString.h"
#import <ctype.h>
#import <float.h>
#import <string.h>
#import <math.h>

time_interval
time_since_unix_epoch (void)
{
  struct timeval tv;

  gettimeofday (&tv, NULL);
  return (tv.tv_sec + tv.tv_usec / 1e6);
} /* time_since_unix_epoch */

struct timeval
timeval_from_time_interval (time_interval when)
{
  double secs = floor (when);
  double frag = when - secs;
  struct timeval tv;

  tv.tv_sec = (secs > LONG_MAX) ? LONG_MAX : secs;
  tv.tv_usec = frag * 1e6;
  return tv;
} /* timeval_from_time_interval */

@implementation TLDate

+(TLDate *) dateWithInterval: (time_interval) t
{
  return [[self gcAlloc] initWithInterval: t];
} /* +dateWithInterval: */

+(TLDate *) dateWithMicroSeconds: (id <TLNumber>) mu
{
  unsigned long mus = [mu unsignedLongValue];
  return ([[self gcAlloc] initWithInterval: mus / 1e6]);
} /* dateWithMicroSeconds: */

+(TLDate *) dateWithString: (id <TLString>) s
{
  return [[self gcAlloc] initWithString: s];
} /* +dateWithString: */

+(TLDate *) dateWithTimeval: (struct timeval *) t
{
  return [[self gcAlloc] initWithInterval: t->tv_sec + t->tv_usec / 1e6];
} /* +dateWithTimeval: */

+(TLDate *) dateWithUnixTime: (time_t) t
{
  return [[self gcAlloc] initWithInterval: t];
} /* +dateWithUnixTime: */

+(TLDate *) distantFuture
{
  return ([[self gcAlloc] initWithInterval: DBL_MAX]);
} /* +distantFuture */

+(TLDate *) distantPast
{
  return ([[self gcAlloc] initWithInterval: 0]);
} /* +distantPast */

+(TLDate *) now
{
  return ([[self gcAlloc] initWithNow]);
} /* +now */

-(int) compare: o
{
  time_interval tv = [o interval];

  return (when > tv ? 1 : when < tv ? -1 : 0);
} /* -compare */

-initWithNow
{
  when = time_since_unix_epoch ();
  return self;
} /* -initWithNow */

-initWithString: (id <TLString>) str
{
  static char *weekday[] =
  {
    "sunday", "monday", "tuesday", "wednesday", "thursday", "friday", "saturday"
  };
  static char *short_weekday[] =
  {
    "sun", "mon", "tue", "wed", "thu", "fri", "sat"
  };
  static char *month[] =
  {
    "january", "february", "march", "april", "may", "june",
    "july", "august", "september", "october", "november", "december"
  };
  static char *short_month[] =
  {
    "jan", "feb", "mar", "apr", "may", "jun",
    "jul", "aug", "sep", "oct", "nov", "dec"
  };
  static unsigned char weekday_len[] = {6, 6, 7, 9, 8, 6, 8};
  static unsigned char month_len[] = {7, 8, 5, 5, 3, 4, 4, 6, 9, 7, 8, 8};

  struct tm tm = { -1, -1, -1, -1, -1, -1, -1, -1, -1};
  const char *s = [str cString];
  int i = 0, j, l = [str length];

  /* Slashes are used in American dates; not in European ones.  */
  int seen_slash = 0;

  while (i < l)
    {
      while (i < l && !isalnum (s[i]))
	i++;
      if (i == l)
	break;

      if (isalpha (s[i]) && l - i >= 3)
	{
	  /* Discern a weekday.  */
	  if (tm.tm_wday == -1)
	    {
	      for (j = 0; j < 7; j++)
		if (!strncasecmp (s + i, short_weekday[j], 3))
		  {
		    i += ((l - i >= weekday_len[j]
			   && !strncasecmp (s + i, weekday[j], weekday_len[j]))
			  ? weekday_len[j] : 3);
		    tm.tm_wday = j;
		    break;
		  }
	      if (j != 7)
		continue;
	    }

	  /* A month!  */
	  if (tm.tm_mon == -1)
	    {
	      for (j = 0; j < 12; j++)
		if (!strncasecmp (s + i, short_month[j], 3))
		  {
		    i += ((l - i >= month_len[j]
			   && !strncasecmp (s + i, month[j], month_len[j]))
			  ? month_len[j] : 3);
		    tm.tm_mon = j;
		    break;
		  }
	      if (j != 12)
		continue;
	    }

	  /* This must be a timezone then, until the end of the string.  */
	  break;
	}

      if (isdigit (s[i]))
	{
	  unsigned int v = s[i++] - '0';

	  while (i < l && isdigit (s[i]))
	    v = v * 10 + s[i++] - '0';

	  if (tm.tm_mon == -1 && tm.tm_mday == -1)
	    {
	      if (i < l && v < 13 && (s[i] == '/' || tm.tm_year != -1))
		{
		  if (s[i] == '/')
		    {
		      /* This is a slash indicating an American date.  */
		      seen_slash = 1;
		    }
		  tm.tm_mon = v - 1;
		  continue;
		}
	      else if (v < 32)
		{
		  tm.tm_mday = v;
		  continue;
		}
	    }

	  if (tm.tm_mon == -1 && tm.tm_mday != -1 && v < 13)
	    tm.tm_mon = v - 1;
	  else if (tm.tm_mday == -1 && tm.tm_mon != -1 && v < 32)
	    tm.tm_mday = v;
	  else if (tm.tm_year == -1
		   && !((i < l && s[i] == ':'
			 && (tm.tm_hour == -1 || tm.tm_min == -1))
			|| (tm.tm_min != -1 && tm.tm_sec == -1 && v < 60)))
	    {
	      if (v < 1900)
		v += 1900;
	      tm.tm_year = v - 1900;
	    }
	  else if (tm.tm_hour == -1)
	    tm.tm_hour = v;
	  else if (tm.tm_min == -1)
	    tm.tm_min = v;
	  else if (tm.tm_sec == -1)
	    tm.tm_sec = v;
	  else
	    {
	      /* Ehhh...  */
	      INIT_RETURN_NIL ();
	    }
	}
    }

  /* Either I points to the end of the string, or to the supposed timezone.  */
  if (tm.tm_year == -1 || tm.tm_mon == -1 || tm.tm_mday == -1
      || tm.tm_hour == -1 || tm.tm_min == -1 || tm.tm_sec == -1)
    INIT_RETURN_NIL ();

#if 0
  /* Adjust for the local time assumption made by mktime().
     XXX This is, of course, utterly wrong in case the DST was different at
     the time of the date from the current DST.*/
  if (!tzname[0])
    tzset ();
  tm.tm_sec -= timezone;
#endif

  when = mktime (&tm);
  if (when == -1) /* XXX */
    INIT_RETURN_NIL ();
  return self;
} /* -initWithString: */

-initWithInterval: (time_interval) t
{
  when = t;
  return self;
} /* -initWithTimeval: */

-(time_interval) interval
{
  return when;
} /* -interval */

-(unsigned long) microSeconds
{
  struct timeval tv = timeval_from_time_interval (when);

  return ((ULONG_MAX - tv.tv_usec) / 1e6 < tv.tv_sec
	  ? ULONG_MAX : tv.tv_usec + 1000000 * tv.tv_sec);
} /* -microSeconds */

-(void) print: (id <TLOutputStream>) stream format: (id <TLString>) format
{
  struct timeval tv = timeval_from_time_interval (when);
  time_t tt = tv.tv_sec;
  struct tm *tm = gmtime (&tt);
  char buf[1000];
  int i;

  /* XXX Bweh!  */
  i = strftime (buf, sizeof (buf), [format cString], tm);

  [(TLString *) [CO_TLString stringWithCString: buf length: i]
		print: stream quoted: 0];
} /* -print:format: */

-(void) print: (id <TLOutputStream>) stream quoted: (BOOL) qp
{
  struct timeval tv = timeval_from_time_interval (when);
  time_t tt = tv.tv_sec;
  struct tm tm = *localtime (&tt);

  if (qp)
    [@"(TLDate \"" print: stream quoted: NO];

  /* XXX */
  formac (stream, @"%d-%d-%d %d:%d:%d.%ld", 1900 + tm.tm_year, 1 + tm.tm_mon,
	  tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec, tv.tv_usec);

  if (qp)
    [@"\")" print: stream quoted: NO];
} /* -print:quoted: */

-(double) seconds
{
  return when;
} /* -seconds */

-(TLString *) stringUsingFormat: (id <TLString>) format
{
  TLMutableString *r = [CO_TLMutableString mutableString];

  [self print: r format: format];
  return r;
} /* -stringUsingFormat: */

-(struct timeval) timeval
{
  return timeval_from_time_interval (when);
} /* -timeval */

@end
