/* Implementation of TLDictionary class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLDictionary.m,v 1.1 1998/01/08 16:11:34 tiggr Exp $  */

#define TLDICTIONARY_DECLARE_PRIVATE_METHODS
#import "tl/support.h"
#import "tl/TLDictionary.h"
#import "tl/TLSymbol.h"
#import <stdlib.h>

struct element
{
  struct element *cdr;
  id key, value;
};

@interface TLDictionaryEnumerator: TLObject
{
  TLDictionary *dict;
  int bucket, depth;
}

+(id <TLEnumerator>) enumeratorWithDictionary: (TLDictionary *) d;

@end

@interface TLDictionaryDictEnumerator: TLDictionaryEnumerator
						<TLDictionaryEnumerator>
{
  struct element *e;
}

@end

/* Declare the proper return value for the bucket enumerator.  This is not
   implemented separately, but only here to keep the compiler happy.
   XXX Barf!  GCC 2.6.3 does not understand this: it picks the declaration
   by TLDictionaryEnumerator to check the types...  */
@interface TLDictionaryDictEnumerator (DeclarationOnly)

+(id <TLDictionaryEnumerator>) enumeratorWithDictionary: (TLDictionary *) d;

@end

@interface TLDictionaryKeyEnumerator: TLDictionaryEnumerator <TLEnumerator>
@end

@interface TLDictionaryValueEnumerator: TLDictionaryEnumerator <TLEnumerator>
@end

@implementation TLDictionary

+(TLDictionary *) dictionary
{
  return ([[self gcAlloc] init]);
} /* +dictionary */

-(boolean) _bucketBeyondEnd: (int) bucket
{
  return (bucket >= nbucket);
} /* _bucketBeyondEnd: */

-(struct element *) _elementAtIndex: (int *) bucket : (int *) depth
{
  struct element *e;
  int i;

  if (*bucket < nbucket)
    {
      for (i = 0, e = buckets[*bucket]; e && i < *depth; e = e->cdr, i++);
      if (!e)
	{
	  *depth = 0;
	  while (++*bucket < nbucket)
	    if (buckets[*bucket])
	      {
		e = buckets[*bucket];
		break;
	      }
	}
      if (e)
	{
	  if (e->cdr)
	    (*depth)++;
	  else
	    {
	      (*bucket)++;
	      *depth = 0;
	    }
	  return (e);
	}
    }
  return (NULL);
} /* _elementAtIndex:: */

-(id <TLDictionaryEnumerator>) enumerator
{
  static id tldict_enumerator_class;
  if (!tldict_enumerator_class)
    tldict_enumerator_class = [TLDictionaryDictEnumerator self];

  return ((id) [tldict_enumerator_class enumeratorWithDictionary: self]);
}

-_keyAtIndex: (int *) bucket : (int *) depth
{
  struct element *e = [self _elementAtIndex: bucket : depth];
  return (e ? e->key : nil);
} /* -_keyAtIndex:: */

-(id <TLEnumerator>) keyEnumerator
{
  static id tldict_enumerator_class;
  if (!tldict_enumerator_class)
    tldict_enumerator_class = [TLDictionaryKeyEnumerator self];

  return ([tldict_enumerator_class enumeratorWithDictionary: self]);
}

-(int) length
{
  return nobject;
} /* -length */

-member: k
{
  unsigned int hv = [k hash];
  struct element *e;

  if (nbucket)
    for (e = buckets[hv % nbucket]; e; e = e->cdr)
      if ([k equal: e->key])
	return (e->value ? e->value : Qt);
  return (nil);
} /* -member: */

-objectForKey: k
{
  unsigned int hv = [k hash];
  struct element *e;

  if (nbucket)
    for (e = buckets[hv % nbucket]; e; e = e->cdr)
      if ([k equal: e->key])
	return (e->value);
  return (nil);
} /* -objectForKey: */

-(void) print: (id <TLMutableStream>) stream quoted: (BOOL) qp
{
  id <TLString> format = qp ? @" (%# . %#)" : @" (%@ . %@)";

  [stream writeBytes: 13 fromBuffer: "(TLDictionary"];

  if (nbucket)
    {
      struct element *e;
      int i;

      for (i = 0; i < nbucket; i++)
	for (e = buckets[i]; e; e = e->cdr)
	  formac (stream, format, e->key, e->value);
    }

  [stream writeBytes: (nbucket ? 2 : 1) fromBuffer: ")\n"];
} /* -print:quoted: */

-(void) reconfigure
{
  unsigned int i, nn = 2 * nbucket + 1;
  struct element **nb = xcalloc (nn, sizeof (*nb));
  struct element *e, *ne;

  for (i = 0; i < nbucket; i++)
    for (e = buckets[i]; e; e = ne)
      {
	unsigned int hvb = [e->key hash] % nn;
	ne = e->cdr;
	e->cdr = nb[hvb];
	nb[hvb] = e;
      }

  xfree (buckets);
  buckets = nb;
  nbucket = nn;
} /* -reconfigure */

-removeKey: k
{
  struct element **pe;
  id o;

  if (!nbucket)
    return (nil);

  for (pe = &buckets[[k hash] % nbucket];
       *pe && ![k equal: (*pe)->key]; pe = &(*pe)->cdr);
  if (!*pe)
    o = nil;
  else
    {
      struct element *e = *pe;
      *pe = e->cdr;
      o = e->value;
      xfree (e);
    }
  return (o);
} /* -removeKey: */

-setObject: o forKey: k
{
  struct element **pe;
  id po;

  if (!k)
    [self error: "(-setObject:forKey:): NIL key for object %#", o];

  if (2 * nbucket <= nobject)
    [self reconfigure];

  for (pe = &buckets[[k hash] % nbucket];
       *pe && ![k equal: (*pe)->key]; pe = &(*pe)->cdr);
  if (!*pe)
    {
      *pe = xmalloc (sizeof (**pe));
      (*pe)->cdr = NULL;
      ASGN_IVAR ((*pe)->key, k);
      ASGN_IVAR ((*pe)->value, o);
      nobject++;
      po = nil;
    }
  else
    {
      po = (*pe)->value;
      ASGN_IVAR ((*pe)->value, o);
    }
  return (po);
} /* -setObject:forKey: */

-(id <TLEnumerator>) valueEnumerator
{
  static id tldict_enumerator_class;
  if (!tldict_enumerator_class)
    tldict_enumerator_class = [TLDictionaryValueEnumerator self];

  return ([tldict_enumerator_class enumeratorWithDictionary: self]);
}

-_valueAtIndex: (int *) bucket : (int *) depth
{
  struct element *e = [self _elementAtIndex: bucket : depth];
  return (e ? e->value : nil);
} /* -_valueAtIndex:: */

/******************** garbage collection ********************/

-(void) dealloc
{
  xfree (buckets);
} /* -dealloc */

-(void) gcReference
{
  struct element *e;
  unsigned int i;

  for (i = 0; i < nbucket; i++)
    for (e = buckets[i]; e; e = e->cdr)
      {
	MARK (e->key);
	MARK (e->value);
      }
} /* -gcReference */

@end

@implementation TLDictionaryEnumerator

/******************** private methods ********************/

-initWithDictionary: (TLDictionary *) d
{
  ASGN_IVAR (dict, d);
  return (self);
} /* -initWithDictionary: */

/******************** public methods ********************/

+(TLDictionaryEnumerator *) enumeratorWithDictionary: (TLDictionary *) d
{
  return ([[self gcAlloc] initWithDictionary: d]);
} /* +enumeratorWithDictionary: */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (dict);
} /* -gcReference */

@end

@implementation TLDictionaryKeyEnumerator

/******************** TLEnumerator ********************/

-notEndP
{
  return ([dict _bucketBeyondEnd: bucket] ? nil : Qt);
} /* -notEndP */

-nextObject
{
  return ([dict _keyAtIndex: &bucket : &depth]);
} /* -nextObject */

@end

@implementation TLDictionaryValueEnumerator

/******************** TLEnumerator ********************/

-notEndP
{
  return ([dict _bucketBeyondEnd: bucket] ? nil : Qt);
} /* -notEndP */

-nextObject
{
  return ([dict _valueAtIndex: &bucket : &depth]);
} /* -nextObject */

@end

@implementation TLDictionaryDictEnumerator

/******************** TLEnumerator ********************/

-notEndP
{
  return (e ? Qt : nil);
} /* -notEndP */

-nextObject
{
  e = [dict _elementAtIndex: &bucket : &depth];
  return (e ? e->key : nil);
} /* -nextObject */

/******************** TLDictionaryEnumerator ********************/

-key
{
  return (e ? e->key : nil);
} /* -key */

-value
{
  return (e ? e->value : nil);
} /* -value */

@end
