/* Implementation of TLObject class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLObject.m,v 1.2 1998/01/09 11:21:18 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLFILEStream.h"
#import "tl/TLPatchedRoots.h"
#import "tl/TLGC.h"
#import "tl/TLDate.h"
#import <stdio.h>
#import <string.h>
#import <time.h>
#if HAVE_STRINGS_H
#import <strings.h>
#endif

#if DEBUG_GC
/* Control at run-time up to the level of the compile-time setting.  */
int debug_gc = 0;
#endif

/* The list protecting objects pointed to from the stack.  */
struct tlgcs *_tlgc_stack;

/* Vector of objects locked against garbage collection.  */
TLVector *_tlgc_locked;

/* Vector of objects retained and thus locked against garbage collection.
   XXX Maybe this should be an rchashtable.  */
TLVector *_tlgc_retained;

/* Whether we're allowed to quit and whether a quit was requested.  */
int tl_quit_inhibit, tl_quit_request;

/* The number of objects allocated since the last partial GC run.  */
unsigned long tlgc_alloc_since_partial;

/* The number of objects allocated since the last complete GC run.  */
unsigned long tlgc_alloc_since_complete;

/* The number of allocated objects.  */
unsigned long tlgc_num_alloc;

/* The total number of objects, both allocated and free.  This does not
   count the size taken by chunks on the free chunk list.  */
unsigned long tlgc_num_total;

/* The threshold for TLGC_ALLOC_SINCE_PARTIAL before `-[TLCons eval]' will
   initiate a GC run.  */
unsigned long tlgc_partial_threshold;

/* The time limit on a GC run initiated by running over
   TLGC_PARTIAL_THRESHOLD.  */
unsigned long tlgc_partial_time_limit;

/* The threshold for TLGC_ALLOC_SINCE_COMPLETE plus TLGC_ALLOC_SINCE_PARTIAL
   before `-[TLCons eval]' will initiate a GC run.  */
unsigned long tlgc_total_threshold;

/* The time limit on a GC run initiated by running over
   TLGC_TOTAL_THRESHOLD.  */
unsigned long tlgc_total_time_limit;

/* The amount of time (seconds) spent in GC since the last completion.  */
time_interval tlgc_this_time;

/* The same information, split in the three phases.  */
time_interval tlgc_this_protect, tlgc_this_mark, tlgc_this_sweep;

/* The amount of time (seconds) spent in total in all completed GC runs.  */
time_interval tlgc_total_time;

/* The same information, split in the three phases.  */
time_interval tlgc_total_protect, tlgc_total_mark, tlgc_total_sweep;

/* The number of completed and partial GC runs.  */
int tlgc_num_run_complete, tlgc_num_run_partial;

/* The hard limit on the number of objects.  */
unsigned long tlgc_alloc_limit;

/* The number of times `gcMark' has been invoked.  */
unsigned long tlgc_num_mark;

/* Which pass of the garbage collector is running.  */
static enum
{
  /* Not doing anything.  */
  TLGC_NOT,

  /* We're protecting (graying) otherwise poosibly non-marked objects.  */
  TLGC_PROTECT,

  /* Marking gray objects.  */
  TLGC_MARK_GRAY,

  /* Sweeping: deallocating whites  */
  TLGC_SWEEP_WHITE,
} tlgc_pass;

/* The stucture of a generic tl object.  */
typedef struct tlobject_defs
{
  @defs (TLObject);
} tlobject_defs;

/* Struct used to inspect allocation slots.  Minimum slot size is the size
   of this struct.  */
typedef struct tlgc_vacant
{
  tlobject_defs o;

#if DEBUG_GC > 0
  /* The saved isa of a deallocated object.  */
  struct objc_class *saved_isa;
#endif

  struct tlgc_vacant *next;
} tlgc_vacant;

/* TLObjects are allocated from chunks.  Each chunk holds CAP objects
   (of ITEM_SIZE bytes, where ITEM_SIZE is defined in the chunk_list, see
   below).  The minimum item size is sizeof (TLGC_VACANT).  A TLNonObject is
   held in every vacant slot.  */
typedef struct chunk
{
  /* Doubly linked list of all chunks in this chunk list.  */
  struct chunk *next, **previous;

  /* Doubly linked list of all chunks in this chunk list which have
     vacancies.  */
  struct chunk *next_vacant, **previous_vacant;

  /* The first vacant object in this chunk.  */
  tlgc_vacant *vacant_object;

  /* The number of allocated items in this chunk, and its capacity.  */
  unsigned int num, cap;

  /* The objects in this chunk.  */
  ALIGN_TYPE data[0];
} chunk;

/* The head of a list of chunks holding objects of the same size.  */
typedef struct slot_entry
{
  /* The first chunk in this list and the first chunk with vacancies.  */
  chunk *first, *first_vacant;
} slot_entry;

/* The array of slot_entry's, and its number.  */
static slot_entry *chunks;
static unsigned int num_slots;

/* The free list of chunks, maintained through their NEXT_VACANT field.  */
static chunk *chunks_free;

/* The number of chunks in CHUNKS_FREE.  */
static int chunks_free_num;

/* Return the index for an object of size N.  */
#define INDEX_FOR_SIZE(N)  \
  (((N) + sizeof (ALIGN_TYPE) - 1) / sizeof (ALIGN_TYPE))

/* The smallest legal index value.  */
#define SMALLEST_INDEX  INDEX_FOR_SIZE (sizeof (tlgc_vacant))

/* Return the size of the items at slot index N.  */
#define SIZE_FOR_INDEX(N)  ((N) * sizeof (ALIGN_TYPE))

/* Return the rounded size for an object of size N.  */
#define ROUNDED_SIZE(N)  \
  (((N) + sizeof (ALIGN_TYPE) - 1) / sizeof (ALIGN_TYPE) * sizeof (ALIGN_TYPE))

/* The class object for vacant slots.  */
static id tl_nonobject_class;

/* Estimates for the number of operations which can be performed in a
   certain amount of time: The number of times per second a `gcMark' can be
   invoked, and the number of objects per second which can be handled in the
   sweep phase.  The initial values do not matter, as these numbers are
   adjusted continuously.  */
static unsigned long tlgc_marks_per_sec = 100000;
static unsigned long tlgc_sweeps_per_sec = 100000;

/* Statistics on object colours during garbage collection.  */
static unsigned long tlgc_num[TLOC_NUM];

/* The number of bytes in a page.  */
static long page_size;

/* Interface to an object that does not implement anything, apart from
   `doesNotRecognize:'.  Used to fill previously allocated but currently
   vacant slots in the heap.  */
@interface TLNonObject
{
  /* We have an ISA, just to be conventional.  XXX Somehow, we can't do this
     by stating `@defs (TLObject)'.  */
  struct objc_class *isa;
}

-(retval_t) forward: (SEL) sel : (FORWARD_ARGUMENTS) arg_frame;
+(retval_t) forward: (SEL) sel : (FORWARD_ARGUMENTS) arg_frame;

@end

#if WANT_NIL_OBJECT && GNU_RUNTIME
/* XXX This should not be here, as it duplicates the _objc_nil_object.  */
static id tll_nil_object;

static void
tll_set_nil_object (id o)
{
  tll_nil_object = o;
  objc_set_nil_object (tll_nil_object);
} /* tll_set_nil_object */
#endif

/* The array used as a stack for gray objects.  All gray objects should be
   here.  */
static id *gray_objects;
static int gray_num, gray_cap;

/* Add an object to the gray stack.  */
static inline void
gray_stack_add (id o)
{
  if (gray_num == gray_cap)
    {
      gray_cap = gray_cap ? gray_cap * 2 : 1024;
      gray_objects = xrealloc (gray_objects, gray_cap * sizeof (*gray_objects));
    }
  gray_objects[gray_num++] = o;
} /* gray_stack_add */

#if DEBUG_GC > 0
void
dump_protection_stack (int verbosity)
{
  struct tlgcs *gc_stack;
  id *vars;
  int i;

  for (gc_stack = _tlgc_stack; gc_stack; gc_stack = gc_stack->next)
    {
      if (verbosity)
	printf ("stack %p var %p num %d\n", gc_stack, gc_stack->v, gc_stack->n);
      for (vars = gc_stack->v, i = 0; i < gc_stack->n; vars++, i++)
	{
	  tlgc_vacant *o = (void *) *vars;

	  if (verbosity || (*(void **) o == tl_nonobject_class))
	    {
	      printf (" %d %p %s", i, *vars,
		      class_get_class_name (*(void **) o));
	      if (*(void **) o == tl_nonobject_class)
		printf (" *NONOBJECT*");
	      else if (verbosity > 1 && [*vars isKindOf: [TLObject self]])
		{
		  printf (" colour %d", o->o._tlobject_colour);
		  switch (o->o._tlobject_colour)
		    {
		    case TLOC_GRAY: printf (" gray"); break;
		    case TLOC_WHITE: printf (" white"); break;
		    case TLOC_BLACK: printf (" black"); break;
		    case TLOC_PINK: printf (" pink"); break;
		    default: printf (" *BAD COLOUR*");
		    }
		}
	      printf ("\n");
	    }
	}
    }
} /* dump_protection_stack */

void
dump_layout (void)
{
  int slot;

  for (slot = SMALLEST_INDEX; slot < num_slots; slot++)
    {
      slot_entry *l = &chunks[slot];

      printf ("slot index=%d", slot);

      if (!l)
	printf (": empty\n");
      else
	{
	  chunk *ch;

	  printf (" chunk_list=%p item_size=%ld\n", l,
		  (long) SIZE_FOR_INDEX (slot));

	  for (ch = l->first; ch; ch = ch->next)
	    printf ("  chunk %p num=%d cap=%d\n", ch, ch->num, ch->cap);
	}
    }
}

void
gc_check_chunks_integrity (void)
{
  int slot;

  for (slot = SMALLEST_INDEX; slot < num_slots; slot++)
    {
      slot_entry *l = &chunks[slot];

      if (l)
	{
	  chunk **vch, **nvch;

	  /* Check integrity of vacancy list.  */
	  for (vch = &l->first_vacant; *vch; vch = nvch)
	    {
	      if ((*vch)->previous_vacant != vch)
		abort ();
	      nvch = &(*vch)->next_vacant;
	      if (*nvch && (*nvch)->previous_vacant != nvch)
		abort ();
	    }

	  /* Check integrity of list.  */
	  for (vch = &l->first; *vch; vch = nvch)
	    {
	      if ((*vch)->previous != vch)
		abort ();
	      nvch = &(*vch)->next;
	      if (*nvch && (*nvch)->previous != nvch)
		abort ();
	    }
	}
    }
} /* gc_check_chunks_integrity */

#endif

/* Run a partial GC for (hopefully) at most USEC micro seconds _elapsed_
   time.  Run til completion if !USEC.  */
static id
tl_incremental_gc (unsigned long usec)
{
  /* Sweep phase administration.  */
  /* The index in CHUNKS of the current chunk list.  */
  static int sweep_slot;
  /* The index of the next object to be investigated in the current chunk.  */
  static int sweep_index;
  /* The current chunk in the current chunk list.  */
  static chunk *sweep_chunk;

  time_interval sec = usec / 1e6;
  time_interval sec_left = sec, sec_total;
  time_interval sec_protect = 0, sec_mark = 0, sec_sweep = 0;
  time_interval enter, start, now;
  unsigned long num_sweep = 0;
  int i;

  QUIT_DISABLE;

  enter = time_since_unix_epoch ();

  tlgc_num_mark = 0;
  tlgc_num_run_partial++;
  tlgc_alloc_since_complete += tlgc_alloc_since_partial;
  tlgc_alloc_since_partial = 0;

  if (tlgc_pass == TLGC_NOT)
    {
      start = time_since_unix_epoch ();

      for (i = 0; i < sizeof (tlgc_num) / sizeof (*tlgc_num); i++)
	tlgc_num[i] = 0;

      tlgc_pass = TLGC_PROTECT;
      tl_garbage_protect ();

      now = time_since_unix_epoch ();
      sec_protect = now - start;
      tlgc_this_protect += sec_protect;
      sec_left -= sec_protect;

      tlgc_pass = TLGC_MARK_GRAY;
    }

  if (tlgc_pass == TLGC_MARK_GRAY && (!sec || sec_left > 0))
    {
      unsigned long mark_estimate, mark_rate;

      start = time_since_unix_epoch ();
      mark_estimate = sec ? tlgc_marks_per_sec * sec_left : -1;

      while (tlgc_num_mark < mark_estimate && gray_num)
	{
	  struct tlgc_vacant *o = (void *) gray_objects[--gray_num];
	  TL_COLOUR_SET (&o->o, TLOC_BLACK);
	  [(id) o gcReference];
	}

      now = time_since_unix_epoch ();
      sec_mark = now - start;
      tlgc_this_mark += sec_mark;
      mark_rate = tlgc_num_mark / sec_mark;
      tlgc_marks_per_sec = (tlgc_marks_per_sec + mark_rate) / 2;
      sec_left -= sec_mark;

      if (tlgc_num_mark < mark_estimate)
	/* We're done with marking!  */
	tlgc_pass = TLGC_SWEEP_WHITE;
    }

  if (tlgc_pass == TLGC_SWEEP_WHITE && (!sec || sec_left > 0))
    {
      unsigned long sweep_estimate, sweep_rate;
      tlgc_vacant *v;
      int limit;

      start = time_since_unix_epoch ();

      sweep_estimate = sec ? tlgc_sweeps_per_sec * sec_left : -1;

      if (!sweep_slot)
	{
	  sweep_chunk = NULL;
	  sweep_index = 0;

	  /* Initialize for the first chunk.  */
	  for (sweep_slot = SMALLEST_INDEX;
	       sweep_slot < num_slots; sweep_slot++)
	    if (chunks[sweep_slot].first)
	      {
		sweep_chunk = chunks[sweep_slot].first;
		break;
	      }
	}

      while (sweep_chunk && num_sweep < sweep_estimate)
	{
#if DEBUG_GC > 3
	  if (debug_gc > 3)
	    printf ("sweep chunk=%x start_index=%d\n",
		    sweep_chunk, sweep_index);
#endif

	  for (v = (void *) ((char *) &sweep_chunk->data
			     + sweep_index * SIZE_FOR_INDEX (sweep_slot)),
		 limit = sweep_chunk->cap;
	       sweep_index < limit && num_sweep < sweep_estimate;
	       sweep_index++,
		 v = (void *) ((char *) v + SIZE_FOR_INDEX (sweep_slot)))
	    {
	      num_sweep++;

	      if (v->o.isa == tl_nonobject_class)
		tlgc_num[TLOC_PINK]++;
	      else switch (TL_COLOUR (&v->o))
		{
		case TLOC_GRAY:
		  /* This object has been allocated between GC marking and
		     this sweep.  Consider it having been allocated during
		     the next mark run.  It will be deallocated, if dead,
		     during the next sweep run.  */
		  tlgc_num[TLOC_GRAY]++;
		  break;

		case TLOC_BLACK:
		  /* This is a live object.  */
		  TL_COLOUR_SET (&v->o, TLOC_WHITE);
		  tlgc_num[TLOC_BLACK]++;
		  break;

		case TLOC_WHITE:
		  /* This is a dead object.  */
		  [(TLObject *) v dealloc];

		  tlgc_num[TLOC_WHITE]++;
		  tlgc_num_alloc--;

#if DEBUG_GC > 0
		  v->saved_isa = v->o.isa;
#endif
		  v->o.isa = tl_nonobject_class;

		  if (!--sweep_chunk->num)
		    {
		      /* This chunk now is empty.  This means all remaining
			 objects in this chunk are already pink.  Remove
			 this chunk from the chunk list.  */
#if DEBUG_GC > 4
		      if (debug_gc > 4)
			printf ("%x: no objects in chunk for slot %d\n",
				sweep_chunk, sweep_slot);
#endif
		      *sweep_chunk->previous_vacant = sweep_chunk->next_vacant;
		      if (sweep_chunk->next_vacant)
			sweep_chunk->next_vacant->previous_vacant
			  = sweep_chunk->previous_vacant;
		      *sweep_chunk->previous = sweep_chunk->next;
		      if (sweep_chunk->next)
			sweep_chunk->next->previous = sweep_chunk->previous;

		      /* Put it on the free chunks list.  */
		      sweep_chunk->next_vacant = chunks_free;
		      sweep_chunk->previous_vacant = NULL;
		      chunks_free = sweep_chunk;
		      chunks_free_num++;

		      tlgc_num_total -= sweep_chunk->cap;
		      sweep_index = limit;
		    }
		  else
		    {
		      if (!sweep_chunk->vacant_object)
			{
			  /* This chunk was full.  Not anymore.  */
			  slot_entry *slot = &chunks[sweep_slot];

#if DEBUG_GC > 4
			  if (debug_gc > 4)
			    printf ("%x: first vacancy in chunk for slot %d\n",
				    sweep_chunk, sweep_slot);
#endif
			  sweep_chunk->next_vacant = slot->first_vacant;
			  if (sweep_chunk->next_vacant)
			    sweep_chunk->next_vacant->previous_vacant
			      = &sweep_chunk->next_vacant;
			  slot->first_vacant = sweep_chunk;
			  sweep_chunk->previous_vacant = &slot->first_vacant;
			}

		      v->next = sweep_chunk->vacant_object;
		      sweep_chunk->vacant_object = v;
		    }
		  break;

		default:
		  abort ();
		}
	    }
	  if (sweep_index >= limit)
	    {
	      sweep_index = 0;
	      sweep_chunk = sweep_chunk->next;

	      if (!sweep_chunk)
		{
		  for (sweep_slot++; sweep_slot < num_slots; sweep_slot++)
		    if (chunks[sweep_slot].first)
		      {
			sweep_chunk = chunks[sweep_slot].first;
			break;
		      }
		}
	    }
	}

      now = time_since_unix_epoch ();
      sec_sweep = now- start;
      tlgc_this_sweep += sec_sweep;
      sweep_rate = num_sweep / sec_sweep;
      tlgc_sweeps_per_sec = (tlgc_sweeps_per_sec + sweep_rate) / 2;
      sec_left -= sec_sweep;

      if (num_sweep != sweep_estimate)
	{
	  /* We're done with sweeping!  */
	  tlgc_pass = TLGC_NOT;
	  sweep_slot = 0;

#if DEBUG_GC > 2
	  /* Check that all objects are not black.  */
	  if (debug_gc > 2)
	    {
	      int slot, printed = 0;

	      for (slot = SMALLEST_INDEX; slot < num_slots; slot++)
		{
		  slot_entry *l = &chunks[slot];
		  chunk *ch;

		  if (l)
		    for (ch = l->first; ch; ch = ch->next)
		      {
			int n;

			for (n = 0; n < ch->cap; n++)
			  {
			    tlgc_vacant *v;

			    v = (void *) ((char *) &ch->data[0]
					  + n * SIZE_FOR_INDEX (slot));
			    if (TL_COLOUR (v) == TLOC_BLACK)
			      {
				printf ("black objects chunk=%x first n=%d\n",
					ch, n);
				printed = 1;
				break;
			      }
			  }
		      }
		}

	      if (printed)
		dump_layout ();
	    }
#endif

#if DEBUG_GC > 1
	  if (debug_gc > 1)
	    {
	      unsigned long free, total;

	      free = tlgc_num[TLOC_WHITE] + tlgc_num[TLOC_PINK];
	      total = free + tlgc_num[TLOC_GRAY] + tlgc_num[TLOC_BLACK];

	      fprintf (stderr,
		       "stats: white: %lu gray: %lu black: %lu pink: %lu "
		       "f/t: %lu/%lu (%.2f)\ntimes: %.3f %.3f %.3f\n",
		       tlgc_num[TLOC_WHITE], tlgc_num[TLOC_GRAY],
		       tlgc_num[TLOC_BLACK], tlgc_num[TLOC_PINK],
		       free, total, (double) (total - free) / total,
		       tlgc_this_protect * 1000, tlgc_this_mark * 1000,
		       tlgc_this_sweep * 1000);
	    }
#endif

	  tlgc_num_run_complete++;
	  tlgc_alloc_since_complete = 0;
	  tlgc_this_protect = tlgc_this_mark = tlgc_this_sweep = 0;

#if 0
	  /* We don't want this: program's calling abort because they use
             too much memory.  */
	  if (tlgc_alloc_limit && tlgc_num_alloc >= tlgc_alloc_limit)
	    if (tlgc_num[TLOC_GRAY])
	      {
		formac (V_stderr_, "gc: forcing gc due to limit: %lu >= %lu\n",
			tlgc_num_alloc, tlgc_alloc_limit);
		tl_incremental_gc (0);
	      }
#endif
	}
    }

  now = time_since_unix_epoch ();
  sec_total = now - enter;
  tlgc_this_time += sec_total;

#if DEBUG_GC > 1
  if (debug_gc > 1)
    fprintf (stderr, "gc (%.3f): %.3f %.3f(%lu) %.3f(%lu) %d %.3f/%.3f %.3f\n",
	     sec * 1000, sec_protect * 1000, sec_mark * 1000, tlgc_num_mark,
	     sec_sweep * 1000, num_sweep, chunks_free_num, sec_total * 1000,
	     tlgc_this_time * 1000, sec_left * 1000);
#endif

  if (!tlgc_alloc_since_complete)
    tlgc_this_time = 0;

#if DEBUG_GC > 5
  if (debug_gc > 5)
    gc_check_chunks_integrity ();
#endif

  QUIT_ENABLE;

  return (nil);
} /* tl_incremental_gc */

/* Create a new instance of the CLASS.  */
static id
tl_default_object_alloc (struct objc_class *class)
{
  chunk *c;
  tlgc_vacant *v;
  slot_entry *slot;
  unsigned int item_size = ROUNDED_SIZE (class_get_instance_size (class));
  unsigned int i, slot_index;

#if DEBUG_GC > 2
  if (debug_gc > 2)
    {
      if (item_size < class_get_instance_size (class))
	abort ();
      /* XXX YYY This is only true for the double ALIGN_TYPE! */
      if (item_size & 7)
	atoi ("0");
    }
#endif

  if (item_size < SIZE_FOR_INDEX (SMALLEST_INDEX))
    item_size = SIZE_FOR_INDEX (SMALLEST_INDEX);
  slot_index = (INDEX_FOR_SIZE (item_size)
		/* XXX item_size / sizeof (ALIGN_TYPE) */);

  QUIT_DISABLE;

  tlgc_alloc_since_partial++;
  tlgc_num_alloc++;

  /* Possibly extend the number of slots in CHUNKS to accommodate a new
     chunk at index SLOT_INDEX.  */
  if (slot_index >= num_slots)
    {
#if DEBUG_GC > 4
      if (debug_gc > 4)
	printf ("resize chunks to index %d\n", slot_index);
#endif
      chunks = xrealloc (chunks, (slot_index + 1) * sizeof (*chunks));
      bzero ((void *) (chunks + num_slots),
	     (slot_index + 1 - num_slots) * sizeof (*chunks));

      {
	int i;

	for (i = SMALLEST_INDEX; i < num_slots; i++)
	  if (chunks[i].first)
	    {
	      chunks[i].first->previous = &chunks[i].first;
	      if (chunks[i].first_vacant)
		chunks[i].first_vacant->previous_vacant
		  = &chunks[i].first_vacant;
	    }
      }

      num_slots = slot_index + 1;
    }

  /* Allocate a new chunk if this slot does not hold any chunks with
     vacancies.  */
  slot = &chunks[slot_index];
  if (!slot->first_vacant)
    {
      if (chunks_free)
	{
	  c = chunks_free;
#if DEBUG_GC > 4
	  if (debug_gc > 4)
	    printf ("%x: reusing chunk for slot %d\n", c, slot_index);
#endif
	  chunks_free = c->next_vacant;
	  chunks_free_num--;
#if DEBUG_GC > 0
	  if (!chunks_free_num && chunks_free)
	    abort ();
#endif
	}
      else
	{
	  c = xmalloc (page_size);
#if DEBUG_GC > 4
	  if (debug_gc > 4)
	    printf ("%x: allocated chunk for slot %d\n", c, slot_index);
#endif
	}

      c->next = slot->first;
      if (c->next)
	c->next->previous = &c->next;
      slot->first = c;
      c->previous = &slot->first;

      c->next_vacant = NULL;
      slot->first_vacant = c;
      c->previous_vacant = &slot->first_vacant;

      c->num = 0;
      c->vacant_object = (void *) &c->data;
      c->cap = (page_size - ((char *) &c->data - (char *) c)) / item_size;
      tlgc_num_total += c->cap;

      for (i = 0, v = c->vacant_object; i < c->cap; v = v->next, i++)
	{
	  v->o.isa = tl_nonobject_class;
	  v->next = (void *) ((char *) v + item_size);
	}
      ((tlgc_vacant *) ((char *) v - item_size))->next = NULL;
    }

  /* Allocate the new object and initialize the TLObject part.
     Initialization is performed here instead of in `-init' to ensure
     consistency of the allocation data structures, in case an object's
     init does not invoke super's implementation.  */
  c = slot->first_vacant;
  v = c->vacant_object;
  c->vacant_object = v->next;
  c->num++;
  if (!c->vacant_object)
    {
#if DEBUG_GC > 0
      if (c->num != c->cap)
	abort ();
#endif
#if DEBUG_GC > 4
      if (debug_gc > 4)
	printf ("%x: no more vacancies in chunk (slot=%d)\n", c, slot_index);
#endif
      slot->first_vacant = c->next_vacant;
      if (slot->first_vacant)
	slot->first_vacant->previous_vacant = &slot->first_vacant;
      c->previous_vacant = NULL;
      c->next_vacant = NULL;
    }

  bzero ((char *) v + sizeof (v->o.isa), item_size - sizeof (v->o.isa));
  v->o.isa = class;

  /* Put this gray object on the gray list.  */
  gray_stack_add ((void *) v);

  tlgc_alloc_since_partial++;

  QUIT_ENABLE;

#if DEBUG_GC > 2
  if (((char *) v - (char *) &c->data) % item_size)
    abort ();
#endif

#if DEBUG_GC > 5
  if (debug_gc > 5)
    gc_check_chunks_integrity ();
#endif

  return ((id) v);
} /* tl_default_object_alloc */

static id
tl_default_garbage_protect (void)
{
  struct tlgcs *gc_stack;
  id *vars;
  int i;

  /* Walk the stack.  */
  for (gc_stack = _tlgc_stack; gc_stack; gc_stack = gc_stack->next)
    for (vars = gc_stack->v, i = 0; i < gc_stack->n; vars++, i++)
      MARK (*vars);

  MARK (_tlgc_locked);
  MARK (_tlgc_retained);

#if WANT_NIL_OBJECT && GNU_RUNTIME
  /* Mark the NIL object.  Since it must implement gcMark by ignoring it,
     some wizardy is necessary.  Or simply this extra method.  With the test
     for a non-TLObject NIL object.  */
  if (tll_nil_object)
    [tll_nil_object __protectAgainstGC];
#endif

  return (nil);
} /* tl_default_garbage_protect */

/* Pointer to the function to create a new instance of a class.  */
id (*tl_objc_object_alloc) (struct objc_class *) = tl_default_object_alloc;

/* Pointer to the function to do a garbage collection run.  */
id (*tl_garbage_collect) (unsigned long) = tl_incremental_gc;

/* Pointer to a function invoked to perform custom garbage protection.  */
id (*tl_garbage_protect) (void) = tl_default_garbage_protect;

@implementation TLObject

+evalWithArguments: (TLCons *) args
{
  [self error: "+evalWithArguments: not implemented (programmer error)"];
  return (nil);
} /* +evalWithArguments: */

+gc
{
  return (tl_garbage_collect (0));
} /* +gc */

+gc: (unsigned long) usec
{
  return (tl_garbage_collect (usec));
} /* +gc: */

+gcAlloc
{
  return (tl_objc_object_alloc (self));
} /* +gcAlloc */

+initialize
{
  if (!tl_nonobject_class)
    {
      tl_nonobject_class = (id) objc_get_class ("TLNonObject");

#if HAVE_GETPAGESIZE
      page_size = getpagesize ();
#else
      page_size = sysconf (_SC_PAGE_SIZE);
#endif
    }
  return (self);
} /* +initialize */

+init
{
  _tlgc_locked = [CO_TLVector vectorWithCapacity: 0];

  return (self);
} /* +init */

-(int) compare: o
{
  [self warning: "compare not properly implemented"];
  return (o != self);
} /* -compare: */

-(void) gcLock
{
#if PARANOID_GCLOCK
  if (TL_LOCKED_P (self))
    abort ();
  TL_LOCKED_SET (self);
#endif

  [_tlgc_locked addElement: self];

  /* If we're not already at least gray, we should be since garbage
     protection could have been performed already.  */
  [self gcMark];
} /* -gcLock */

-(void) gcMark
{
  tlgc_num_mark++;
  if (TL_COLOUR (self) == TLOC_WHITE)
    {
      TL_COLOUR_SET (self, TLOC_GRAY);
      gray_stack_add (self);
    }
#if DEBUG_GC > 10
  else if (debug_gc > 10 && TL_COLOUR (self) == TLOC_GRAY)
    {
      int j;

      for (j = 0; j < gray_num; j++)
	if (gray_objects[j] == self)
	  break;
      if (j == gray_num)
	abort ();
    }
#endif
} /* -gcMark */

-(void) gcReference
{
#if DEBUG_GC > 0
  if (TL_COLOUR (self) != TLOC_BLACK)
    abort ();
#endif
} /* -gcReference */

-(void) gcUnlock
{
#if PARANOID_GCLOCK
  if (!TL_LOCKED_P (self))
    abort ();
  TL_LOCKED_CLEAR (self);
#endif
  [_tlgc_locked removeElementIdenticalTo: self];
} /* -gcLock */

/* XXX There's this bug somewhere in the GNU CC or runtime, or whatever,
   that causes some methods not to show up in the method list of a class.
   Which is exactly what happens with [Object self].  */
-self
{
  return (self);
} /* -self */

/******************** overriding allocation routines ********************/

-(void) dealloc
{
  /* Subclasses do not need to invoke this (i.e. super's) method.  */
#if PARANOID_GCLOCK
  if (tlgc_pass != TLGC_SWEEP_WHITE)
    abort ();
#endif
} /* -dealloc */

-free
{
  return (nil);
} /* -free */

-(oneway void) release
{
  if (!TL_REFCOUNT (self))
    [self warning: "release called with zero refcount"];
  else if (TL_REFCOUNT (self) != TL_REFCOUNT_MAX)
    {
      TL_RELEASE (self);
      if (!TL_REFCOUNT (self))
	[_tlgc_retained removeElementIdenticalTo: self];
    }
} /* -release */

-retain
{
  if (!TL_REFCOUNT (self))
    [_tlgc_retained addElement: self];
  if (TL_REFCOUNT (self) < TL_REFCOUNT_MAX)
    TL_RETAIN (self);
  return (self);
} /* -retain */

-(unsigned int) retainCount
{
  return (TL_REFCOUNT (self) - 1);
} /* -retainCount */

@end

/* Implementation of the TLNonObject class.  */
@implementation TLNonObject

-(retval_t) forward: (SEL) sel : (FORWARD_ARGUMENTS) arg_frame;
{
  formac (V_stderr_, @"Message `%s' sent to unallocated instance %lx\n",
	  sel_get_name (sel), (long) self);
  abort ();
} /* -forward:: */

+(retval_t) forward: (SEL) sel : (FORWARD_ARGUMENTS) arg_frame;
{
  formac (V_stderr_, @"Message `%s' sent to TLNonObject class\n",
	  sel_get_name (sel));
  abort ();
} /* +forward:: */

@end
