/* Implementation of TLStringStream class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLStringStream.m,v 1.1 1998/01/08 16:12:00 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLStringStream.h"
#import "tl/TLString.h"

@implementation TLConstantStringStream

-close
{
  next = len;
  return (self);
} /* -close */

-(int) fileDescriptor
{
  return (-1);
} /* -fileDescriptor */

-flushInput
{
  return (self);
} /* -flushInput */

-(void) gcReference
{
  MARK (string);
} /* -gcReference */

-initWithString: (id <TLString>) s
{
  ASGN_IVAR (string, s);
  len = [s length];
  c_string = (void *) [s cString];
  return (self);
} /* -initWithString: */

-(int) readByte
{
  return (next == len ? TL_EOF : ((unsigned char *) c_string)[next++]);
} /* -readByte */

-(int) readBytes: (int) n intoBuffer: (char *) b
{
  if (next + n > len)
    n = len - next;
  memcpy (b, c_string + next, n);
  return (n);
} /* -readBytes:intoBuffer: */

-(int) _writeRange: (id <TLRange>) r toStream: (id <TLOutputStream>) s
{
  int n, start = [r _start], length = [r length];

  next = start;
  if (start > len)
    return 0;

  if (start + length > len)
    length = len - start;
  n = [s writeBytes: length fromBuffer: c_string + start];
  if (n > 0)
    next += n;

  return n;
} /* _writeRange:toStream: */

-(long) _seek: (long) new_offset from: (int) pos
{
  switch (pos)
    {
    case TLSEEK_ABSOLUTE:
      break;
    case TLSEEK_FROM_END:
      new_offset = len - new_offset;
      break;
    case TLSEEK_RELATIVE:
      new_offset = next + new_offset;
      if (new_offset >= 0)
	break;
      /* Fall through.  */
    default:
      [self error: "bad TLSEEK %d (to %d)", pos, new_offset];
      return -1;
    }
  next = new_offset;
  return next;
} /* _seek:from: */

-streamp
{
  return (self);
} /* -streamp */

-(long) _tell
{
  return next;
} /* -_tell */

@end

@implementation TLMutableStringStream

-initWithString: (TLMutableString *) s
{
  ASGN_IVAR (string, s);
  return (self);
} /* -initWithString: */

/******************** TLStream ********************/

-close
{
  id s = string;
  string = nil;
  return (s ? self : nil);
} /* -close */

-(int) fileDescriptor
{
  return (-1);
} /* -fileDescriptor */

-streamp
{
  return (self);
} /* -streamp */

/******************** TLOutputStream ********************/

-flushOutput
{
  return (string && [string flushOutput] ? self : nil);
} /* -flushOutput */

-(int) writeByte: (char) c
{
  if (!string)
    return (TL_EOF);
  return ([string appendCChar: c]);
} /* -writeByte */

-(int) writeBytes: (int) n fromBuffer: (const char *) b
{
  if (!string)
    return (-1);
  [string appendCString: b length: n];
  return (n);
} /* -writeBytes:fromBuffer: */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (string);
} /* -gcReference */

@end

@implementation TLStringStream

+(id <TLMutableStream>) mutableStreamWithString: (TLMutableString *) s
{
  return ([(TLMutableStringStream *) [TLMutableStringStream gcAlloc]
	   initWithString: s]);
} /* -streamWithString: */

+(id <TLSeekableStream, TLInputStream>) streamWithString: (id <TLString>) s
{
  return ([(TLConstantStringStream *) [TLConstantStringStream gcAlloc]
	   initWithString: s]);
} /* -streamWithString: */

@end
