/* Implementation of TLL subroutines.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: subr.m,v 1.7 1998/05/12 20:03:34 tiggr Exp $  */

#import "tl/tl.h"
#import <time.h>
#import <string.h>
#import <sys/param.h>
#import <sys/resource.h>
#import <sys/wait.h>
#if HAVE_STRINGS_H
#import <strings.h>
#endif

/* Symbols of subroutines whose name is ever referenced within the Library
   in Objective-C code.  */
TLSymbol *Qquote;

/* Define the subroutines in this file.  */
static struct tl_subdef subdefs[] =
{
  DEFSUB (nil, @"*",			Fmul,		      0, ARGS_NUM_ANY),
  DEFSUB (nil, @"+",			Fadd,		      0, ARGS_NUM_ANY),
  DEFSUB (nil, @"++",			Fplusplus,	      1, ARGS_UNEVAL),
  DEFSUB (nil, @"-",			Fsub,		      0, ARGS_NUM_ANY),
  DEFSUB (nil, @"--",			Fminusminus,	      1, ARGS_UNEVAL),
  DEFSUB (nil, @"/",			Fdiv,		      2, ARGS_NUM_ANY),
  DEFSUB (nil, @"/=",			Fne,		      2, 2),
  DEFSUB (nil, @"1+",			Foneplus,	      1, 1),
  DEFSUB (nil, @"1-",			Foneminus,	      1, 1),
  DEFSUB (nil, @"<",			Flt,		      2, 2),
  DEFSUB (nil, @"<=",			Fle,		      2, 2),
  DEFSUB (nil, @"=",			Fnumeric_eq,	      2, 2),
  DEFSUB (nil, @">",			Fgt,		      2, 2),
  DEFSUB (nil, @">=",			Fge,		      2, 2),
  DEFSUB (nil, @"and",			Fand,		      0, ARGS_UNEVAL),
  DEFSUB (nil, @"car",			Fcar,		      1, 1),
  DEFSUB (nil, @"cdr",			Fcdr,		      1, 1),
  DEFSUB (nil, @"cond",			Fcond,		      0, ARGS_UNEVAL),
  DEFSUB (nil, @"cons",			Fcons,		      2, 2),
  DEFSUB (nil, @"defun",		Fdefun,		      2, ARGS_UNEVAL),
  DEFSUB (nil, @"eq",			Feq,		      2, 2),
  DEFSUB (nil, @"eval-string",		Feval_string,	      1, 1),
  DEFSUB (nil, @"exit",			Fexit,		      0, 1),
  DEFSUB (nil, @"fd-stream",		Ffd_stream,	      2, 3),
  DEFSUB (nil, @"formac",		Fformac,	      2, ARGS_NUM_ANY),
  DEFSUB (nil, @"garbage-collect",	Fgarbage_collect,     0, 1),
  DEFSUB (nil, @"getenv",		Fgetenv,	      1, 1),
  DEFSUB (nil, @"if",			Fif,		      2, ARGS_UNEVAL),
  DEFSUB (nil, @"intern",		Fintern,	      1, 1),
  DEFSUB (nil, @"lambda",		Flambda,	      2, ARGS_UNEVAL),
  DEFSUB (nil, @"let",			Flet,		      0, ARGS_UNEVAL),
  DEFSUB (nil, @"let*",			Flet_star,	      0, ARGS_UNEVAL),
  DEFSUB (nil, @"mapcar",		Fmapcar,	      2, 2),
  DEFSUB (nil, @"nconc",		Fnconc,		      0, ARGS_NUM_ANY),
  DEFSUB (nil, @"not",			Fnot,		      1, 1),
  DEFSUB (nil, @"or",			For,		      0, ARGS_UNEVAL),
  DEFSUB (nil, @"perform",		Fperform,	      2, ARGS_NUM_ANY),
  DEFSUB (nil, @"progn",		Fprogn,		      0, ARGS_UNEVAL),
  DEFSUB (&Qquote, @"quote",		Fquote,		      1, ARGS_UNEVAL),
  DEFSUB (nil, @"random",		Frandom,	      0, 0),
  DEFSUB (nil, @"range",		Frange,		      0, 3),
  DEFSUB (nil, @"setq",			Fsetq,		      0, ARGS_UNEVAL),
  DEFSUB (nil, @"sleep",		Fsleep,		      1, 1),
  DEFSUB (nil, @"time",			Ftime,		      1, ARGS_UNEVAL),
  DEFSUB (nil, @"vector",		Fvector,	      0, ARGS_NUM_ANY),
  DEFSUB (nil, @"while",		Fwhile,		      1, ARGS_UNEVAL),
  DEFSUB (0, 0, 0, 0, 0)
};

void
tll_subr_init (void)
{
  [TLLSubroutine registerSubroutines: subdefs];
} /* tll_subr_init */

/******************** general ********************/

id
Fcar (id cons)
  /* (car list) */
{
  return ([cons car]);
} /* Fcar */

id
Fcdr (id cons)
  /* (cdr list) */
{
  return ([cons cdr]);
} /* Fcdr */

id
Fcond (id in_args)
  /* (cond CLAUSE ...) */
{
  id retval = nil, args = in_args, clause, body;
  GCDECL1;

  GCPRO1 (in_args);

  while (args)
    {
      DECONS (args, clause, args);
      DECONS (clause, clause, body);
      retval = EVAL (clause);
      if (retval)
	{
	  if (body)
	    retval = Fprogn (body);
	  break;
	}
    }

  GCUNPRO;
  return (retval);
} /* Fcond */

id
Fcons (id car, id cdr)
  /* (cons car-elt cdr-elt) */
{
  return (CONS (car, cdr));
} /* Fcons */

id
Fdefun (id args)
  /* (defun sym args [body]) */
{
  id sym, arglist;

  DECONS (args, sym, args);
  DECONS (args, arglist, args);
  [sym setFunValue: [TLLLambda lambdaWithArguments: arglist body: args]];
  return (sym);
} /* Fdefun */

id
Feq (id arg1, id arg2)
{
  return (arg1 == arg2 ? Qt : nil);
} /* Feq */

id
Feval_string (id string)
{
  id r, s, lexer = [TLLLex lexerWithString: string];
  GCDECL1;

  /* The lexer will protect the string.  */
  GCPRO1 (lexer);

  for (r = nil, s = [lexer read]; s != Qlex_eof; s = [lexer read])
    r = EVAL (s);

  GCUNPRO;
  return (r);
} /* Feval_string */

id
Fexit (id <TLNumber> value)
{
  exit (value ? [value integerPIntValue] : 0);
} /* Fexit */

struct open_symbol
{
  TLSymbol **sym;
  const char *name;
};

static TLSymbol *Qo_rdonly, *Qo_wronly, *Qo_rdwr, *Qo_nonblock, *Qo_append;
static TLSymbol *Qo_creat, *Qo_trunc, *Qo_excl, *Qo_noctty, *Qo_sync;

static const struct open_symbol open_symbols[] =
{
  {&Qo_rdonly, "o-rdonly"},
  {&Qo_wronly, "o-wronly"},
  {&Qo_rdwr, "o-rdwr"},
  {&Qo_nonblock, "o-nonblock"},
  {&Qo_append, "o-append"},
  {&Qo_creat, "o-creat"},
  {&Qo_trunc, "o-trunc"},
  {&Qo_excl, "o-excl"},
  {&Qo_noctty, "o-noctty"},
  {&Qo_sync, "o-sync"},
  {0, 0}
};

TLFDStream *
Ffd_stream (id <TLString> name, TLCons *mode, id <TLNumber> protection)
{
  TLSymbol *s;
  int m, p;

  if (!Qo_rdonly)
    {
      int i;

      for (i = 0; open_symbols[i].name; i++)
	*open_symbols[i].sym = [CO_TLSymbol symbolWithName:
				[CO_TLString stringWithCString:
				 open_symbols[i].name]];
    }

  p = protection ? [protection integerPIntValue] : 0666;

  for (m = 0; mode;)
    {
      DECONS (mode, s, mode);

      if (s == Qo_rdonly) m |= O_RDONLY;
      else if (s == Qo_wronly) m |= O_WRONLY;
      else if (s == Qo_rdwr) m |= O_RDWR;
      else if (s == Qo_append) m |= O_APPEND;
      else if (s == Qo_creat) m |= O_CREAT;
      else if (s == Qo_trunc) m |= O_TRUNC;
      else if (s == Qo_excl) m |= O_EXCL;
#ifdef O_SYNC
      else if (s == Qo_sync) m |= O_SYNC;
#else
#if 0
      /* On Rhapsody we want something like this.  */
      else if (s == Qo_async) m |= O_ASYNC;
      else if (s == Qo_fsync) m |= O_FSYNC;
#endif
#endif
      else
	[TLLSubroutine error: "%#: bad fd-stream flag", s];
    }

  return ([TLFDStream mutableStreamWithFileNamed: name mode: m protection: p]);
} /* Ffd_stream */

id
Fformac (id args)
  /* (formac stream format ...)
     Like format, but with C escaping.  */
{
  id <TLOutputStream> stream = [args _elementAtIndex: 0];
  id <TLString> format = [args _elementAtIndex: 1];
  va_list ap;

  memset (&ap, 0, sizeof (ap));
  [args removeElementsFromIndex: 0 range: 2];

  return (llvformac (stream, format, ap, args));
}

id
Fgarbage_collect (id <TLNumber> usec)
  /* (garbage-collect &optional usec) */
{
  return ([TLObject gc: (usec ? [usec unsignedLongValue] : 0)]);
} /* Fgarbage_collect */

id
Fgetenv (id <TLString> n)
{
  char *v = getenv ([n cString]);
  return (v ? [CO_TLString stringWithCString: v] : nil);
} /* Fgetenv */

TLSymbol *
Fintern (id <TLString> name)
{
  return ([CO_TLSymbol symbolWithName: name]);
} /* Fintern */

id
Flambda (id args)
  /* (lambda (arg1 ...) form ...) */
{
  id arglist;

  DECONS (args, arglist, args);
  return ([TLLLambda lambdaWithArguments: arglist body: args]);
} /* Flambda */

id
Flet (id in_args)
  /* (let (binding ...) form ...)
     With binding == `var' or `(var value)'.  Return value of last form.  */
{
  id retval, body = in_args, b, next_b, *bind_values;
  int i, num_bindings, entry_binding_stack;
  TLCons *bindings;
  GCDECL2;

  if (!in_args || ({DECONS (in_args, bindings, body); !bindings;}))
    [TLLSubroutine error: "bad #arguments to let"];

  num_bindings = [bindings length];
  bind_values = alloca (num_bindings * sizeof (*bind_values));

  GCPRO2 (in_args, in_args);
  _gcpro2.n = 0;
  _gcpro2.v = bind_values;

  for (b = bindings; b; _gcpro2.n++, b = next_b)
    {
      DECONS (b, b, next_b);
      if (SYMBOLP (b))
	bind_values[_gcpro2.n] = nil;
      else
	{
	  id sym, val;
	  DECONS (b, sym, val);
	  if ([val cdr])
	    [TLLSubroutine error: "multiform let binding #%d", _gcpro2.n];
	  bind_values[_gcpro2.n] = EVAL ([val car]);
	  ASGN_SPROT (bind_values[_gcpro2.n]);
	}
    }

  for (b = bindings, entry_binding_stack = -1, i = 0; b; i++, b = next_b)
    {
      DECONS (b, b, next_b);
      if (!SYMBOLP (b))
	b = [b car];
      if (entry_binding_stack == -1)
	entry_binding_stack = [b pushVarValue: bind_values[i]];
      else
	[b pushVarValue: bind_values[i]];
    }

  retval = Fprogn (body);

  [CO_TLSymbol popVarValues: entry_binding_stack];
  GCUNPRO;
  return (retval);
} /* Flet */

id
Flet_star (id in_args)
  /* (let* (binding ...) form ...)
     With binding == `var' or `(var value)'.  Return value of last form.  */
{
  id retval, body = in_args, bindings, b, next_b;
  int i, entry_binding_stack = -1;
  GCDECL1;

  if (!in_args || ({DECONS (in_args, bindings, body); !bindings;}))
    [TLLSubroutine error: "bad #arguments to let*"];

  GCPRO1 (in_args);

  for (b = bindings; b; b = next_b)
    {
      DECONS (b, b, next_b);
      if (SYMBOLP (b))
	i = [b pushVarValue: nil];
      else
	{
	  id sym, val;
	  DECONS (b, sym, val);
	  if ([val cdr])
	    [TLLSubroutine error: "multiform let* binding"];
	  i = [sym pushVarValue: EVAL ([val car])];
	}
      if (entry_binding_stack == -1)
	entry_binding_stack = i;
    }

  retval = Fprogn (body);

  [CO_TLSymbol popVarValues: entry_binding_stack];
  GCUNPRO;
  return (retval);
} /* Flet_star */

id
Fmapcar (id sym, id seq)
  /* XXX Implemented in lisp.tl.  */
  /* (mapcar function sequence) */
{
  return ([seq mapcar: sym]);
} /* Fmapcar */

id
Fnconc (id argvec, int num_args)
{
  id first, last;
  int i;

  for (i = 0, first = last = nil; i < num_args; i++)
    {
      id o = [argvec _elementAtIndex: i];
      if (o)
	{
	  if (last)
	    [last nconc: o];
	  else
	    first = o;
	  last = o;
	}
    }
  return (first);
} /* Fnconc */

id
Fprogn (id in_args)
/* (progn form ...)
   Return the value of the last form executed.  */
{
  id form, args = in_args, retval = nil;
  GCDECL1;

  GCPRO1 (in_args);

  while (args)
    {
      DECONS (args, form, args);
      retval = EVAL (form);
    }

  GCUNPRO;
  return (retval);
} /* Fprogn */

id
Fquote (id args)
/* (quote value) */
{
  return ([args car]);
} /* Fquote */

id
Frandom (void)
{
  static int initialized = 0;

  if (!initialized)
    {
      initialized = 1;
      srandom (time (0));
    }
  return ([CO_TLNumber numberWithInt: random () + 231 * random ()]);
} /* Frandom */

id
Frange (id start, id length, id end)
{
  return (end ? [TLRange rangeWithStart: [start integerPIntValue]
		 end: [end integerPIntValue]]
	  : (length ? [TLRange rangeWithStart: [start integerPIntValue]
		       length: [length integerPIntValue]]
	     : (start ? [TLRange rangeWithStart: [start integerPIntValue]
			 length: -1] : tll_full_range)));
} /* Frange */

id
Fsetq (id in_args)
/* (setq var val var val ...) */
{
  id retval = nil, args = in_args;
  GCDECL1;

  GCPRO1 (in_args);

  while (args)
    {
      id sym, val;

      DECONS (args, sym, args);
      DECONS (args, val, args);
      retval = EVAL (val);
      [sym setVarValue: retval];
    }

  GCUNPRO;
  return (retval);
} /* Fsetq */

id
Fsleep (id <TLNumber> seconds)
{
  sleep ([seconds intValue]);
  return (nil);
} /* Fsleep */

id
Ftime (id in_args)
  /* (time form ...)
     Evaluate all forms and return the time needed for the execution.  */
{
  double start = get_run_time ();
  Fprogn (in_args);
  return ([CO_TLNumber numberWithDouble: get_run_time () - start]);
} /* Ftime */

id
Fvector (id args, int num_args)
  /* (vector ...) */
{
  return (args);
} /* Fvector */

id
Fwhile (id args)
/* (while condition body)
   Return the value of `(progn BODY)'.  (Since upon return, the CONDITION is
   always NIL and not very interesting to return.)  */
{
  id condition, body, retval;
  GCDECL1;

  GCPRO1 (args);

  DECONS (args, condition, body);
  for (retval = nil; EVAL (condition); retval = Fprogn (body));

  GCUNPRO;
  return (retval);
} /* Fwhile */

/******************** arithmetic ********************/

enum op_kind
{
  OP_ADD,
  OP_SUB,
  OP_MUL,
  OP_DIV,
  OP_REM,
  OP_AND,
  OP_OR,
  OP_XOR,
};

static id
compute (id args, int num_args, enum op_kind op)
{
  id val;
  int i;

  switch (op)
    {
    case OP_MUL:
      val = tll_small_int[1];
      break;
    case OP_AND:
      val = tll_small_int[-1];
      break;
    default:
      val = tll_small_int[0];
      break;
    }

  for (i = 0; i < num_args; i++)
    {
      id arg = [args _elementAtIndex: i];

      switch (op)
	{
	case OP_ADD:
	  val = [val numberByAdd: arg];
	  break;
	case OP_SUB:
	  if (i || num_args == 1)
	    val = [val numberBySubtract: arg];
	  else
	    val = arg;
	  break;
	case OP_MUL:
	  val = [val numberByMultiply: arg];
	  break;
	case OP_DIV:
	  val = i ? [val numberByDivide: arg] : arg;
	  break;
	case OP_REM:
	  val = i ? [val numberByModulo: arg] : arg;
	  break;
	case OP_AND:
	  val = [val numberByAnd: arg];
	  break;
	case OP_OR:
	  val = [val numberByOr: arg];
	  break;
	case OP_XOR:
	  val = [val numberByXor: arg];
	  break;
	}
    }

  return (val);
} /* compute */

id
Fadd (id args, int num_args)
  /* (+ a ...) */
{
  return (compute (args, num_args, OP_ADD));
} /* Fadd */

id
Fdiv (id args, int num_args)
  /* (/ a b ...) */
{
  return (compute (args, num_args, OP_DIV));
} /* Fdiv */

id
Fmul (id args, int num_args)
  /* (* a ...) */
{
  return (compute (args, num_args, OP_MUL));
} /* Fmul */

id
Fsub (id args, int num_args)
  /* (- a ...)
     One arg: -a.  Multiple args: a - arg1 - ... - argn.  */
{
  return (compute (args, num_args, OP_SUB));
} /* Fsub */

id
Flt (id a1, id a2)
  /* (< a b) */
{
  return ([a1 lessThan: a2]);
} /* Flt */

id
Fle (id a1, id a2)
  /* (<= a b) */
{
  return ([a1 lessThanOrEqual: a2]);
} /* Fle */

id
Fnumeric_eq (id a1, id a2)
  /* (= a b) */
{
  return ([a1 equal: a2]);
} /* Fnumeric_eq */

id
Fne (id a1, id a2)
  /* (/= a b) */
{
  return ([a1 notEqual: a2]);
} /* Fne */

id
Fge (id a1, id a2)
  /* (>= a b) */
{
  return ([a1 greaterThanOrEqual: a2]);
} /* Fge */

id
Fgt (id a1, id a2)
  /* (> a b) */
{
  return ([a1 greaterThan: a2]);
} /* Fgt */

id
Foneplus (id arg)
  /* (1+ num) */
{
  return ([arg numberByAddingInt: 1]);
} /* Foneplus */

id
Foneminus (id arg)
  /* (1- num) */
{
  return ([arg numberByAddingInt: -1]);
} /* Foneminus */

id
modify_by_add (int value, id in_args)
{
  id retval = nil, arg, next_arg;
  GCDECL1;

  GCPRO1 (in_args);

  for (arg = in_args; arg; arg = next_arg)
    {
      DECONS (arg, arg, next_arg);
      retval = [EVAL (arg) numberByAddingInt: value];
      [arg setVarValue: retval];
    }

  GCUNPRO;
  return (retval);
} /* modify_by_add */

id
Fplusplus (id in_args)
  /* (++ ...) */
{
  return (modify_by_add (1, in_args));
} /* Fplusplus */

id
Fminusminus (id in_args)
  /* (-- ...) */
{
  return (modify_by_add (-1, in_args));
} /* Fminusminus */

/* Logic.  */

id
Fand (id args)
  /* (and ...) */
{
  id retval = Qt;
  id arg, next_arg;

  if (args)
    {
      GCDECL2;

      GCPRO2 (args, retval);

      for (arg = args; retval && arg; arg = next_arg)
	{
	  DECONS (arg, arg, next_arg);
	  retval = EVAL (arg);
	}

      GCUNPRO;
    }

  return (retval);
} /* Fand */

id
Fif (id args)
  /* (id condition then &rest else) */
{
  id cond, then_part, else_part, retval;
  GCDECL1;

  GCPRO1 (args);

  DECONS (args, cond, then_part);
  DECONS (then_part, then_part, else_part);

  retval = EVAL (cond) ? EVAL (then_part) : Fprogn (else_part);

  GCUNPRO;
  return (retval);
} /* Fif */

id
Fnot (id arg)
{
  return (arg ? nil : Qt);
} /* Fnot */

id
For (id args)
  /* (or ...) */
{
  id retval = nil;
  id arg, next_arg;

  if (args)
    {
      GCDECL2;

      GCPRO2 (args, retval);

      for (arg = args; !retval && arg; arg = next_arg)
	{
	  DECONS (arg, arg, next_arg);
	  retval = EVAL (arg);
	}

      GCUNPRO;
    }

  return (retval);
} /* For */

/******************** Objective-C ********************/

id
Fperform (id in_args, int num_args)
  /* (perform receiver selector arg ...) */
{
  id receiver, *argv = alloca ((num_args - 2) * sizeof (*argv));
  TLSymbol *selector;
  int i;

  receiver = [in_args _elementAtIndex: 0];
  selector = [in_args _elementAtIndex: 1];
  for (i = 2; i < num_args; i++)
    argv[i - 2] = [in_args _elementAtIndex: i];

  return (tll_invoke_method (receiver, selector, argv, num_args - 2, 0));
} /* Fperform */
