<copyright> BucketDictElement class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: BucketDictElement.t,v 1.13 1999/09/08 19:36:12 tiggr Exp $</id>
    </copyright>

implementation class
BucketDictElement: BucketElement

end;

implementation instance
BucketDictElement
{
  <doc> This bucket element's key.  </doc>
  All key;

  <doc> The value in this bucket element.  </doc>
  public Any value;
}

<doc> Designated initializer.  </doc>
id
  initWith (All, All) (k, v)
{
  (key, value) = (k, Any (v));
  = self;
}

<doc> Apple the {block} to {value} and pass to {next}.  </doc>
void
  do Block block
{
  [block eval value];
  if (next != nil)
    [next do block];
}

<doc> Apple the {block} to {key} and pass to {next}.  </doc>
void
  doKeys Block block
{
  [block eval key];
  if (next != nil)
    [next do block];
}

<doc> Return the {key}, with a suitable type.  </doc>
Any
  key
{
  = Any (key);
}

<doc> Return the value associated with the key {k}, asking the {next}
    element if this element does not match.

    The implementation by {BucketDictElement}, considers its {key} and
    returns its {value}.  </doc>
Any
  member All k
{
  = key == k || [key equal k] ? value : !next ? nil : [next member k];
}

<doc> Like member, but using the selector {cmp} to have the objects
    compare themselves.  </doc>
Any
  member All k
   equal selector cmp
{
  = (key == k || [key perform cmp with k] ? value
     : !next ? nil : [next member k equal cmp]);
}

Any
  memq All k
{
  = key == k ? value : !next ? nil : [next memq k];
}

<doc> Add the {(k, v)} pair to this bucket, if the key is not already
    present.  Return the number by which this bucket's length has
    increased.  </doc>
int
  add (All, All) (k, v)
{
  if (k == key || [k equal key])
    value = Any (v);
  else if (!next)
    {
      next = [[isa alloc] initWith (k, v)];
      = 1;
    }
  else
    = [next add (k, v)];
}

<doc> Add the {(k, v)} pair to this bucket, if the key is not already
    present.  Return the number by which this bucket's length has
    increased.  </doc>
int
  addq (All, All) (k, v)
{
  if (k == key)
    value = Any (v);
  else if (!next)
    {
      next = [[isa alloc] initWith (k, v)];
      = 1;
    }
  else
    = [next addq (k, v)];
}

<doc> Remove the object with the key equal to {k}.  Return the replacement
    for this element, and the number of bucket elements that were removed
    from this bucket list (max 1).  </doc>
(id, int)
  remove All k
{
  int n;

  if (key == k || [key equal k])
    return (next, 1);

  if (next != nil)
    (next, n) = [next remove k];

  return (self, n);
}

<doc> Remove the object with the identical key {k}.  Return the
    replacement for this element, and the number of bucket elements that
    were removed from this bucket list (max 1).  </doc>
(id, int)
  removeq All k
{
  int n;

  if (key == k)
    return (next, 1);

  if (next != nil)
    (next, n) = [next removeq k];

  return (self, n);
}

void
  encodeUsingCoder Encoder coder
{
  if (![coder hasBeenCodedFor [BucketDictElement self]])
    {
      [super encodeUsingCoder coder];

      // Encoding should discern between classes and instance, not us.
      // Tue Jul  1 12:05:04 1997, tiggr@natlab.research.philips.com
      [coder encode State (key)];
      [coder encode value];
    }
}

void
  initWithCoder Decoder coder
{
  if (![coder hasBeenCodedFor [BucketDictElement self]])
    {
      [super initWithCoder coder];

      key = [coder decode];
      value = [coder decode];
    }
}

<doc> Starting with this bucket element, remove the objects of which the
    {value} is {gc_dead}.  Return the replacement for this element, and
    the number of bucket elements that were removed from this bucket list.
    </doc>
(id, int)
  gc_mark_values
{
  int n;

  if (next != nil)
    (next, n) = [next gc_mark_values];

  if ([value gc_dead_p])
    return (next, n + 1);

  return (self, n);
}

<doc> Similar to {gc_mark_values}, but consider the liveness of the {key}
    instead of the {value}.  </doc>
(id, int)
  gc_mark_keys
{
  int n;

  if (next != nil)
    (next, n) = [next gc_mark_keys];

  if ([key gc_dead_p])
    return (next, n + 1);

  return (self, n);
}

<doc> Rehash the key of the receiving element.  </doc>
int
  rehash
{
  = [key hash];
}

<doc> Rehashq the key of the receiving element.  </doc>
int
  rehashq
{
  = [key hashq];
}

end;
