<copyright> BucketElement class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: BucketElement.t,v 1.12 1999/09/08 19:36:12 tiggr Exp $</id>
    </copyright>

implementation class
BucketElement: State

end;

implementation instance
BucketElement
{
  <doc> The next element in this bucket.  </doc>
  public id next;
}

<doc> Apple the {block} to {self} and pass to {next}.  </doc>
void
  do Block block
{
  [block eval self];
  if (next != nil)
    [next do block];
}

<doc> Return the value associated with the {key}, asking the {next}
    element if this element does not match.

    The implementation by {BucketElement} considers itself to be both the
    key and the value.  </doc>
Any
  member All key
pre
  key != nil
{
  = (key == self || [key equal self] ? Any (self)
     : !next ? nil : [next member key]);
}

<doc> Like member, but using the selector {cmp} to have the objects
    compare themselves.  </doc>
Any
  member All key
   equal selector cmp
pre
  key != nil
{
  = (key == self || [key perform cmp with self] ? Any (self)
     : !next ? nil : [next member key equal cmp]);
}

<doc> Like {member}, but use reference equality instead of the {equal}
    method.  </doc>
Any
  memq All key
pre
  key != nil
{
  = self == key ? Any (self) : !next ? nil : [next memq key];
}

<doc> Add the {elt} to this bucket, if it is not already present.  Return
    the number by which this bucket's length has increased.  </doc>
int
  addElement id elt
{
  if (elt != self && ![elt equal self])
    if (!next)
      {
	next = elt;
	= 1;
      }
    else
      = [next addElement elt];
}

<doc> Add the {elt} to this bucket, if it is not already present.  Return
    the number by which this bucket's length has increased.  </doc>
int
  addqElement id elt
{
  if (elt != self)
    if (!next)
      {
        next = elt;
	= 1;
      }
    else
      = [next addqElement elt];
}

<doc> Remove the {elt} from this bucket, if present.  Return the number by
    which the length of this bucket has decreased, and the replacement
    remainder of the bucket list.  </doc>
(id, int) (replacement, decrease)
  remove id elt
post
  !decrease -> replacement == self
{
  if (elt == self || [elt equal self])
    = (next, 1);
  else
    {
      if (!!next)
	{
	  /* What is faster?  A check and branch (frequently taken) or a
	     (frequent) overwrite with the same value, dirtying a cache
	     line?  */
	  id new_next;
	  (new_next, decrease) = [next remove elt];
	  if (next != new_next)
	    next = new_next;
	}
      replacement = self;
    }
}

<doc> Remove the {elt} from this bucket, if present.  Return the number by
    which the length of this bucket has decreased, and the replacement
    remainder of the bucket list.  </doc>
(id, int) (replacement, decrease)
  removeq id elt
post
  !decrease -> replacement == self
{
  if (elt == self)
    = (next, 1);
  else
    {
      if (!!next)
	{
	  /* What is faster?  A check and branch (frequently taken) or a
	     (frequent) overwrite with the same value, dirtying a cache
	     line?  */
	  id new_next;
	  (new_next, decrease) = [next removeq elt];
	  if (next != new_next)
	    next = new_next;
	}
      replacement = self;
    }
}

<doc> For resizing a hashtable, feed this element and those following
    elements, to the hashtable using the hashtable's {resizing_add}.
    </doc>
void
  resizing_feed HashTable ht
{
  id n = next;

  [ht resizing_add self];

  if (n != nil)
    [n resizing_feed ht];
}

<doc> While resizing a hashtable, accept a new {next}.  </doc>
void
  resizing_add id n
{
  next = n;
}

<doc> Rehash the key of the receiving element.  </doc>
int
  rehash
{
  = [self hash];
}

<doc> Rehashq the key of the receiving element.  </doc>
int
  rehashq
{
  = [self hashq];
}

void
  encodeUsingCoder Encoder coder
{
  if (![coder hasBeenCodedFor [BucketElement self]])
    {
      [super encodeUsingCoder coder];

      [coder encode next];
    }
}

void
  initWithCoder Decoder coder
{
  if (![coder hasBeenCodedFor [BucketElement self]])
    {
      [super initWithCoder coder];

      next = [coder decode];
    }
}

<doc> Starting with this bucket element, remove those bucket elements of
    which the objects are {gc_dead}.  Return the replacement for this
    element, and the number of bucket elements that were removed from this
    bucket list.  </doc>
(id, int)
  gc_mark_values
{
  int n;

  if (next != nil)
    (next, n) = [next gc_mark_values];

  if ([self gc_dead_p])
    return (next, n + 1);

  = (self, n);
}

end;
